<?php

namespace App\Exports;

use App\Helpers\Helper;
use App\Models\Base\CfdiDownload;
use App\Models\Catalogs\Deduction;
use App\Models\Catalogs\OtherPaymentType;
use App\Models\Catalogs\Perception;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithColumnFormatting;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStrictNullComparison;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Events\AfterSheet;
use Maatwebsite\Excel\Events\BeforeExport;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;

class CfdiDownloadsEmployeePayrollExport implements FromCollection, WithMapping, WithHeadings, WithColumnFormatting, WithTitle, WithStrictNullComparison, WithEvents
{
    protected $request;

    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        if (empty($this->request->filter_date_from)) {
            $this->request->request->add([
                'filter_date_from' => Helper::date(\Date::parse('first day of this month'))
            ]);
        }
        if (empty($this->request->filter_date_to)) {
            $this->request->request->add([
                'filter_date_to' => Helper::date(\Date::parse('last day of this month'))
            ]);
        }
        return CfdiDownload::filter($this->request->all())->sortable(['date' => 'desc'])
            ->with('cfdiDownloadPayments')
            ->get();
    }

    /**
     * @return string
     */
    public function title(): string
    {
        return __('base/cfdi_download.document_title');
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        $tmp = [
            __('base/cfdi_download.column_type'),
            __('base/cfdi_download.column_cfdi_type'),
            __('base/cfdi_download.column_serie'),
            __('base/cfdi_download.column_folio'),
            __('base/cfdi_download.column_date'),
            __('base/cfdi_download.column_time'),
            __('base/cfdi_download.column_uuid'),
            __('base/cfdi_download.column_rfc_emisor'),
            __('base/cfdi_download.column_emisor'),
            __('base/cfdi_download.column_employer_register'),
            __('base/cfdi_download.column_rfc_receptor'),
            __('base/cfdi_download.column_employee_code'),
            __('base/cfdi_download.column_employee'),
            __('base/cfdi_download.column_curp'),
            __('base/cfdi_download.column_nss'),
            __('base/cfdi_download.column_recruitment_regime'),
            __('base/cfdi_download.column_department'),
            __('base/cfdi_download.column_job'),
            __('base/cfdi_download.column_job_risk_classe'),
            __('base/cfdi_download.column_date_start_work'),
            __('base/cfdi_download.column_antiquity'),
            __('base/cfdi_download.column_contract_type'),
            __('base/cfdi_download.column_workday_type'),
            __('base/cfdi_download.column_frequency_payment'),
            __('base/cfdi_download.column_bank'),
            __('base/cfdi_download.column_bank_account'),
            __('base/cfdi_download.column_sdi'),
            __('base/cfdi_download.column_base_salary'),
            __('base/cfdi_download.column_date_start_payment'),
            __('base/cfdi_download.column_date_end_payment'),
            __('base/cfdi_download.column_date_payment'),
            __('base/cfdi_download.column_payment_days'),
            __('base/cfdi_download.column_amount_total_perceptions'),
            __('base/cfdi_download.column_amount_total_deductions'),
            __('base/cfdi_download.column_amount_total_other_payment_types'),
            __('base/cfdi_download.column_amount_total'),
            __('base/cfdi_download.column_amount_isr'),
            __('base/cfdi_download.column_rfc_pac'),
            __('base/cfdi_download.column_status'),
            __('base/cfdi_download.column_is_cancelable'),
            __('base/cfdi_download.column_expense_type'),
            __('base/cfdi_download.column_operational_type'),
            __('base/cfdi_download.column_comment_2'),
        ];
        $perceptions = Perception::active()->orderBy('code')->orderBy('name')->get();
        foreach($perceptions as $perception){
            array_push($tmp, '['.$perception->code.'] ' . str_limit($perception->name,40) . '('.__('base/cfdi_download.entry_amount_taxed').')');
            array_push($tmp, '['.$perception->code.'] ' . str_limit($perception->name,40) . '('.__('base/cfdi_download.entry_line_amount_exempt').')');
        }
        $deductions = Deduction::active()->orderBy('code')->orderBy('name')->get();
        foreach($deductions as $deduction){
            array_push($tmp, '['.$deduction->code.'] ' . str_limit($deduction->name,40));
        }
        $other_payment_types = OtherPaymentType::active()->orderBy('code')->orderBy('name')->get();
        foreach($other_payment_types as $other_payment_type){
            array_push($tmp, '['.$other_payment_type->code.'] ' . str_limit($other_payment_type->name,40));
        }
        return $tmp;
    }

    /**
     * @return array
     */
    public function map($row): array
    {
        //Leer XML para obtener los conceptos
        $path_files = Helper::setDirectory(CfdiDownload::PATH_FILES, $row->company_id) . '/';
        $file_xml = $path_files . $row->file_xml;

        //Obtener datos de la nomina
        $data_employee_payroll = [];
        $text_file_not_found = '';
        if (!empty($file_xml)) {
            if (\Storage::exists($file_xml) && Helper::validateXmlToArrayCfdi33($file_xml)) {
                $data_employee_payroll = Helper::parseXmlToArrayCfdi33EmployeePayroll($file_xml);
            }
            if (!\Storage::exists($file_xml)) {
                $text_file_not_found = '***' . __('general.text_file_xml_not_found') . '***';
            }
        }

        //sumatoria de ISR
        $amount_isr = 0;
        if(!empty($data_employee_payroll['deductions'])){
            foreach($data_employee_payroll['deductions'] as $key => $deduction){
                if(in_array($key,['002','101'])){
                    $amount_isr += $deduction;
                }
            }
        }

        $tmp = [
            $row->type_name,
            $row->cfdiType->name ?? '',
            $row->serie,
            $row->folio,
            \App\Helpers\Helper::date(\Date::parse($row->date)),
            \Date::parse($row->date)->format('H:i:s'),
            $row->uuid,
            $row->rfc_emisor,
            $row->emisor,
            $data_employee_payroll['employer_register'] ?? $text_file_not_found,
            $row->rfc_receptor,
            $data_employee_payroll['employee_code'] ?? '',
            $row->receptor,
            $data_employee_payroll['curp'] ?? '',
            $data_employee_payroll['nss'] ?? '',
            $data_employee_payroll['recruitment_regime'] ?? '',
            $data_employee_payroll['department'] ?? '',
            $data_employee_payroll['job'] ?? '',
            $data_employee_payroll['job_risk_classe'] ?? '',
            $data_employee_payroll['date_start_work'] ?? '',
            $data_employee_payroll['antiquity'] ?? '',
            $data_employee_payroll['contract_type'] ?? '',
            $data_employee_payroll['workday_type'] ?? '',
            $data_employee_payroll['frequency_payment'] ?? '',
            $data_employee_payroll['bank'] ?? '',
            $data_employee_payroll['bank_account'] ?? '',
            $data_employee_payroll['sdi'] ?? '',
            $data_employee_payroll['base_salary'] ?? '',
            $data_employee_payroll['date_start_payment'] ?? '',
            $data_employee_payroll['date_end_payment'] ?? '',
            $data_employee_payroll['date_payment'] ?? '',
            $data_employee_payroll['payment_days'] ?? '',
            $data_employee_payroll['amount_total_perceptions'] ?? '',
            $data_employee_payroll['amount_total_deductions'] ?? '',
            $data_employee_payroll['amount_total_other_payment_types'] ?? '',
            $row->amount_total,
            $amount_isr,
            $row->rfc_pac,
            $row->status,
            $row->is_cancelable,
            $row->expenseType->name ?? '',
            $row->operationalType->name ?? '',
            $row->comment_2,
        ];

        $perceptions = Perception::active()->orderBy('code')->orderBy('name')->get();
        foreach($perceptions as $perception){
            if(!empty($data_employee_payroll['perceptions'][$perception->code .'_1'])){
                array_push($tmp, $data_employee_payroll['perceptions'][$perception->code .'_1']);
            }else{
                array_push($tmp, '');
            }
            if(!empty($data_employee_payroll['perceptions'][$perception->code .'_2'])){
                array_push($tmp, $data_employee_payroll['perceptions'][$perception->code .'_2']);
            }else{
                array_push($tmp, '');
            }
        }
        $deductions = Deduction::active()->orderBy('code')->orderBy('name')->get();
        foreach($deductions as $deduction){
            if(!empty($data_employee_payroll['deductions'][$deduction->code])){
                array_push($tmp, $data_employee_payroll['deductions'][$deduction->code]);
            }else{
                array_push($tmp, '');
            }
        }
        $other_payment_types = OtherPaymentType::active()->orderBy('code')->orderBy('name')->get();
        foreach($other_payment_types as $other_payment_type){
            if(!empty($data_employee_payroll['other_payment_types'][$other_payment_type->code])){
                array_push($tmp, $data_employee_payroll['other_payment_types'][$other_payment_type->code]);
            }else{
                array_push($tmp, '');
            }
        }

        return $tmp;
    }

    /**
     * @return array
     */
    public function columnFormats(): array
    {
        return [
            'AA' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'AB' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'AG' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'AH' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'AI' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'AJ' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'AK' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
        ];
    }

    /**
     * @return array
     */
    public function registerEvents(): array
    {
        return [
            BeforeExport::class => function (BeforeExport $event) {
                $event->getWriter()->getDelegate()
                    ->getProperties()
                    ->setCreator(config('app.name'))
                    ->setLastModifiedBy(config('app.name'))
                    ->setTitle(__('base/cfdi_download.document_title_payroll_type') . '-' . config('app.name'))
                    ->setDescription(
                        __('base/cfdi_download.document_title_payroll_type') . '-' . config('app.name')
                    );
            },
            // Array callable, refering to a static method.
            AfterSheet::class => [self::class, 'afterSheet'],

        ];
    }

    public static function AfterSheet(AfterSheet $event)
    {

        //Default

        //Titulo
        $title_cell_range = 'A1:AQ1';
        $title_cell_range_left = 'AR1:IN1';
        $title_style = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
                'vertical' => \PhpOffice\PhpSpreadsheet\Style\Alignment::VERTICAL_CENTER,
            ],
            'font' => [
                'name' => 'Arial',
                'size' => 10,
                'bold' => true,
                'color' => ['argb' => '0A7864'],
            ],
            'borders' => [
                'bottom' => [
                    'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_MEDIUM,
                    'color' => ['argb' => 'C5DAD6'],
                ],
            ]
        ];
        $title_style_left = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_LEFT,
                'vertical' => \PhpOffice\PhpSpreadsheet\Style\Alignment::VERTICAL_CENTER,
            ],
            'font' => [
                'name' => 'Arial',
                'size' => 10,
                'bold' => true,
                'color' => ['argb' => '0A7864'],
            ],
            'borders' => [
                'bottom' => [
                    'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_MEDIUM,
                    'color' => ['argb' => 'C5DAD6'],
                ],
            ]
        ];
        $event->getDelegate()->getStyle($title_cell_range)->applyFromArray($title_style);
        $event->getDelegate()->getStyle($title_cell_range)->getFill()
            ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
            ->getStartColor()->setARGB('EEEEEE');
        $event->getDelegate()->getStyle($title_cell_range_left)->applyFromArray($title_style_left);
        $event->getDelegate()->getStyle($title_cell_range_left)->getFill()
            ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
            ->getStartColor()->setARGB('EEEEEE');
        $event->getDelegate()->getRowDimension('1')->setRowHeight(22);
        //$event->getDelegate()->getColumnDimension('B')->setWidth(50);

        //autosize
        foreach(range('A','Z') as $columnID) {
            $event->getDelegate()->getColumnDimension($columnID)->setAutoSize(true);
        }
        foreach(range('A','A') as $columnID) {
            foreach(range('A','Q') as $columnID2) {
                $event->getDelegate()->getColumnDimension($columnID . $columnID2)->setAutoSize(true);
            }
        }
        foreach(range('A','A') as $columnID) {
            foreach(range('R','Z') as $columnID2) {
                $event->getDelegate()->getColumnDimension($columnID . $columnID2)->setWidth(25);
            }
        }
        foreach(range('B','H') as $columnID) {
            foreach(range('A','Z') as $columnID2) {
                $event->getDelegate()->getColumnDimension($columnID . $columnID2)->setWidth(25);
            }
        }
        foreach(range('I','I') as $columnID) {
            foreach(range('A','M') as $columnID2) {
                $event->getDelegate()->getColumnDimension($columnID . $columnID2)->setWidth(25);
            }
        }

        //Ajuste otras columnas
        $style_center = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
            ],
        ];
        $style_right = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_RIGHT,
            ],
        ];
        $event->getDelegate()->getStyle('A2:A1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('B2:B1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('C2:C1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('D2:D1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('E2:E1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('F2:F1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('G2:G1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('H2:H1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('J2:J1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('K2:K1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('L2:L1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('P2:P1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('S2:S1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('T2:T1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('U2:U1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('V2:V1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('W2:W1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('X2:X1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('Y2:Y1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('Z2:Z1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('AC2:AC1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('AD2:AD1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('AE2:AE1000')->applyFromArray($style_center);

    }
}
