<?php

namespace App\Exports;

use App\Helpers\Helper;
use App\Models\Base\CfdiDownload;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithColumnFormatting;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStrictNullComparison;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Events\AfterSheet;
use Maatwebsite\Excel\Events\BeforeExport;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;

class CfdiDownloadsExport implements FromCollection, WithMapping, WithHeadings, WithColumnFormatting, ShouldAutoSize, WithTitle, WithStrictNullComparison, WithEvents
{
    protected $request;

    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        if (empty($this->request->filter_date_from)) {
            $this->request->request->add([
                'filter_date_from' => Helper::date(\Date::parse('first day of this month'))
            ]);
        }
        if (empty($this->request->filter_date_to)) {
            $this->request->request->add([
                'filter_date_to' => Helper::date(\Date::parse('last day of this month'))
            ]);
        }
        return CfdiDownload::filter($this->request->all())->sortable(['date' => 'desc'])
            ->with('cfdiDownloadPayments')
            ->with('cfdiType')
            ->with('paymentWay')
            ->with('paymentMethod')
            ->with('cfdiUse')
            ->with('currency')
            ->get();
    }

    /**
     * @return string
     */
    public function title(): string
    {
        return __('base/cfdi_download.document_title');
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        return [
            __('base/cfdi_download.column_type'),
            __('base/cfdi_download.column_cfdi_type'),
            __('base/cfdi_download.column_serie'),
            __('base/cfdi_download.column_folio'),
            __('base/cfdi_download.column_date'),
            __('base/cfdi_download.column_time'),
            __('base/cfdi_download.column_uuid'),
            __('base/cfdi_download.column_rfc_emisor'),
            __('base/cfdi_download.column_emisor'),
            __('base/cfdi_download.column_rfc_receptor'),
            __('base/cfdi_download.column_receptor'),
            __('base/cfdi_download.column_payment_way'),
            __('base/cfdi_download.column_payment_method'),
            __('base/cfdi_download.column_cfdi_use'),
            __('base/cfdi_download.column_currency'),
            __('base/cfdi_download.column_currency_value'),
            __('base/cfdi_download.column_amount_untaxed'),
            __('base/cfdi_download.column_amount_discount'),
            #__('base/cfdi_download.column_amount_iva'),
            __('base/cfdi_download.column_amount_iva_16'),
            __('base/cfdi_download.column_amount_iva_15'),
            __('base/cfdi_download.column_amount_iva_11'),
            __('base/cfdi_download.column_amount_iva_10'),
            __('base/cfdi_download.column_amount_iva_8'),
            __('base/cfdi_download.column_amount_ieps'),
            __('base/cfdi_download.column_amount_tax'),
            __('base/cfdi_download.column_amount_ret_iva'),
            __('base/cfdi_download.column_amount_ret_isr'),
            __('base/cfdi_download.column_amount_tax_ret'),
            __('base/cfdi_download.column_amount_total'),
            __('base/cfdi_download.column_balance'),
            __('base/cfdi_download.column_rfc_pac'),
            __('base/cfdi_download.column_status'),
            __('base/cfdi_download.column_is_cancelable'),
            __('base/cfdi_download.column_expense_type'),
            __('base/cfdi_download.column_operational_type'),
            __('base/cfdi_download.column_comment_0'),
            __('base/cfdi_download.column_comment_2'),
            __('base/cfdi_download.column_cfdi_download_lines'),
            __('base/cfdi_download.column_payments'),
        ];
    }

    /**
     * @return array
     */
    public function map($row): array
    {
        //Leer XML para obtener los conceptos
        $path_files = Helper::setDirectory(CfdiDownload::PATH_FILES,$row->company_id) . '/';
        $file_xml = $path_files . $row->file_xml;
        $cfdi_download_lines = '';
        $file_valid = false;
        if (!empty($row->file_xml) && \Storage::exists($file_xml) && Helper::validateXmlToArrayCfdi33($file_xml)) {
                $file_valid = true;
        }

        if ($file_valid) {
            $cfdi_download_lines = Helper::parseXmlToArrayCfdi33Conceptos($file_xml);
        }

        //Obtiene los pagos
        $payments = '';
        if ($row->cfdiDownloadPayments->isNotEmpty()) {
            foreach ($row->cfdiDownloadPayments as $payment) {
                $payments .= (!empty($payments) ? ' * ' : '') . Helper::convertSqlToDate($payment->date_payment) . '/' . ($payment->paymentWay->name ?? '') . '/' . money($payment->amount_payment,$row->currency->code ?? 'MXN',true);
            }
        }

        //Obtener datos de impuestos
        #$iva = 0;
        $iva_16 = 0;
        $iva_15 = 0;
        $iva_11 = 0;
        $iva_10 = 0;
        $iva_8 = 0;
        $ieps = 0;
        $ret_iva = 0;
        $ret_isr = 0;
        if ($file_valid) {
            $tmp = Helper::parseXmlToArrayCfdi33Impuestos($file_xml);
            #$iva = $tmp['iva'];
            $iva_16 = $tmp['iva_16'];
            $iva_15 = $tmp['iva_15'];
            $iva_11 = $tmp['iva_11'];
            $iva_10 = $tmp['iva_10'];
            $iva_8 = $tmp['iva_8'];
            $ieps = $tmp['ieps'];
            $ret_iva = $tmp['ret_iva'];
            $ret_isr = $tmp['ret_isr'];

            if($row->cfdiType->name ?? '' == 'P'){
                $tmp2 = [];
                if($tmp['version'] == '4.0'){
                    $tmp2 = Helper::parseXmlToArrayCfdi40PagosImpuestos($file_xml);
                }elseif($tmp['version'] == '3.3'){
                    $tmp2 = Helper::parseXmlToArrayCfdi33PagosImpuestos($file_xml, $path_files);
                }

                #$iva = $tmp['iva'];
                $iva_16 += $tmp2['iva_16'] ?? 0;
                $iva_15 += $tmp2['iva_15'] ?? 0;
                $iva_11 += $tmp2['iva_11'] ?? 0;
                $iva_10 += $tmp2['iva_10'] ?? 0;
                $iva_8 += $tmp2['iva_8'] ?? 0;
                $ieps += $tmp2['ieps'] ?? 0;
                $ret_iva += $tmp2['ret_iva'] ?? 0;
                $ret_isr += $tmp2['ret_isr'] ?? 0;

                $row->amount_tax = $iva_16 + $iva_15 + $iva_11 + $iva_10 + $iva_8 + $ieps;
                $row->amount_tax_ret = $ret_iva + $ret_isr;
            }
        }

        return [
            $row->type_name,
            $row->cfdiType->name ?? '',
            $row->serie,
            $row->folio,
            \App\Helpers\Helper::date(\Date::parse($row->date)),
            \Date::parse($row->date)->format('H:i:s'),
            $row->uuid,
            $row->rfc_emisor,
            $row->emisor,
            $row->rfc_receptor,
            $row->receptor,
            $row->paymentWay->name ?? '',
            $row->paymentMethod->code ?? '',
            $row->cfdiUse->name ?? '',
            $row->currency->code ?? '',
            $row->currency_value ?? 1,
            $row->amount_untaxed + $row->amount_discount,
            $row->amount_discount,
            #$iva,
            $iva_16,
            $iva_15,
            $iva_11,
            $iva_10,
            $iva_8,
            $ieps,
            $row->amount_tax,
            $ret_iva,
            $ret_isr,
            $row->amount_tax_ret,
            $row->amount_total,
            $row->balance,
            $row->rfc_pac,
            $row->status,
            $row->is_cancelable,
            $row->expenseType->name ?? '',
            $row->operationalType->name ?? '',
            $row->comment,
            $row->comment_2,
            $cfdi_download_lines,
            $payments
        ];
    }

    /**
     * @return array
     */
    public function columnFormats(): array
    {
        return [
            'P' => '"$"#,##0.0000_-',
            'Q' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'R' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'S' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'T' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'U' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'V' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'W' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'X' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'Y' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'Z' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'AA' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'AB' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'AC' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'AD' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
        ];
    }

    /**
     * @return array
     */
    public function registerEvents(): array
    {
        return [
            BeforeExport::class => function (BeforeExport $event) {
                $event->getWriter()->getDelegate()
                    ->getProperties()
                    ->setCreator(config('app.name'))
                    ->setLastModifiedBy(config('app.name'))
                    ->setTitle(__('base/cfdi_download.document_title') . '-' . config('app.name'))
                    ->setDescription(
                        __('base/cfdi_download.document_title') . '-' . config('app.name')
                    );
            },
            // Array callable, refering to a static method.
            AfterSheet::class => [self::class, 'afterSheet'],

        ];
    }

    public static function AfterSheet(AfterSheet $event)
    {

        //Default

        //Titulo
        $title_cell_range = 'A1:AM1';
        $title_style = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
                'vertical' => \PhpOffice\PhpSpreadsheet\Style\Alignment::VERTICAL_CENTER,
            ],
            'font' => [
                'name' => 'Arial',
                'size' => 10,
                'bold' => true,
                'color' => ['argb' => '0A7864'],
            ],
            'borders' => [
                'bottom' => [
                    'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_MEDIUM,
                    'color' => ['argb' => 'C5DAD6'],
                ],
            ]
        ];
        $event->getDelegate()->getStyle($title_cell_range)->applyFromArray($title_style);
        $event->getDelegate()->getStyle($title_cell_range)->getFill()
            ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
            ->getStartColor()->setARGB('EEEEEE');
        $event->getDelegate()->getRowDimension('1')->setRowHeight(22);
        //$event->getDelegate()->getColumnDimension('B')->setWidth(50);

        //autosize
        foreach(range('A','Z') as $columnID) {
            $event->getDelegate()->getColumnDimension($columnID)->setAutoSize(true);
        }
        foreach(range('A','A') as $columnID) {
            foreach(range('A','K') as $columnID2) {
                $event->getDelegate()->getColumnDimension($columnID . $columnID2)->setAutoSize(true);
            }
        }
        $event->getDelegate()->getColumnDimension('AL')->setAutoSize(false)->setWidth(50);
        $event->getDelegate()->getColumnDimension('AM')->setAutoSize(false)->setWidth(50);

        //Ajuste otras columnas
        $style_center = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
            ],
        ];
        $style_right = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_RIGHT,
            ],
        ];
        $event->getDelegate()->getStyle('A2:A1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('B2:B1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('C2:C1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('D2:D1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('E2:E1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('F2:F1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('G2:G1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('H2:H1000')->applyFromArray($style_center);
        //I
        $event->getDelegate()->getStyle('J2:J1000')->applyFromArray($style_center);
        //K
        //L
        //M
        //N
        $event->getDelegate()->getStyle('O2:O1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('Z2:Z1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('AA2:AA1000')->applyFromArray($style_center);

    }
}
