<?php

namespace App\Exports;

use App\Helpers\Helper;
use App\Models\Sales\CustomerInvoice;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithColumnFormatting;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStrictNullComparison;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Events\AfterSheet;
use Maatwebsite\Excel\Events\BeforeExport;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;

class CustomerInvoicesExport implements FromCollection, WithMapping, WithHeadings, WithColumnFormatting, ShouldAutoSize, WithTitle, WithStrictNullComparison, WithEvents
{
    protected $request;
    private $list_status = [];
    private $document_type_code = 'customer.invoice';

    public function __construct(Request $request)
    {
        $this->request = $request;
        $this->list_status = [
            CustomerInvoice::DRAFT => __('sales/customer_invoice.text_status_draft'),
            CustomerInvoice::OPEN => __('sales/customer_invoice.text_status_open'),
            CustomerInvoice::TO_PAY => __('sales/customer_invoice.text_status_to_pay'),
            CustomerInvoice::PAID => __('sales/customer_invoice.text_status_paid'),
            CustomerInvoice::CANCEL => __('sales/customer_invoice.text_status_cancel'),
            CustomerInvoice::CANCEL_PER_AUTHORIZED => __('sales/customer_invoice.text_status_cancel_per_authorized'),
        ];
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        if (empty($this->request->filter_date_from)) {
            $this->request->request->add([
                'filter_date_from' => Helper::date(\Date::parse('first day of this month'))
            ]);
        }
        if (empty($this->request->filter_date_to)) {
            $this->request->request->add([
                'filter_date_to' => Helper::date(\Date::parse('last day of this month'))
            ]);
        }
        $this->request->request->add(['filter_document_type_code' => $this->document_type_code]); //Filtra tipo de documento
        return CustomerInvoice::filter($this->request->all())->sortable(['date' => 'desc'])->get();
    }

    /**
     * @return string
     */
    public function title(): string
    {
        return __('sales/customer_invoice.document_title');
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        return [
            __('sales/customer_invoice.column_serie'),
            __('sales/customer_invoice.column_folio'),
            __('sales/customer_invoice.column_date'),
            __('sales/customer_invoice.column_time'),
            __('sales/customer_invoice.column_uuid'),
            __('sales/customer.column_taxid'),
            __('sales/customer_invoice.column_customer'),
            __('sales/customer_invoice.column_salesperson'),
            __('sales/customer_invoice.column_branch_office'),
            __('sales/customer_invoice.column_payment_term'),
            __('sales/customer_invoice.column_payment_way'),
            __('sales/customer_invoice.column_payment_method'),
            __('sales/customer_invoice.column_cfdi_use'),
            __('sales/customer_invoice.column_reference'),
            __('sales/customer_invoice.column_date_due'),
            __('sales/customer_invoice.column_currency'),
            __('sales/customer_invoice.column_amount_total'),
            __('sales/customer_invoice.column_balance'),
            __('sales/customer_invoice.column_status'),
        ];
    }

    /**
     * @return array
     */
    public function map($row): array
    {
        return [
            $row->serie,
            $row->folio,
            \App\Helpers\Helper::date(\Date::parse($row->date)),
            \Date::parse($row->date)->format('H:i:s'),
            $row->customerInvoiceCfdi->uuid ?? '',
            $row->customer->taxid,
            $row->customer->name,
            $row->salesperson->name ?? '',
            $row->branchOffice->name,
            $row->paymentTerm->name,
            $row->paymentWay->name,
            $row->paymentMethod->code,
            $row->cfdiUse->name,
            $row->reference,
            \App\Helpers\Helper::convertSqlToDate($row->date_due),
            $row->currency->code,
            $row->amount_total,
            $row->balance,
            $this->list_status[$row->status],
        ];
    }

    /**
     * @return array
     */
    public function columnFormats(): array
    {
        return [
            'Q' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'R' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
        ];
    }

    /**
     * @return array
     */
    public function registerEvents(): array
    {
        return [
            BeforeExport::class => function (BeforeExport $event) {
                $event->getWriter()->getDelegate()
                    ->getProperties()
                    ->setCreator(config('app.name'))
                    ->setLastModifiedBy(config('app.name'))
                    ->setTitle(__('sales/customer_invoice.document_title') . '-' . config('app.name'))
                    ->setDescription(
                        __('sales/customer_invoice.document_title') . '-' . config('app.name')
                    );
            },
            // Array callable, refering to a static method.
            AfterSheet::class => [self::class, 'afterSheet'],

        ];
    }

    public static function AfterSheet(AfterSheet $event)
    {

        //Default

        //Titulo
        $title_cell_range = 'A1:S1';
        $title_style = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
                'vertical' => \PhpOffice\PhpSpreadsheet\Style\Alignment::VERTICAL_CENTER,
            ],
            'font' => [
                'name' => 'Arial',
                'size' => 10,
                'bold' => true,
                'color' => ['argb' => '0A7864'],
            ],
            'borders' => [
                'bottom' => [
                    'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_MEDIUM,
                    'color' => ['argb' => 'C5DAD6'],
                ],
            ]
        ];
        $event->getDelegate()->getStyle($title_cell_range)->applyFromArray($title_style);
        $event->getDelegate()->getStyle($title_cell_range)->getFill()
            ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
            ->getStartColor()->setARGB('EEEEEE');
        $event->getDelegate()->getRowDimension('1')->setRowHeight(22);
        //$event->getDelegate()->getColumnDimension('B')->setWidth(50);

        //Ajuste otras columnas
        $style_center = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
            ],
        ];
        $style_right = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_RIGHT,
            ],
        ];
        $event->getDelegate()->getStyle('A2:A1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('B2:B1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('C2:C1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('D2:D1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('E2:E1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('F2:F1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('L2:L1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('O2:O1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('P2:P1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('Q2:Q1000')->applyFromArray($style_right);
        $event->getDelegate()->getStyle('R2:R1000')->applyFromArray($style_right);
        $event->getDelegate()->getStyle('S2:S1000')->applyFromArray($style_center);

    }
}
