<?php

namespace App\Exports;

use App\Helpers\Helper;
use App\Models\Sales\CustomerRemission;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithColumnFormatting;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStrictNullComparison;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Events\AfterSheet;
use Maatwebsite\Excel\Events\BeforeExport;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;

class CustomerRemissionsExport implements FromCollection, WithMapping, WithHeadings, WithColumnFormatting, ShouldAutoSize, WithTitle, WithStrictNullComparison, WithEvents
{
    protected $request;
    private $list_status = [];
    private $document_type_code = 'customer.remission';

    public function __construct(Request $request)
    {
        $this->request = $request;
        $this->list_status = [
            CustomerRemission::OPEN => __('sales/customer_remission.text_status_open'),
            CustomerRemission::BILLED => __('sales/customer_remission.text_status_billed'),
            CustomerRemission::CANCEL => __('sales/customer_remission.text_status_cancel'),
        ];
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        if (empty($this->request->filter_date_from)) {
            $this->request->request->add([
                'filter_date_from' => Helper::date(\Date::parse('first day of this month'))
            ]);
        }
        if (empty($this->request->filter_date_to)) {
            $this->request->request->add([
                'filter_date_to' => Helper::date(\Date::parse('last day of this month'))
            ]);
        }
        $this->request->request->add(['filter_document_type_code' => $this->document_type_code]); //Filtra tipo de documento
        return CustomerRemission::filter($this->request->all())->sortable(['date' => 'desc'])->get();
    }

    /**
     * @return string
     */
    public function title(): string
    {
        return __('sales/customer_remission.document_title');
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        return [
            __('sales/customer_remission.column_name'),
            /*__('sales/customer_remission.column_folio'),*/
            __('sales/customer_remission.column_date'),
            __('sales/customer_remission.column_time'),
            /*__('sales/customer_remission.column_uuid'),*/
            __('sales/customer.column_taxid'),
            __('sales/customer_remission.column_customer'),
            __('sales/customer_remission.column_salesperson'),
            __('sales/customer_remission.column_branch_office'),
            __('sales/customer_remission.column_payment_term'),
            __('sales/customer_remission.column_payment_way'),
            __('sales/customer_remission.column_payment_method'),
            __('sales/customer_remission.column_cfdi_use'),
            __('sales/customer_remission.column_expiration_date'),
            __('sales/customer_remission.column_currency'),
            __('sales/customer_remission.column_amount_total'),
            __('sales/customer_remission.column_customer_invoice'),
            __('sales/customer_remission.column_status'),
        ];
    }

    /**
     * @return array
     */
    public function map($row): array
    {
        return [
            $row->name,
            \App\Helpers\Helper::date(\Date::parse($row->date)),
            \Date::parse($row->date)->format('H:i:s'),
            $row->customer->taxid,
            $row->customer->name,
            $row->salesperson->name ?? '',
            $row->branchOffice->name,
            $row->paymentTerm->name ?? '',
            $row->paymentWay->name ?? '',
            $row->paymentMethod->code ?? '',
            $row->cfdiUse->name ?? '',
            $row->expiration_date ? \App\Helpers\Helper::convertSqlToDate($row->expiration_date) : '',
            $row->currency->code,
            $row->amount_total,
            $row->customerInvoice->name ?? '',
            $this->list_status[$row->status],
        ];
    }

    /**
     * @return array
     */
    public function columnFormats(): array
    {
        return [
            'N' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'O' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
        ];
    }

    /**
     * @return array
     */
    public function registerEvents(): array
    {
        return [
            BeforeExport::class => function (BeforeExport $event) {
                $event->getWriter()->getDelegate()
                    ->getProperties()
                    ->setCreator(config('app.name'))
                    ->setLastModifiedBy(config('app.name'))
                    ->setTitle(__('sales/customer_remission.document_title') . '-' . config('app.name'))
                    ->setDescription(
                        __('sales/customer_remission.document_title') . '-' . config('app.name')
                    );
            },
            // Array callable, refering to a static method.
            AfterSheet::class => [self::class, 'afterSheet'],

        ];
    }

    public static function AfterSheet(AfterSheet $event)
    {

        //Default

        //Titulo
        $title_cell_range = 'A1:P1';
        $title_style = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
                'vertical' => \PhpOffice\PhpSpreadsheet\Style\Alignment::VERTICAL_CENTER,
            ],
            'font' => [
                'name' => 'Arial',
                'size' => 10,
                'bold' => true,
                'color' => ['argb' => '0A7864'],
            ],
            'borders' => [
                'bottom' => [
                    'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_MEDIUM,
                    'color' => ['argb' => 'C5DAD6'],
                ],
            ]
        ];
        $event->getDelegate()->getStyle($title_cell_range)->applyFromArray($title_style);
        $event->getDelegate()->getStyle($title_cell_range)->getFill()
            ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
            ->getStartColor()->setARGB('EEEEEE');
        $event->getDelegate()->getRowDimension('1')->setRowHeight(22);
        //$event->getDelegate()->getColumnDimension('B')->setWidth(50);

        //Ajuste otras columnas
        $style_center = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
            ],
        ];
        $style_right = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_RIGHT,
            ],
        ];
        $event->getDelegate()->getStyle('A2:A1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('B2:B1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('C2:C1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('D2:D1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('J2:J1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('K2:K1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('L2:L1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('M2:M1000')->applyFromArray($style_right);
        $event->getDelegate()->getStyle('N2:N1000')->applyFromArray($style_right);
        $event->getDelegate()->getStyle('O2:O1000')->applyFromArray($style_center);

    }
}
