<?php

namespace App\Exports;

use App\Helpers\Helper;
use App\Models\Base\Employee;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithColumnFormatting;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStrictNullComparison;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Events\AfterSheet;
use Maatwebsite\Excel\Events\BeforeExport;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;

class EmployeesExport implements FromCollection, WithMapping, WithHeadings, WithColumnFormatting, ShouldAutoSize, WithTitle, WithStrictNullComparison, WithEvents
{
    protected $request;

    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        $results = Employee::filter($this->request->all());
        if(isset($this->request->filter_status) && $this->request->filter_status != 'all'){
            $results = $results->where('employees.status','=',(int)$this->request->filter_status);
        }
        $results = $results->with('frequencyPayment')
            ->with('recruitmentRegime')
            ->sortable('name')
            ->get();

        return $results;
    }

    /**
     * @return string
     */
    public function title(): string
    {
        return __('base/employee.document_title');
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        return [
            __('base/employee.column_code'),
            __('base/employee.column_firstname'),
            __('base/employee.column_lastname'),
            __('base/employee.column_lastname2'),
            __('base/employee.column_taxid'),
            __('base/employee.column_tax_regimen'),
            __('base/employee.column_curp'),
            __('base/employee.column_postcode'),
            __('base/employee.column_nss'),
            __('base/employee.column_email'),
            __('base/employee.column_recruitment_regime'),
            __('base/employee.column_date_start_work'),
            __('base/employee.column_unionized'),
            __('base/employee.column_workday_type'),
            __('base/employee.column_contract_type'),
            __('base/employee.column_employee_category'),
            __('base/employee.column_department'),
            __('base/employee.column_job'),
            __('base/employee.column_job_risk_classe'),
            __('base/employee.column_frequency_payment'),
            __('base/employee.column_bank'),
            __('base/employee.column_bank_account'),
            __('base/employee.column_base_salary'),
            __('base/employee.column_sdi'),
            __('base/employee.column_country'),
            __('base/employee.column_state'),
        ];
    }

    /**
     * @return array
     */
    public function map($row): array
    {
        return [
            $row->code,
            $row->firstname,
            $row->lastname,
            $row->lastname2,
            $row->taxid,
            $row->taxRegimen->name ?? null,
            $row->curp,
            $row->postcode,
            $row->nss,
            $row->email,
            $row->recruitmentRegime->name ?? null,
            $row->date_start_work ? Helper::convertSqlToDate($row->date_start_work) : null,
            $row->unionized ? __('general.text_yes') : __('general.text_no'),
            $row->workdayType->name ?? null,
            $row->contractType->name ?? null,
            $row->employeeCategory->name ?? null,
            $row->department,
            $row->job,
            $row->jobRiskClasse->name ?? '',
            $row->frequencyPayment->name ?? '',
            $row->bank->name ?? '',
            $row->bank_account,
            $row->base_salary,
            $row->sdi,
            $row->country->name ?? '',
            $row->state->name ?? '',
        ];
    }

    /**
     * @return array
     */
    /**
     * @return array
     */
    public function columnFormats(): array
    {
        return [
            'W' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
            'X' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
        ];
    }

    /**
     * @return array
     */
    public function registerEvents(): array
    {
        return [
            BeforeExport::class => function (BeforeExport $event) {
                $event->getWriter()->getDelegate()
                    ->getProperties()
                    ->setCreator(config('app.name'))
                    ->setLastModifiedBy(config('app.name'))
                    ->setTitle(__('base/employee.document_title') . '-' . config('app.name'))
                    ->setDescription(
                        __('base/employee.document_title') . '-' . config('app.name')
                    );
            },
            // Array callable, refering to a static method.
            AfterSheet::class => [self::class, 'afterSheet'],

        ];
    }

    public static function AfterSheet(AfterSheet $event)
    {

        //Default

        //Titulo
        $title_cell_range = 'A1:Z1';
        $title_style = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
                'vertical' => \PhpOffice\PhpSpreadsheet\Style\Alignment::VERTICAL_CENTER,
            ],
            'font' => [
                'name' => 'Arial',
                'size' => 10,
                'bold' => true,
                'color' => ['argb' => '0A7864'],
            ],
            'borders' => [
                'bottom' => [
                    'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_MEDIUM,
                    'color' => ['argb' => 'C5DAD6'],
                ],
            ]
        ];
        $event->getDelegate()->getStyle($title_cell_range)->applyFromArray($title_style);
        $event->getDelegate()->getStyle($title_cell_range)->getFill()
            ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
            ->getStartColor()->setARGB('EEEEEE');
        $event->getDelegate()->getRowDimension('1')->setRowHeight(22);
        //$event->getDelegate()->getColumnDimension('A')->setWidth(50);

        //Ajuste otras columnas
        $style_center = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
            ],
        ];
        $style_right = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_RIGHT,
            ],
        ];
        $event->getDelegate()->getStyle('A2:A1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('E2:E1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('G2:G1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('I2:I1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('L2:L1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('M2:M1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('N2:N1000')->applyFromArray($style_center);

    }

}
