<?php

namespace App\Exports;

use App\Models\Catalogs\Product;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithColumnFormatting;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStrictNullComparison;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Events\AfterSheet;
use Maatwebsite\Excel\Events\BeforeExport;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;

class ProductsExport implements FromCollection, WithMapping, WithHeadings, WithColumnFormatting, ShouldAutoSize, WithTitle, WithStrictNullComparison, WithEvents
{
    protected $request;

    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return Product::filter($this->request->all())->sortable('name')->get();
    }

    /**
     * @return string
     */
    public function title(): string
    {
        return __('catalogs/product.document_title');
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        return [
            __('catalogs/product.column_code'),
            __('catalogs/product.column_name'),
            __('catalogs/product.column_description'),
            __('catalogs/product.column_model'),
            __('catalogs/product.column_manufacturer'),
            __('catalogs/product.column_unit_measure'),
            __('catalogs/product.column_sat_product'),
            __('catalogs/product.column_price'),
            __('catalogs/product.entry_taxes'),
        ];
    }

    /**
     * @return array
     */
    public function map($row): array
    {
        $taxes = [];
        foreach($row->taxes as $tax)
            $taxes[] = $tax->name;

        return [
            $row->code,
            $row->name,
            $row->description,
            $row->model,
            $row->manufacturer,
            $row->unitMeasure->code . '-' . $row->unitMeasure->name,
            $row->satProduct->code,
            $row->price,
            implode(',',$taxes)
        ];
    }

    /**
     * @return array
     */
    public function columnFormats(): array
    {
        return [
            'H' => NumberFormat::FORMAT_CURRENCY_USD_SIMPLE,
        ];
    }

    /**
     * @return array
     */
    public function registerEvents(): array
    {
        return [
            BeforeExport::class => function (BeforeExport $event) {
                $event->getWriter()->getDelegate()
                    ->getProperties()
                    ->setCreator(config('app.name'))
                    ->setLastModifiedBy(config('app.name'))
                    ->setTitle(__('catalogs/product.document_title') . '-' . config('app.name'))
                    ->setDescription(
                        __('catalogs/product.document_title') . '-' . config('app.name')
                    );
            },
            // Array callable, refering to a static method.
            AfterSheet::class => [self::class, 'afterSheet'],

        ];
    }

    public static function AfterSheet(AfterSheet $event)
    {

        //Default

        //Titulo
        $title_cell_range = 'A1:I1';
        $title_style = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
                'vertical' => \PhpOffice\PhpSpreadsheet\Style\Alignment::VERTICAL_CENTER,
            ],
            'font' => [
                'name' => 'Arial',
                'size' => 10,
                'bold' => true,
                'color' => ['argb' => '0A7864'],
            ],
            'borders' => [
                'bottom' => [
                    'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_MEDIUM,
                    'color' => ['argb' => 'C5DAD6'],
                ],
            ]
        ];
        $event->getDelegate()->getStyle($title_cell_range)->applyFromArray($title_style);
        $event->getDelegate()->getStyle($title_cell_range)->getFill()
            ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
            ->getStartColor()->setARGB('EEEEEE');
        $event->getDelegate()->getRowDimension('1')->setRowHeight(22);
        //$event->getDelegate()->getColumnDimension('B')->setWidth(50);

        //Ajuste otras columnas
        $style_center = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
            ],
        ];
        $style_right = [
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_RIGHT,
            ],
        ];
        $event->getDelegate()->getStyle('A2:A1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('D2:D1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('E2:E1000')->applyFromArray($style_center);
        $event->getDelegate()->getStyle('H2:H1000')->applyFromArray($style_right);

    }
}
