<?php

namespace App\Helpers;

use SoapVar;
use SoapClient;
use SoapHeader;
use ZipArchive;
use App\Models\Base\Pac;
use Illuminate\Support\Str;
use App\Models\Base\Company;
use App\Models\Catalogs\Tax;
use App\Models\Catalogs\CfdiUse;
use App\Models\Catalogs\Project;
use NumberToWords\NumberToWords;
use App\Models\Base\BranchOffice;
use App\Models\Base\CfdiDownload;
use App\Models\Base\DocumentType;
use App\Models\Catalogs\CfdiType;
use App\Models\Catalogs\Currency;
use Chumper\Zipper\Facades\Zipper;
use App\Models\Catalogs\PaymentWay;
use App\Models\Catalogs\SatProduct;
use App\Models\Catalogs\TaxRegimen;
use App\Models\Catalogs\UnitMeasure;
use App\Models\Catalogs\CfdiRelation;
use App\Models\Sales\CustomerInvoice;
use App\Models\Sales\CustomerPayment;
use Illuminate\Support\Facades\Crypt;
use App\Models\Catalogs\PaymentMethod;
use Symfony\Component\Process\Process;
use App\Models\Sales\CustomerInvoiceCfdi;
use Symfony\Component\Process\Exception\ProcessFailedException;

class Helper
{

    public static function testEnvironment(){
        if(config('app.env') == 'local' || config('app.env') == 'development' || config('app.debug')){
            return true;
        }
        return false;
    }

    /**
     * Funcion para subir archivos solo imagenes
     *
     * @param $key
     * @param $path
     * @param string $disk
     * @return string
     * @throws \Exception
     */
    public static function uploadFileImage($key, $path, $company_id = '')
    {
        request()->file($key)->store(self::setDirectory($path, $company_id));

        return request()->file($key)->hashName();
    }

    /**
     * Funcion para subir archivos solo imagenes
     *
     * @param $key
     * @param $path
     * @param string $disk
     * @return string
     * @throws \Exception
     */
    public static function uploadFileImageUsers($key, $path)
    {
        request()->file($key)->store($path);

        return request()->file($key)->hashName();
    }

    /**
     * Retorna el direactorio de la empresa si no existe lo crea
     *
     * @param string $disk
     * @return string
     * @throws \Exception
     */
    public static function directoryCompany($company_id = ''){
        try {

            $company = self::defaultCompany($company_id);
            $company_path = 'default';
            if(!empty($company)){
                if(!empty($company->path_files)){
                    $company_path = $company->path_files;
                }else if(!empty($company->taxid)){
                    $company_path =  str_replace('&','0',$company->taxid);
                }
            }

            if(!\Storage::exists($company_path)){
                \Storage::makeDirectory($company_path, 0777, true); //creates directory
                @shell_exec('chmod -R 777 ' . \Storage::path($company_path));
            }

            return $company_path;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Crear directorio con el default de la empresa, si no existe lo crea
     *
     * @param string $disk
     * @param $path
     * @return string
     * @throws \Exception
     */
    public static function setDirectory($path,$company_id = ''){
        $company_path = self::directoryCompany($company_id);
        $path = $company_path . '/' . $path;
        if(!\Storage::exists($path)){
            \Storage::makeDirectory($path, 0777, true); //creates directory
            @shell_exec('chmod -R 777 ' . \Storage::path($path));
        }
        return $path;
    }

    /**
     * Logo de la empresa
     *
     * @param bool $default
     * @param string $company_id
     * @return string
     * @throws \Exception
     */
    public static function logoCompany($default = false,$company_id = ''){
        $company = self::defaultCompany($company_id);
        $path = public_path('img/logo/factura-16-logo-horizontal.png');
        if (!empty($company->image) && !$default) {
            if(\Storage::exists(self::setDirectory(Company::PATH_IMAGES,$company_id). '/' . $company->image)){
                $path = \Storage::path(self::setDirectory(Company::PATH_IMAGES,$company_id). '/' . $company->image);
            }
        }
        $image     = \Image::make($path)->encode()->getEncoded();
        $extension = \File::extension($path);
        return 'data:image/' . $extension . ';base64,' . base64_encode($image);
    }

    /**
     * Logo de la empresa
     *
     * @param bool $default
     * @param string $company_id
     * @return string
     * @throws \Exception
     */
    public static function logoCompanyDocs($company_id = ''){
        $company = self::defaultCompany($company_id);
        $path = public_path('img/logo/no-logo-horizontal.png');
        if (!empty($company->image)) {
            if(\Storage::exists(self::setDirectory(Company::PATH_IMAGES,$company_id). '/' . $company->image)){
                $path = \Storage::path(self::setDirectory(Company::PATH_IMAGES,$company_id). '/' . $company->image);
            }
        }
        $image     = \Image::make($path)->encode()->getEncoded();
        $extension = \File::extension($path);
        return 'data:image/' . $extension . ';base64,' . base64_encode($image);
    }

    /**
     * Logo de la empresa por sucursal
     *
     * @param bool $default
     * @param string $company_id
     * @return string
     * @throws \Exception
     */
    public static function logoCompanyDocsBranchOffices($company_id = '', $branch_office_id= ''){
        if(!empty($branch_office_id)){
            $branch_office = BranchOffice::findOrFail($branch_office_id);
            if(!empty($branch_office->logo_docs)){
                if(\Storage::exists(self::setDirectory(BranchOffice::PATH_LOGO_DOCS,$branch_office->company_id). '/' . $branch_office->logo_docs)){
                    $path = \Storage::path(self::setDirectory(BranchOffice::PATH_LOGO_DOCS,$branch_office->company_id). '/' . $branch_office->logo_docs);
                    $image     = \Image::make($path)->encode()->getEncoded();
                    $extension = \File::extension($path);
                    return 'data:image/' . $extension . ';base64,' . base64_encode($image);
                }
            }
        }
        return self::logoCompanyDocs($company_id);
    }

    /**
     * Logo de la empresa por proyecto
     *
     * @param bool $default
     * @param string $company_id
     * @return string
     * @throws \Exception
     */
    public static function logoCompanyDocsProjects($company_id = '', $project_id= ''){
        if(!empty($project_id)){
            $project = Project::findOrFail($project_id);
            if(!empty($project->logo_docs)){
                if(\Storage::exists(self::setDirectory(Project::PATH_LOGO_DOCS,$project->company_id). '/' . $project->logo_docs)){
                    $path = \Storage::path(self::setDirectory(Project::PATH_LOGO_DOCS,$project->company_id). '/' . $project->logo_docs);
                    $image     = \Image::make($path)->encode()->getEncoded();
                    $extension = \File::extension($path);
                    return 'data:image/' . $extension . ';base64,' . base64_encode($image);
                }
            }
        }
        return self::logoCompanyDocs($company_id);
    }

    /**
     * Estatus a cadena
     *
     * @param $status
     * @return array|null|string
     */
    public static function statusHuman($status)
    {
        return $status ? __('general.text_enabled') : __('general.text_disabled');
    }

    public static function statusTenant($status)
    {
        return $status ? __('general.text_enabled') : __('general.text_suspended');
    }

    /**
     * Estatus a cadena enriquecida
     *
     * @param $status
     * @return array|null|string
     */
    public static function statusHumanRich($status)
    {
        return $status ? '<label class="label label-success">'.__('general.text_enabled').'</label>' : '<label class="label label-default">' . __('general.text_disabled').'</label>';
    }

    /**
     * Funcion obtenida para convertir el numero de serie de un certificado de la corona.com.mx
     *
     * http://www.lacorona.com.mx/fortiz/sat/cfdcvali.phps
     *
     * @param $serial_number
     * @return string
     */
    public static function certificateNumber($serial_number): string
    {
        $ser = '';
        for($i=0;$i<strlen($serial_number);$i++){
            $ser .= ((($i%2)) ? $serial_number[$i] : '');
        }

        /*$hex = self::bcdechex($serial_number);
        $ser = '';
        for ($i = 1; $i < @strlen($hex); $i = $i + 2) {
            $ser .= substr($hex, $i, 1);
        }*/

        return $ser;
    }

    public static function bcdechex($dec)
    {
        $last = bcmod($dec, 16);
        $remain = bcdiv(bcsub($dec, $last), 16);
        if ($remain == 0) {
            return dechex($last);
        } else {
            return self::bcdechex($remain) . dechex($last);
        }
    }

    /**
     * Formato de numero
     *
     * @param $val
     * @param $decimal_place
     * @param bool $format
     * @return string
     */
    public static function numberFormat($val, $decimal_place=0, $format = true)
    {
        $val = (double)$val;
        $decimal_place = (int)$decimal_place;

        return number_format($val, $decimal_place, setting('decimal_mark', '.'),
            $format ? setting('thousands_separator', ',') : '');
    }

    /**
     * Formato de numero para porcentajes
     *
     * @param $val
     * @param $decimal_place
     * @return string
     */
    public static function numberFormatPercent($val, $decimal_place=0)
    {
        return self::numberFormat($val, $decimal_place) . '%';
    }

    /**
     * Formato de numero para porcentajes
     *
     * @param $val
     * @param $decimal_place
     * @return string
     */
    public static function numberFormatPercentRealDecimalPlace($val)
    {
        $decimal_place = 2;
        $tmp = (float)$val;
        $tmp = explode('.', $tmp);
        if(!empty($tmp[1])){
            $decimal_place = strlen($tmp[1]);
        }
        if($tmp[1] ?? '0' == '0'){
            $decimal_place = 0;
        }
        return self::numberFormatPercent($val, $decimal_place);
    }

    /**
     * Formato de numero para monedas
     *
     * @param $val
     * @param int $decimal_place
     * @param string $currency
     * @return string
     */
    public static function numberFormatMoney($val, $decimal_place = 2, $currency = 'MXN')
    {
        $symbol = '$';
        $symbol_position = 'L';
        $decimal_mark = setting('decimal_mark', '.');
        $thousands_separator = setting('thousands_separator', '.');
        if ($currency == 'USD') {
            $symbol = '$';
            $symbol_position = 'L';
        } elseif ($currency == 'EUR') {
            $symbol = '€';
            $symbol_position = 'L';
        }

        $val = (double)$val;
        $decimal_place = (int)$decimal_place;

        $val = number_format($val, $decimal_place, $decimal_mark, $thousands_separator);

        return ($symbol_position == 'L' ? $symbol : '') . $val . ($symbol_position == 'R' ? $symbol : '');
    }

    public static function numberFormatMoneyRealDecimalPlace($val,$currency='MXN'){
        $decimal_place = 2;
        $tmp = (float)$val;
        $tmp = explode('.', $tmp);
        if(!empty($tmp[1])){
            $decimal_place = strlen($tmp[1]);
        }
        return self::numberFormatMoney($val, $decimal_place, $currency);

    }
    public static function numberFormatRealDecimalPlace($val, $format = true){
        $decimal_place = 2;
        $tmp = (float)$val;
        $tmp = explode('.', $tmp);
        if(!empty($tmp[1])){
            $decimal_place = strlen($tmp[1]);
        }
        return self::numberFormat($val, $decimal_place, $format);

    }

    /**
     * Crea fecha a partir del formato configurado
     *
     * @param $date
     * @return mixed
     */
    public static function createDate($date)
    {
        return \Date::createFromFormat('!' . setting('date_format', 'd-m-Y'), $date);
    }

    /**
     * Crea fecha tiempo a partir del formato configurado
     *
     * @param $datetime
     * @return mixed
     */
    public static function createDateTime($datetime)
    {
        return \Date::createFromFormat(setting('datetime_format', 'd-m-Y H:i:s'), $datetime);
    }

    /**
     * Crea fecha a partir de una fecha en formato BD
     *
     * @param $date
     * @return mixed
     */
    public static function createDateFromSql($date)
    {
        return \Date::createFromFormat('!Y-m-d', $date);
    }

    /**
     * Crea fecha tiempo a aprtir de una fecha en formato BD
     *
     * @param $datetime
     * @return mixed
     */
    public static function createDateTimeFromSql($datetime)
    {
        return \Date::parse($datetime);
    }

    /**
     * Convierte fecha a sql para guardar
     *
     * @param $date
     * @return mixed
     */
    public static function dateToSql($date)
    {
        return $date->format('Y-m-d');
    }

    /**
     * Convierte fecha tiempo a sql para guardar
     *
     * @param $date
     * @return mixed
     */
    public static function dateTimeToSql($date)
    {
        return $date->format('Y-m-d H:i:s');
    }

    /**
     * Convierte fecha a sql para guardar
     *
     * @param $date
     * @return mixed
     */
    public static function date($date)
    {
        return $date->format(setting('date_format', 'd-m-Y'));
    }

    /**
     * Convierte fecha tiempo a sql para guardar
     *
     * @param $date
     * @return mixed
     */
    public static function dateTime($date)
    {
        return $date->format(setting('datetime_format', 'd-m-Y H:i:s'));
    }

    /**
     * Crear fecha a partir de fecha configurada y convierte a sql para guardar
     *
     * @param $date
     * @return mixed
     */
    public static function convertDateToSql($date)
    {
        return self::dateToSql(self::createDate($date));
    }

    /**
     * Crear fecha a partir de fecha configurada y convierte a sql para guardar
     *
     * @param $datetime
     * @return mixed
     */
    public static function convertDateTimeToSql($datetime)
    {
        return self::dateTimeToSql(self::createDateTime($datetime));
    }

    /**
     * Crear fecha a partir de fecha configurada y convierte a sql para guardar
     *
     * @param $date
     * @return mixed
     */
    public static function convertSqlToDate($date)
    {
        return self::date(self::createDateFromSql($date));
    }

    /**
     * Crear fecha a partir de fecha configurada y convierte a sql para guardar
     *
     * @param $datetime
     * @return mixed
     */
    public static function convertSqlToDateTime($datetime)
    {
        return self::dateTime(self::createDateTimeFromSql($datetime));
    }

    /**
     * Consecutivo por tipo de documento
     *
     * @param $code
     * @return array
     * @throws \Exception
     */
    public static function getNextDocumentTypeByCode($code, $company_id ='', $draft = false, $branch_office_id = '')
    {
        try {
            $data = [];
            if(!empty(setting('folios_per_branch_office'))){
                $document_type = DocumentType::where('code', '=', $code)->where('company_id', '=', $company_id)->where('branch_office_id', '=', $branch_office_id)->first();
            }else {
                $document_type = DocumentType::where('code', '=', $code)->where('company_id', '=', $company_id)->first();
            }
            if (!empty($document_type)) {
                $data['id'] = $document_type->id;
                if($draft){
                    $document_type->draft_current_number += $document_type->increment_number;
                    $data['serie'] = $document_type->draft_prefix;
                    $data['folio'] = $document_type->draft_current_number;
                    $data['name'] = $document_type->draft_prefix . $document_type->draft_current_number;
                }else{
                    $document_type->current_number += $document_type->increment_number;
                    $data['serie'] = $document_type->prefix;
                    $data['folio'] = $document_type->current_number;
                    $data['name'] = $document_type->prefix . $document_type->current_number;
                }
                $document_type->update();
            } else {
                if(!empty(setting('folios_per_branch_office'))){
                    throw new \Exception(sprintf(__('base/document_type.error_next_document_type_per_branch_office'),route('document-types.index')));
                }else{
                    throw new \Exception(__('base/document_type.error_next_document_type'));
                }
            }
            if (empty($data['id']) || empty($data['name'])) {
                throw new \Exception(__('base/document_type.error_next_document_type'));
            }
            return $data;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Convierte numero a letras
     *
     * @param $number
     * @param string $currency_code
     * @param int $decimal_place
     * @return string
     */
    public static function numberToWordCurrency($number, $currency_code = 'MXN', $decimal_place = 2)
    {
        $numberToWords = new NumberToWords();
        $numberTransformer = $numberToWords->getNumberTransformer('es');
        $postfix_plural_currency = ['MXN' => 'Pesos', 'USD' => 'Dolares', 'EUR' => 'Euros'];
        $postfix_singular_currency = ['MXN' => 'Peso', 'USD' => 'Dolar', 'EUR' => 'Euro'];
        $tmp = self::numberFormat($number, $decimal_place, false);
        $tmp = explode(setting('decimal_mark', '.'), $tmp);
        $number_1 = ($tmp[0] ?? 0);
        $number_2 = ($tmp[1] ?? 0);
        $str = $numberTransformer->toWords($number_1);
        $str_postfix = $postfix_plural_currency[$currency_code] ?? 'XXX';
        if ((int)$number_1 <= 1) {
            $str_postfix = $postfix_singular_currency[$currency_code] ?? 'XXX';
        }
        $str = $str . ' ' . $str_postfix . ' ' . $number_2 . '/100 ' . $currency_code;
        return mb_strtoupper($str);
    }

    /**
     * Crea directorios para CFDI's
     *
     * @return string
     * @throws \Exception
     */
    public static function makeDirectoryCfdi($path_xml)
    {
        try {
            $tmp_path = date('Y') . '/' . date('m');
            if (!\Storage::exists($path_xml . $tmp_path)) {
                \Storage::makeDirectory($path_xml . $tmp_path, 0777, true, true);
                @shell_exec('chmod -R 777 ' . \Storage::path($path_xml . $tmp_path));
            }

            return $tmp_path;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Funcion para convertir el saldo a pesos solo cuando el pago es en MXN
     *
     * @param $currency
     * @param $amount
     * @param $currency_code
     * @param null $currency_value
     * @param int $decimal_place
     * @return float|int
     */
    public static function convertBalanceCurrency($currency, $amount, $currency_code, $currency_value = null,$decimal_place=2)
    {
        if (!empty($currency) && $currency->code == 'MXN' && $currency_code != 'MXN' && !empty($currency_value)) {
            $amount = self::roundDown($amount * $currency_value, $decimal_place);
        }

        return $amount;
    }

    /**
     * Funcion para convertir el saldo a pesos solo cuando el pago es en MXN
     *
     * @param $currency
     * @param $amount
     * @param $currency_code
     * @param null $currency_value
     * @param int $decimal_place
     * @return float|int
     */
    public static function invertBalanceCurrency($currency, $amount, $currency_code, $currency_value = null,$decimal_place=2)
    {
        if (!empty($currency) && $currency->code != $currency_code && !empty($currency_value)) {
            $amount = self::roundDown($amount / $currency_value, $decimal_place);
        }

        return $amount;
    }

    /**
     * Redondeo hacia abajo
     *
     * @param $decimal
     * @param $precision
     * @return float|int
     */
    public static function roundDown($val, $precision)
    {
        return bcdiv($val, 1, $precision);
        //return round($val, $precision, PHP_ROUND_HALF_DOWN);
    }

    /**
     * Obtiene empresa default
     *
     * @return mixed
     */
    public static function defaultCompany($company_id = ''){
        if(empty($company_id)){
            if(!empty(\Auth::user()->company_id) && !empty(setting('multi_taxid'))){
                return Company::findOrFail(\Auth::user()->company_id);
            }
            return Company::orderBy('id')->get()->first();
        }else{
            return Company::findOrFail($company_id);
        }
    }

    /**
     * Obtiene empresa default
     *
     * @return mixed
     */
    public static function firstCompany(){
        return Company::orderBy('id')->get()->first();
    }

    /**
     * Regresa configuracion general
     *
     * @return \Illuminate\Foundation\Application|mixed
     */
    public static function companyColorPrimary($company_id = ''){
        $company = self::defaultCompany($company_id);

        return !empty($company->color_primary) ? $company->color_primary : setting('color_primary','#999999');
    }

    /**
     * Regresa configuracion general
     *
     * @return \Illuminate\Foundation\Application|mixed
     */
    public static function companyProductPriceDecimalPlace($company_id = ''){
        $company = self::defaultCompany($company_id);

        return !empty($company->product_price_decimal_place) ? $company->product_price_decimal_place : setting('product_price_decimal_place',2);
    }

    /**
     * Regresa configuracion general
     *
     * @return \Illuminate\Foundation\Application|mixed
     */
    public static function companyPdfTemplate($company_id = ''){
        $company = self::defaultCompany($company_id);

        return !empty($company->pdf_template) ? $company->pdf_template : setting('pdf_template','default');
    }

    /**
     * Regresa configuracion general
     *
     * @return \Illuminate\Foundation\Application|mixed
     */
    public static function companyComplementoCfdiIedu($company_id = ''){
        $company = self::defaultCompany($company_id);

        return $company->complemento_cfdi_iedu != '' ? $company->complemento_cfdi_iedu : setting('complemento_cfdi_iedu',0);
    }

    /**
     * Regresa configuracion general
     *
     * @return \Illuminate\Foundation\Application|mixed
     */
    public static function companyComplementoCfdiForeignTrade($company_id = ''){
        $company = self::defaultCompany($company_id);

        return $company->complemento_cfdi_foreign_trade != '' ? $company->complemento_cfdi_foreign_trade : setting('complemento_cfdi_foreign_trade',0);
    }

    /**
     * Regresa configuracion general
     *
     * @return \Illuminate\Foundation\Application|mixed
     */
    public static function companyModuleCustomerQuotations($company_id = ''){
        $company = self::defaultCompany($company_id);

        return $company->module_customer_quotations != '' ? $company->module_customer_quotations : setting('module_customer_quotations',0);
    }

    /**
     * Regresa configuracion general
     *
     * @return \Illuminate\Foundation\Application|mixed
     */
    public static function companyModuleCustomerRemissions($company_id = ''){
        $company = self::defaultCompany($company_id);

        return $company->module_customer_remissions != '' ? $company->module_customer_remissions : setting('module_customer_remissions',0);
    }

    /**
     * Regresa configuracion general
     *
     * @return \Illuminate\Foundation\Application|mixed
     */
    public static function companyModuleCqTermOfSale($company_id = ''){
        $company = self::defaultCompany($company_id);

        return $company->modulecq_terms_of_sale != '' ? $company->modulecq_terms_of_sale : setting('modulecq_terms_of_sale','');
    }

    /**
     * Regresa configuracion general
     *
     * @return \Illuminate\Foundation\Application|mixed
     */
    public static function companyModulePayroll($company_id = ''){
        $company = self::defaultCompany($company_id);

        return $company->module_payroll != '' ? $company->module_payroll : setting('module_payroll',0);
    }

    /**
     * Regresa configuracion general
     *
     * @return \Illuminate\Foundation\Application|mixed
     */
    public static function companyModulePayrLegend($company_id = ''){
        $company = self::defaultCompany($company_id);

        return $company->modulepayr_legend != '' ? $company->modulepayr_legend : setting('modulepayr_legend','');
    }

    /**
     * Eliminar archivos temporales
     */
    public static function deleteTempFiles(){
        $files = \Storage::files('temp/');
        $count = 1;
        if(!empty($files)) {
            foreach ($files as $file) {
                if ($file != 'temp/.gitignore') {
                    $time = \Storage::lastModified($file);
                    $date = \Date::createFromTimestamp($time);
                    if ($date->lessThan(\Date::now()->subMonths(1))) { //Solo elimina archivos mayores a 1 mes
                        @\Storage::delete($file);
                        if ($count > 10) { //Solo elimina 10 archivos por usuario
                            break;
                        }
                        $count++;
                    }
                }
            }
        }
        $directories = \Storage::directories();
        if(!empty($directories)) {
            foreach ($directories as $key => $directory) {
                $files = \Storage::files($directory . '/files/cfdi_download_requests/');
                if(!empty($files)) {
                    foreach ($files as $key2 => $file) {
                        $time = \Storage::lastModified($file);
                        $date = \Date::createFromTimestamp($time);
                        if ($date->lessThan(\Date::now()->subMonths(2))) { //Solo elimina archivos mayores a 2 meses
                        @\Storage::delete($file);
                            if ($key2 > 10) { //Solo elimina 10 archivos por usuario
                                break;
                            }
                        }
                    }
                }
                break;
            }
        }
    }

    /**
     * Limpiar arreglo de datos
     *
     * @param $rows
     * @return mixed
     */
    public static function clearArrayImport($rows){
        //Limpieza de datos
        foreach ($rows as $key => $row) {
            foreach($row as $key2 => $value){ //Elimina salto de linea
                $rows[$key][$key2] = trim(str_replace(array("\r\n", "\r", "\n"), '', $value));
            }
            $rows[$key] = collect(array_map('trim', $row->toArray()));
            if(strlen(implode($row->toArray())) == 0){ //Elimina campos vacios
                unset($rows[$key]);
            }
        }

        return $rows;
    }

    public static function validateInsignaLcoRfc($taxid){
        if(!empty(setting('validate_taxid_insigna_url')) && !empty(setting('validate_taxid_insigna_user')) && !empty(setting('validate_taxid_insigna_password'))) {
            try {
                //Inicializa el wsseheader
                $wsse_user = setting('validate_taxid_insigna_user');
                $wsse_password = Crypt::decryptString(setting('validate_taxid_insigna_password'));
                $wsseHeader =
                    '<wsse:Security env:mustUnderstand="1"   xmlns:SOAP-ENV="http://schemas.xmlsoap.org/soap/envelope/" ' . ' xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/" ' . ' xmlns:wsu="http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd"' . ' xmlns:wsse="http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd"> ' . ' <wsse:UsernameToken><wsse:Username>' .
                    $wsse_user .
                    '</wsse:Username> ' . ' <wsse:Password Type="http://docs.oasis-open.org/wss/2004/01/oasis-200401-' . 'wss-username-token-profile-1.0#PasswordText">' .
                    $wsse_password .
                    '</wsse:Password></wsse:UsernameToken></wsse:Security>';
                $wsseHeader = new SoapVar($wsseHeader, XSD_ANYXML);
                $wsseHeader = new SoapHeader('http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd', 'Security', $wsseHeader, true);
                //ws
                $client = new SoapClient(setting('validate_taxid_insigna_url'), [
                    'soap_version' => SOAP_1_2,
                    'cache_wsdl' => WSDL_CACHE_NONE,
                    'ssl_method' => SOAP_SSL_METHOD_TLS
                ]);
                //

                $parametros[] = new SoapVar($taxid, XSD_STRING, null, null, 'ns1:rfc');
                $param[] = new SoapVar($parametros, SOAP_ENC_OBJECT, null, null, 'ns1:validateLRFCWrapper');


                $client->__setSoapHeaders($wsseHeader);
                $response = $client->validateLRFC(new SoapVar($param, SOAP_ENC_OBJECT));
                $codigo_int = (int)$response->return->responseCode;
                if ($codigo_int != 1000) {
                    //throw new \Exception($response->return->responseDescription);
                    return false;
                }
                if (isset($response->return->sncf) && $response->return->sncf == 'N') {
                    return true;
                }

                return false;
            } catch (\SoapFault $e) {
                if (isset($e->detail->OperationFailed)) {
                    throw new \Exception($e->detail->OperationFailed->errorDescription);
                }
                throw new \Exception($e->getMessage());
            } catch (\Exception $e) {
                throw $e;
            }
        }

        return true;
    }

    public static function validateXmlToArrayCfdi33($file_xml)
    {
        $data = [];
        $xml = new \DOMDocument('1.0', 'UTF-8');
        //Revisa y limpia archivo
        $file_xml = \Storage::get($file_xml);
        if (empty($file_xml)) {
            return false;
        }elseif(strlen($file_xml) < 50){
            return false;
        }elseif(preg_match('/a no puedes descargar/i', $file_xml)) {
            return false;
        }elseif(preg_match('/<html /i', $file_xml)){
            return false;
        }
        $file_xml = preg_replace('/[^\x{0009}\x{000a}\x{000d}\x{0020}-\x{D7FF}\x{E000}-\x{FFFD}]+/u', ' ', $file_xml);
        if(empty($file_xml)){
            return false;
        }
        libxml_use_internal_errors(true);
        //$file_xml = \CfdiUtils\Cleaner\Cleaner::staticClean($file_xml);
        $xml->loadXML($file_xml);
        if($xml->getElementsByTagName('Comprobante')->length == 0){
            return false;
        }

        return true;
    }

    public static function validateXmlToArrayCfdi332($file_xml)
    {
        $data = [];
        $xml = new \DOMDocument('1.0', 'UTF-8');
        //Revisa y limpia archivo
        $file_xml = \Storage::get($file_xml);
        if (empty($file_xml)) {
            return false;
        }elseif(strlen($file_xml) < 50){
            return false;
        }elseif(preg_match('/a no puedes descargar/i', $file_xml)) {
            throw new \Exception('Ya no puedes descargar más documentos. Por seguridad únicamente se permite descargar un máximo de 2,000 archivos por día.');
        }elseif(preg_match('/<html /i', $file_xml)){
            throw new \Exception('Error en la descarga');
        }
        $file_xml = preg_replace('/[^\x{0009}\x{000a}\x{000d}\x{0020}-\x{D7FF}\x{E000}-\x{FFFD}]+/u', ' ', $file_xml);
        if(empty($file_xml)){
            return false;
        }
        libxml_use_internal_errors(true);
        //$file_xml = \CfdiUtils\Cleaner\Cleaner::staticClean($file_xml);
        $xml->loadXML($file_xml);
        if($xml->getElementsByTagName('Comprobante')->length == 0){
            return false;
        }

        return true;
    }

    public static function parseXmlToArrayCfdi33($file_xml, $generate_reconcileds = false){
        $data = [];
        $xml = new \DOMDocument('1.0', 'UTF-8');
        //Revisa y limpia archivo
        $file_xml = \Storage::get($file_xml);
        $file_xml = preg_replace('/[^\x{0009}\x{000a}\x{000d}\x{0020}-\x{D7FF}\x{E000}-\x{FFFD}]+/u', ' ', $file_xml);
        //$file_xml = \CfdiUtils\Cleaner\Cleaner::staticClean($file_xml);
        libxml_use_internal_errors(true);
        $xml->loadXML($file_xml);
        $xml_array = self::domNodeToArray($xml->getElementsByTagName('Comprobante')->item(0));
        //
        $comprobante = $xml_array['@attributes'];
        $emisor = !empty($xml_array['cfdi:Emisor']['@attributes']) ? $xml_array['cfdi:Emisor']['@attributes'] : [];
        $receptor = !empty($xml_array['cfdi:Receptor']['@attributes']) ? $xml_array['cfdi:Receptor']['@attributes'] : [];
        $impuestos = !empty($xml_array['cfdi:Impuestos']['@attributes']) ? $xml_array['cfdi:Impuestos']['@attributes'] : [];
        $tfd = !empty($xml_array['cfdi:Complemento']['tfd:TimbreFiscalDigital']['@attributes']) ? $xml_array['cfdi:Complemento']['tfd:TimbreFiscalDigital']['@attributes'] : [];
        $cfdi_relacionados = !empty($xml_array['cfdi:CfdiRelacionados']['@attributes']) ? $xml_array['cfdi:CfdiRelacionados']['@attributes'] : [];

        //fix
        if(!empty($comprobante['TipoDeComprobante'])){
            if(strtolower($comprobante['TipoDeComprobante']) == 'ingreso'){
                $comprobante['TipoDeComprobante'] = 'I';
            }
            if(strtolower($comprobante['TipoDeComprobante']) == 'pago'){
                $comprobante['TipoDeComprobante'] = 'P';
            }
            if(strtolower($comprobante['TipoDeComprobante']) == 'egreso'){
                $comprobante['TipoDeComprobante'] = 'E';
            }
            if(strtolower($comprobante['TipoDeComprobante']) == 'traslado'){
                $comprobante['TipoDeComprobante'] = 'T';
            }
        }

        //Obtiene datos
        $payment_way = PaymentWay::whereCode($comprobante['FormaPago'] ?? 'XXX')->first();
        $payment_method = PaymentMethod::whereCode($comprobante['MetodoPago'] ?? 'XXX')->first();
        $currency = Currency::whereCode($comprobante['Moneda'] ?? 'XXX')->first();
        $cfdi_use = CfdiUse::whereCode($receptor['UsoCFDI'] ?? 'XXX')->first();
        $cfdi_type = CfdiType::whereCode($comprobante['TipoDeComprobante'] ?? 'XXX')->first();
        $cfdi_relation = CfdiRelation::whereCode($cfdi_relacionados['TipoRelacion'] ?? 'XXX')->first();
        $tax_regimen_emisor = TaxRegimen::whereCode($emisor['RegimenFiscal'] ?? 'XXX')->first();
        $tax_regimen_receptor = TaxRegimen::whereCode($emisor['RegimenFiscalReceptor'] ?? 'XXX')->first();

        //
        $data['date'] = str_replace(['T','Z'],[' ',''],$comprobante['Fecha']);
        $data['date_certification'] = !empty($tfd['FechaTimbrado']) ? str_replace('T',' ',$tfd['FechaTimbrado']) : null;
        $data['version'] = $comprobante['Version'] ?? ($comprobante['version'] ?? '');
        $data['name'] = ($comprobante['Serie'] ?? '') . ($comprobante['Folio'] ?? '');
        $data['serie'] = $comprobante['Serie'] ?? '';
        $data['folio'] = $comprobante['Folio'] ?? null;
        $data['rfc_emisor'] = $emisor['Rfc'] ?? '';
        $data['emisor'] = $emisor['Nombre'] ?? '';
        $data['rfc_receptor'] = $receptor['Rfc'] ?? '';
        $data['receptor'] = $receptor['Nombre'] ?? '';
        $data['numid'] = $receptor['NumRegIdTrib'] ?? '';
        $data['country_code'] = $receptor['ResidenciaFiscal'] ?? '';
        $data['regimen_code_emisor'] = $emisor['RegimenFiscal'] ?? '';
        $data['regimen_code_receptor'] = $receptor['RegimenFiscalReceptor'] ?? '';
        $data['regimen_emisor'] = $tax_regimen_emisor->name ?? '';
        $data['regimen_receptor'] = $tax_regimen_receptor->name ?? '';
        $data['domicilio_cp'] = $receptor['DomicilioFiscalReceptor'] ?? '';
        $data['payment_way_id'] = !empty($payment_way) ? $payment_way->id : null;
        $data['payment_method_id'] = !empty($payment_method) ? $payment_method->id : null;
        $data['payment_method_code'] = !empty($payment_method) ? $payment_method->code : null;
        $data['cfdi_use_id'] = !empty($cfdi_use) ? $cfdi_use->id : null;
        $data['currency_id'] = !empty($currency) ? $currency->id : null;
        $data['currency_code'] = !empty($currency) ? $currency->code : null;
        $data['currency_value'] = (double)($comprobante['TipoCambio'] ?? 1);
        $data['currency_value'] = $data['currency_value'] == 0 ? 1 : $data['currency_value'];
        $data['amount_discount'] = (double)($comprobante['Descuento'] ?? '');
        $data['amount_untaxed'] = (double)$comprobante['SubTotal'] - $data['amount_discount'];
        $data['amount_tax'] = (double)($impuestos['TotalImpuestosTrasladados'] ?? '');
        $data['amount_tax_ret'] = (double)($impuestos['TotalImpuestosRetenidos'] ?? '');
        $data['amount_total'] = (double)$comprobante['Total'];
        $data['balance'] = (double)$comprobante['Total'];
        $data['cfdi_type_id'] = !empty($cfdi_type) ? $cfdi_type->id : null;
        $data['tipo_de_comprobante'] = $comprobante['TipoDeComprobante'] ?? '';
        $data['rfc_pac'] = $tfd['RfcProvCertif'] ?? '';
        $data['uuid'] = !empty($tfd['UUID']) ? strtoupper(trim($tfd['UUID'])) : '';
        $data['cfdi_relation_id'] = !empty($cfdi_relation) ? $cfdi_relation->id : null;
        $data_lines = [];
        $data_reconcileds = [];

        //Fix serie y folio
        $data['folio'] = preg_replace('/[^0-9a-zA-Z]/', '', $data['folio']);
        $data['serie'] .= preg_replace('/[^0-9a-zA-Z]/', '', $data['folio']);
        $data['folio'] = !empty($data['folio']) ? (int)$data['folio'] : null;

        //impuestos
        $data['impuestos'] = [
            'iva' => 0,
            'iva_16' => 0,
            'iva_15' => 0,
            'iva_11' => 0,
            'iva_10' => 0,
            'iva_8' => 0,
            'ieps' => 0,
            'ret_iva' => 0,
            'ret_isr' => 0
        ];

        //Monto total de pago para pagos
        if(in_array($comprobante['TipoDeComprobante'], ['P'])){
            $tmp_nodo_pagos = [];
            $nodo_pagos = [];
            $pagos = [];
            $tmp_nodo_reconcileds = [];
            $nodo_reconcileds = [];
            $reconcileds = [];
            $traslados = [];
            $retenciones = [];

            //
            if(!empty($xml_array['cfdi:Complemento']['pago10:Pagos']['pago10:Pago'])){
                $tmp_nodo_pagos = $xml_array['cfdi:Complemento']['pago10:Pagos']['pago10:Pago'];
            }elseif(!empty($xml_array['cfdi:Complemento']['pago10:Pagos']['Pago'])){
                $tmp_nodo_pagos = $xml_array['cfdi:Complemento']['pago10:Pagos']['Pago'];
            }elseif(!empty($xml_array['cfdi:Complemento']['pago20:Pagos']['pago20:Pago'])){
                $tmp_nodo_pagos = $xml_array['cfdi:Complemento']['pago20:Pagos']['pago20:Pago'];
            }elseif(!empty($xml_array['cfdi:Complemento']['pago20:Pagos']['Pago'])){
                $tmp_nodo_pagos = $xml_array['cfdi:Complemento']['pago20:Pagos']['Pago'];
            }
            if(!empty($tmp_nodo_pagos)){
                if(!empty($tmp_nodo_pagos['@attributes'])){
                    $nodo_pagos[] = $tmp_nodo_pagos;
                    $pagos[] = $tmp_nodo_pagos['@attributes'];

                }elseif(!empty($tmp_nodo_pagos)){
                    foreach ($tmp_nodo_pagos as $tmp) {
                        $nodo_pagos[] = $tmp;
                        $pagos[] = $tmp['@attributes'];
                    }
                }
            }
            //Obtiene datos generales del pago
            if(!empty($pagos) && count($pagos) == 1){
                $pago = $pagos[0];
                if(empty($data['payment_way_id']) && !empty($pago['FormaDePagoP'])){
                    $payment_way = PaymentWay::whereCode($pago['FormaDePagoP'] ?? 'XXX')->first();
                    $data['payment_way_id'] = !empty($payment_way) ? $payment_way->id : null;
                }
                if(empty($data['currency_id']) && !empty($pago['MonedaP'])){
                    $currency = Currency::whereCode($pago['MonedaP'] ?? 'XXX')->first();
                    $data['currency_id'] = !empty($currency) ? $currency->id : null;
                    $data['currency_value'] = (double)($pago['TipoCambioP'] ?? 1);
                }
                if(!empty($pago['FechaPago'])){
                    $data['date_payment'] = str_replace(['T','Z'],[' ',''],$pago['FechaPago']);
                }
                if(!empty($pago['Monto'])){
                    $data['amount_total'] = (double)$pago['Monto'];
                    $data['balance'] = 0;
                }
            }elseif(!empty($pagos) && count($pagos) > 1){
                $currency = Currency::whereCode('MXN')->first();
                $data['currency_id'] = !empty($currency) ? $currency->id : null;
                $data['currency_value'] = 1;
                foreach($pagos as $pago){
                    if(!empty($pago['Monto'])){
                        $data['amount_total'] += (double)$pago['Monto'] * (double)($pago['TipoCambioP'] ?? 1);
                        $data['balance'] += 0;
                    }
                }
            }

            //
            if($generate_reconcileds){
                if(!empty($nodo_pagos)){
                    foreach($nodo_pagos as $nodo_pago){
                        //
                        $reconcileds = [];
                        $pago = $nodo_pago['@attributes'];

                        //
                        if(!empty($nodo_pago['pago10:DoctoRelacionado'])){
                            $tmp_nodo_reconcileds = $nodo_pago['pago10:DoctoRelacionado'];
                        }elseif(!empty($nodo_pago['DoctoRelacionado'])){
                            $tmp_nodo_reconcileds = $nodo_pago['DoctoRelacionado'];
                        }elseif(!empty($nodo_pago['pago20:DoctoRelacionado'])){
                            $tmp_nodo_reconcileds = $nodo_pago['pago20:DoctoRelacionado'];
                        }
                        if(!empty($tmp_nodo_reconcileds)){
                            if(!empty($tmp_nodo_reconcileds['@attributes'])){
                                $nodo_reconcileds[] = $tmp_nodo_reconcileds;
                                $reconcileds[] = $tmp_nodo_reconcileds['@attributes'];
                            }elseif(!empty($tmp_nodo_reconcileds)){
                                foreach ($tmp_nodo_reconcileds as $tmp) {
                                    $nodo_reconcileds[] = $tmp;
                                    $reconcileds[] = $tmp['@attributes'];
                                }
                            }
                        }

                        //
                        if(!empty($reconcileds)){
                            foreach($reconcileds as $reconciled){
                                $customer_invoice_cfdi = CustomerInvoiceCfdi::where('uuid','=',$reconciled['IdDocumento'])->first();

                                //Fix serie y folio
                                $reconciled['Serie'] = $reconciled['Serie'] ?? '';
                                $reconciled['Folio'] = preg_replace('/[^0-9]/', '', $reconciled['Folio'] ?? '');
                                $reconciled['Serie'] .= empty($reconciled['Serie']) ? preg_replace('/[^a-zA-Z]/', '', $reconciled['Folio'] ?? '') : $reconciled['Serie'];
                                $reconciled['name'] = $reconciled['Serie'] . $reconciled['Folio'];

                                $data_reconcileds[] = [
                                    'name' => $customer_invoice_cfdi->customerInvoice->name ?? (!empty($reconciled['name']) ? $reconciled['name'] : $reconciled['IdDocumento']),
                                    'reconciled_id' => $customer_invoice_cfdi->customerInvoice->id ?? null,
                                    'currency_value' => !empty($reconciled['TipoCambioDR']) ? 1 / $reconciled['TipoCambioDR'] : (!empty($reconciled['EquivalenciaDR']) ? 1 / $reconciled['EquivalenciaDR'] : 1),
                                    'amount_reconciled' => $reconciled['ImpPagado'] ?? 0,
                                    'last_balance' =>  $reconciled['ImpSaldoAnt'] ?? 0,
                                    'number_of_payment' => $reconciled['NumParcialidad'] ?? '',
                                    'uuid_related' => $reconciled['IdDocumento'] ?? '',
                                    'serie_related' => $reconciled['Serie'] ?? '',
                                    'folio_related' => !empty($reconciled['Folio']) ? (int)$reconciled['Folio'] : null,
                                    'currency_code_related' =>  $reconciled['MonedaDR'] ?? '',
                                    'currency_code_origin' =>  $pago['MonedaP'] ?? '',
                                    'payment_method_code_related' =>  $reconciled['MetodoDePagoDR'] ?? '', //Solo CFDI 3.3
                                    'current_balance' =>  $reconciled['ImpSaldoInsoluto'] ?? 0,
                                ];
                            }
                        }
                    }
                }
            }
        }

        return ['data' => $data, 'data_lines' => $data_lines, 'data_reconcileds' => $data_reconcileds];
    }

    public static function parseXmlToArrayCfdi33Relacionados($file_xml){
        $data = [];
        $xml = new \DOMDocument('1.0', 'UTF-8');
        //Revisa y limpia archivo
        $file_xml = \Storage::get($file_xml);
        $file_xml = preg_replace('/[^\x{0009}\x{000a}\x{000d}\x{0020}-\x{D7FF}\x{E000}-\x{FFFD}]+/u', ' ', $file_xml);
        //$file_xml = \CfdiUtils\Cleaner\Cleaner::staticClean($file_xml);
        libxml_use_internal_errors(true);
        $xml->loadXML($file_xml);
        $xml_array = self::domNodeToArray($xml->getElementsByTagName('Comprobante')->item(0));
        //

        $tmp_nodo_cfdi_relacionados = [];
        $cfdi_relacionados = [];
        if(!empty($xml_array['cfdi:CfdiRelacionados']['cfdi:CfdiRelacionado'])){
            $tmp_nodo_cfdi_relacionados = $xml_array['cfdi:CfdiRelacionados']['cfdi:CfdiRelacionado'];

        }elseif(!empty($xml_array['cfdi:CfdiRelacionados']['CfdiRelacionado'])){
            $tmp_nodo_cfdi_relacionados = $xml_array['cfdi:CfdiRelacionados']['CfdiRelacionado'];
        }
        if(!empty($tmp_nodo_cfdi_relacionados)){
            if(!empty($tmp_nodo_cfdi_relacionados['@attributes'])){
                $cfdi_relacionados[] = $tmp_nodo_cfdi_relacionados['@attributes'];

            }elseif(!empty($tmp_nodo_cfdi_relacionados)){
                foreach ($tmp_nodo_cfdi_relacionados as $tmp) {
                    $cfdi_relacionados[] = $tmp['@attributes'];
                }
            }
        }

        if(!empty($cfdi_relacionados)){
            foreach($cfdi_relacionados as $cfdi_relacionado){
                $data[] = $cfdi_relacionado['UUID'];
            }
        }

        return $data;
    }

    public static function parseXmlToArrayCfdi33Conceptos($file_xml){
        $data = '';
        $xml = new \DOMDocument('1.0', 'UTF-8');
        //Revisa y limpia archivo
        $file_xml = \Storage::get($file_xml);
        $file_xml = preg_replace('/[^\x{0009}\x{000a}\x{000d}\x{0020}-\x{D7FF}\x{E000}-\x{FFFD}]+/u', ' ', $file_xml);
        //$file_xml = \CfdiUtils\Cleaner\Cleaner::staticClean($file_xml);
        libxml_use_internal_errors(true);
        $xml->loadXML($file_xml);
        $xml_array = self::domNodeToArray($xml->getElementsByTagName('Comprobante')->item(0));
        //
        $comprobante = $xml_array['@attributes'];
        if(in_array($comprobante['TipoDeComprobante'], ['I','E']) && isset($xml_array['cfdi:Conceptos'])){
            //Lista de conceptos
            $tmp_nodo_conceptos = [];
            $nodo_conceptos = [];
            $conceptos = [];
            if(!empty($xml_array['cfdi:Conceptos']['cfdi:Concepto'])){
                $tmp_nodo_conceptos[] = $xml_array['cfdi:Conceptos']['cfdi:Concepto'];

            }elseif(!empty($xml_array['cfdi:Conceptos']['Concepto'])){
                $tmp_nodo_conceptos[] = $xml_array['cfdi:Conceptos']['Concepto'];
            }
            if(!empty($tmp_nodo_conceptos)){
                foreach($tmp_nodo_conceptos as $tmp_nodo_concepto){
                    if(!empty($tmp_nodo_concepto['@attributes'])){
                        $nodo_conceptos[] = $tmp_nodo_concepto;
                        $conceptos[] = $tmp_nodo_concepto['@attributes'];

                    }elseif(!empty($tmp_nodo_concepto)){
                        foreach ($tmp_nodo_concepto as $tmp) {
                            $nodo_conceptos[] = $tmp;
                            $conceptos[] = $tmp['@attributes'];
                        }
                    }
                }
            }
            if(!empty($conceptos)){
                foreach($conceptos as $concepto){
                    $data .= !empty($concepto['Descripcion']) ? (!empty($conceptos) ? ' * ' : '') . $concepto['Descripcion'] : '';
                }
            }
        }

        return $data;
    }

    public static function parseXmlToArrayCfdi33Conceptos2($file_xml){
        $data = [];
        $accumulated_taxes = [];

        $xml = new \DOMDocument('1.0', 'UTF-8');
        //Revisa y limpia archivo
        $file_xml = \Storage::get($file_xml);
        $file_xml = preg_replace('/[^\x{0009}\x{000a}\x{000d}\x{0020}-\x{D7FF}\x{E000}-\x{FFFD}]+/u', ' ', $file_xml);
        //$file_xml = \CfdiUtils\Cleaner\Cleaner::staticClean($file_xml);
        libxml_use_internal_errors(true);
        $xml->loadXML($file_xml);
        $xml_array = self::domNodeToArray($xml->getElementsByTagName('Comprobante')->item(0));
        //
        $comprobante = $xml_array['@attributes'];
        if(in_array($comprobante['TipoDeComprobante'], ['I','E','T']) && isset($xml_array['cfdi:Conceptos'])){
            //Lista de conceptos
            $tmp_nodo_conceptos = [];
            $nodo_conceptos = [];
            $conceptos = [];
            if(!empty($xml_array['cfdi:Conceptos']['cfdi:Concepto'])){
                $tmp_nodo_conceptos[] = $xml_array['cfdi:Conceptos']['cfdi:Concepto'];

            }elseif(!empty($xml_array['cfdi:Conceptos']['Concepto'])){
                $tmp_nodo_conceptos[] = $xml_array['cfdi:Conceptos']['Concepto'];
            }
            if(!empty($tmp_nodo_conceptos)){
                foreach($tmp_nodo_conceptos as $tmp_nodo_concepto){
                    if(!empty($tmp_nodo_concepto['@attributes'])){
                        $nodo_conceptos[] = $tmp_nodo_concepto;
                        $conceptos[] = $tmp_nodo_concepto['@attributes'];

                    }elseif(!empty($tmp_nodo_concepto)){
                        foreach ($tmp_nodo_concepto as $tmp) {
                            $nodo_conceptos[] = $tmp;
                            $conceptos[] = $tmp['@attributes'];
                        }
                    }
                }
            }
            if(!empty($nodo_conceptos)){
                foreach($nodo_conceptos as $nodo_concepto){
                    $concepto = $nodo_concepto['@attributes'];
                    $importe = !empty($concepto['Importe']) ? $concepto['Importe'] : (!empty($concepto['importe']) ? $concepto['importe'] : 0);
                    $unit_measure = UnitMeasure::whereCode($concepto['ClaveUnidad'] ?? 'XXX')->first();
                    $sat_product = SatProduct::whereCode($concepto['ClaveProdServ'] ?? 'XXX')->first();
                    $amount_discount = $concepto['Descuento'] ?? 0;
                    $amount_untaxed = $importe - $amount_discount;
                    $price_unit = $concepto['ValorUnitario'];
                    $discount = $amount_discount > 0 ? $amount_discount / $importe : 0;
                    $price_reduce = ($price_unit * (100 - $discount) / 100);;
                    $amount_tax = 0;
                    $amount_tax_ret = 0;
                    $taxes = [];

                    //Traslados
                    $tmp_nodo_traslados = [];
                    $traslados = [];
                    if(!empty($nodo_concepto['cfdi:Impuestos']['cfdi:Traslados']['cfdi:Traslado'])){
                        $tmp_nodo_traslados = $nodo_concepto['cfdi:Impuestos']['cfdi:Traslados']['cfdi:Traslado'];

                    }elseif(!empty($nodo_concepto['cfdi:Impuestos']['cfdi:Traslados']['Traslado'])){
                        $tmp_nodo_traslados = $nodo_concepto['cfdi:Impuestos']['cfdi:Traslados']['Traslado'];
                    }
                    if(!empty($tmp_nodo_traslados)){
                        if(!empty($tmp_nodo_traslados['@attributes'])){
                            $traslados[] = $tmp_nodo_traslados['@attributes'];

                        }elseif(!empty($tmp_nodo_traslados)){
                            foreach ($tmp_nodo_traslados as $tmp) {
                                $traslados[] = $tmp['@attributes'];
                            }
                        }
                    }
                    if(!empty($traslados)){
                        foreach($traslados as $traslado){
                            $traslado['TasaOCuota'] = $traslado['TasaOCuota'] ?? 0;
                            $traslado['Importe'] = $traslado['Importe'] ?? 0;
                            $amount_tax += $traslado['Importe'];
                            $tax = Tax::where('factor','=',$traslado['TipoFactor'] ?? 'XXX')
                            ->where('rate','=',$traslado['TasaOCuota'] > 0 ? $traslado['TasaOCuota']*100 : $traslado['TasaOCuota'])
                            ->where('code','=',$traslado['Impuesto'] ?? 'XXX')
                            ->first();
                            if(!empty($tax)){
                                $taxes[] = $tax->id;
                                $amount_base_tmp = $amount_untaxed;
                                $amount_tax_tmp = $traslado['Importe'];
                                $amount_base_tmp += $accumulated_taxes[$tax->id]['amount_base'] ?? 0;
                                $amount_tax_tmp += $accumulated_taxes[$tax->id]['amount_tax'] ?? 0;
                                $accumulated_taxes[$tax->id] = [
                                    'name' => $tax->name,
                                    'tax_id' => $tax->id,
                                    'amount_base' => $amount_base_tmp,
                                    'amount_tax' => $amount_tax_tmp,
                                ];
                            }
                        }
                    }

                    //Retenciones
                    $tmp_nodo_retenciones = [];
                    $retenciones = [];
                    if(!empty($nodo_concepto['cfdi:Impuestos']['cfdi:Retenciones']['cfdi:Retencion'])){
                        $tmp_nodo_retenciones = $nodo_concepto['cfdi:Impuestos']['cfdi:Retenciones']['cfdi:Retencion'];
                    }elseif(!empty($nodo_concepto['cfdi:Impuestos']['cfdi:Retenciones']['Retencion'])){
                        $tmp_nodo_retenciones = $nodo_concepto['cfdi:Impuestos']['cfdi:Retenciones']['Retencion'];
                    }
                    if(!empty($tmp_nodo_retenciones)){
                        if(!empty($tmp_nodo_retenciones['@attributes'])){
                            $retenciones[] = $tmp_nodo_retenciones['@attributes'];

                        }elseif(!empty($tmp_nodo_retenciones)){
                            foreach ($tmp_nodo_retenciones as $tmp) {
                                $retenciones[] = $tmp['@attributes'];
                            }
                        }
                    }
                    if(!empty($retenciones)){
                        foreach($retenciones as $retencion){
                            $retencion['TasaOCuota'] = $retencion['TasaOCuota'] ?? 0;
                            $retencion['Importe'] = $retencion['Importe'] ?? 0;
                            $amount_tax_ret += $retencion['Importe'] * (-1);
                            $tax = Tax::where('factor','=',$retencion['TipoFactor'] ?? 'XXX')
                            ->where('rate','=',$retencion['TasaOCuota'] > 0 ? $retencion['TasaOCuota']*100 * (-1) : $retencion['TasaOCuota'] * (-1))
                            ->where('code','=',$retencion['Impuesto'] ?? 'XXX')
                            ->first();
                            if(!empty($tax)){
                                $taxes[] = $tax->id;
                                $amount_base_tmp = $amount_untaxed;
                                $amount_tax_tmp = $retencion['Importe'] * (-1);
                                $amount_base_tmp += $accumulated_taxes[$tax->id]['amount_base'] ?? 0;
                                $amount_tax_tmp += $accumulated_taxes[$tax->id]['amount_tax'] ?? 0;
                                $accumulated_taxes[$tax->id] = [
                                    'name' => $tax->name,
                                    'tax_id' => $tax->id,
                                    'amount_base' => $amount_base_tmp,
                                    'amount_tax' => $amount_tax_tmp,
                                ];
                            }
                        }
                    }

                    //
                    $data[] = [
                        'code' => $concepto['NoIdentificacion'] ?? '',
                        'name' => $concepto['Descripcion'],
                        'product_id' => null,
                        'sat_product_id' => !empty($sat_product) ? $sat_product->id : null,
                        'unit_measure_id' => !empty($unit_measure) ? $unit_measure->id : null,
                        'quantity' => $concepto['Cantidad'],
                        'price_unit' => $price_unit,
                        'discount' => round($discount,4),
                        'price_reduce' => round($price_reduce,6),
                        'amount_discount' => $amount_discount,
                        'amount_untaxed' => $amount_untaxed,
                        'amount_tax' => $amount_tax,
                        'amount_tax_ret' => $amount_tax_ret,
                        'amount_total' => $amount_untaxed + $amount_tax + $amount_tax_ret,
                        'taxes' => $taxes,
                    ];
                }
            }
        }

        return ['data' => $data, 'accumulated_taxes' => $accumulated_taxes];
    }

    public static function parseXmlToArrayCfdi33Conceptos3($file_xml){
        $data = [];
        $xml = new \DOMDocument('1.0', 'UTF-8');
        //Revisa y limpia archivo
        $file_xml = \Storage::get($file_xml);
        $file_xml = preg_replace('/[^\x{0009}\x{000a}\x{000d}\x{0020}-\x{D7FF}\x{E000}-\x{FFFD}]+/u', ' ', $file_xml);
        //$file_xml = \CfdiUtils\Cleaner\Cleaner::staticClean($file_xml);
        libxml_use_internal_errors(true);
        $xml->loadXML($file_xml);
        $xml_array = self::domNodeToArray($xml->getElementsByTagName('Comprobante')->item(0));
        //
        $comprobante = $xml_array['@attributes'];
        if(in_array($comprobante['TipoDeComprobante'], ['I','E']) && isset($xml_array['cfdi:Conceptos'])){
            //Lista de conceptos
            $tmp_nodo_conceptos = [];
            $nodo_conceptos = [];
            $conceptos = [];
            if(!empty($xml_array['cfdi:Conceptos']['cfdi:Concepto'])){
                $tmp_nodo_conceptos[] = $xml_array['cfdi:Conceptos']['cfdi:Concepto'];

            }elseif(!empty($xml_array['cfdi:Conceptos']['Concepto'])){
                $tmp_nodo_conceptos[] = $xml_array['cfdi:Conceptos']['Concepto'];
            }
            if(!empty($tmp_nodo_conceptos)){
                foreach($tmp_nodo_conceptos as $tmp_nodo_concepto){
                    if(!empty($tmp_nodo_concepto['@attributes'])){
                        $nodo_conceptos[] = $tmp_nodo_concepto;
                        $conceptos[] = $tmp_nodo_concepto['@attributes'];

                    }elseif(!empty($tmp_nodo_concepto)){
                        foreach ($tmp_nodo_concepto as $tmp) {
                            $nodo_conceptos[] = $tmp;
                            $conceptos[] = $tmp['@attributes'];
                        }
                    }
                }
            }
            if(!empty($conceptos)){
                foreach($conceptos as $concepto){
                    $data[] = [
                        'ClaveProdServ' => !empty($concepto['ClaveProdServ']) ? $concepto['ClaveProdServ'] : '',
                        'descripcion' => !empty($concepto['Descripcion']) ? $concepto['Descripcion'] : '',
                    ];
                }
            }
        }

        return $data;
    }

    public static function parseXmlToArrayCfdi33BaseImpuestos($file_xml){
        $data = [
            'base_iva' => 0,
            'base_iva_16' => 0,
            'base_iva_15' => 0,
            'base_iva_11' => 0,
            'base_iva_10' => 0,
            'base_iva_8' => 0,
            'base_iva_0' => 0,
            'base_iva_exempt' => 0,
            'base_ieps' => 0,
            'base_ret_iva' => 0,
            'base_ret_isr' => 0

        ];
        $xml = new \DOMDocument('1.0', 'UTF-8');
        //Revisa y limpia archivo
        $file_xml = \Storage::get($file_xml);
        $file_xml = preg_replace('/[^\x{0009}\x{000a}\x{000d}\x{0020}-\x{D7FF}\x{E000}-\x{FFFD}]+/u', ' ', $file_xml);
        //$file_xml = \CfdiUtils\Cleaner\Cleaner::staticClean($file_xml);
        libxml_use_internal_errors(true);
        $xml->loadXML($file_xml);
        $xml_array = self::domNodeToArray($xml->getElementsByTagName('Comprobante')->item(0));
        //
        $comprobante = $xml_array['@attributes'];
        if(in_array($comprobante['TipoDeComprobante'], ['I','E']) && isset($xml_array['cfdi:Conceptos'])){

            //Lista de conceptos
            $tmp_nodo_conceptos = [];
            $nodo_conceptos = [];
            if(!empty($xml_array['cfdi:Conceptos']['cfdi:Concepto'])){
                $tmp_nodo_conceptos[] = $xml_array['cfdi:Conceptos']['cfdi:Concepto'];

            }elseif(!empty($xml_array['cfdi:Conceptos']['Concepto'])){
                $tmp_nodo_conceptos[] = $xml_array['cfdi:Conceptos']['Concepto'];
            }
            if(!empty($tmp_nodo_conceptos)){
                foreach($tmp_nodo_conceptos as $tmp_nodo_concepto){
                    if(!empty($tmp_nodo_concepto['@attributes'])){
                        $nodo_conceptos[] = $tmp_nodo_concepto;
                    }elseif(!empty($tmp_nodo_concepto)){
                        foreach ($tmp_nodo_concepto as $tmp) {
                            $nodo_conceptos[] = $tmp;
                        }
                    }
                }
            }

            //Impuestos por concepto
            foreach($nodo_conceptos as $nodo_concepto){
                //Traslados
                $tmp_nodo_traslados = [];
                $traslados = [];
                if(!empty($nodo_concepto['cfdi:Impuestos']['cfdi:Traslados']['cfdi:Traslado'])){
                    $tmp_nodo_traslados = $nodo_concepto['cfdi:Impuestos']['cfdi:Traslados']['cfdi:Traslado'];

                }elseif(!empty($nodo_concepto['cfdi:Impuestos']['cfdi:Traslados']['Traslado'])){
                    $tmp_nodo_traslados = $nodo_concepto['cfdi:Impuestos']['cfdi:Traslados']['Traslado'];
                }
                if(!empty($tmp_nodo_traslados)){
                    if(!empty($tmp_nodo_traslados['@attributes'])){
                        $traslados[] = $tmp_nodo_traslados['@attributes'];

                    }elseif(!empty($tmp_nodo_traslados)){
                        foreach ($tmp_nodo_traslados as $tmp) {
                            $traslados[] = $tmp['@attributes'];
                        }
                    }
                }
                if(!empty($traslados)){
                    foreach($traslados as $traslado){
                        $traslado['TasaOCuota'] = $traslado['TasaOCuota'] ?? null;
                        $traslado['TipoFactor'] = $traslado['TipoFactor'] ?? null;
                        $traslado['Base'] = !empty($traslado['Base']) ? (double)$traslado['Base'] : (!empty($traslado['base']) ? (double)$traslado['base'] : 0);


                        if ($traslado['Impuesto'] == '002') {
                            $data['base_iva'] += (double)$traslado['Base'];
                        }
                        if ($traslado['Impuesto'] == '002' && $traslado['TipoFactor'] == 'Tasa' && (double)$traslado['TasaOCuota'] == 0.16) {
                            $data['base_iva_16'] += (double)$traslado['Base'];
                        }
                        if ($traslado['Impuesto'] == '002' && $traslado['TipoFactor'] == 'Tasa' && (double)$traslado['TasaOCuota'] == 0.15) {
                            $data['base_iva_15'] += (double)$traslado['Base'];
                        }
                        if ($traslado['Impuesto'] == '002' && $traslado['TipoFactor'] == 'Tasa' && (double)$traslado['TasaOCuota'] == 0.11) {
                            $data['base_iva_11'] += (double)$traslado['Base'];
                        }
                        if ($traslado['Impuesto'] == '002' && $traslado['TipoFactor'] == 'Tasa' && (double)$traslado['TasaOCuota'] == 0.10) {
                            $data['base_iva_10'] += (double)$traslado['Base'];
                        }
                        if ($traslado['Impuesto'] == '002' && $traslado['TipoFactor'] == 'Tasa' && (double)$traslado['TasaOCuota'] == 0.08) {
                            $data['base_iva_8'] += (double)$traslado['Base'];
                        }
                        if ($traslado['Impuesto'] == '002' && $traslado['TipoFactor'] == 'Tasa' && (double)$traslado['TasaOCuota'] == 0) {
                            $data['base_iva_0'] += (double)$traslado['Base'];
                        }
                        if ($traslado['Impuesto'] == '002' && $traslado['TipoFactor'] == 'Exento') {
                            $data['base_iva_exempt'] += (double)$traslado['Base'];
                        }
                        if ($traslado['Impuesto'] == '003') {
                            $data['base_ieps'] += (double)$traslado['Base'];
                        }
                    }
                }

                //Retenciones
                $tmp_nodo_retenciones = [];
                $retenciones = [];
                if(!empty($nodo_concepto['cfdi:Impuestos']['cfdi:Retenciones']['cfdi:Retencion'])){
                    $tmp_nodo_retenciones = $nodo_concepto['cfdi:Impuestos']['cfdi:Retenciones']['cfdi:Retencion'];
                }elseif(!empty($nodo_concepto['cfdi:Impuestos']['cfdi:Retenciones']['Retencion'])){
                    $tmp_nodo_retenciones = $nodo_concepto['cfdi:Impuestos']['cfdi:Retenciones']['Retencion'];
                }
                if(!empty($tmp_nodo_retenciones)){
                    if(!empty($tmp_nodo_retenciones['@attributes'])){
                        $retenciones[] = $tmp_nodo_retenciones['@attributes'];

                    }elseif(!empty($tmp_nodo_retenciones)){
                        foreach ($tmp_nodo_retenciones as $tmp) {
                            $retenciones[] = $tmp['@attributes'];
                        }
                    }
                }
                if(!empty($retenciones)){
                    foreach($retenciones as $retencion){
                        $retencion['TasaOCuota'] = $retencion['TasaOCuota'] ?? null;
                        $retencion['TipoFactor'] = $retencion['TipoFactor'] ?? null;
                        $retencion['Base'] = !empty($retencion['Base']) ? (double)$retencion['Base'] : (!empty($retencion['base']) ? (double)$retencion['base'] : 0);

                        if ($retencion['Impuesto'] == '002') {
                            $data['base_ret_iva'] += (double)$retencion['Base'];
                        }
                        if ($retencion['Impuesto'] == '001') {
                            $data['base_ret_isr'] += (double)$retencion['Base'];
                        }
                    }
                }
            }
        }

        return $data;
    }

    public static function parseXmlToArrayCfdi33BaseImpuestos2($file_xml){

        $taxes = Tax::populateSelect()->get();
        if(!empty($taxes)){
            foreach($taxes as $key => $tax){
                $taxes[$key]->amount_base = 0;
            }
        }

        $xml = new \DOMDocument('1.0', 'UTF-8');
        //Revisa y limpia archivo
        $file_xml = \Storage::get($file_xml);
        $file_xml = preg_replace('/[^\x{0009}\x{000a}\x{000d}\x{0020}-\x{D7FF}\x{E000}-\x{FFFD}]+/u', ' ', $file_xml);
        //$file_xml = \CfdiUtils\Cleaner\Cleaner::staticClean($file_xml);
        libxml_use_internal_errors(true);
        $xml->loadXML($file_xml);
        $xml_array = self::domNodeToArray($xml->getElementsByTagName('Comprobante')->item(0));
        //
        $comprobante = $xml_array['@attributes'];
        if(in_array($comprobante['TipoDeComprobante'], ['I','E']) && isset($xml_array['cfdi:Conceptos'])){

            //Lista de conceptos
            $tmp_nodo_conceptos = [];
            $nodo_conceptos = [];
            if(!empty($xml_array['cfdi:Conceptos']['cfdi:Concepto'])){
                $tmp_nodo_conceptos[] = $xml_array['cfdi:Conceptos']['cfdi:Concepto'];

            }elseif(!empty($xml_array['cfdi:Conceptos']['Concepto'])){
                $tmp_nodo_conceptos[] = $xml_array['cfdi:Conceptos']['Concepto'];
            }
            if(!empty($tmp_nodo_conceptos)){
                foreach($tmp_nodo_conceptos as $tmp_nodo_concepto){
                    if(!empty($tmp_nodo_concepto['@attributes'])){
                        $nodo_conceptos[] = $tmp_nodo_concepto;
                    }elseif(!empty($tmp_nodo_concepto)){
                        foreach ($tmp_nodo_concepto as $tmp) {
                            $nodo_conceptos[] = $tmp;
                        }
                    }
                }
            }

            //Impuestos por concepto
            foreach($nodo_conceptos as $nodo_concepto){
                //Traslados
                $tmp_nodo_traslados = [];
                $traslados = [];
                if(!empty($nodo_concepto['cfdi:Impuestos']['cfdi:Traslados']['cfdi:Traslado'])){
                    $tmp_nodo_traslados = $nodo_concepto['cfdi:Impuestos']['cfdi:Traslados']['cfdi:Traslado'];

                }elseif(!empty($nodo_concepto['cfdi:Impuestos']['cfdi:Traslados']['Traslado'])){
                    $tmp_nodo_traslados = $nodo_concepto['cfdi:Impuestos']['cfdi:Traslados']['Traslado'];
                }
                if(!empty($tmp_nodo_traslados)){
                    if(!empty($tmp_nodo_traslados['@attributes'])){
                        $traslados[] = $tmp_nodo_traslados['@attributes'];

                    }elseif(!empty($tmp_nodo_traslados)){
                        foreach ($tmp_nodo_traslados as $tmp) {
                            $traslados[] = $tmp['@attributes'];
                        }
                    }
                }
                if (!empty($traslados)) {
                    foreach ($traslados as $traslado) {
                        $tasaOCuota = (double) ($traslado['TasaOCuota'] ?? null);
                        $traslado['TasaOCuota'] = $traslado['TasaOCuota'] ?? null;
                        $traslado['TipoFactor'] = $traslado['TipoFactor'] ?? null;
                        $traslado['Base'] = !empty($traslado['Base']) ? (double)$traslado['Base'] : (!empty($traslado['base']) ? (double)$traslado['base'] : 0);

                        if(!empty($taxes)){
                            foreach($taxes as $key => $tax){
                                $rate = abs($tax->rate / 100);
                                if ($traslado['Impuesto'] == $tax->code && $traslado['TipoFactor'] == $tax->factor && abs($tasaOCuota - $rate) <= 0.001 && $tax->type == 'T') {
                                    $taxes[$key]->amount_base += (double)$traslado['Base'];
                                }
                            }
                        }
                    }
                }


                //Retenciones
                $tmp_nodo_retenciones = [];
                $retenciones = [];
                if(!empty($nodo_concepto['cfdi:Impuestos']['cfdi:Retenciones']['cfdi:Retencion'])){
                    $tmp_nodo_retenciones = $nodo_concepto['cfdi:Impuestos']['cfdi:Retenciones']['cfdi:Retencion'];
                }elseif(!empty($nodo_concepto['cfdi:Impuestos']['cfdi:Retenciones']['Retencion'])){
                    $tmp_nodo_retenciones = $nodo_concepto['cfdi:Impuestos']['cfdi:Retenciones']['Retencion'];
                }
                if(!empty($tmp_nodo_retenciones)){
                    if(!empty($tmp_nodo_retenciones['@attributes'])){
                        $retenciones[] = $tmp_nodo_retenciones['@attributes'];

                    }elseif(!empty($tmp_nodo_retenciones)){
                        foreach ($tmp_nodo_retenciones as $tmp) {
                            $retenciones[] = $tmp['@attributes'];
                        }
                    }
                }
                if(!empty($retenciones)){
                    foreach($retenciones as $retencion){
                        $tasaOCuota = (double) ($retencion['TasaOCuota'] ?? null);
                        $retencion['TasaOCuota'] = $retencion['TasaOCuota'] ?? null;
                        $retencion['TipoFactor'] = $retencion['TipoFactor'] ?? null;
                        $retencion['Base'] = !empty($retencion['Base']) ? (double)$retencion['Base'] : (!empty($retencion['base']) ? (double)$retencion['base'] : 0);

                        if($tasaOCuota == 0.106666){
                            $tasaOCuota = 0.106667;
                        }
                        if(!empty($taxes)){
                            foreach($taxes as $key => $tax){
                                $rate = abs($tax->rate / 100);
                                if ($retencion['Impuesto'] == $tax->code && $retencion['TipoFactor'] == $tax->factor && abs($tasaOCuota - $rate) <= 0.001 && $tax->type == 'R') {
                                    $taxes[$key]->amount_base += (double)$retencion['Base'];
                                }
                            }
                        }
                    }
                }
            }
        }

        return $taxes;
    }

    public static function parseXmlToArrayCfdi33Impuestos($file_xml){
        $data = [
            'version' => '',
            'iva' => 0,
            'iva_16' => 0,
            'iva_15' => 0,
            'iva_11' => 0,
            'iva_10' => 0,
            'iva_8' => 0,
            'ieps' => 0,
            'ret_iva' => 0,
            'ret_isr' => 0
        ];
        $xml = new \DOMDocument('1.0', 'UTF-8');
        //Revisa y limpia archivo
        $file_xml = \Storage::get($file_xml);
        $file_xml = preg_replace('/[^\x{0009}\x{000a}\x{000d}\x{0020}-\x{D7FF}\x{E000}-\x{FFFD}]+/u', ' ', $file_xml);
        //$file_xml = \CfdiUtils\Cleaner\Cleaner::staticClean($file_xml);
        libxml_use_internal_errors(true);
        $xml->loadXML($file_xml);
        $xml_array = self::domNodeToArray($xml->getElementsByTagName('Comprobante')->item(0));

        //
        $comprobante = $xml_array['@attributes'];
        $data['version'] = $comprobante['Version'] ?? ($comprobante['version'] ?? '');

        //Traslados
        $tmp_nodo_traslados = [];
        $traslados = [];
        if(!empty($xml_array['cfdi:Impuestos']['cfdi:Traslados']['cfdi:Traslado'])){
            $tmp_nodo_traslados = $xml_array['cfdi:Impuestos']['cfdi:Traslados']['cfdi:Traslado'];

        }elseif(!empty($xml_array['cfdi:Impuestos']['cfdi:Traslados']['Traslado'])){
            $tmp_nodo_traslados = $xml_array['cfdi:Impuestos']['cfdi:Traslados']['Traslado'];
        }
        if(!empty($tmp_nodo_traslados)){
            if(!empty($tmp_nodo_traslados['@attributes'])){
                $traslados[] = $tmp_nodo_traslados['@attributes'];

            }elseif(!empty($tmp_nodo_traslados)){
                foreach ($tmp_nodo_traslados as $tmp) {
                    $traslados[] = $tmp['@attributes'];
                }
            }
        }
        if(!empty($traslados)){
            foreach($traslados as $traslado){
                $traslado['TasaOCuota'] = $traslado['TasaOCuota'] ?? null;
                $traslado['TipoFactor'] = $traslado['TipoFactor'] ?? null;
                $traslado['Base'] = !empty($traslado['Base']) ? (double)$traslado['Base'] : (!empty($traslado['base']) ? (double)$traslado['base'] : 0);


                if ($traslado['Impuesto'] == '002' || $traslado['Impuesto'] == 'IVA') {
                    $data['iva'] += !empty($traslado['Importe']) ? (double)$traslado['Importe'] : (!empty($traslado['importe']) ? (double)$traslado['importe'] : 0);
                }
                if ($traslado['Impuesto'] == '002' && $traslado['TipoFactor'] == 'Tasa' && (double)$traslado['TasaOCuota'] == 0.16) {
                    $data['iva_16'] += !empty($traslado['Importe']) ? (double)$traslado['Importe'] : (!empty($traslado['importe']) ? (double)$traslado['importe'] : 0);
                }
                if ($traslado['Impuesto'] == 'IVA' && (double)$traslado['Tasa'] == 16) {
                    $data['iva_16'] += !empty($traslado['Importe']) ? (double)$traslado['Importe'] : (!empty($traslado['importe']) ? (double)$traslado['importe'] : 0);
                }
                if ($traslado['Impuesto'] == '002' && $traslado['TipoFactor'] == 'Tasa' && (double)$traslado['TasaOCuota'] == 0.15) {
                    $data['iva_15'] += !empty($traslado['Importe']) ? (double)$traslado['Importe'] : (!empty($traslado['importe']) ? (double)$traslado['importe'] : 0);
                }
                if ($traslado['Impuesto'] == 'IVA' && (double)$traslado['Tasa'] == 15) {
                    $data['iva_15'] += !empty($traslado['Importe']) ? (double)$traslado['Importe'] : (!empty($traslado['importe']) ? (double)$traslado['importe'] : 0);
                }
                if ($traslado['Impuesto'] == '002' && $traslado['TipoFactor'] == 'Tasa' && (double)$traslado['TasaOCuota'] == 0.11) {
                    $data['iva_11'] += !empty($traslado['Importe']) ? (double)$traslado['Importe'] : (!empty($traslado['importe']) ? (double)$traslado['importe'] : 0);
                }
                if ($traslado['Impuesto'] == 'IVA' && (double)$traslado['Tasa'] == 11) {
                    $data['iva_11'] += !empty($traslado['Importe']) ? (double)$traslado['Importe'] : (!empty($traslado['importe']) ? (double)$traslado['importe'] : 0);
                }
                if ($traslado['Impuesto'] == '002' && $traslado['TipoFactor'] == 'Tasa' && (double)$traslado['TasaOCuota'] == 0.10) {
                    $data['iva_10'] += !empty($traslado['Importe']) ? (double)$traslado['Importe'] : (!empty($traslado['importe']) ? (double)$traslado['importe'] : 0);
                }
                if ($traslado['Impuesto'] == 'IVA' && (double)$traslado['Tasa'] == 10) {
                    $data['iva_10'] += !empty($traslado['Importe']) ? (double)$traslado['Importe'] : (!empty($traslado['importe']) ? (double)$traslado['importe'] : 0);
                }
                if ($traslado['Impuesto'] == '002' && $traslado['TipoFactor'] == 'Tasa' && (double)$traslado['TasaOCuota'] == 0.08) {
                    $data['iva_8'] += !empty($traslado['Importe']) ? (double)$traslado['Importe'] : (!empty($traslado['importe']) ? (double)$traslado['importe'] : 0);
                }
                if ($traslado['Impuesto'] == 'IVA' && (double)$traslado['Tasa'] == 8) {
                    $data['iva_8'] += !empty($traslado['Importe']) ? (double)$traslado['Importe'] : (!empty($traslado['importe']) ? (double)$traslado['importe'] : 0);
                }
                if ($traslado['Impuesto'] == '003') {
                    $data['ieps'] += !empty($traslado['Importe']) ? (double)$traslado['Importe'] : (!empty($traslado['importe']) ? (double)$traslado['importe'] : 0);
                }
                if ($traslado['Impuesto'] == 'IEPS') {
                    $data['ieps'] += !empty($traslado['Importe']) ? (double)$traslado['Importe'] : (!empty($traslado['importe']) ? (double)$traslado['importe'] : 0);
                }
            }
        }

        //Retenciones
        $tmp_nodo_retenciones = [];
        $retenciones = [];
        if(!empty($xml_array['cfdi:Impuestos']['cfdi:Retenciones']['cfdi:Retencion'])){
            $tmp_nodo_retenciones = $xml_array['cfdi:Impuestos']['cfdi:Retenciones']['cfdi:Retencion'];
        }elseif(!empty($xml_array['cfdi:Impuestos']['cfdi:Retenciones']['Retencion'])){
            $tmp_nodo_retenciones = $xml_array['cfdi:Impuestos']['cfdi:Retenciones']['Retencion'];
        }
        if(!empty($tmp_nodo_retenciones)){
            if(!empty($tmp_nodo_retenciones['@attributes'])){
                $retenciones[] = $tmp_nodo_retenciones['@attributes'];

            }elseif(!empty($tmp_nodo_retenciones)){
                foreach ($tmp_nodo_retenciones as $tmp) {
                    $retenciones[] = $tmp['@attributes'];
                }
            }
        }
        if(!empty($retenciones)){
            foreach($retenciones as $retencion){
                $retencion['TasaOCuota'] = $retencion['TasaOCuota'] ?? null;
                $retencion['TipoFactor'] = $retencion['TipoFactor'] ?? null;
                $retencion['Base'] = !empty($retencion['Base']) ? (double)$retencion['Base'] : (!empty($retencion['base']) ? (double)$retencion['base'] : 0);

                if ($retencion['Impuesto'] == '002') {
                    $data['ret_iva'] += !empty($retencion['Importe']) ? (double)$retencion['Importe'] : (!empty($retencion['importe']) ? (double)$retencion['importe'] : 0);
                }
                if ($retencion['Impuesto'] == 'IVA') {
                    $data['ret_iva'] += !empty($retencion['Importe']) ? (double)$retencion['Importe'] : (!empty($retencion['importe']) ? (double)$retencion['importe'] : 0);
                }
                if ($retencion['Impuesto'] == '001') {
                    $data['ret_isr'] += !empty($retencion['Importe']) ? (double)$retencion['Importe'] : (!empty($retencion['importe']) ? (double)$retencion['importe'] : 0);
                }
                if ($retencion['Impuesto'] == 'ISR') {
                    $data['ret_isr'] += !empty($retencion['Importe']) ? (double)$retencion['Importe'] : (!empty($retencion['importe']) ? (double)$retencion['importe'] : 0);
                }
            }
        }

        return $data;
    }

    public static function parseXmlToArrayCfdi40PagosImpuestos($file_xml){
        $data = [
            'iva' => 0,
            'iva_16' => 0,
            'iva_15' => 0,
            'iva_11' => 0,
            'iva_10' => 0,
            'iva_8' => 0,
            'ieps' => 0,
            'ret_iva' => 0,
            'ret_isr' => 0
        ];
        $xml = new \DOMDocument('1.0', 'UTF-8');
        //Revisa y limpia archivo
        $file_xml = \Storage::get($file_xml);
        $file_xml = preg_replace('/[^\x{0009}\x{000a}\x{000d}\x{0020}-\x{D7FF}\x{E000}-\x{FFFD}]+/u', ' ', $file_xml);
        //$file_xml = \CfdiUtils\Cleaner\Cleaner::staticClean($file_xml);
        libxml_use_internal_errors(true);
        $xml->loadXML($file_xml);
        $xml_array = self::domNodeToArray($xml->getElementsByTagName('Comprobante')->item(0));

        //Pagos
        $tmp_nodo_pagos = [];
        $nodo_pagos = [];
        $pagos = [];
        $traslados = [];
        $retenciones = [];
        if(!empty($xml_array['cfdi:Complemento']['pago20:Pagos']['pago20:Pago'])){
            $tmp_nodo_pagos = $xml_array['cfdi:Complemento']['pago20:Pagos']['pago20:Pago'];
        }elseif(!empty($xml_array['cfdi:Complemento']['pago20:Pagos']['Pago'])){
            $tmp_nodo_pagos = $xml_array['cfdi:Complemento']['pago20:Pagos']['Pago'];
        }
        if(!empty($tmp_nodo_pagos)){
            if(!empty($tmp_nodo_pagos['@attributes'])){
                $nodo_pagos[] = $tmp_nodo_pagos;
                $pagos[] = $tmp_nodo_pagos['@attributes'];

            }elseif(!empty($tmp_nodo_pagos)){
                foreach ($tmp_nodo_pagos as $tmp) {
                    $nodo_pagos[] = $tmp;
                    $pagos[] = $tmp['@attributes'];
                }
            }
        }

        if(!empty($nodo_pagos)){
            foreach($nodo_pagos as $nodo_pago){
                $pago = $nodo_pago['@attributes'];

                //Traslados
                $tmp_nodo_traslados = [];
                if(!empty($nodo_pago['pago20:ImpuestosP']['pago20:TrasladosP']['pago20:TrasladoP'])){
                    $tmp_nodo_traslados = $nodo_pago['pago20:ImpuestosP']['pago20:TrasladosP']['pago20:TrasladoP'];
                }
                if(!empty($tmp_nodo_traslados)){
                    if(!empty($tmp_nodo_traslados['@attributes'])){
                        $importe = !empty($tmp_nodo_traslados['@attributes']['ImporteP']) ? (double)$tmp_nodo_traslados['@attributes']['ImporteP'] : (!empty($tmp_nodo_traslados['@attributes']['importeP']) ? (double)$tmp_nodo_traslados['@attributes']['importeP'] : 0);
                        if(count($nodo_pagos) > 1){
                            $importe *= (double)($pago['TipoCambioP'] ?? 1);
                            if(!empty($tmp_nodo_traslados['@attributes']['ImporteP']))
                                $tmp_nodo_traslados['@attributes']['ImporteP'] = $importe;
                            if(!empty($tmp_nodo_traslados['@attributes']['importeP']))
                                $tmp_nodo_traslados['@attributes']['importeP'] = $importe;
                        }
                        $traslados[] = $tmp_nodo_traslados['@attributes'];

                    }elseif(!empty($tmp_nodo_traslados)){
                        foreach ($tmp_nodo_traslados as $tmp) {
                            $importe = !empty($tmp['@attributes']['ImporteP']) ? (double)$tmp['@attributes']['ImporteP'] : (!empty($tmp['@attributes']['importeP']) ? (double)$tmp['@attributes']['importeP'] : 0);
                            if(count($nodo_pagos) > 1){
                                $importe *= (double)($pago['TipoCambioP'] ?? 1);
                                if(!empty($tmp['@attributes']['ImporteP']))
                                    $tmp['@attributes']['ImporteP'] = $importe;
                                if(!empty($tmp['@attributes']['importeP']))
                                    $tmp['@attributes']['importeP'] = $importe;
                            }
                            $traslados[] = $tmp['@attributes'];
                        }
                    }
                }
                //Retenciones
                $tmp_nodo_retenciones = [];
                if(!empty($nodo_pago['pago20:ImpuestosP']['pago20:RetencionesP']['pago20:RetencionP'])){
                    $tmp_nodo_retenciones = $nodo_pago['pago20:ImpuestosP']['pago20:RetencionesP']['pago20:RetencionP'];
                }
                if(!empty($tmp_nodo_retenciones)){
                    if(!empty($tmp_nodo_retenciones['@attributes'])){
                        $retenciones[] = $tmp_nodo_retenciones['@attributes'];

                    }elseif(!empty($tmp_nodo_retenciones)){
                        foreach ($tmp_nodo_retenciones as $tmp) {
                            $retenciones[] = $tmp['@attributes'];
                        }
                    }
                }


            }
        }

        //Traslados
        if(!empty($traslados)){
            foreach($traslados as $traslado){
                $traslado['TasaOCuotaP'] = $traslado['TasaOCuotaP'] ?? null;
                $traslado['TipoFactorP'] = $traslado['TipoFactorP'] ?? null;
                $traslado['BaseP'] = !empty($traslado['BaseP']) ? (double)$traslado['BaseP'] : (!empty($traslado['BaseP']) ? (double)$traslado['BaseP'] : 0);

                if ($traslado['ImpuestoP'] == '002' && $traslado['TipoFactorP'] == 'Tasa' && (double)$traslado['TasaOCuotaP'] == 0.16) {
                    $data['iva_16'] += !empty($traslado['ImporteP']) ? (double)$traslado['ImporteP'] : (!empty($traslado['importeP']) ? (double)$traslado['importeP'] : 0);
                }
                if ($traslado['ImpuestoP'] == '002' && $traslado['TipoFactorP'] == 'Tasa' && (double)$traslado['TasaOCuotaP'] == 0.15) {
                    $data['iva_15'] += !empty($traslado['ImporteP']) ? (double)$traslado['ImporteP'] : (!empty($traslado['importeP']) ? (double)$traslado['importeP'] : 0);
                }
                if ($traslado['ImpuestoP'] == '002' && $traslado['TipoFactorP'] == 'Tasa' && (double)$traslado['TasaOCuotaP'] == 0.11) {
                    $data['iva_11'] += !empty($traslado['ImporteP']) ? (double)$traslado['ImporteP'] : (!empty($traslado['importeP']) ? (double)$traslado['importeP'] : 0);
                }
                if ($traslado['ImpuestoP'] == '002' && $traslado['TipoFactorP'] == 'Tasa' && (double)$traslado['TasaOCuotaP'] == 0.10) {
                    $data['iva_10'] += !empty($traslado['ImporteP']) ? (double)$traslado['ImporteP'] : (!empty($traslado['importeP']) ? (double)$traslado['importeP'] : 0);
                }
                if ($traslado['ImpuestoP'] == '002' && $traslado['TipoFactorP'] == 'Tasa' && (double)$traslado['TasaOCuotaP'] == 0.08) {
                    $data['iva_8'] += !empty($traslado['ImporteP']) ? (double)$traslado['ImporteP'] : (!empty($traslado['importeP']) ? (double)$traslado['importeP'] : 0);
                }
                if ($traslado['ImpuestoP'] == '003') {
                    $data['ieps'] += !empty($traslado['ImporteP']) ? (double)$traslado['ImporteP'] : (!empty($traslado['importeP']) ? (double)$traslado['importeP'] : 0);
                }
                if ($traslado['ImpuestoP'] == 'IEPS') {
                    $data['ieps'] += !empty($traslado['ImporteP']) ? (double)$traslado['ImporteP'] : (!empty($traslado['importeP']) ? (double)$traslado['importeP'] : 0);
                }
            }
        }

        //Retenciones
        if(!empty($retenciones)){
            foreach($retenciones as $retencion){
                $retencion['TasaOCuotaP'] = $retencion['TasaOCuotaP'] ?? null;
                $retencion['TipoFactorP'] = $retencion['TipoFactorP'] ?? null;
                $retencion['BaseP'] = !empty($retencion['BaseP']) ? (double)$retencion['BaseP'] : (!empty($retencion['BaseP']) ? (double)$retencion['BaseP'] : 0);

                if ($retencion['ImpuestoP'] == '002') {
                    $data['ret_iva'] += !empty($retencion['ImporteP']) ? (double)$retencion['ImporteP'] : (!empty($retencion['importeP']) ? (double)$retencion['importeP'] : 0);
                }
                if ($retencion['ImpuestoP'] == '001') {
                    $data['ret_isr'] += !empty($retencion['ImporteP']) ? (double)$retencion['ImporteP'] : (!empty($retencion['importeP']) ? (double)$retencion['importeP'] : 0);
                }
            }
        }

        return $data;
    }

    public static function parseXmlToArrayCfdi33PagosImpuestos($file_xml, $path_files = ''){
        $data = [
            'iva' => 0,
            'iva_16' => 0,
            'iva_15' => 0,
            'iva_11' => 0,
            'iva_10' => 0,
            'iva_8' => 0,
            'ieps' => 0,
            'ret_iva' => 0,
            'ret_isr' => 0
        ];
        $xml = new \DOMDocument('1.0', 'UTF-8');
        //Revisa y limpia archivo
        $file_xml = \Storage::get($file_xml);
        $file_xml = preg_replace('/[^\x{0009}\x{000a}\x{000d}\x{0020}-\x{D7FF}\x{E000}-\x{FFFD}]+/u', ' ', $file_xml);
        //$file_xml = \CfdiUtils\Cleaner\Cleaner::staticClean($file_xml);
        libxml_use_internal_errors(true);
        $xml->loadXML($file_xml);
        $xml_array = self::domNodeToArray($xml->getElementsByTagName('Comprobante')->item(0));

        //Pagos
        $tmp_nodo_pagos = [];
        $nodo_pagos = [];
        $pagos = [];
        $traslados = [];
        $retenciones = [];
        if(!empty($xml_array['cfdi:Complemento']['pago10:Pagos']['pago10:Pago'])){
            $tmp_nodo_pagos = $xml_array['cfdi:Complemento']['pago10:Pagos']['pago10:Pago'];
        }elseif(!empty($xml_array['cfdi:Complemento']['pago10:Pagos']['Pago'])){
            $tmp_nodo_pagos = $xml_array['cfdi:Complemento']['pago10:Pagos']['Pago'];
        }
        if(!empty($tmp_nodo_pagos)){
            if(!empty($tmp_nodo_pagos['@attributes'])){
                $nodo_pagos[] = $tmp_nodo_pagos;
                $pagos[] = $tmp_nodo_pagos['@attributes'];

            }elseif(!empty($tmp_nodo_pagos)){
                foreach ($tmp_nodo_pagos as $tmp) {
                    $nodo_pagos[] = $tmp;
                    $pagos[] = $tmp['@attributes'];
                }
            }
        }

        if(!empty($nodo_pagos)){
            foreach($nodo_pagos as $nodo_pago){
                //
                $reconcileds = [];
                $pago = $nodo_pago['@attributes'];

                //
                if(!empty($nodo_pago['pago10:DoctoRelacionado'])){
                    $tmp_nodo_reconcileds = $nodo_pago['pago10:DoctoRelacionado'];
                }elseif(!empty($nodo_pago['DoctoRelacionado'])){
                    $tmp_nodo_reconcileds = $nodo_pago['DoctoRelacionado'];
                }elseif(!empty($nodo_pago['pago20:DoctoRelacionado'])){
                    $tmp_nodo_reconcileds = $nodo_pago['pago20:DoctoRelacionado'];
                }
                if(!empty($tmp_nodo_reconcileds)){
                    if(!empty($tmp_nodo_reconcileds['@attributes'])){
                        $nodo_reconcileds[] = $tmp_nodo_reconcileds;
                        $reconcileds[] = $tmp_nodo_reconcileds['@attributes'];
                    }elseif(!empty($tmp_nodo_reconcileds)){
                        foreach ($tmp_nodo_reconcileds as $tmp) {
                            $nodo_reconcileds[] = $tmp;
                            $reconcileds[] = $tmp['@attributes'];
                        }
                    }
                }

                //
                if(!empty($reconcileds)){
                    foreach($reconcileds as $reconciled){
                        $uuid = $reconciled['IdDocumento'];
                        $customer_invoice_cfdi = CustomerInvoiceCfdi::where('uuid','=',$reconciled['IdDocumento'])->with('customerInvoice')->first();
                        $cfdi_download = CfdiDownload::where('uuid','=',$reconciled['IdDocumento'])->first();

                        $amountReconciled = $reconciled['ImpPagado'] ?? 0;
                        $currency_value = (!empty($reconciled['TipoCambioDR']) && (($pago['MonedaP'] ?? '') != ($reconciled['MonedaDR'] ?? ''))) ? 1 / $reconciled['TipoCambioDR'] : 1;

                        $percentPayment = 0;
                        if(!empty($cfdi_download)){
                            $percentPayment = $amountReconciled / $cfdi_download->amount_total;
                            if (!empty($cfdi_download->file_xml)) {
                                $file_xml = $path_files . $cfdi_download->file_xml;
                                if (\Storage::exists($file_xml) && Helper::validateXmlToArrayCfdi33($file_xml)) {
                                    $tmp = Helper::parseXmlToArrayCfdi33Impuestos($file_xml);
                                    $data['iva_16'] += round($tmp['iva_16'] * $percentPayment * $currency_value, 2);
                                    $data['iva_15'] = round($tmp['iva_15'] * $percentPayment * $currency_value, 2);
                                    $data['iva_11 '] = round($tmp['iva_11'] * $percentPayment * $currency_value, 2);
                                    $data['iva_10'] = round($tmp['iva_10'] * $percentPayment * $currency_value, 2);
                                    $data['iva_8'] = round($tmp['iva_8'] * $percentPayment * $currency_value, 2);
                                    $data['ieps'] = round($tmp['ieps'] * $percentPayment * $currency_value, 2);
                                    $data['ret_iva'] = round($tmp['ret_iva'] * $percentPayment * $currency_value, 2);
                                    $data['ret_isr'] = round($tmp['ret_isr'] * $percentPayment * $currency_value, 2);
                                }
                            }
                        }elseif(!empty($customer_invoice_cfdi)){
                            $percentPayment = $amountReconciled / $customer_invoice_cfdi->customerInvoice->amount_total;
                            if (!empty($customer_invoice_cfdi->file_xml_pac)) {
                                $path_files = '';

                                if($customer_invoice_cfdi->customerInvoice->documentType->code == 'customer.invoice'){
                                    $path_files = CustomerInvoice::PATH_XML_FILES_CI;
                                }
                                if($customer_invoice_cfdi->customerInvoice->documentType->code == 'customer.credit_note'){
                                    $path_files = CustomerInvoice::PATH_XML_FILES_CCN;
                                }
                                if($customer_invoice_cfdi->customerInvoice->documentType->code == 'customer.lease'){
                                    $path_files = CustomerInvoice::PATH_XML_FILES_LEA;
                                }
                                if($customer_invoice_cfdi->customerInvoice->documentType->code == 'customer.fee'){
                                    $path_files = CustomerInvoice::PATH_XML_FILES_FEE;
                                }
                                if($customer_invoice_cfdi->customerInvoice->documentType->code == 'customer.transfer'){
                                    $path_files = CustomerInvoice::PATH_XML_FILES_CTR;
                                    if($customer_invoice_cfdi->customerInvoice->document_type2 == 'invoice'){
                                        $path_files = CustomerInvoice::PATH_XML_FILES_CI;
                                    }
                                }
                                $path_files = Helper::setDirectory($path_files, $customer_invoice_cfdi->customerInvoice->company_id) . '/';
                                $file_xml = $path_files . $customer_invoice_cfdi->file_xml_pac;
                                if (\Storage::exists($file_xml) && Helper::validateXmlToArrayCfdi33($file_xml)) {
                                    $tmp = Helper::parseXmlToArrayCfdi33Impuestos($file_xml);
                                    $data['iva_16'] += round($tmp['iva_16'] * $percentPayment * $currency_value, 2);
                                    $data['iva_15'] = round($tmp['iva_15'] * $percentPayment * $currency_value, 2);
                                    $data['iva_11 '] = round($tmp['iva_11'] * $percentPayment * $currency_value, 2);
                                    $data['iva_10'] = round($tmp['iva_10'] * $percentPayment * $currency_value, 2);
                                    $data['iva_8'] = round($tmp['iva_8'] * $percentPayment * $currency_value, 2);
                                    $data['ieps'] = round($tmp['ieps'] * $percentPayment * $currency_value, 2);
                                    $data['ret_iva'] = round($tmp['ret_iva'] * $percentPayment * $currency_value, 2);
                                    $data['ret_isr'] = round($tmp['ret_isr'] * $percentPayment * $currency_value, 2);
                                }
                            }

                        }
                    }
                }


            }
        }

        return $data;
    }

    public static function parseXmlToArrayCfdi33EmployeePayroll($file_xml){
        $data = [];
        $xml = new \DOMDocument('1.0', 'UTF-8');
        //Revisa y limpia archivo
        $file_xml = \Storage::get($file_xml);
        $file_xml = preg_replace('/[^\x{0009}\x{000a}\x{000d}\x{0020}-\x{D7FF}\x{E000}-\x{FFFD}]+/u', ' ', $file_xml);
        //$file_xml = \CfdiUtils\Cleaner\Cleaner::staticClean($file_xml);
        libxml_use_internal_errors(true);
        $xml->loadXML($file_xml);
        $xml_array = self::domNodeToArray($xml->getElementsByTagName('Comprobante')->item(0));

        $data['employer_register'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Emisor']['@attributes']['RegistroPatronal']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Emisor']['@attributes']['RegistroPatronal'] : '';
        $data['employee_code'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['NumEmpleado']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['NumEmpleado'] : '';
        $data['curp'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['Curp']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['Curp'] : '';
        $data['nss'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['NumSeguridadSocial']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['NumSeguridadSocial'] : '';
        $data['recruitment_regime'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['TipoRegimen']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['TipoRegimen'] : '';
        $data['department'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['Departamento']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['Departamento'] : '';
        $data['job'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['Puesto']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['Puesto'] : '';
        $data['job_risk_classe'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['RiesgoPuesto']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['RiesgoPuesto'] : '';
        $data['date_start_work'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['FechaInicioRelLaboral']) ? self::convertSqlToDate($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['FechaInicioRelLaboral']) : '';
        $data['antiquity'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['Antigüedad']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['Antigüedad'] : '';
        $data['contract_type'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['TipoContrato']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['TipoContrato'] : '';
        $data['workday_type'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['TipoJornada']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['TipoJornada'] : '';
        $data['frequency_payment'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['PeriodicidadPago']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['PeriodicidadPago'] : '';
        $data['bank'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['Banco']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['Banco'] : '';
        $data['bank_account'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['CuentaBancaria']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['CuentaBancaria'] : '';
        $data['sdi'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['SalarioDiarioIntegrado']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['SalarioDiarioIntegrado'] : '';
        $data['base_salary'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['SalarioBaseCotApor']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Receptor']['@attributes']['SalarioBaseCotApor'] : '';
        $data['date_start_payment'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['@attributes']['FechaInicialPago']) ? self::convertSqlToDate($xml_array['cfdi:Complemento']['nomina12:Nomina']['@attributes']['FechaInicialPago']) : '';
        $data['date_end_payment'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['@attributes']['FechaFinalPago']) ? self::convertSqlToDate($xml_array['cfdi:Complemento']['nomina12:Nomina']['@attributes']['FechaFinalPago']) : '';
        $data['date_payment'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['@attributes']['FechaPago']) ? self::convertSqlToDate($xml_array['cfdi:Complemento']['nomina12:Nomina']['@attributes']['FechaPago']) : '';
        $data['payment_days'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['@attributes']['NumDiasPagados']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['@attributes']['NumDiasPagados'] : '';
        $data['amount_total_perceptions'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['@attributes']['TotalPercepciones']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['@attributes']['TotalPercepciones'] : '';
        $data['amount_total_deductions'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['@attributes']['TotalDeducciones']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['@attributes']['TotalDeducciones'] : '';
        $data['amount_total_other_payment_types'] = !empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['@attributes']['TotalOtrosPagos']) ? $xml_array['cfdi:Complemento']['nomina12:Nomina']['@attributes']['TotalOtrosPagos'] : '';

        //Percepciones
        $tmp = [];
        if(!empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Percepciones']['nomina12:Percepcion'])){
            $perceptions = $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Percepciones']['nomina12:Percepcion'];
            if(!empty($perceptions['@attributes'])){
                $perception = $perceptions;
                $tmp[$perception['@attributes']['TipoPercepcion'] . '_1'] = isset($tmp[$perception['@attributes']['TipoPercepcion'] . '_1']) ? $tmp[$perception['@attributes']['TipoPercepcion'] . '_1']+$perception['@attributes']['ImporteGravado'] : $perception['@attributes']['ImporteGravado'];
                $tmp[$perception['@attributes']['TipoPercepcion'] . '_2'] = isset($tmp[$perception['@attributes']['TipoPercepcion'] . '_2']) ? $tmp[$perception['@attributes']['TipoPercepcion'] . '_2']+$perception['@attributes']['ImporteExento'] : $perception['@attributes']['ImporteExento'];
            }else {
                foreach ($perceptions as $perception) {
                    $tmp[$perception['@attributes']['TipoPercepcion'] . '_1'] = isset($tmp[$perception['@attributes']['TipoPercepcion'] . '_1']) ? $tmp[$perception['@attributes']['TipoPercepcion'] . '_1']+$perception['@attributes']['ImporteGravado'] : $perception['@attributes']['ImporteGravado'];
                    $tmp[$perception['@attributes']['TipoPercepcion'] . '_2'] = isset($tmp[$perception['@attributes']['TipoPercepcion'] . '_2']) ? $tmp[$perception['@attributes']['TipoPercepcion'] . '_2']+$perception['@attributes']['ImporteExento'] : $perception['@attributes']['ImporteExento'];
                }
            }
        }
        $data['perceptions'] = $tmp;
        //Deducciones
        $tmp = [];
        if(!empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Deducciones']['nomina12:Deduccion'])){
            $deductions = $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:Deducciones']['nomina12:Deduccion'];
            if(!empty($deductions['@attributes'])){
                $deduction = $deductions;
                $tmp[$deduction['@attributes']['TipoDeduccion']] = isset($tmp[$deduction['@attributes']['TipoDeduccion']]) ? $tmp[$deduction['@attributes']['TipoDeduccion']]+$deduction['@attributes']['Importe'] : $deduction['@attributes']['Importe'];
            }else {
                foreach ($deductions as $deduction) {
                    $tmp[$deduction['@attributes']['TipoDeduccion']] = isset($tmp[$deduction['@attributes']['TipoDeduccion']]) ? $tmp[$deduction['@attributes']['TipoDeduccion']]+$deduction['@attributes']['Importe'] : $deduction['@attributes']['Importe'];
                }
            }
        }
        $data['deductions'] = $tmp;
        //Otros pagos
        $tmp = [];
        if(!empty($xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:OtrosPagos']['nomina12:OtroPago'])){
            $other_payment_types = $xml_array['cfdi:Complemento']['nomina12:Nomina']['nomina12:OtrosPagos']['nomina12:OtroPago'];
            if(!empty($other_payment_types['@attributes'])){
                $other_payment_type = $other_payment_types;
                $tmp[$other_payment_type['@attributes']['TipoOtroPago']] = isset($tmp[$other_payment_type['@attributes']['TipoOtroPago']]) ? $tmp[$other_payment_type['@attributes']['TipoOtroPago']]+$other_payment_type['@attributes']['Importe'] : $other_payment_type['@attributes']['Importe'];
            }else {
                foreach ($other_payment_types as $other_payment_type) {
                    $tmp[$other_payment_type['@attributes']['TipoOtroPago']] = isset($tmp[$other_payment_type['@attributes']['TipoOtroPago']]) ? $tmp[$other_payment_type['@attributes']['TipoOtroPago']]+$other_payment_type['@attributes']['Importe'] : $other_payment_type['@attributes']['Importe'];
                }
            }
        }
        $data['other_payment_types'] = $tmp;

        return $data;
    }

    public static function domNodeToArray($node)
    {
        $output = array();
        switch ($node->nodeType) {
            case XML_CDATA_SECTION_NODE:
            case XML_TEXT_NODE:
                $output = trim($node->textContent);
                break;
            case XML_ELEMENT_NODE:
                for ($i = 0, $m = $node->childNodes->length; $i < $m; $i++) {
                    $child = $node->childNodes->item($i);
                    $v     = self::domNodeToArray($child);
                    if (isset($child->tagName)) {
                        $t = $child->tagName;
                        if (!isset($output[$t])) {
                            $output[$t] = array();
                        }
                        $output[$t][] = $v;
                    } elseif ($v) {
                        $output = (string)$v;
                    }
                }
                if (is_array($output)) {
                    if ($node->attributes->length) {
                        $a = array();
                        foreach ($node->attributes as $attrName => $attrNode) {
                            $a[ucfirst($attrName)] = (string)$attrNode->value;
                        }
                        $output['@attributes'] = $a;
                    }
                    foreach ($output as $t => $v) {
                        if (is_array($v) && count($v) == 1 && $t != '@attributes') {
                            $output[$t] = $v[0];
                        }
                    }
                }
                break;
        }

        return $output;
    }

    /**
     * Valida que ya no se permita descargar mas cfdi si no tiene permiso
     *
     */
    public static function effectiveDateCfdiDownload(){
        if(!empty(setting('effective_date_cfdi_download'))){
            $effective_date = self::createDateFromSql(setting('effective_date_cfdi_download'));
            if($effective_date->lessThan(\Date::today())){
                return false;
            }
        }

        return true;
    }

    public static function validateEffectiveDateCfdiDownload(){
        if(!empty(setting('effective_date_cfdi_download'))){
            $effective_date = self::createDateFromSql(setting('effective_date_cfdi_download'));
            $pre_effective_date = $effective_date->copy()->subDays(30);
            $post_effective_date = $effective_date->copy()->addDays(5);
            if($pre_effective_date->lessThan(\Date::today()) && $effective_date->greaterThanOrEqualTo(\Date::today())){
                flash(sprintf(__('base/cfdi_download.error_pre_effective_date_cfdi_download'), Helper::convertSqlToDate(setting('effective_date_cfdi_download'))) )->info();
            }
            if($effective_date->lessThan(\Date::today()) && $post_effective_date->greaterThanOrEqualTo(\Date::today())){
                flash(sprintf(__('base/cfdi_download.error_effective_date_cfdi_download'), Helper::convertSqlToDate(setting('effective_date_cfdi_download'))) )->error();
            }
        }
    }

    public static function validateQtyCfdiDownload($year){
        $data['qty'] = 20000;
        $data['error'] = false;
        $data['msg'] = '';

        $cfdi_download_count = CfdiDownload::whereYear('date', $year)->count();
        $data['qty'] = empty(setting('qty_cfdi_download')) ? $data['qty'] : (int) setting('qty_cfdi_download');
        if($cfdi_download_count > $data['qty']){
            $data['msg'] = sprintf(__('base/cfdi_download.error_qty_cfdi_download'), Helper::numberFormat($data['qty']));
            $data['error'] = true;
        }

        return $data;
    }

    public static function replyMails()
    {
        $reply = [];
        $company = Helper::defaultCompany(); //Empresa
        if(!empty($company->email)){
            $reply[$company->email] = $company->email;
        }
        if(empty($reply)){
            $reply[\Auth::user()->email] = \Auth::user()->email;
        }

        return $reply;
    }
}
