<?php

namespace App\Helpers;

use App\Models\Base\Company;
use App\Models\Base\DocumentType;
use App\Models\Base\Pac;
use App\Models\Sales\CustomerInvoice;
use App\Models\Sales\CustomerPayment;
use CfdiUtils\Cfdi;
use CfdiUtils\ConsultaCfdiSat\RequestParameters;
use CfdiUtils\ConsultaCfdiSat\WebService;
use Chumper\Zipper\Facades\Zipper;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Str;
use NumberToWords\NumberToWords;
use SoapClient;
//use SWServices\Cancelation\CancelationService;
//use SWServices\Stamp\StampService as StampService;
use ZipArchive;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Process\Process;

class PacHelper
{
    private static $finkokErrors = [
        '203' => 'No encontrado o no corresponde en el emisor',
        '204' => 'UUID No aplicable para cancelación',
        '205' => 'UUID No existe',
        '207' => 'No se especificó el motivo de cancelación o el motivo no es valido',
        '208' => 'Folio Sustitución invalido',
        '209' => 'Folio Sustitución no requerido',
        '210' => 'La fecha de solicitud de cancelación es mayor a la fecha de declaración',
        '211' => 'La fecha de solicitud de cancelación límite para factura global',
        '212' => 'Relación no valida o inexistente',
        '301' => 'XML mal formado',
        '302' => 'Sello mal formado o inválido',
        '303' => 'Sello no corresponde al emisor',
        '304' => 'Certificado revocado o caduco',
        '305' => 'Certificado inválido',
        '310' => 'Uso de certificado de e.firma inválido',
        '311' => 'Clave de motivo de cancelación no válida',
        '312' => 'UUID no relacionado de acuerdo a la clave de motivo de cancelación',
        '708' => 'No se pudo conectar al SAT'
    ];

    /**
     * Almacenamiento local para archivos del SAT
     *
     * @return \CfdiUtils\XmlResolver\XmlResolver
     */
    public static function resourcePathCfdiUtils()
    {
        $path = \Storage::path('sat');
        return new \CfdiUtils\XmlResolver\XmlResolver($path);
    }

    /**
     * Valida acciones para timbrar
     *
     * @throws \Exception
     */
    public static function validateSatActions($company = null)
    {
        try {
            //Valida datos de empresa
            if (!empty($company)) {
                //Valida archivo cer
                if (empty($company->file_cer)) {
                    throw new \Exception(__('base/company.error_file_cer_empty'));
                }
                if (!\Storage::exists($company->pathFileCer()) || !\Storage::exists($company->pathFileCerPem())) {
                    throw new \Exception(__('base/company.error_file_cer_exists'));
                }
                //Valida archivo key
                if (empty($company->file_key)) {
                    throw new \Exception(__('base/company.error_file_key_empty'));
                }
                if (!\Storage::exists($company->pathFileKeyPassPem())) {
                    throw new \Exception(__('base/company.error_file_key_exists'));
                }
                //Valida vigencia de certificado
                if (!empty($company->date_start) && !empty($company->date_end)) {
                    if (\Date::parse($company->date_start)->greaterThan(\Date::now())) {
                        throw new \Exception(sprintf(__('base/company.error_file_cer_validity'),
                            Helper::convertSqlToDateTime($company->date_start),
                            Helper::convertSqlToDateTime($company->date_end)));
                    }
                    if (\Date::parse($company->date_end)->lessThan(\Date::now())) {
                        throw new \Exception(sprintf(__('base/company.error_file_cer_validity'),
                            Helper::convertSqlToDateTime($company->date_start),
                            Helper::convertSqlToDateTime($company->date_end)));
                    }
                }
            } else {
                throw new \Exception(__('base/company.error_company_exists'));
            }

            //Valida datos de pac
            $pac = Pac::findOrFail(setting('default_pac_id')); //PAC
            if (!empty($pac)) {
                $class_pac = $pac->code;
                if (!method_exists(self::class, $class_pac)) {
                    throw new \Exception(__('base/pac.error_pac_class_exists'));
                }
            } else {
                throw new \Exception(__('base/pac.error_pac_exists'));
            }
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Valida acciones para cancelar timbrar
     *
     * @throws \Exception
     */
    public static function validateSatCancelActions($company = null, $pac)
    {
        try {
            //Valida datos de empresa
            if (!empty($company)) {
                //Valida archivo cer
                if (empty($company->file_pfx)) {
                    throw new \Exception(__('base/company.error_file_pfx_empty'));
                }
                if (!\Storage::exists($company->pathFilePfx())) {
                    throw new \Exception(__('base/company.error_file_pfx_exists'));
                }
            } else {
                throw new \Exception(__('base/company.error_company_exists'));
            }

            //Valida datos de pac
            if (!empty($pac)) {
                $class_pac = $pac->code . 'Cancel';
                if (!method_exists(self::class, $class_pac)) {
                    throw new \Exception(__('base/pac.error_pac_class_exists'));
                }
            } else {
                throw new \Exception(__('base/pac.error_pac_exists'));
            }
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para timbrado de pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function pacTest($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC
            $file_xml_pac = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . Str::random(40) . '.xml';
            $uuid = '9FB6ED1A-5F37-4FEF-980A-'.strtoupper(Str::random(12));

            //Creación del nodo de LeyendasFiscales
            $timbreFiscalDigital = new \CfdiUtils\Nodes\Node(
                'tfd:TimbreFiscalDigital', // nombre del elemento raíz
                [ // nodos obligatorios de XML y del nodo
                    'xmlns:tfd' => 'http://www.sat.gob.mx/TimbreFiscalDigital',
                    'xsi:schemaLocation' => 'http://www.sat.gob.mx/TimbreFiscalDigital'
                        . ' http://www.sat.gob.mx/sitio_internet/cfd/TimbreFiscalDigital/TimbreFiscalDigitalv11.xsd',
                    'Version' => '1.1',
                    'UUID' => $uuid,
                    'FechaTimbrado' => \Date::now()->format('Y-m-d\TH:i:s'),
                    'RfcProvCertif' => 'SFE0807172W7',
                    'SelloCFD' => 'W2Fr9AiEuUIFJUVRWMXWMHndDcwvpNCu2g0uTE58wutNkUgjq3J+5f7Kl/ygpAlQggmJB9dKBd2UsYjd94dGTvIso26CFdmW3QY+KBa5d/qpFBsnLxVq+NgP4l2MpAzMMlzD4AsyaTSPnKc6/xmFzIQszCEQ0DsQO+twW0VsxrI=',
                    'NoCertificadoSAT' => '20001000000300022779',
                    'SelloSAT' => 'a0U3KSPH6ouIzOnO3A5KUG+wKJ0Kg77SeDLdw8c4a4bLaib5dCxh4nv1Y/GFq7fgm3DQDCmPTzmcJ8BATwO8r9+e36Zcw0kHcVlJ8+VRz/oHYWyCgq4etsTybXtrqcnLeyCy6Oi38Yk/2lvEwnvtqSFSXrNTL/jA5ZEyeZxpwerGmuQ/NmUC4Ta5+DjjON7bZomfMNBMoAsnLj5uTEQa03mZGhbPg+h/RMLZ7GA2VXLFhxj68YXt/uR5tu4kJTjwTn0ZAK83wzYp68V2TJdEaW3JMLw/5EwXpqctV2drek3u2gt63S9Po2FHpmXDKVNM6LTw57iRQ8tnLSCeKP68Nw==',
                ]
            );

            //Agregar el nodo $timbreFiscalDigital a los complementos del CFDI
            $comprobante = $creator->comprobante();
            $comprobante->addComplemento($timbreFiscalDigital);

            //Guardar XML ya timbrado
            $creator->saveXml(\Storage::path($tmp['path_xml']) . $file_xml_pac);

            //Actualiza los datos para guardar
            $tmp['uuid'] = $uuid;
            $tmp['date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['file_xml_pac'] = $file_xml_pac;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function pacTestStatus($tmp, $pac)
    {
        try {
            $data_status_sat = [
                'cancelable' => 1,
                'status_cancelled' => true,
                'pac_is_cancelable' => 'Pruebas',
                'pac_status' => 'Pruebas',
                'pac_status_cancelation' => 'Pruebas'
            ];

            return $data_status_sat;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function pacTestCancel($tmp, $pac)
    {
        try {
            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = 'ok';

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para timbrado con Edicom pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function edicomTest($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC
            //
            $context_options = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true
                )
            );
            $ssl_context = stream_context_create($context_options);
            $client = new SoapClient($pac->ws_url, array('stream_context' => $ssl_context));
            $params = array(
                'user' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'file' => \Storage::get($tmp['path_xml'] . $tmp['file_xml']),
            );
            //Metodo de timbrado
            $response = $client->__soapCall('getCfdiTest', array('parameters' => $params));
            $result = $response->getCfdiTestReturn;

            //Crear archivo Zip con el contenido de la respuesta
            $path_temp = \Storage::path('temp-xml/');
            $file_zip = Str::random(40) . '.zip';
            \Storage::put('temp-xml/' . $file_zip, $result);

            //Renombra los archivos dentro del zip y los extrae en la carpeta de CFDI
            $file_xml_pac = Str::random(40) . '.xml';
            $list_tmp = Zipper::make($path_temp . $file_zip)->listFiles();
            $zip = new ZipArchive;
            $res = $zip->open($path_temp . $file_zip);
            if ($res == true) {
                if (!empty($list_tmp[0])) {
                    $zip->renameName($list_tmp[0], $file_xml_pac);
                }
                $zip->close();
            }
            Zipper::make($path_temp . $file_zip)->extractTo(\Storage::path($tmp['path_xml']) . Helper::makeDirectoryCfdi($tmp['path_xml']));

            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = $tfd['UUID'];
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID en pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function edicomTestStatus($tmp,$company = null, $pac)
    {
        try {
            //
            $context_options = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true
                )
            );
            $ssl_context = stream_context_create($context_options);
            $client = new SoapClient($pac->ws_url_cancel, array('stream_context' => $ssl_context));
            $params = [
                'user' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'rfcE' => $company->taxid,
                'rfcR' => $tmp['rfcR'],
                'uuid' => $tmp['uuid'],
                'total' => $tmp['total'],
                'test' => true,
            ];
            $response = $client->__soapCall('getCFDiStatus', array('parameters' => $params));
            $result = $response->getCFDiStatusReturn;
            $data_status_sat = [];
            if(isset($result->status)){
                //Valida si el CFDI puede ser cancelado 1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado
                $cancelable = 1;
                if(isset($result->isCancelable) && in_array($result->isCancelable,['Cancelable con aceptación'])){
                    $cancelable = 2;
                }
                if(isset($result->isCancelable) && in_array($result->isCancelable,['No cancelable'])){
                    $cancelable = 3;
                }
                if(isset($result->status) && in_array($result->status,['Cancelado','No Encontrado'])){
                    $cancelable = 3;
                }
                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false;
                if(isset($result->cancelStatus) && in_array($result->cancelStatus,['Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }
                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_is_cancelable' => $result->isCancelable ?? '',
                    'pac_status' => $result->status ?? '',
                    'pac_status_cancelation' => $result->cancelStatus ?? ''
                ];
            }
            return $data_status_sat;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado Edicom de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function edicomTestCancel($tmp,$company = null, $pac)
    {
        try {
            //
            $context_options = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true
                )
            );
            $ssl_context = stream_context_create($context_options);
            $client = new SoapClient($pac->ws_url_cancel, array('stream_context' => $ssl_context));
            $params = array(
                'user' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'rfcE' => $company->taxid,
                'rfcR' => $tmp['rfcR'],
                'uuid' => $tmp['uuid'],
                'total' => $tmp['total'],
                'pfx' => \Storage::get($company->pathFilePfx()),
                'pfxPassword' => Crypt::decryptString($company->password_key),
                'motivo' => $tmp['reason_cancellation_code'],
                'sustitucion' => $tmp['reason_cancellation_uuid'],
                'test' => true,
            );
            //Metodo de cancelacion de timbrado
            //$response = $client->__soapCall('cancelCFDiAsync', array('parameters' => $params));
            //$result = $response->cancelCFDiAsyncReturn;
            $tmp_response = 'ok';
            //if(!empty($result->ack)){
            //    $tmp_response = $result->ack;
            //}

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para timbrado con Edicom
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function edicom($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC
            //
            $context_options = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true
                )
            );
            $ssl_context = stream_context_create($context_options);
            $client = new SoapClient($pac->ws_url, array('stream_context' => $ssl_context));
            $params = array(
                'user' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'file' => \Storage::get($tmp['path_xml'] . $tmp['file_xml']),
            );
            //Metodo de timbrado
            $response = $client->__soapCall('getCfdi', array('parameters' => $params));
            $result = $response->getCfdiReturn;

            //Crear archivo Zip con el contenido de la respuesta
            $path_temp = \Storage::path('temp-xml/');
            $file_zip = Str::random(40) . '.zip';
            \Storage::put('temp-xml/' . $file_zip, $result);

            //Renombra los archivos dentro del zip y los extrae en la carpeta de CFDI
            $file_xml_pac = Str::random(40) . '.xml';
            $list_tmp = Zipper::make($path_temp . $file_zip)->listFiles();
            $zip = new ZipArchive;
            $res = $zip->open($path_temp . $file_zip);
            if ($res == true) {
                if (!empty($list_tmp[0])) {
                    $zip->renameName($list_tmp[0], $file_xml_pac);
                }
                $zip->close();
            }
            Zipper::make($path_temp . $file_zip)->extractTo(\Storage::path($tmp['path_xml']) . Helper::makeDirectoryCfdi($tmp['path_xml']));

            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = $tfd['UUID'];
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID en pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function edicomStatus($tmp,$company = null, $pac)
    {
        try {
            //
            $context_options = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true
                )
            );
            $ssl_context = stream_context_create($context_options);
            $client = new SoapClient($pac->ws_url_cancel, array('stream_context' => $ssl_context));
            $params = [
                'user' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'rfcE' => $company->taxid,
                'rfcR' => $tmp['rfcR'],
                'uuid' => $tmp['uuid'],
                'total' => $tmp['total'],
                'test' => false,
            ];
            $response = $client->__soapCall('getCFDiStatus', array('parameters' => $params));
            $result = $response->getCFDiStatusReturn;
            $data_status_sat = [];
            if(isset($result->status)){
                //Valida si el CFDI puede ser cancelado 1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado
                $cancelable = 1;
                if(isset($result->isCancelable) && in_array($result->isCancelable,['Cancelable con aceptación'])){
                    $cancelable = 2;
                }
                if(isset($result->isCancelable) && in_array($result->isCancelable,['No cancelable'])){
                    $cancelable = 3;
                }
                if(isset($result->status) && in_array($result->status,['Cancelado','No Encontrado'])){
                    $cancelable = 3;
                }
                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false;
                if(isset($result->cancelStatus) && in_array($result->cancelStatus,['Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }
                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_is_cancelable' => $result->isCancelable ?? '',
                    'pac_status' => $result->status ?? '',
                    'pac_status_cancelation' => $result->cancelStatus ?? ''
                ];
            }
            return $data_status_sat;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado Edicom de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function edicomCancel($tmp,$company = null, $pac)
    {
        try {
            //
            $context_options = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true
                )
            );
            $ssl_context = stream_context_create($context_options);
            $client = new SoapClient($pac->ws_url_cancel, array('stream_context' => $ssl_context));
            $params = array(
                'user' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'rfcE' => $company->taxid,
                'rfcR' => $tmp['rfcR'],
                'uuid' => $tmp['uuid'],
                'total' => $tmp['total'],
                'pfx' => \Storage::get($company->pathFilePfx()),
                'pfxPassword' => Crypt::decryptString($company->password_key),
                'motivo' => $tmp['reason_cancellation_code'],
                'sustitucion' => $tmp['reason_cancellation_uuid'],
                'test' => false,
            );
            //Metodo de cancelacion de timbrado
            $response = $client->__soapCall('cancelCFDiAsync', array('parameters' => $params));
            $result = $response->cancelCFDiAsyncReturn;
            $tmp_response = 'ok';
            if(!empty($result->ack)){
                $tmp_response = $result->ack;
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para timbrado con Finkok pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function finkokTest($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC

            //
            $client = new SoapClient($pac->ws_url);
            $params = [
                'username' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'xml' => \Storage::get($tmp['path_xml'] . $tmp['file_xml']),
            ];
            //Metodo de timbrado
            $response = $client->__soapCall('stamp', ['parameters' => $params]);

            //Valida respuesta
            if (isset($response->stampResult->Incidencias->Incidencia)) {
                throw new \Exception($response->stampResult->Incidencias->Incidencia->MensajeIncidencia . ': ' . $response->stampResult->Incidencias->Incidencia->ExtraInfo);
            }else{
                $tmp_xml = $response->stampResult->xml;
                $file_xml_pac = Str::random(40) . '.xml';
                \Storage::put($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac, $tmp_xml);
            }

            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = $tfd['UUID'];
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID en pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function finkokTestStatus($tmp,$company = null, $pac)
    {
        try {

            $client = new SoapClient($pac->ws_url_cancel);
            $params = [
                'username' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'taxpayer_id' => $company->taxid,
                'rtaxpayer_id' => $tmp['rfcR'],
                'uuid' => $tmp['uuid'],
                'total' => $tmp['total'],
            ];
            $response = $client->__soapCall('get_sat_status', ['parameters' => $params]);

            //Valida respuesta
            $data_status_sat = [];
            if(isset($response->get_sat_statusResult->sat)){
                //Valida si el CFDI puede ser cancelado 1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado
                $cancelable = 1;
                if(isset($response->get_sat_statusResult->sat->EsCancelable) && in_array($response->get_sat_statusResult->sat->EsCancelable,['Cancelable con aceptación'])){
                    $cancelable = 2;
                }
                if(isset($response->get_sat_statusResult->sat->EsCancelable) && in_array($response->get_sat_statusResult->sat->EsCancelable,['No cancelable'])){
                    $cancelable = 3;
                }
                if(isset($response->get_sat_statusResult->sat->Estado) && in_array($response->get_sat_statusResult->sat->Estado,['Cancelado','No Encontrado'])){
                    $cancelable = 3;
                }
                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false;
                if(isset($response->get_sat_statusResult->sat->EstatusCancelacion) && in_array($response->get_sat_statusResult->sat->EstatusCancelacion,['Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }
                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_is_cancelable' => $response->get_sat_statusResult->sat->EsCancelable ?? '',
                    'pac_status' => $response->get_sat_statusResult->sat->Estado ?? '',
                    'pac_status_cancelation' => $response->get_sat_statusResult->sat->EstatusCancelacion ?? ''
                ];
            }
            return $data_status_sat;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado Finkok de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function finkokTestCancel($tmp,$company = null, $pac)
    {
        try {

            //Crear archivos key.finkok.pass.pem con la clave del usuario como lo solicita finkok
            //Convertir en PEM con contraseña
            $path_file_key = $company->pathFileKey();
            $path_file_key_pem = $path_file_key . '.pem';
            $path_file_key_finkok_pass_pem = $path_file_key . '.finkok.pass.pem';
            if (\Storage::exists($path_file_key) && \Storage::exists($path_file_key_pem) && !\Storage::exists($path_file_key_finkok_pass_pem)) {
                $pass = Crypt::decryptString($pac->password);
                if(PHP_OS_FAMILY !== "Windows"){
                    $pass = str_replace('$','\$', $pass);
                }
                if(config('app.shared_hosting')){
                    system('openssl rsa -in "' . \Storage::path($path_file_key_pem) . '" -des3 -out "' . \Storage::path($path_file_key_finkok_pass_pem) . '" -passout pass:"' . $pass.'"');
                }else {
                    $process = new Process('openssl rsa -in "' . \Storage::path($path_file_key_pem) . '" -des3 -out "' . \Storage::path($path_file_key_finkok_pass_pem) . '" -passout pass:"' . $pass.'"');
                    $process->run();
                    if (!$process->isSuccessful()) {
                        throw new ProcessFailedException($process);
                    }
                }
            }

            //Verison a partir de 31/12/2022
            /*$uuids = [$tmp['uuid']];
            $client = new SoapClient($pac->ws_url_cancel);
            $params = [
                'UUIDS' => ['uuids' => $uuids],
                'username' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'taxpayer_id' => $company->taxid,
                'cer' => \Storage::get($company->pathFileCerPem()),
                'key' => \Storage::exists($path_file_key_finkok_pass_pem) ? \Storage::get($path_file_key_finkok_pass_pem) : null,
            ];*/

            //Verison a partir de 01/01/2022
            $client = new SoapClient($pac->ws_url_cancel);
            $params = [
                'UUIDS' => [
                    'UUID' => [
                        'UUID' => $tmp['uuid'],
                        'Motivo' => $tmp['reason_cancellation_code'],
                        'FolioSustitucion' => $tmp['reason_cancellation_uuid']
                    ]
                ],
                'username' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'taxpayer_id' => $company->taxid,
                'cer' => \Storage::get($company->pathFileCerPem()),
                'key' => \Storage::exists($path_file_key_finkok_pass_pem) ? \Storage::get($path_file_key_finkok_pass_pem) : null,
            ];

            //Metodo de cancelacion de timbrado
            $response = $client->__soapCall('cancel', ['parameters' => $params]);

            //Valida respuesta
            $tmp_response = 'ok';
            if(isset($response->cancelResult->Folios->Folio->EstatusUUID) && in_array($response->cancelResult->Folios->Folio->EstatusUUID,['201','202'])){
                if(!empty($response->cancelResult->Acuse)){
                    $tmp_response = $response->cancelResult->Acuse;
                }else{
                    $tmp_response = 'Sin acuse';
                }
            }else{ //En caso de errores
                if(!empty($response->cancelResult->Folios->Folio->EstatusUUID)){
                    $ms_error = $response->cancelResult->Folios->Folio->EstatusUUID;
                    if (!empty($response->cancelResult->Folios->Folio->EstatusCancelacion)){
                        $ms_error = $ms_error . ' - ' . $response->cancelResult->Folios->Folio->EstatusCancelacion;
                    }else if (!empty(self::$finkokErrors[$ms_error])){
                        $ms_error = $ms_error . ' - ' . self::$finkokErrors[$ms_error];
                    }
                    throw new \Exception($ms_error);
                }elseif(!empty($response->cancelResult->CodEstatus)){
                    $ms_error = $response->cancelResult->CodEstatus;
                    if (!empty(self::$finkokErrors[$ms_error])){
                        $ms_error = $ms_error . ' - ' . self::$finkokErrors[$ms_error];
                    }
                    throw new \Exception($ms_error);
                }
                throw new \Exception(__('general.error_ws_cfdi'));
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para timbrado con Finkok pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function finkok($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC

            //
            /*$options = [
                'cache_wsdl'=>WSDL_CACHE_NONE,
                'stream_context' => stream_context_create([
                    'ssl' => [
                        'verify_peer' => false,
                        'verify_peer_name' => false,
                        'allow_self_signed' => true,
                    ]
                ])
            ];*/
            $client = new SoapClient($pac->ws_url);
            $params = [
                'username' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'xml' => \Storage::get($tmp['path_xml'] . $tmp['file_xml']),
            ];
            //Metodo de timbrado
            $response = $client->__soapCall('stamp', ['parameters' => $params]);

            //if(\Auth::user()->email == 'sksistemas@hotmail.com'){
                //$request = $client->__getLastRequest();
                //$file = fopen(\Storage::path($tmp['path_xml'] . "request" . $tmp['file_xml']), "w");
                //fwrite($file, 'request: ' .$request);
                //fclose($file);
                //dd($response);
            //}

            //Valida respuesta
            if (isset($response->stampResult->Incidencias->Incidencia)) {
                throw new \Exception($response->stampResult->Incidencias->Incidencia->MensajeIncidencia . ': ' . $response->stampResult->Incidencias->Incidencia->ExtraInfo);
            }else{
                $tmp_xml = $response->stampResult->xml;
                $file_xml_pac = Str::random(40) . '.xml';
                \Storage::put($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac, $tmp_xml);
            }

            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = $tfd['UUID'];
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID en pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function finkokStatus($tmp,$company = null, $pac)
    {
        try {

            $client = new SoapClient($pac->ws_url_cancel);
            $params = [
                'username' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'taxpayer_id' => $company->taxid,
                'rtaxpayer_id' => $tmp['rfcR'],
                'uuid' => $tmp['uuid'],
                'total' => $tmp['total'],
            ];
            $response = $client->__soapCall('get_sat_status', ['parameters' => $params]);

            //Valida respuesta
            $data_status_sat = [];
            if(isset($response->get_sat_statusResult->sat)){
                //Valida si el CFDI puede ser cancelado 1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado
                $cancelable = 1;
                if(isset($response->get_sat_statusResult->sat->EsCancelable) && in_array($response->get_sat_statusResult->sat->EsCancelable,['Cancelable con aceptación'])){
                    $cancelable = 2;
                }
                if(isset($response->get_sat_statusResult->sat->EsCancelable) && in_array($response->get_sat_statusResult->sat->EsCancelable,['No cancelable'])){
                    $cancelable = 3;
                }
                if(isset($response->get_sat_statusResult->sat->Estado) && in_array($response->get_sat_statusResult->sat->Estado,['Cancelado','No Encontrado'])){
                    $cancelable = 3;
                }
                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false;
                if(isset($response->get_sat_statusResult->sat->EstatusCancelacion) && in_array($response->get_sat_statusResult->sat->EstatusCancelacion,['Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }
                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_is_cancelable' => $response->get_sat_statusResult->sat->EsCancelable ?? '',
                    'pac_status' => $response->get_sat_statusResult->sat->Estado ?? '',
                    'pac_status_cancelation' => $response->get_sat_statusResult->sat->EstatusCancelacion ?? ''
                ];
            }
            return $data_status_sat;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado Finkok de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function finkokCancel($tmp,$company = null, $pac)
    {
        try {

            //Crear archivos key.finkok.pass.pem con la clave del usuario como lo solicita finkok
            //Convertir en PEM con contraseña
            $path_file_key = $company->pathFileKey();
            $path_file_key_pem = $path_file_key . '.pem';
            $path_file_key_finkok_pass_pem = $path_file_key . '.finkok.pass.pem';
            if (\Storage::exists($path_file_key) && \Storage::exists($path_file_key_pem) && !\Storage::exists($path_file_key_finkok_pass_pem)) {
                $pass = Crypt::decryptString($pac->password);
                if(PHP_OS_FAMILY !== "Windows"){
                    $pass = str_replace('$','\$', $pass);
                }
                if(config('app.shared_hosting')){
                    system('openssl rsa -in "' . \Storage::path($path_file_key_pem) . '" -des3 -out "' . \Storage::path($path_file_key_finkok_pass_pem) . '" -passout pass:"' . $pass.'"');
                }else {
                    $process = new Process('openssl rsa -in "' . \Storage::path($path_file_key_pem) . '" -des3 -out "' . \Storage::path($path_file_key_finkok_pass_pem) . '" -passout pass:"' . $pass.'"');
                    $process->run();
                    if (!$process->isSuccessful()) {
                        throw new ProcessFailedException($process);
                    }
                }
            }

            //Version hasta 31/12/2021
            /*$uuids = [$tmp['uuid']];
            $client = new SoapClient($pac->ws_url_cancel);
            $params = [
                'UUIDS' => ['uuids' => $uuids],
                'username' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'taxpayer_id' => $company->taxid,
                'cer' => \Storage::get($company->pathFileCerPem()),
                'key' => \Storage::exists($path_file_key_finkok_pass_pem) ? \Storage::get($path_file_key_finkok_pass_pem) : null,
            ];*/

            //Verison a partir de 01/01/2022
            $client = new SoapClient($pac->ws_url_cancel);
            $params = [
                'UUIDS' => [
                    'UUID' => [
                        'UUID' => $tmp['uuid'],
                        'Motivo' => $tmp['reason_cancellation_code'],
                        'FolioSustitucion' => $tmp['reason_cancellation_uuid']
                    ]
                ],
                'username' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'taxpayer_id' => $company->taxid,
                'cer' => \Storage::get($company->pathFileCerPem()),
                'key' => \Storage::exists($path_file_key_finkok_pass_pem) ? \Storage::get($path_file_key_finkok_pass_pem) : null,
            ];

            //Metodo de cancelacion de timbrado
            $response = $client->__soapCall('cancel', ['parameters' => $params]);

            //Valida respuesta
            $tmp_response = 'ok';
            if(isset($response->cancelResult->Folios->Folio->EstatusUUID) && in_array($response->cancelResult->Folios->Folio->EstatusUUID,['201','202'])){
                if(!empty($response->cancelResult->Acuse)){
                    $tmp_response = $response->cancelResult->Acuse;
                }else{
                    $tmp_response = 'Sin acuse';
                }
            }else{ //En caso de errores
                if(!empty($response->cancelResult->Folios->Folio->EstatusUUID)){
                    $ms_error = $response->cancelResult->Folios->Folio->EstatusUUID;
                    if (!empty($response->cancelResult->Folios->Folio->EstatusCancelacion)){
                        $ms_error = $ms_error . ' - ' . $response->cancelResult->Folios->Folio->EstatusCancelacion;
                    }else if (!empty(self::$finkokErrors[$ms_error])){
                        $ms_error = $ms_error . ' - ' . self::$finkokErrors[$ms_error];
                    }
                    throw new \Exception($ms_error);
                }elseif(!empty($response->cancelResult->CodEstatus)){
                    $ms_error = $response->cancelResult->CodEstatus;
                    if (!empty(self::$finkokErrors[$ms_error])){
                        $ms_error = $ms_error . ' - ' . self::$finkokErrors[$ms_error];
                    }
                    throw new \Exception($ms_error);
                }
                throw new \Exception(__('general.error_ws_cfdi'));
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para timbrado con Timbox pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function timboxTest($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC

            //
            $client = new SoapClient($pac->ws_url,[
                'trace' => 1,
                'use' => SOAP_LITERAL
            ]);
            $params = [
                'username' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'xml' => base64_encode(\Storage::get($tmp['path_xml'] . $tmp['file_xml'])),
            ];

            //Metodo de timbrado
            $response = $client->__soapCall('timbrar_cfdi', $params);

            $tmp_xml = $response->xml;
            $file_xml_pac = Str::random(40) . '.xml';
            \Storage::put($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac, $tmp_xml);


            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = $tfd['UUID'];
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID en pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function timboxTestStatus($tmp,$company = null, $pac)
    {
        try {

            $client = new SoapClient($pac->ws_url_cancel);
            $params = [
                'username' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'uuid' => $tmp['uuid'],
                'rfc_emisor' => $company->taxid,
                'rfc_receptor' => $tmp['rfcR'],
                'total' => $tmp['total'],
            ];
            $response = $client->__soapCall('consultar_estatus', $params);
            $data_status_sat = [];
            if(isset($response->estado)){
                //Valida si el CFDI puede ser cancelado 1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado
                $cancelable = 1;
                if(in_array($response->es_cancelable,['Cancelable con aceptación'])){
                    $cancelable = 2;
                }
                if(in_array($response->es_cancelable,['No cancelable'])){
                    $cancelable = 3;
                }
                if(in_array($response->estado,['Cancelado','No Encontrado'])){
                    $cancelable = 3;
                }
                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false;
                if(in_array($response->estatus_cancelacion,['Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }

                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_is_cancelable' => $response->es_cancelable ?? '',
                    'pac_status' => $response->estado ?? '',
                    'pac_status_cancelation' => $response->estatus_cancelacion  ?? ''
                ];

            }
            return $data_status_sat;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado Timbox de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function timboxTestCancel($tmp,$company = null, $pac)
    {
        try {

            //
            $uuids = [
                [
                    'uuid' => $tmp['uuid'],
                    'rfc_receptor' => $tmp['rfcR'],
                    'total' => $tmp['total'],
                    'motivo' => $tmp['reason_cancellation_code'],
                    'folio_sustituto' => $tmp['reason_cancellation_uuid']
                ]
            ];
            $client = new SoapClient($pac->ws_url_cancel);
            $params = [
                'username' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'rfc_emisor' => $company->taxid,
                'folios' => $uuids,
                'cert_pem' => \Storage::get($company->pathFileCerPem()),
                'llave_pem' => \Storage::get($company->pathFileKeyPem()),
                /*'pfxbase64' => \Storage::get($company->pathFilePfx()),
                'pfxpassword' => Crypt::decryptString($company->password_key),*/
            ];
            //Metodo de cancelacion de timbrado
            $response = $client->__soapCall('cancelar_cfdi', $params);

            //Valida respuesta
            $tmp_response = 'ok';
            if(!empty($response->acuse_cancelacion)){
                $tmp_response = $response->acuse_cancelacion;
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para timbrado con Timbox
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function timbox($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC

            //
            $client = new SoapClient($pac->ws_url,[
                'trace' => 1,
                'use' => SOAP_LITERAL
            ]);
            $params = [
                'username' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'xml' => base64_encode(\Storage::get($tmp['path_xml'] . $tmp['file_xml'])),
            ];

            //Metodo de timbrado
            $response = $client->__soapCall('timbrar_cfdi', $params);

            $tmp_xml = $response->xml;
            $file_xml_pac = Str::random(40) . '.xml';
            \Storage::put($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac, $tmp_xml);


            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = $tfd['UUID'];
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function timboxStatus($tmp,$company = null, $pac)
    {
        try {

            $client = new SoapClient($pac->ws_url_cancel);
            $params = [
                'username' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'uuid' => $tmp['uuid'],
                'rfc_emisor' => $company->taxid,
                'rfc_receptor' => $tmp['rfcR'],
                'total' => $tmp['total'],
            ];
            $response = $client->__soapCall('consultar_estatus', $params);
            $data_status_sat = [];
            if(isset($response->estado)){
                //Valida si el CFDI puede ser cancelado 1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado
                $cancelable = 1;
                if(in_array($response->es_cancelable,['Cancelable con aceptación'])){
                    $cancelable = 2;
                }
                if(in_array($response->es_cancelable,['No cancelable'])){
                    $cancelable = 3;
                }
                if(in_array($response->estado,['Cancelado','No Encontrado'])){
                    $cancelable = 3;
                }
                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false;
                if(in_array($response->estatus_cancelacion,['Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }

                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_is_cancelable' => $response->es_cancelable ?? '',
                    'pac_status' => $response->estado ?? '',
                    'pac_status_cancelation' => $response->estatus_cancelacion  ?? ''
                ];

            }
            return $data_status_sat;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado Timbox
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function timboxCancel($tmp,$company = null, $pac)
    {
        try {

            //
            $uuids = [
                [
                    'uuid' => $tmp['uuid'],
                    'rfc_receptor' => $tmp['rfcR'],
                    'total' => $tmp['total'],
                    'motivo' => $tmp['reason_cancellation_code'],
                    'folio_sustituto' => $tmp['reason_cancellation_uuid']
                ]
            ];
            $client = new SoapClient($pac->ws_url_cancel);
            $params = [
                'username' => $pac->username,
                'password' => Crypt::decryptString($pac->password),
                'rfc_emisor' => $company->taxid,
                'folios' => $uuids,
                'cert_pem' => \Storage::get($company->pathFileCerPem()),
                'llave_pem' => \Storage::get($company->pathFileKeyPem()),
                /*'pfxbase64' => \Storage::get($company->pathFilePfx()),
                'pfxpassword' => Crypt::decryptString($company->password_key),*/
            ];
            //Metodo de cancelacion de timbrado
            $response = $client->__soapCall('cancelar_cfdi', $params);

            //Valida respuesta
            $tmp_response = 'ok';
            if(!empty($response->acuse_cancelacion)){
                $tmp_response = $response->acuse_cancelacion;
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Funcion de apoyo para parsear la respuesta de HTTP regresada por CURL
     */
    private static function parseResponseComdig($response, $headerSize){
        $header_text = substr($response, 0, $headerSize);
        $body = substr($response, $headerSize);
        foreach (explode("\r\n", $header_text) as $i => $line)
            if ($i === 0)
                $headers['http_code'] = $line;
            else
            {
                if(!empty($line)){
                    list ($key, $value) = explode(': ', $line);
                    $headers[$key] = $value;
                }
            }
        return array($body,$headers);
    }

    /**
     * Funcion de apoyo para parsear la respuesta de HTTP regresada por CURL
     */
    private static function parseResponseBodyComdig($response_body){

        $bodys=[];
        foreach (explode("\n", $response_body) as $i => $line)
            if(!empty($line)){
                list ($key, $value) = explode(': ', $line);
                $bodys[$key] = $value;
            }
        return $bodys;
    }

    /**
     * Funcion de apoyo para parsear el acuse de cancelacion del SAT
     */
    private static function parseAckComdig($acuse_string){
        $acuse=new \SimpleXMLElement($acuse_string);
        $uuids = array();
        foreach($acuse->Folios as $folio){
            $uuid = (string)$folio->UUID;
            $uuids[$uuid] = (string)$folio->EstatusUUID;
        }
        return $uuids;
    }


    /**
     * Clase para timbrado con Comercio digital pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function comdigTest($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC

            $ch = curl_init($pac->ws_url);
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, \Storage::get($tmp['path_xml'] . $tmp['file_xml']));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt(
                $ch,
                CURLOPT_HTTPHEADER,
                [
                    'Expect:',
                    'Content-Type: text/xml',
                    'usrws: '.$pac->username,
                    'pwdws: '.Crypt::decryptString($pac->password),
                    'tipo: XML'
                ]
            );
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 20);
            curl_setopt($ch, CURLOPT_HEADER,1);
            $response = curl_exec($ch);
            $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
            curl_close($ch);

            list($response_body, $headers) = self::parseResponseComdig($response,$headerSize);
            $codigo = $headers['codigo'];
            $codigo_int = (int)$codigo;
            if($codigo_int != 0){
                throw new \Exception($headers['errmsg']);
            }

            $tmp_xml = $response_body;
            $file_xml_pac = Str::random(40) . '.xml';
            \Storage::put($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac, $tmp_xml);

            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = $tfd['UUID'];
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        }catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function comdigTestStatus($tmp,$company = null, $pac)
    {
        try {

            $response = new \stdClass();
            $response->estado = 'No proporcionado';
            $response->estatus_cancelacion = 'No proporcionado';
            $data_status_sat = [];
            if(isset($response->estado)){
                //Valida si el CFDI puede ser cancelado 1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado
                $cancelable = 1;
                /*if(in_array($response->estado,['Cancelable con aceptación'])){
                    $cancelable = 2;
                }
                if(in_array($response->estado,['No cancelable'])){
                    $cancelable = 3;
                }
                if(in_array($response->estado,['Cancelado','No Encontrado'])){
                    $cancelable = 3;
                }*/
                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false;
                /*if(in_array($response->estatus_cancelacion,['Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }*/

                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_is_cancelable' => $response->estado ?? '',
                    'pac_status' => $response->estado ?? '',
                    'pac_status_cancelation' => $response->estatus_cancelacion  ?? ''
                ];

            }
            return $data_status_sat;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado Comercio Digital de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function comdigTestCancel($tmp,$company = null, $pac)
    {
        try {
            //Verison a partir de 31/12/2022
            //$request = "USER=".$pac->username."\nPWDW=".Crypt::decryptString($pac->password)."\nRFCE=".$company->taxid."\nUUID=".$tmp['uuid']."\nPWDK=".Crypt::decryptString($company->password_key)."\nKEYF=".base64_encode(\Storage::get($company->pathFileKey()))."\nCERT=".base64_encode(\Storage::get($company->pathFileCer()))."\nTIPO=".$tmp['cfdi_version']."\nACUS=SI\nRFCR=".$tmp['rfcR']."\nTOTAL=".$tmp['total']."\nTIPOC=".$tmp['cfdi_type']."\nEMAILE=".$company->email."\n";

            //Verison a partir de 01/01/2022
            $request = "USER=".$pac->username."\nPWDW=".Crypt::decryptString($pac->password)."\nRFCE=".$company->taxid."\nUUID=".$tmp['uuid']."\nPWDK=".Crypt::decryptString($company->password_key)."\nKEYF=".base64_encode(\Storage::get($company->pathFileKey()))."\nCERT=".base64_encode(\Storage::get($company->pathFileCer()))."\nTIPO1=cfdi\nACUS=SI\nRFCR=".$tmp['rfcR']."\nTOTAL=".$tmp['total']."\nTIPOC=".$tmp['cfdi_type']."\nEMAILE=".$company->email."\n"."MOTIVO=".$tmp['reason_cancellation_code']."\n"."UUIDREL=".$tmp['reason_cancellation_uuid']."\n";

            $ch = curl_init($pac->ws_url_cancel);
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $request);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt(
                $ch,
                CURLOPT_HTTPHEADER,
                [
                    'Expect:',
                    'Content-Type: text/plain',
                    'usrws: '.$pac->username,
                    'pwdws: '.Crypt::decryptString($pac->password),
                ]
            );
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 20);
            curl_setopt($ch, CURLOPT_HEADER,1);
            $response = curl_exec($ch);
            $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
            curl_close($ch);

            list($response_body, $headers) = self::parseResponseComdig($response,$headerSize);
            $codigo = $headers['codigo'];
            $codigo_int = (int)$codigo;
            if($codigo_int != 0){
                throw new \Exception($headers['errmsg']);
            }

            //Valida respuesta
            $tmp_response = 'ok';
            if(!empty($response_body)){
                $tmp_response = $response_body;
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        }catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para timbrado con Comercio digital pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function comdig($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC

            $ch = curl_init($pac->ws_url);
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, \Storage::get($tmp['path_xml'] . $tmp['file_xml']));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt(
                $ch,
                CURLOPT_HTTPHEADER,
                [
                    'Expect:',
                    'Content-Type: text/xml',
                    'usrws: '.$pac->username,
                    'pwdws: '.Crypt::decryptString($pac->password),
                    'tipo: XML'
                ]
            );
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 20);
            curl_setopt($ch, CURLOPT_HEADER,1);
            $response = curl_exec($ch);
            $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
            curl_close($ch);

            list($response_body, $headers) = self::parseResponseComdig($response,$headerSize);
            $codigo = $headers['codigo'];
            $codigo_int = (int)$codigo;
            if($codigo_int != 0){
                throw new \Exception($headers['errmsg']);
            }

            $tmp_xml = $response_body;
            $file_xml_pac = Str::random(40) . '.xml';
            \Storage::put($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac, $tmp_xml);

            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = $tfd['UUID'];
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        }catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function comdigStatus($tmp,$company = null, $pac)
    {
        try {

            $request = "USER=".$pac->username."\nPWDW=".Crypt::decryptString($pac->password)."\nRFCR=".$tmp['rfcR']."\nRFCE=".$company->taxid."\nTOTAL=".$tmp['total']."\nUUID=".$tmp['uuid']."\n";
            $ch = curl_init('https://cancela.comercio-digital.mx/arws/consultaEstatus');
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $request);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt(
                $ch,
                CURLOPT_HTTPHEADER,
                [
                    'Expect:',
                    'Content-Type: text/plain',
                    'usrws: '.$pac->username,
                    'pwdws: '.Crypt::decryptString($pac->password),
                ]
            );
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 20);
            curl_setopt($ch, CURLOPT_HEADER,1);
            $response = curl_exec($ch);
            $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
            curl_close($ch);

            list($response_body, $headers) = self::parseResponseComdig($response,$headerSize);
            $codigo = isset($headers['codigo']) ? $headers['codigo'] : '';
            $codigo_int = (int)$codigo;
            if($codigo_int != 0){
                throw new \Exception($headers['errmsg']);
            }

            //$uuids = self::parseAckComdig($response_body);
            $bodys = self::parseResponseBodyComdig($response_body);
            $response = new \stdClass();
            $response->estado = isset($bodys['Es cancelable']) ? $bodys['Es cancelable'] : '';
            $response->estatus_cancelacion = isset($bodys['Estatus Cancelacion']) ? $bodys['Estatus Cancelacion'] : '';
            $data_status_sat = [];
            if(isset($response->estado)){
                //Valida si el CFDI puede ser cancelado 1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado
                $cancelable = 1;
                if(in_array($response->estado,['Cancelable con aceptación'])){
                    $cancelable = 2;
                }
                if(in_array($response->estado,['No cancelable'])){
                    $cancelable = 3;
                }
                if(in_array($response->estado,['Cancelado','No Encontrado'])){
                    $cancelable = 3;
                }
                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false;
                if(in_array($response->estatus_cancelacion,['Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }

                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_is_cancelable' => $response->estado ?? '',
                    'pac_status' => isset($bodys['Estatus Cfdi']) ? $bodys['Estatus Cfdi'] : '',
                    'pac_status_cancelation' => $response->estatus_cancelacion  ?? ''
                ];

            }
            return $data_status_sat;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado Comercio Digital de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function comdigCancel($tmp,$company = null, $pac)
    {
        try {
            //Verison a partir de 31/12/2022
            //$request = "USER=".$pac->username."\nPWDW=".Crypt::decryptString($pac->password)."\nRFCE=".$company->taxid."\nUUID=".$tmp['uuid']."\nPWDK=".Crypt::decryptString($company->password_key)."\nKEYF=".base64_encode(\Storage::get($company->pathFileKey()))."\nCERT=".base64_encode(\Storage::get($company->pathFileCer()))."\nTIPO=".$tmp['cfdi_version']."\nACUS=SI\nRFCR=".$tmp['rfcR']."\nTOTAL=".$tmp['total']."\nTIPOC=".$tmp['cfdi_type']."\nEMAILE=".$company->email."\n";

            //Verison a partir de 01/01/2022
            $request = "USER=".$pac->username."\nPWDW=".Crypt::decryptString($pac->password)."\nRFCE=".$company->taxid."\nUUID=".$tmp['uuid']."\nPWDK=".Crypt::decryptString($company->password_key)."\nKEYF=".base64_encode(\Storage::get($company->pathFileKey()))."\nCERT=".base64_encode(\Storage::get($company->pathFileCer()))."\nTIPO1=cfdi\nACUS=SI\nRFCR=".$tmp['rfcR']."\nTOTAL=".$tmp['total']."\nTIPOC=".$tmp['cfdi_type']."\nEMAILE=".$company->email."\n"."MOTIVO=".$tmp['reason_cancellation_code']."\n"."UUIDREL=".$tmp['reason_cancellation_uuid']."\n";

            $ch = curl_init($pac->ws_url_cancel);
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $request);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt(
                $ch,
                CURLOPT_HTTPHEADER,
                [
                    'Expect:',
                    'Content-Type: text/plain',
                    'usrws: '.$pac->username,
                    'pwdws: '.Crypt::decryptString($pac->password),
                ]
            );
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 20);
            curl_setopt($ch, CURLOPT_HEADER,1);
            $response = curl_exec($ch);
            $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
            curl_close($ch);

            list($response_body, $headers) = self::parseResponseComdig($response,$headerSize);
            $codigo = $headers['codigo'];
            $codigo_int = (int)$codigo;
            if($codigo_int != 0){
                throw new \Exception($headers['errmsg']);
            }

            //Valida respuesta
            $tmp_response = 'ok';
            if(!empty($response_body)){
                $tmp_response = $response_body;
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        }catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para timbrado con Multifacturas pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function multifacturasTest($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC

            //
            $client = new SoapClient($pac->ws_url,[
                'trace' => 1,
                'use' => SOAP_LITERAL
            ]);
            $params = [
                'rfc' => $pac->username,
                'clave' => Crypt::decryptString($pac->password),
                'xml' => base64_encode(\Storage::get($tmp['path_xml'] . $tmp['file_xml'])),
                'produccion' => 'NO',
            ];

            //Metodo de timbrado
            $response = $client->__soapCall('timbrar33b64', $params);
            $codigo_int = (int)$response->codigo_mf_numero;
            if($codigo_int != 0){
                throw new \Exception($response->codigo_mf_texto);
            }
            $tmp_xml = base64_decode($response->cfdi);
            $file_xml_pac = Str::random(40) . '.xml';
            \Storage::put($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac, $tmp_xml);

            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = strtoupper($tfd['UUID']);
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID en pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function multifacturasTestStatus($tmp,$company = null, $pac)
    {
        try {

            $response = new \stdClass();
            $response->estado = 'No proporcionado';
            $response->estatus_cancelacion = 'No proporcionado';
            $data_status_sat = [];
            if(isset($response->estado)){
                //Valida si el CFDI puede ser cancelado 1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado
                $cancelable = 1;
                /*if(in_array($response->estado,['Cancelable con aceptación'])){
                    $cancelable = 2;
                }
                if(in_array($response->estado,['No cancelable'])){
                    $cancelable = 3;
                }
                if(in_array($response->estado,['Cancelado','No Encontrado'])){
                    $cancelable = 3;
                }*/
                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false;
                /*if(in_array($response->estatus_cancelacion,['Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }*/

                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_is_cancelable' => $response->estado ?? '',
                    'pac_status' => $response->estado ?? '',
                    'pac_status_cancelation' => $response->estatus_cancelacion  ?? ''
                ];

            }
            return $data_status_sat;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado Multifacturas de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function multifacturasTestCancel($tmp,$company = null, $pac)
    {
        try {
            //
            $client = new SoapClient($pac->ws_url_cancel);
            $params = [
                'PAC' => [
                    'usuario' => $pac->username,
                    'pass' => Crypt::decryptString($pac->password),
                ],
                'usuario' => $pac->username,
                'pass' => Crypt::decryptString($pac->password),
                'accion' => 'cancelar',
                'uuid' => $tmp['uuid'],
                'rfc' => $company->taxid,
                'b64Cer' => base64_encode(\Storage::get($company->pathFileCer())),
                'b64Key' => base64_encode(\Storage::get($company->pathFileKey())),
                'password' => Crypt::decryptString($company->password_key),
                'motivo' => $tmp['reason_cancellation_code'],
                'folioSustitucion' => $tmp['reason_cancellation_uuid'],
                'produccion' => 'NO'
            ];
            //Metodo de cancelacion de timbrado
            $response = $client->__soapCall('cancelarCfdi', ['datos' => $params]);
            $codigo_int = (int)$response->codigo_mf_numero;
            if($codigo_int != 0){
                throw new \Exception($response->codigo_mf_texto);
            }

            //Valida respuesta
            $tmp_response = 'ok';
            if(!empty($response->acuse)){
                $tmp_response = (!empty($response->codigo_respuesta_sat_texto) ? $response->codigo_respuesta_sat_texto : '-') .base64_decode($response->acuse);
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para timbrado con Multifacturas pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function multifacturas($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC

            //
            $client = new SoapClient($pac->ws_url,[
                'trace' => 1,
                'use' => SOAP_LITERAL
            ]);
            $params = [
                'rfc' => $pac->username,
                'clave' => Crypt::decryptString($pac->password),
                'xml' => base64_encode(\Storage::get($tmp['path_xml'] . $tmp['file_xml'])),
                'produccion' => 'SI',
            ];

            //Metodo de timbrado
            $response = $client->__soapCall('timbrar33b64', $params);
            $codigo_int = (int)$response->codigo_mf_numero;
            if($codigo_int != 0){
                throw new \Exception($response->codigo_mf_texto);
            }
            $tmp_xml = base64_decode($response->cfdi);
            $file_xml_pac = Str::random(40) . '.xml';
            \Storage::put($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac, $tmp_xml);

            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = strtoupper($tfd['UUID']);
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID en pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function multifacturasStatus($tmp,$company = null, $pac)
    {
        try {

            $response = new \stdClass();
            $response->estado = 'No proporcionado';
            $response->estatus_cancelacion = 'No proporcionado';
            $data_status_sat = [];
            if(isset($response->estado)){
                //Valida si el CFDI puede ser cancelado 1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado
                $cancelable = 1;
                /*if(in_array($response->estado,['Cancelable con aceptación'])){
                    $cancelable = 2;
                }
                if(in_array($response->estado,['No cancelable'])){
                    $cancelable = 3;
                }
                if(in_array($response->estado,['Cancelado','No Encontrado'])){
                    $cancelable = 3;
                }*/
                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false;
                /*if(in_array($response->estatus_cancelacion,['Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }*/

                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_is_cancelable' => $response->estado ?? '',
                    'pac_status' => $response->estado ?? '',
                    'pac_status_cancelation' => $response->estatus_cancelacion  ?? ''
                ];

            }
            return $data_status_sat;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado Multifacturas de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function multifacturasCancel($tmp,$company = null, $pac)
    {
        try {
            //
            $client = new SoapClient($pac->ws_url_cancel);
            $params = [
                'PAC' => [
                    'usuario' => $pac->username,
                    'pass' => Crypt::decryptString($pac->password),
                ],
                'usuario' => $pac->username,
                'pass' => Crypt::decryptString($pac->password),
                'accion' => 'cancelar',
                'uuid' => $tmp['uuid'],
                'rfc' => $company->taxid,
                'b64Cer' => base64_encode(\Storage::get($company->pathFileCer())),
                'b64Key' => base64_encode(\Storage::get($company->pathFileKey())),
                'password' => Crypt::decryptString($company->password_key),
                'motivo' => $tmp['reason_cancellation_code'],
                'folioSustitucion' => $tmp['reason_cancellation_uuid'],
                'produccion' => 'SI'
            ];
            //Metodo de cancelacion de timbrado
            $response = $client->__soapCall('cancelarCfdi', ['datos' => $params]);
            $codigo_int = (int)$response->codigo_mf_numero;
            if($codigo_int != 0){
                throw new \Exception($response->codigo_mf_texto);
            }

            //Valida respuesta
            $tmp_response = 'ok';
            if(!empty($response->acuse)){
                $tmp_response = (!empty($response->codigo_respuesta_sat_texto) ? $response->codigo_respuesta_sat_texto : '-') .base64_decode($response->acuse);
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para timbrado con Sifei pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function sifeiTest($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC
            //
            $context_options = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true
                )
            );
            $ssl_context = stream_context_create($context_options);
            $client = new SoapClient($pac->ws_url, array('stream_context' => $ssl_context));
            $params = array(
                'Usuario' => $pac->username,
                'Password' => Crypt::decryptString($pac->password),
                'archivoXMLZip' => \Storage::get($tmp['path_xml'] . $tmp['file_xml']),
                'Serie'=>'',
                'IdEquipo'=> $pac->team_id
            );
            //Metodo de timbrado
            $response = $client->__soapCall('getCFDI', array('parameters' => $params));
            $result = $response->return;

            //Crear archivo Zip con el contenido de la respuesta
            $path_temp = \Storage::path('temp-xml/');
            $file_zip = Str::random(40) . '.zip';
            \Storage::put('temp-xml/' . $file_zip, $result);

            //Renombra los archivos dentro del zip y los extrae en la carpeta de CFDI
            $file_xml_pac = Str::random(40) . '.xml';
            $list_tmp = Zipper::make($path_temp . $file_zip)->listFiles();
            $zip = new ZipArchive;
            $res = $zip->open($path_temp . $file_zip);
            if ($res == true) {
                if (!empty($list_tmp[0])) {
                    $zip->renameName($list_tmp[0], $file_xml_pac);
                }
                $zip->close();
            }
            Zipper::make($path_temp . $file_zip)->extractTo(\Storage::path($tmp['path_xml']) . Helper::makeDirectoryCfdi($tmp['path_xml']));

            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = $tfd['UUID'];
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        } catch (\SoapFault $e) {
            if(!empty($e->detail->SifeiException->error)){
                throw new \Exception($e->detail->SifeiException->codigo . ' - ' . $e->detail->SifeiException->error);
            }
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID en pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function sifeiTestStatus($tmp,$company = null, $pac)
    {
        try {
            //
            $context_options = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true,
                )
            );
            $ssl_context = stream_context_create($context_options);
            $client = new SoapClient($pac->ws_url_cancel, array('stream_context' => $ssl_context));
            $params = [
                'usuarioSIFEI' => $pac->username,
                'passwordSIFEI' => Crypt::decryptString($pac->password),
                're' => $company->taxid,
                'rr' => $tmp['rfcR'],
                'id' => $tmp['uuid'],
                'tt' => $tmp['total'],
                'fe' => $tmp['fe'],
            ];
            $response = $client->__soapCall('consultaSATCFDI', array('parameters' => $params));
            $xml_response = new \DOMDocument('1.0', 'UTF-8');
            libxml_use_internal_errors(true);
            $xml_response->loadXML($response->return);
            $data_status_sat = [];
            if(isset($xml_response->getElementsByTagName('CodigoEstatus')->item(0)->nodeValue) /*&& $xml_response->getElementsByTagName('CodigoEstatus')->item(0)->nodeValue == 'S - Comprobante obtenido satisfactoriamente.'*/){
                $is_cancelable = $xml_response->getElementsByTagName('EsCancelable')->item(0)->nodeValue;
                $status = $xml_response->getElementsByTagName('Estado')->item(0)->nodeValue;
                $cancel_status = $xml_response->getElementsByTagName('EstatusCancelacion')->item(0)->nodeValue;

                //Valida si el CFDI puede ser cancelado 1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado
                $cancelable = 1;
                if(isset($result->isCancelable) && in_array($is_cancelable,['Cancelable con aceptación'])){
                    $cancelable = 2;
                }
                if(isset($result->isCancelable) && in_array($is_cancelable,['No Cancelable'])){
                    $cancelable = 3;
                }
                if(isset($result->status) && in_array($status,['Cancelado','No Encontrado'])){
                    $cancelable = 3;
                }
                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false;
                if(isset($cancel_status) && in_array($cancel_status,['Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }
                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_is_cancelable' => $is_cancelable,
                    'pac_status' => $status,
                    'pac_status_cancelation' => $cancel_status
                ];
            }

            return $data_status_sat;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado Sifei de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function sifeiTestCancel($tmp,$company = null, $pac)
    {
        try {
            //
            $context_options = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true
                )
            );
            $ssl_context = stream_context_create($context_options);
            $client = new SoapClient($pac->ws_url_cancel, array('stream_context' => $ssl_context));
            //Version hasta 31/12/2021
            /*$params = array(
                'usuarioSIFEI' => $pac->username,
                'passwordSifei' => Crypt::decryptString($pac->password),
                'rfcEmisor' => $company->taxid,
                'pfx' => \Storage::get($company->pathFilePfx()),
                'passwordPfx' => Crypt::decryptString($company->password_key),
                'uuids' => $tmp['uuid'],
            );*/

            //Verison a partir de 01/01/2022
            $params = array(
                'usuarioSIFEI' => $pac->username,
                'passwordSifei' => Crypt::decryptString($pac->password),
                'rfcEmisor' => $company->taxid,
                'pfx' => \Storage::get($company->pathFilePfx()),
                'passwordPfx' => Crypt::decryptString($company->password_key),
                'uuids' => '|' . $tmp['uuid'] . '|' .$tmp['reason_cancellation_code'] . '|' . $tmp['reason_cancellation_uuid'] . '|',
            );

            //Metodo de cancelacion de timbrado
            $response = $client->__soapCall('cancelaCFDI', array('parameters' => $params));
            $tmp_response = 'ok';
            if(!empty($response->return)){
                $tmp_response = $response->return;
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para timbrado con Sifei
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function sifei($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC
            //
            $context_options = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true
                )
            );
            $ssl_context = stream_context_create($context_options);
            $client = new SoapClient($pac->ws_url, array('stream_context' => $ssl_context));
            $params = array(
                'Usuario' => $pac->username,
                'Password' => Crypt::decryptString($pac->password),
                'archivoXMLZip' => \Storage::get($tmp['path_xml'] . $tmp['file_xml']),
                'Serie'=>'',
                'IdEquipo' => $pac->team_id
            );
            //Metodo de timbrado
            $response = $client->__soapCall('getCFDI', array('parameters' => $params));
            $result = $response->return;

            //Crear archivo Zip con el contenido de la respuesta
            $path_temp = \Storage::path('temp-xml/');
            $file_zip = Str::random(40) . '.zip';
            \Storage::put('temp-xml/' . $file_zip, $result);

            //Renombra los archivos dentro del zip y los extrae en la carpeta de CFDI
            $file_xml_pac = Str::random(40) . '.xml';
            $list_tmp = Zipper::make($path_temp . $file_zip)->listFiles();
            $zip = new ZipArchive;
            $res = $zip->open($path_temp . $file_zip);
            if ($res == true) {
                if (!empty($list_tmp[0])) {
                    $zip->renameName($list_tmp[0], $file_xml_pac);
                }
                $zip->close();
            }
            Zipper::make($path_temp . $file_zip)->extractTo(\Storage::path($tmp['path_xml']) . Helper::makeDirectoryCfdi($tmp['path_xml']));

            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = $tfd['UUID'];
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        } catch (\SoapFault $e) {
            if(!empty($e->detail->SifeiException->error)){
                throw new \Exception($e->detail->SifeiException->codigo . ' - ' . $e->detail->SifeiException->error);
            }
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID en pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function sifeiStatus($tmp,$company = null, $pac)
    {
        try {
            //
            $context_options = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true,
                )
            );
            $ssl_context = stream_context_create($context_options);
            $client = new SoapClient($pac->ws_url_cancel, array('stream_context' => $ssl_context));
            $params = [
                'usuarioSIFEI' => $pac->username,
                'passwordSIFEI' => Crypt::decryptString($pac->password),
                're' => $company->taxid,
                'rr' => $tmp['rfcR'],
                'id' => $tmp['uuid'],
                'tt' => $tmp['total'],
                'fe' => $tmp['fe'],
            ];
            $response = $client->__soapCall('consultaSATCFDI', array('parameters' => $params));
            $xml_response = new \DOMDocument('1.0', 'UTF-8');
            libxml_use_internal_errors(true);
            $xml_response->loadXML($response->return);
            $data_status_sat = [];
            if(isset($xml_response->getElementsByTagName('CodigoEstatus')->item(0)->nodeValue) /*&& $xml_response->getElementsByTagName('CodigoEstatus')->item(0)->nodeValue == 'S - Comprobante obtenido satisfactoriamente.'*/){
                $is_cancelable = $xml_response->getElementsByTagName('EsCancelable')->item(0)->nodeValue;
                $status = $xml_response->getElementsByTagName('Estado')->item(0)->nodeValue;
                $cancel_status = $xml_response->getElementsByTagName('EstatusCancelacion')->item(0)->nodeValue;

                //Valida si el CFDI puede ser cancelado 1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado
                $cancelable = 1;
                if(isset($result->isCancelable) && in_array($is_cancelable,['Cancelable con aceptación'])){
                    $cancelable = 2;
                }
                if(isset($result->isCancelable) && in_array($is_cancelable,['No Cancelable'])){
                    $cancelable = 3;
                }
                if(isset($result->status) && in_array($status,['Cancelado','No Encontrado'])){
                    $cancelable = 3;
                }
                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false;
                if(isset($cancel_status) && in_array($cancel_status,['Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }
                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_is_cancelable' => $is_cancelable,
                    'pac_status' => $status,
                    'pac_status_cancelation' => $cancel_status
                ];
            }

            return $data_status_sat;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado Sifei
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function sifeiCancel($tmp,$company = null, $pac)
    {
        try {
            //
            $context_options = array(
                'ssl' => array(
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true
                )
            );
            $ssl_context = stream_context_create($context_options);
            $client = new SoapClient($pac->ws_url_cancel, array('stream_context' => $ssl_context));
            //Verison a partir de 31/12/2022
            /*$params = array(
                'usuarioSIFEI' => $pac->username,
                'passwordSifei' => Crypt::decryptString($pac->password),
                'rfcEmisor' => $company->taxid,
                'pfx' => \Storage::get($company->pathFilePfx()),
                'passwordPfx' => Crypt::decryptString($company->password_key),
                'uuids' => $tmp['uuid'],
            );*/
            //Verison a partir de 01/01/2022
            $params = array(
                'usuarioSIFEI' => $pac->username,
                'passwordSifei' => Crypt::decryptString($pac->password),
                'rfcEmisor' => $company->taxid,
                'pfx' => \Storage::get($company->pathFilePfx()),
                'passwordPfx' => Crypt::decryptString($company->password_key),
                'uuids' => '|' . $tmp['uuid'] . '|' .$tmp['reason_cancellation_code'] . '|' . $tmp['reason_cancellation_uuid'] . '|',
            );
            //Metodo de cancelacion de timbrado
            $response = $client->__soapCall('cancelaCFDI', array('parameters' => $params));
            $tmp_response = 'ok';
            if(!empty($response->return)){
                $tmp_response = $response->return;
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para timbrado con SW Sapien pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    /*public static function swsapienTest($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC

            //
            $params = array(
                'url'=>$pac->ws_url,
                'user' => $pac->username,
                'password'=> Crypt::decryptString($pac->password)
            );
            $xml = \Storage::get($tmp['path_xml'] . $tmp['file_xml']);
            $stamp = StampService::Set($params);
            $result = $stamp::StampV3($xml);
            if(isset($result->status) && $result->status == 'error'){
                throw new \Exception(($result->message ?? '') . ' - ' . ($result->messageDetail ?? ''));
            }
            $tmp_xml = $result->data->cfdi;
            $file_xml_pac = Str::random(40) . '.xml';
            \Storage::put($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac, $tmp_xml);

            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = $tfd['UUID'];
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        } catch (\Exception $e) {
            throw $e;
        }
    }*/

    /**
     * Clase para obtener el estatus del UUID en pruebas directamente del SAT
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    /*public static function swsapienTestStatus($tmp,$company = null, $pac)
    {
        try {
            $cfdi = Cfdi::newFromString(\Storage::get($tmp['file_xml_pac']));
            $request = RequestParameters::createFromCfdi($cfdi);
            $service = new WebService();
            $response = $service->request($request);

            $data_status_sat = [];
            if(isset($response)){
                //Estatus para cancelar el CFDI
                //1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado

                //
                $status = $response->getCfdi(); //[Vigente, Cancelado, No Encontrado]
                $is_cancelable = null;
                $cancel_status = null;
                if(!in_array($status, ['No Encontrado'])){
                    $is_cancelable = $response->getCancellable(); // [Cancelable con aceptación, No cancelable, Cancelable sin aceptación]
                    $cancel_status = $response->getCancellationStatus(); // [(null), En proceso, Plazo vencido, Solicitud rechazada, Cancelado sin aceptación, Cancelado con aceptación]
                }

                $cancelable = 1; //Se cancela normal [Vigente] [Cancelable sin aceptación]
                //if(in_array($is_cancelable, ['Cancelable con aceptación'])){
                    //$cancelable = 2; //Se cancelar y se espera aceptacion
                //}
                //if(in_array($is_cancelable, ['No cancelable'])){
                    //$cancelable = 3; //No es cancelable
                //}
                //if(in_array($status, ['Cancelado','No Encontrado'])){
                    //$cancelable = 3; //No es cancelable
                //}

                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false; // No cancelado [En proceso, Solicitud rechazada]
                //if(in_array($cancel_status,[null, 'Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    //$status_cancelled = true;
                //}

                //
                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_status' => $status,
                    'pac_is_cancelable' => $is_cancelable,
                    'pac_status_cancelation' => $cancel_status
                ];

            }
            return $data_status_sat;
        } catch (\Exception $e) {
            throw $e;
        }
    }*/

    /**
     * Clase para cancelar timbrado SW Sapien de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    /*public static function swsapienTestCancel($tmp,$company = null, $pac)
    {
        try {
            //
            $params = array(
                'url'=>$pac->ws_url_cancel,
                'token'=>$pac->token,
                'uuid'=> $tmp['uuid'],
                'password'=> Crypt::decryptString($company->password_key),
                'rfc'=> $company->taxid,
                'b64Cer' => base64_encode(\Storage::get($company->pathFileCer())),
                'b64Key' => base64_encode(\Storage::get($company->pathFileKey())),
                'motivo' => $tmp['reason_cancellation_code'],
                'folioSustitucion' => $tmp['reason_cancellation_uuid']
            );
            $cancelationService = CancelationService::Set($params);
            $result = $cancelationService::CancelationByCSD();

            //En case error
            if(isset($result->status) && $result->status == 'error'){
                throw new \Exception(($result->message ?? '') . ' - ' . ($result->messageDetail ?? ''));
            }
            //Valida el estatus de respuesta
            $uuid = $tmp['uuid'];
            if(isset($result->data->uuid->$uuid)){
                $response_code = $result->data->uuid->$uuid;
                if(!in_array($response_code,['201'])){
                    $response_msg = 'Error de cancelación identificado';
                    if($response_code == '202'){
                        $response_msg = 'UUID Previamente cancelado';
                    }elseif($response_code == '203'){
                        $response_msg = 'UUID No corresponde el RFC del emisor y de quien solicita la cancelación.';
                    }elseif($response_code == '205'){
                        $response_msg = 'No Existe';
                    }
                    throw new \Exception($response_msg);
                }
            }

            $tmp_response = 'ok';
            if(!empty($result->data->acuse)){
                $tmp_response = $result->data->acuse;
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        } catch (\Exception $e) {
            throw $e;
        }
    }*/

    /**
     * Clase para timbrado con SW Sapien
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    /*public static function swsapien($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC

            //
            $params = array(
                'url'=>$pac->ws_url,
                'user' => $pac->username,
                'password'=> Crypt::decryptString($pac->password)
            );
            $xml = \Storage::get($tmp['path_xml'] . $tmp['file_xml']);
            $stamp = StampService::Set($params);
            $result = $stamp::StampV3($xml);
            if(isset($result->status) && $result->status == 'error'){
                throw new \Exception(($result->message ?? '') . ' - ' . ($result->messageDetail ?? ''));
            }
            $tmp_xml = $result->data->cfdi;
            $file_xml_pac = Str::random(40) . '.xml';
            \Storage::put($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac, $tmp_xml);

            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = $tfd['UUID'];
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        } catch (\Exception $e) {
            throw $e;
        }
    }*/

    /**
     * Clase para obtener el estatus del UUID en pruebas directamente del SAT
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    /*public static function swsapienStatus($tmp,$company = null, $pac)
    {
        try {
            $cfdi = Cfdi::newFromString(\Storage::get($tmp['file_xml_pac']));
            $request = RequestParameters::createFromCfdi($cfdi);
            $service = new WebService();
            $response = $service->request($request);

            $data_status_sat = [];
            if(isset($response)){
                //Estatus para cancelar el CFDI
                //1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado

                //
                $status = $response->getCfdi(); //[Vigente, Cancelado, No Encontrado]
                $is_cancelable = null;
                $cancel_status = null;
                if(!in_array($status, ['No Encontrado'])){
                    $is_cancelable = $response->getCancellable(); // [Cancelable con aceptación, No cancelable, Cancelable sin aceptación]
                    $cancel_status = $response->getCancellationStatus(); // [(null), En proceso, Plazo vencido, Solicitud rechazada, Cancelado sin aceptación, Cancelado con aceptación]
                }

                $cancelable = 1; //Se cancela normal [Vigente] [Cancelable sin aceptación]
                if(in_array($is_cancelable, ['Cancelable con aceptación'])){
                    $cancelable = 2; //Se cancelar y se espera aceptacion
                }
                if(in_array($is_cancelable, ['No cancelable'])){
                    $cancelable = 3; //No es cancelable
                }
                if(in_array($status, ['Cancelado','No Encontrado'])){
                    $cancelable = 3; //No es cancelable
                }

                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false; // No cancelado [En proceso, Solicitud rechazada]
                if(in_array($cancel_status,[null, 'Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }

                //
                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_status' => $status,
                    'pac_is_cancelable' => $is_cancelable,
                    'pac_status_cancelation' => $cancel_status
                ];

            }
            return $data_status_sat;
        } catch (\Exception $e) {
            throw $e;
        }
    }*/

    /**
     * Clase para cancelar timbrado SW Sapien de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    /*public static function swsapienCancel($tmp,$company = null, $pac)
    {
        try {
            //
            $params = array(
                'url'=>$pac->ws_url_cancel,
                'token'=>$pac->token,
                'uuid'=> $tmp['uuid'],
                'password'=> Crypt::decryptString($company->password_key),
                'rfc'=> $company->taxid,
                'b64Cer' => base64_encode(\Storage::get($company->pathFileCer())),
                'b64Key' => base64_encode(\Storage::get($company->pathFileKey())),
                'motivo' => $tmp['reason_cancellation_code'],
                'folioSustitucion' => $tmp['reason_cancellation_uuid']
            );
            $cancelationService = CancelationService::Set($params);
            $result = $cancelationService::CancelationByCSD();

            //En case error
            if(isset($result->status) && $result->status == 'error'){
                throw new \Exception(($result->message ?? '') . ' - ' . ($result->messageDetail ?? ''));
            }
            //Valida el estatus de respuesta
            $uuid = $tmp['uuid'];
            if(isset($result->data->uuid->$uuid)){
                $response_code = $result->data->uuid->$uuid;
                if(!in_array($response_code,['201'])){
                    $response_msg = 'Error de cancelación identificado';
                    if($response_code == '202'){
                        $response_msg = 'UUID Previamente cancelado';
                    }elseif($response_code == '203'){
                        $response_msg = 'UUID No corresponde el RFC del emisor y de quien solicita la cancelación.';
                    }elseif($response_code == '205'){
                        $response_msg = 'No Existe';
                    }
                    throw new \Exception($response_msg);
                }
            }

            $tmp_response = 'ok';
            if(!empty($result->data->acuse)){
                $tmp_response = $result->data->acuse;
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        } catch (\Exception $e) {
            throw $e;
        }
    }*/

    /**
     * Clase para timbrado con Invoice One pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function invoiceOneTest($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC

            //
            $client = new SoapClient($pac->ws_url);
            $params = [
                'usuario' => $pac->username,
                'contrasena' => Crypt::decryptString($pac->password),
                'xmlComprobante' => \Storage::get($tmp['path_xml'] . $tmp['file_xml']),
            ];
            //Metodo de timbrado
            $response = $client->__soapCall('ObtenerCFDIPrueba', ['parameters' => $params]);

            //Valida respuesta
            if (isset($response->ObtenerCFDIPruebaResult->MensajeError)) {
                throw new \Exception($response->ObtenerCFDIPruebaResult->MensajeExepcionReal);
            }else{
                $tmp_xml = $response->ObtenerCFDIPruebaResult->Xml;
                $file_xml_pac = Str::random(40) . '.xml';
                \Storage::put($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac, $tmp_xml);
            }

            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = $tfd['UUID'];
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID Directamente del sat
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function invoiceOneTestStatus($tmp, $pac)
    {

        try {
            $cfdi = Cfdi::newFromString(\Storage::get($tmp['file_xml_pac']));
            $request = RequestParameters::createFromCfdi($cfdi);
            $service = new WebService();
            $response = $service->request($request);

            $data_status_sat = [];
            if(isset($response)){
                //Estatus para cancelar el CFDI
                //1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado

                //
                $status = $response->getCfdi(); //[Vigente, Cancelado, No Encontrado]
                $is_cancelable = null;
                $cancel_status = null;
                if(!in_array($status, ['No Encontrado'])){
                    $is_cancelable = $response->getCancellable(); // [Cancelable con aceptación, No cancelable, Cancelable sin aceptación]
                    $cancel_status = $response->getCancellationStatus(); // [(null), En proceso, Plazo vencido, Solicitud rechazada, Cancelado sin aceptación, Cancelado con aceptación]
                }

                $cancelable = 1; //Se cancela normal [Vigente] [Cancelable sin aceptación]
                if(in_array($is_cancelable, ['Cancelable con aceptación'])){
                    $cancelable = 2; //Se cancelar y se espera aceptacion
                }
                if(in_array($is_cancelable, ['No cancelable'])){
                    $cancelable = 3; //No es cancelable
                }
                if(in_array($status, ['Cancelado','No Encontrado'])){
                    $cancelable = 3; //No es cancelable
                }

                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false; // No cancelado [En proceso, Solicitud rechazada]
                if(in_array($cancel_status,[null, 'Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }

                //
                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_status' => $status,
                    'pac_is_cancelable' => $is_cancelable,
                    'pac_status_cancelation' => $cancel_status
                ];

            }
            return $data_status_sat;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado Invoice One de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function invoiceOneTestCancel($tmp,$company = null, $pac)
    {
        try {
            //
            $client = new SoapClient($pac->ws_url_cancel);
            $params = [
                'nombreUsuario' => $pac->username,
                'contrasena' => Crypt::decryptString($pac->password),
                'rfcEmisor' => $company->taxid,
                'ListaUUIDsCancelar' => [
                    'UUID' => [
                        'UUID' => $tmp['uuid'],
                        'MotivoCancelacion' => $tmp['reason_cancellation_code'],
                        'FolioSustitucion' => $tmp['reason_cancellation_uuid']
                    ]
                ],
                'pfxBase64' => base64_encode(\Storage::get($company->pathFilePfx())),
                'contrasenaPfx' => Crypt::decryptString($company->password_key),
            ];
            //Metodo de cancelacion de timbrado
            $response = $client->__soapCall('CancelaCFDIPruebas', ['parameters' => $params]);

            //Valida respuesta
            $tmp_response = 'ok';
            if (isset($response->cancelaCFDIPruebasResult->MensajeError)) {
                throw new \Exception($response->cancelaCFDIPruebasResult->MensajeExepcionReal);
            }elseif(isset($response->cancelaCFDIPruebasResult->XmlAcuse)){
                if(!empty($response->cancelaCFDIPruebasResult->XmlAcuse)){
                    $tmp_response = $response->cancelaCFDIPruebasResult->XmlAcuse;
                }else{
                    $tmp_response = 'Sin acuse';
                }
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para timbrado con Invoice One
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function invoiceOne($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC

            //
            $client = new SoapClient($pac->ws_url);
            $params = [
                'usuario' => $pac->username,
                'contrasena' => Crypt::decryptString($pac->password),
                'xmlComprobante' => \Storage::get($tmp['path_xml'] . $tmp['file_xml']),
            ];
            //Metodo de timbrado
            $response = $client->__soapCall('ObtenerCFDI', ['parameters' => $params]);

            //Valida respuesta
            if (isset($response->ObtenerCFDIResult->MensajeError)) {
                throw new \Exception($response->ObtenerCFDIResult->MensajeExepcionReal);
            }else{
                $tmp_xml = $response->ObtenerCFDIResult->Xml;
                $file_xml_pac = Str::random(40) . '.xml';
                \Storage::put($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac, $tmp_xml);
            }

            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = $tfd['UUID'];
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID Directamente del sat
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function invoiceOneStatus($tmp, $pac)
    {

        try {
            $cfdi = Cfdi::newFromString(\Storage::get($tmp['file_xml_pac']));
            $request = RequestParameters::createFromCfdi($cfdi);
            $service = new WebService();
            $response = $service->request($request);

            $data_status_sat = [];
            if(isset($response)){
                //Estatus para cancelar el CFDI
                //1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado

                //
                $status = $response->getCfdi(); //[Vigente, Cancelado, No Encontrado]
                $is_cancelable = null;
                $cancel_status = null;
                if(!in_array($status, ['No Encontrado'])){
                    $is_cancelable = $response->getCancellable(); // [Cancelable con aceptación, No cancelable, Cancelable sin aceptación]
                    $cancel_status = $response->getCancellationStatus(); // [(null), En proceso, Plazo vencido, Solicitud rechazada, Cancelado sin aceptación, Cancelado con aceptación]
                }

                $cancelable = 1; //Se cancela normal [Vigente] [Cancelable sin aceptación]
                if(in_array($is_cancelable, ['Cancelable con aceptación'])){
                    $cancelable = 2; //Se cancelar y se espera aceptacion
                }
                if(in_array($is_cancelable, ['No cancelable'])){
                    $cancelable = 3; //No es cancelable
                }
                if(in_array($status, ['Cancelado','No Encontrado'])){
                    $cancelable = 3; //No es cancelable
                }

                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false; // No cancelado [En proceso, Solicitud rechazada]
                if(in_array($cancel_status,[null, 'Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }

                //
                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_status' => $status,
                    'pac_is_cancelable' => $is_cancelable,
                    'pac_status_cancelation' => $cancel_status
                ];

            }
            return $data_status_sat;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado Invoice One
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function invoiceOneCancel($tmp,$company = null, $pac)
    {
        try {
            //
            $client = new SoapClient($pac->ws_url_cancel);
            $params = [
                'nombreUsuario' => $pac->username,
                'contrasena' => Crypt::decryptString($pac->password),
                'rfcEmisor' => $company->taxid,
                'ListaUUIDsCancelar' => [
                    'UUID' => [
                        'UUID' => $tmp['uuid'],
                        'MotivoCancelacion' => $tmp['reason_cancellation_code'],
                        'FolioSustitucion' => $tmp['reason_cancellation_uuid']
                    ]
                ],
                'pfxBase64' => base64_encode(\Storage::get($company->pathFilePfx())),
                'contrasenaPfx' => Crypt::decryptString($company->password_key),
            ];
            //Metodo de cancelacion de timbrado
            $response = $client->__soapCall('CancelaCFDI', ['parameters' => $params]);

            //Valida respuesta
            $tmp_response = 'ok';
            if (isset($response->cancelaCFDIResult->MensajeError)) {
                throw new \Exception($response->cancelaCFDIResult->MensajeExepcionReal);
            }elseif(isset($response->cancelaCFDIResult->XmlAcuse)){
                if(!empty($response->cancelaCFDIResult->XmlAcuse)){
                    $tmp_response = $response->cancelaCFDIResult->XmlAcuse;
                }else{
                    $tmp_response = 'Sin acuse';
                }
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para timbrado con Timbrador Xpress pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function timbradorXpressTest($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC

            //
            $client = new SoapClient($pac->ws_url);
            $params = [];
            //Metodo de timbrado
            $response = $client->timbrar($pac->token, \Storage::get($tmp['path_xml'] . $tmp['file_xml']));

            //Valida respuesta
            if ($response->code != 200) {
                throw new \Exception($response->message);
            }else{
                $tmp_xml = $response->data;
                $file_xml_pac = Str::random(40) . '.xml';
                \Storage::put($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac, $tmp_xml);
            }

            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = $tfd['UUID'];
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID Directamente del sat
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function timbradorXpressTestStatus($tmp, $company, $pac)
    {

        try {
            $client = new SoapClient($pac->ws_url);
            $params = [];
            //Metodo de timbrado
            $response = $client->consultarEstadoSAT($pac->token, $tmp['uuid'], $company->taxid, $tmp['rfcR'], $tmp['total']);

            $data_status_sat = [];
            if(isset($response)){
                //Estatus para cancelar el CFDI
                //1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado

                //
                $status = $response->Estado; //[Vigente, Cancelado, No Encontrado]
                $is_cancelable = null;
                $cancel_status = null;
                if(!in_array($status, ['No Encontrado'])){
                    $is_cancelable = $response->EsCancelable; // [Cancelable con aceptación, No cancelable, Cancelable sin aceptación]
                    $cancel_status = $response->EstatusCancelacion; // [(null), En proceso, Plazo vencido, Solicitud rechazada, Cancelado sin aceptación, Cancelado con aceptación]
                }

                $cancelable = 1; //Se cancela normal [Vigente] [Cancelable sin aceptación]
                if(in_array($is_cancelable, ['Cancelable con aceptación'])){
                    $cancelable = 2; //Se cancelar y se espera aceptacion
                }
                if(in_array($is_cancelable, ['No cancelable'])){
                    $cancelable = 3; //No es cancelable
                }
                if(in_array($status, ['Cancelado','No Encontrado'])){
                    $cancelable = 3; //No es cancelable
                }

                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false; // No cancelado [En proceso, Solicitud rechazada]
                if(in_array($cancel_status,[null, 'Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }

                //
                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_status' => $status,
                    'pac_is_cancelable' => $is_cancelable,
                    'pac_status_cancelation' => $cancel_status
                ];

            }
            return $data_status_sat;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado Timbrador Xpress de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function timbradorXpressTestCancel($tmp,$company = null, $pac)
    {
        try {
            //
            $client = new SoapClient($pac->ws_url_cancel);
            $params = [];
            //Metodo de cancelacion de timbrado
            $response = $client->cancelarPFX2(
                $pac->token,
                base64_encode(\Storage::get($company->pathFilePfx())),
                Crypt::decryptString($company->password_key),
                $tmp['uuid'],
                $company->taxid,
                $tmp['rfcR'],
                $tmp['total'],
                $tmp['reason_cancellation_code'],
                $tmp['reason_cancellation_uuid']
            );

            //Valida respuesta
            $tmp_response = 'ok';
            if ($response->status != 'success') {
                throw new \Exception($response->message);
            }elseif($response->status == 'success'){
                $resData = json_decode($response->data);
                $tmp_response = $resData->acuse;
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para timbrado con Timbrador Xpress pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function timbradorXpress($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC

            //
            $client = new SoapClient($pac->ws_url);
            $params = [];
            //Metodo de timbrado
            $response = $client->timbrar($pac->token, \Storage::get($tmp['path_xml'] . $tmp['file_xml']));

            //Valida respuesta
            if ($response->code != 200) {
                throw new \Exception($response->message);
            }else{
                $tmp_xml = $response->data;
                $file_xml_pac = Str::random(40) . '.xml';
                \Storage::put($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac, $tmp_xml);
            }

            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = $tfd['UUID'];
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID Directamente del sat
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function timbradorXpressStatus($tmp, $company, $pac)
    {

        try {
            $client = new SoapClient($pac->ws_url);
            $params = [];
            //Metodo de timbrado
            $response = $client->consultarEstadoSAT($pac->token, $tmp['uuid'], $company->taxid, $tmp['rfcR'], $tmp['total']);

            $data_status_sat = [];
            if(isset($response)){
                //Estatus para cancelar el CFDI
                //1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado

                //
                $status = $response->Estado; //[Vigente, Cancelado, No Encontrado]
                $is_cancelable = null;
                $cancel_status = null;
                if(!in_array($status, ['No Encontrado'])){
                    $is_cancelable = $response->EsCancelable; // [Cancelable con aceptación, No cancelable, Cancelable sin aceptación]
                    $cancel_status = $response->EstatusCancelacion; // [(null), En proceso, Plazo vencido, Solicitud rechazada, Cancelado sin aceptación, Cancelado con aceptación]
                }

                $cancelable = 1; //Se cancela normal [Vigente] [Cancelable sin aceptación]
                if(in_array($is_cancelable, ['Cancelable con aceptación'])){
                    $cancelable = 2; //Se cancelar y se espera aceptacion
                }
                if(in_array($is_cancelable, ['No cancelable'])){
                    $cancelable = 3; //No es cancelable
                }
                if(in_array($status, ['Cancelado','No Encontrado'])){
                    $cancelable = 3; //No es cancelable
                }

                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false; // No cancelado [En proceso, Solicitud rechazada]
                if(in_array($cancel_status,[null, 'Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }

                //
                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_status' => $status,
                    'pac_is_cancelable' => $is_cancelable,
                    'pac_status_cancelation' => $cancel_status
                ];

            }
            return $data_status_sat;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado Timbrador Xpress de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function timbradorXpressCancel($tmp,$company = null, $pac)
    {
        try {
            //
            $client = new SoapClient($pac->ws_url_cancel);
            $params = [];
            //Metodo de cancelacion de timbrado
            $response = $client->cancelarPFX2(
                $pac->token,
                base64_encode(\Storage::get($company->pathFilePfx())),
                Crypt::decryptString($company->password_key),
                $tmp['uuid'],
                $company->taxid,
                $tmp['rfcR'],
                $tmp['total'],
                $tmp['reason_cancellation_code'],
                $tmp['reason_cancellation_uuid']
            );

            //Valida respuesta
            $tmp_response = 'ok';
            if ($response->status != 'success') {
                throw new \Exception($response->message);
            }elseif($response->status == 'success'){
                $resData = json_decode($response->data);
                $tmp_response = $resData->acuse;
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID Directamente del sat
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function satStatus($tmp)
    {

        try {
            $cfdi = Cfdi::newFromString(\Storage::get($tmp['file_xml_pac']));
            $request = RequestParameters::createFromCfdi($cfdi);
            $service = new WebService();
            $response = $service->request($request);

            $data_status_sat = [];
            if(isset($response)){
                //Estatus para cancelar el CFDI
                //1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado

                //
                $status = $response->getCfdi(); //[Vigente, Cancelado, No Encontrado]
                $is_cancelable = null;
                $cancel_status = null;
                if(!in_array($status, ['No Encontrado'])){
                    $is_cancelable = $response->getCancellable(); // [Cancelable con aceptación, No cancelable, Cancelable sin aceptación]
                    $cancel_status = $response->getCancellationStatus(); // [(null), En proceso, Plazo vencido, Solicitud rechazada, Cancelado sin aceptación, Cancelado con aceptación]
                }

                $cancelable = 1; //Se cancela normal [Vigente] [Cancelable sin aceptación]
                if(in_array($is_cancelable, ['Cancelable con aceptación'])){
                    $cancelable = 2; //Se cancelar y se espera aceptacion
                }
                if(in_array($is_cancelable, ['No cancelable'])){
                    $cancelable = 3; //No es cancelable
                }
                if(in_array($status, ['Cancelado','No Encontrado'])){
                    $cancelable = 3; //No es cancelable
                }

                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false; // No cancelado [En proceso, Solicitud rechazada]
                if(in_array($cancel_status,[null, 'Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }

                //
                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_status' => $status,
                    'pac_is_cancelable' => $is_cancelable,
                    'pac_status_cancelation' => $cancel_status
                ];

            }
            return $data_status_sat;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para timbrado con Virtual PAC pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function virtualPacTest($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC

            //
            $client = new SoapClient($pac->ws_url);
            $params = [
                'usuario' => $pac->username,
                'DllVersion' => 'xxx',
                'xml' => \Storage::get($tmp['path_xml'] . $tmp['file_xml']),
            ];

            //Metodo de timbrado
            $response = $client->__soapCall('TimbraCFDI_DEMO', ['parameters' => $params]);
            $xml_response = new \DOMDocument('1.0', 'UTF-8');
            libxml_use_internal_errors(true);
            $xml_response->loadXML($response->TimbraCFDI_DEMOResult->any);
            if(!empty($xml_response->getElementsByTagName('TimbreFiscalDigital')->item(0))){
                $xml_tfd = new \DOMDocument('1.0', 'UTF-8');
                $xml_tfd->loadXML($params['xml']);
                $root = $xml_tfd->getElementsByTagName('Comprobante');
                if ($root->item(0)->getElementsByTagName('Complemento')->length == 0) {
                    $complemento = $xml_tfd->createElement('cfdi:Complemento');
                    $complemento = $xml_tfd->documentElement->appendChild($complemento);
                } else {
                    $complemento = $root->item(0)->getElementsByTagName('Complemento')->item(0);
                }
                $timbrefiscal = $xml_tfd->importNode($xml_response->getElementsByTagName('TimbreFiscalDigital')->item(0), true);
                $timbrefiscal = $complemento->appendChild($timbrefiscal);
                $xml_tfd->formatOutput = true;
            }else{
                throw new \Exception($xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('status') ?? 'Error al timbrar');
            }

            $tmp_xml = $xml_tfd->saveXML();
            $file_xml_pac = Str::random(40) . '.xml';
            \Storage::put($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac, $tmp_xml);


            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = $tfd['UUID'];
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function virtualPacTestStatus($tmp,$company = null, $pac)
    {
        try {

            $client = new SoapClient($pac->ws_url_cancel);
            $params = [
                'usuario' => $pac->username,
                'dllversion' => 'xxx',
                'uuid' => $tmp['uuid'],
                'rfcEmisor' => $company->taxid,
                'rfcReceptor' => $tmp['rfcR'],
                'Total' => $tmp['total'],
            ];
            $response = $client->__soapCall('GetStatusCFDI_DEMO', ['parameters' => $params]);

            $xml_response = new \DOMDocument('1.0', 'UTF-8');
            libxml_use_internal_errors(true);
            $xml_response->loadXML($response->GetStatusCFDI_DEMOResult->any);
            if(!empty($xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('estadocomprobante'))){

            }else{
                throw new \Exception($xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('status') ?? 'Error al obtener el estatus');
            }

            $estadocomprobante = $xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('estadocomprobante');
            $escancelable = $xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('escancelable');
            $estatuscancelacion = $xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('estatuscancelacion');

            $data_status_sat = [];
            if(isset($estadocomprobante)){
                //Valida si el CFDI puede ser cancelado 1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado
                $cancelable = 1;
                if(in_array($escancelable,['Cancelable con aceptación'])){
                    $cancelable = 2;
                }
                if(in_array($escancelable,['No cancelable'])){
                    $cancelable = 3;
                }
                if(in_array($estadocomprobante,['Cancelado','No Encontrado'])){
                    $cancelable = 3;
                }
                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false;
                if(in_array($estatuscancelacion,['Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }

                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_is_cancelable' => $escancelable ?? '',
                    'pac_status' => $estadocomprobante ?? '',
                    'pac_status_cancelation' => $estatuscancelacion  ?? ''
                ];
            }
            return $data_status_sat;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado VirtualPac de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function virtualPacTestCancel($tmp,$company = null, $pac)
    {
        try {

            //
            $client = new SoapClient($pac->ws_url_cancel);
            $params = [
                'usuario' => $pac->username,
                'dllversion' => 'xxx',
                'rfcEmisor' => $company->taxid,
                'motivo' => $tmp['reason_cancellation_code'],
                'csdBase64' => base64_encode(\Storage::get($company->pathFileCer())),
                'keyBase64' => base64_encode(\Storage::get($company->pathFileKey())),
                'keypwd' => Crypt::decryptString($company->password_key),
                'uuid' => $tmp['uuid'],
                'folioSustitucion' => $tmp['reason_cancellation_uuid']
            ];
            //Metodo de cancelacion de timbrado
            $response = $client->__soapCall('CancelaCFDI2022_DEMO', ['parameters' => $params]);

            $xml_response = new \DOMDocument('1.0', 'UTF-8');
            libxml_use_internal_errors(true);
            $xml_response->loadXML($response->CancelaCFDI2022_DEMOResult->any);
            if(!empty($xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('status')) && $xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('status') == 'Solicitud de cancelación recibida'){

            }else{
                throw new \Exception($xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('status') ?? 'Error al cancelar el CFDI');
            }

            //Valida respuesta
            $tmp_response = 'ok';
            if(!empty($xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('status'))){
                $tmp_response = $xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('status');
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para timbrado con Virtual PAC pruebas
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function virtualPac($tmp, &$creator = null)
    {
        try {
            $pac = $tmp['pac']; //PAC

            //
            $client = new SoapClient($pac->ws_url);
            $params = [
                'usuario' => $pac->username,
                'DllVersion' => 'xxx',
                'xml' => \Storage::get($tmp['path_xml'] . $tmp['file_xml']),
            ];

            //Metodo de timbrado
            $response = $client->__soapCall('TimbraCFDI', ['parameters' => $params]);
            $xml_response = new \DOMDocument('1.0', 'UTF-8');
            libxml_use_internal_errors(true);
            $xml_response->loadXML($response->TimbraCFDIResult->any);
            if(!empty($xml_response->getElementsByTagName('TimbreFiscalDigital')->item(0))){
                $xml_tfd = new \DOMDocument('1.0', 'UTF-8');
                $xml_tfd->loadXML($params['xml']);
                $root = $xml_tfd->getElementsByTagName('Comprobante');
                if ($root->item(0)->getElementsByTagName('Complemento')->length == 0) {
                    $complemento = $xml_tfd->createElement('cfdi:Complemento');
                    $complemento = $xml_tfd->documentElement->appendChild($complemento);
                } else {
                    $complemento = $root->item(0)->getElementsByTagName('Complemento')->item(0);
                }
                $timbrefiscal = $xml_tfd->importNode($xml_response->getElementsByTagName('TimbreFiscalDigital')->item(0), true);
                $timbrefiscal = $complemento->appendChild($timbrefiscal);
                $xml_tfd->formatOutput = true;
            }else{
                throw new \Exception($xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('status') ?? 'Error al timbrar');
            }

            $tmp_xml = $xml_tfd->saveXML();
            $file_xml_pac = Str::random(40) . '.xml';
            \Storage::put($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac, $tmp_xml);


            //Obtiene datos del CFDI ya timbrado
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($tmp['path_xml'] . Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac));
            $tfd = $cfdi->getNode()->searchNode('cfdi:Complemento', 'tfd:TimbreFiscalDigital');

            //Actualiza los datos para guardar
            $tmp['uuid'] = $tfd['UUID'];
            $tmp['date'] = str_replace('T', ' ', $tfd['FechaTimbrado']);
            $tmp['file_xml_pac'] = Helper::makeDirectoryCfdi($tmp['path_xml']) . '/' . $file_xml_pac;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para obtener el estatus del UUID
     *
     * @param $tmp
     * @param null $creator
     * @return mixed
     * @throws \Exception
     */
    public static function virtualPacStatus($tmp,$company = null, $pac)
    {
        try {

            $client = new SoapClient($pac->ws_url_cancel);
            $params = [
                'usuario' => $pac->username,
                'dllversion' => 'xxx',
                'uuid' => $tmp['uuid'],
                'rfcEmisor' => $company->taxid,
                'rfcReceptor' => $tmp['rfcR'],
                'Total' => $tmp['total'],
            ];
            $response = $client->__soapCall('GetStatusCFDI', ['parameters' => $params]);

            $xml_response = new \DOMDocument('1.0', 'UTF-8');
            libxml_use_internal_errors(true);
            $xml_response->loadXML($response->GetStatusCFDIResult->any);
            if(!empty($xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('estadocomprobante'))){

            }else{
                throw new \Exception($xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('status') ?? 'Error al obtener el estatus');
            }

            $estadocomprobante = $xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('estadocomprobante');
            $escancelable = $xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('escancelable');
            $estatuscancelacion = $xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('estatuscancelacion');

            $data_status_sat = [];
            if(isset($estadocomprobante)){
                //Valida si el CFDI puede ser cancelado 1 Sin aceptacion, 2 Con Aceptacion, 3 no cancelable o si ya esta cancelado
                $cancelable = 1;
                if(in_array($escancelable,['Cancelable con aceptación'])){
                    $cancelable = 2;
                }
                if(in_array($escancelable,['No cancelable'])){
                    $cancelable = 3;
                }
                if(in_array($estadocomprobante,['Cancelado','No Encontrado'])){
                    $cancelable = 3;
                }
                //Valida si ya fue aceptado el proceso de cancelacion
                $status_cancelled = false;
                if(in_array($estatuscancelacion,['Cancelado con aceptación','Plazo vencido','Cancelado sin aceptación'])){
                    $status_cancelled = true;
                }

                $data_status_sat = [
                    'cancelable' => $cancelable,
                    'status_cancelled' => $status_cancelled,
                    'pac_is_cancelable' => $escancelable ?? '',
                    'pac_status' => $estadocomprobante ?? '',
                    'pac_status_cancelation' => $estatuscancelacion  ?? ''
                ];
            }
            return $data_status_sat;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Clase para cancelar timbrado VirtualPac de pruebas
     *
     * @param $tmp
     * @return mixed
     * @throws \Exception
     */
    public static function virtualPacCancel($tmp,$company = null, $pac)
    {
        try {

            //
            $client = new SoapClient($pac->ws_url_cancel);
            $params = [
                'usuario' => $pac->username,
                'dllversion' => 'xxx',
                'rfcEmisor' => $company->taxid,
                'motivo' => $tmp['reason_cancellation_code'],
                'csdBase64' => base64_encode(\Storage::get($company->pathFileCer())),
                'keyBase64' => base64_encode(\Storage::get($company->pathFileKey())),
                'keypwd' => Crypt::decryptString($company->password_key),
                'uuid' => $tmp['uuid'],
                'folioSustitucion' => $tmp['reason_cancellation_uuid']
            ];
            //Metodo de cancelacion de timbrado
            $response = $client->__soapCall('CancelaCFDI2022', ['parameters' => $params]);

            $xml_response = new \DOMDocument('1.0', 'UTF-8');
            libxml_use_internal_errors(true);
            $xml_response->loadXML($response->CancelaCFDI2022Result->any);
            if(!empty($xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('status')) && $xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('status') == 'Solicitud de cancelación recibida'){

            }else{
                throw new \Exception($xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('status') ?? 'Error al cancelar el CFDI');
            }

            //Valida respuesta
            $tmp_response = 'ok';
            if(!empty($xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('status'))){
                $tmp_response = $xml_response->getElementsByTagName('VirtualPAC')->item(0)->getAttribute('status');
            }

            //Actualiza los datos para guardar
            $tmp['cancel_date'] = Helper::dateTimeToSql(\Date::now());
            $tmp['cancel_response'] = $tmp_response;

            return $tmp;
        } catch (\SoapFault $e) {
            throw new \Exception($e->getMessage());
        } catch (\Exception $e) {
            throw $e;
        }
    }

}
