<?php

namespace App\Http\Controllers\Auth;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Http\Request;
use Hyn\Tenancy\Environment;
use Hyn\Tenancy\Models\Hostname;
use Hyn\Tenancy\Models\Website;

class LoginController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Login Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles authenticating users for the application and
    | redirecting them to your home screen. The controller uses a trait
    | to conveniently provide its functionality to your applications.
    |
    */

    use AuthenticatesUsers;

    /**
     * Where to redirect users after login.
     *
     * @var string
     */
    protected $redirectTo = '/home';

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest')->except('logout');
    }

    /**
     * Valida con el campo de status
     *
     * @param \Illuminate\Http\Request $request
     * @return array
     */
    protected function credentials(\Illuminate\Http\Request $request)
    {
        return array_merge($request->only($this->username(), 'password'), ['status' => 1]);
    }

    /**
     * Valida fecha de expiracion y guarda fecha de logeo
     *
     * @param Request $request
     * @param $user
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function authenticated(Request $request, $user) {
        //Validamos la fecha de expiracion
        if(!empty($user->expiry_date)){
            if(Helper::createDateTimeFromSql($user->expiry_date)->lessThan(\Date::now())){
                $validator = \Validator::make([], []);
                $validator->after(function ($validator) {
                    $validator->errors()->add('expiry_date', __('auth/user.error_login_expiry_date'));
                });
                if ($validator->fails()) {
                    auth()->logout();
                    return redirect('/login')->withInput()->withErrors($validator);
                }
            }
        }

        if(empty($user->email_verified_at)){
            $validator = \Validator::make([], []);
            $validator->after(function ($validator) {
                $validator->errors()->add('email_verified_at', __('auth/user.error_login_email_verified'));
            });
            if ($validator->fails()) {
                auth()->logout();
                return redirect('/login')->withInput()->withErrors($validator);
            }
        }

        //Valida si el inquilino al que pertenece está activo
        if($user->superadmin == 0 && $user->distributor == 0 && !empty($user->sys_customer_id)){
            if($user->sysCustomer->status == 0){
                $validator = \Validator::make([], []);
                $validator->after(function ($validator) {
                    $validator->errors()->add('tenant_suspended', __('auth/user.error_tenant_suspended'));
                });
                if ($validator->fails()) {
                    auth()->logout();
                    return redirect('/login')->withInput()->withErrors($validator);
                }
            }
        }

        //Si ingresa
        $user->last_login = Helper::dateTimeToSql(\Date::now());
        $user->login_iframe = !empty($request->login_iframe) ? 1 : 0;
        $user->save();
    }

    /**
     * Show the application's login form.
     *
     * @return \Illuminate\Http\Response
     */
    public function showLoginFormIframe()
    {
        return view('auth.login_iframe');
    }

    /**
     * Log the user out of the application.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function logout(Request $request)
    {
        $login_iframe = false;
        if (\Auth::check() && !empty(\Auth::user()->login_iframe)) {
            $login_iframe = true;
        }
        $this->guard()->logout();

        $request->session()->invalidate();

        return $this->loggedOut($request) ?: (!$login_iframe ? redirect('/') : redirect('/login-iframe'));
    }
}
