<?php

namespace App\Http\Controllers\Auth;

use App\Models\Auth\User;
use Illuminate\Http\Request;
use App\Jobs\CreateTenantJob;
use App\Models\System\SysCustomer;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Auth\Events\Registered;
use Illuminate\Support\Facades\Validator;
use Illuminate\Foundation\Auth\RegistersUsers;
use Illuminate\Validation\ValidationException;

class RegisterController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Register Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users as well as their
    | validation and creation. By default this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
    */

    use RegistersUsers;

    /**
     * Where to redirect users after registration.
     *
     * @var string
     */
    protected $redirectTo = '/home';

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest');
    }

    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array  $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function validator(array $data)
    {
        return Validator::make($data, [
            'name' => 'required|string|max:200',
            'taxid' => [
                'required',
                'regex:/^[A-Z&Ñ]{3,4}[0-9]{2}(0[1-9]|1[012])(0[1-9]|[12][0-9]|3[01])[A-Z0-9]{2}[0-9A]$/i',
                'unique:system.sys_customers'
            ],
            'phone' => 'required|digits:10',
            'email' => 'required|string|email|max:100|unique:system.sys_customers|unique:system.users',
            'password' => 'required|string|min:5|confirmed',
        ], [
            'name.*' => __('system/sys_customer.error_name'),
            'taxid.required' => __('system/sys_customer.error_taxid'),
            'taxid.regex' => __('system/sys_customer.error_taxid_format'),
            'taxid.unique' => __('system/sys_customer.error_taxid_unique'),
            'phone.*' => __('system/sys_customer.error_phone'),
            'email.required' => __('system/sys_customer.error_email'),
            'email.string' => __('system/sys_customer.error_email'),
            'email.max' => __('system/sys_customer.error_email'),
            'email.email' => __('system/sys_customer.error_email_format'),
            'email.unique' => __('system/sys_customer.error_email_unique'),
            'password.required' => __('system/sys_customer.error_password'),
            'password.string' => __('system/sys_customer.error_password'),
            'password.min' => __('system/sys_customer.error_password_min'),
            'password.confirmed' => __('system/sys_customer.error_password_confirmed'),
        ]);
    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     * @return \App\Auth\User
     */
    protected function create(array $data)
    {
        return User::create([
            'name' => $data['name'],
            'email' => $data['email'],
            'password' => Hash::make($data['password']),
            'email_verified_token' => md5(microtime(true)),
        ]);
    }

    /**
     * Show the application registration form.
     *
     * @return \Illuminate\Http\Response
     */
    public function showRegistrationForm(Request $request)
    {
        $iddi = !empty($request->iddi) ? base64_decode($request->iddi) : null;

        if($iddi){
            $user_dist = User::where('distributor', 1)->where('id', (int) $iddi)->first();
            if(empty($user_dist))
                $iddi = null;
        }

        return view('auth.register', compact('iddi'));
    }

    /**
     * Handle a registration request for the application.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function register(Request $request)
    {
        //
        $request->merge(['taxid' => strtoupper(trim($request->taxid))]);
        $request->merge(['email' => trim($request->email)]);

        $this->validator($request->all())->validate();

        //Validaciones manuales
        /*$validator = \Validator::make([], []);
        //Valida que no exista un directorio con el mismo RFC
        if(\Storage::exists($request->taxid)) {
            $validator->after(function ($validator) {
                $validator->errors()->add('taxid', __('system/sys_customer.error_taxid_directory_exists'));
            });
        }
        if ($validator->fails()) {
            throw new ValidationException($validator);
        }*/

        try{
            //logica
            $request->merge(['created_uid' => null]);
            $request->merge(['updated_uid' => null]);
            $request->merge(['name_user' => $request->name]);
            $request->merge(['sort_order' => (int)$request->sort_order]);
            $request->merge(['status' => 1]);

            //Guardar
            //Registro principal
            event(new Registered($user = $this->create($request->all())));

            //Crea nuevo inquilino
            $request->merge(['password' => Crypt::encryptString($request->password)]);
            $sys_customer = SysCustomer::create($request->input());
            if(!empty($request->iddi)){
                $user_dist = User::where('distributor', 1)->where('id', (int) $request->iddi)->first();
                if(!empty($user_dist)){
                    $sys_customer->distributor_id = (int) $request->iddi;
                    $sys_customer->save();
                }
            }
            CreateTenantJob::dispatch($request->input(), $sys_customer, $user);

            flash(__('auth/user.text_success_email_verified_link_email'))->success();

            //Respuesta
            return response()->json([
                'success' => true
            ]);
        }catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 422);
        }
        //$this->guard()->login($user);
        //return $this->registered($request, $user)?: redirect($this->redirectPath());
    }

    public function confirm(User $user, $token) {
        if ($user->confirm($token)) {
            flash(__('auth/user.text_success_email_verified'))->success();
            $this->guard()->login($user);
            return redirect('base/installation/step-1');
        } else {
            flash(__('auth/user.text_fail_email_verified'))->error();
            return redirect()->route('login');
        }
    }
}
