<?php

namespace App\Http\Controllers\Auth;

use App\Helpers\Helper;
use App\Http\Middleware\CheckDemo;
use App\Http\Middleware\CheckSuperadmin;
use App\Http\Middleware\CheckSuperadminOrDistributor;
use App\Mail\SendAccessFactura16;
use App\Models\Auth\User;
use Illuminate\Auth\Passwords\PasswordBroker;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware(CheckSuperadminOrDistributor::class)->except(['editProfile', 'updateProfile','updateCompany']);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        if (\Auth::user()->superadmin) { //Muestra todos los usuarios superadmin
            $results = User::filter($request->all())->sortable('name')->paginate($limit);
        }elseif(\Auth::user()->distributor){
            $results = User::filter($request->all())->notSuperadmin()->userDistributor()->sortable('name')->paginate($limit);
        }else{
            $results = User::filter($request->all())->notSuperadmin()->sortable('name')->paginate($limit);
        }

        //Vista
        return view('auth.users.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('auth.users.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'email' => 'required|email|unique:system.users',
            'password' => 'required|min:5|confirmed',
            'password_confirmation' => 'required|min:5',
        ], [
            'email.required' => __('auth/user.error_email'),
            'email.email' => __('auth/user.error_email'),
            'email.unique' => __('auth/user.error_email_unique'),
            'password.*' => __('auth/user.error_password'),
            'password_confirmation.*' => __('auth/user.error_password_confirmation'),
        ]);

        //Logica
        $password = $request->password;
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['password' => bcrypt($password)]);
        $request->merge(['expiry_date' => !empty($request->expiry_date) ? Helper::convertDateTimeToSql($request->expiry_date) : null]);
        $request->merge(['status' => 1]); //Por default activo
        //Solo el superadministrador puede crar distribuidores
        $request->merge(['distributor' => 0]);
        $request->merge(['available_folio' => !empty($request->available_folio) && \Auth::user()->superadmin ? (int) $request->available_folio : 0]);
        $request->merge(['p_customers' => !empty($request->p_customers) ? 1 : 0]);
        $request->merge(['p_products' => !empty($request->p_products) ? 1 : 0]);
        $request->merge(['p_customer_quotations' => !empty($request->p_customer_quotations) ? 1 : 0]);
        $request->merge(['p_customer_remissions' => !empty($request->p_customer_remissions) ? 1 : 0]);
        $request->merge(['p_cfdis' => !empty($request->p_cfdis) ? 1 : 0]);
        $request->merge(['p_payrolls' => !empty($request->p_payrolls) ? 1 : 0]);
        $request->merge(['p_tax_mailbox' => !empty($request->p_tax_mailbox) ? 1 : 0]);
        $request->merge(['p_cfdi_downloads' => !empty($request->p_cfdi_downloads) ? 1 : 0]);
        $request->merge(['p_bank_reconciliations' => !empty($request->p_bank_reconciliations) ? 1 : 0]);
        $request->merge(['p_report_income_expenses' => !empty($request->p_report_income_expenses) ? 1 : 0]);
        $request->merge(['p_catalogs' => !empty($request->p_catalogs) ? 1 : 0]);
        $request->merge(['p_branch_offices' => !empty($request->p_branch_offices) ? 1 : 0]);
        $request->merge(['p_companies' => !empty($request->p_companies) ? 1 : 0]);
        $request->merge(['p_add_companies' => !empty($request->p_add_companies) ? 1 : 0]);
        $request->merge(['p_edit_companies' => !empty($request->p_edit_companies) ? 1 : 0]);
        $request->merge(['p_settings' => !empty($request->p_settings) ? 1 : 0]);
        $request->merge(['p_admin' => !empty($request->p_admin) ? 1 : 0]);
        $request->merge(['p_users' => !empty($request->p_users) ? 1 : 0]);

        //Guardar
        //Registro principal
        $user = User::create($request->except(['password_confirmation']));

        //
        if(\Auth::user()->superadmin){
            $user->distributor = !empty($request->distributor) ? 1 : 0;
        }

        //Envia correo con datos de acceso
        if(!empty($request->send_access)) {
            \Mail::to($user->email)->send(new SendAccessFactura16($user, $password));
        }

        $user->email_verified_at = \Date::now();
        $user->save();

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/users');

    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Auth\User  $user
     * @return \Illuminate\Http\Response
     */
    public function show(User $user)
    {
        return redirect('/users');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Auth\User  $user
     * @return \Illuminate\Http\Response
     */
    public function edit(User $user)
    {
        return view('auth.users.edit',compact('user'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Auth\User  $user
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, User $user)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'email' => [
                'required',
                'email',
                Rule::unique('system.users')->ignore($user->id)
            ],
            'password' => 'nullable|min:5|confirmed',
            'password_confirmation' => 'nullable|min:5',
        ], [
            'email.required' => __('auth/user.error_email'),
            'email.email' => __('auth/user.error_email'),
            'email.unique' => __('auth/user.error_email_unique'),
            'password.*' => __('auth/user.error_password'),
            'password_confirmation.*' => __('auth/user.error_password_confirmation'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['expiry_date' => !empty($request->expiry_date) ? Helper::convertDateTimeToSql($request->expiry_date) : null]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        //Solo el superadministrador puede crar distribuidores
        if(\Auth::user()->superadmin){
            $request->merge(['distributor' => !empty($request->distributor) ? 1 : 0]);
        }else{
            $request->merge(['distributor' => $user->distributor]);
        }
        $request->merge(['available_folio' => !empty($request->available_folio) && \Auth::user()->superadmin ? (int) $request->available_folio : $user->available_folio]);
        $request->merge(['p_customers' => !empty($request->p_customers) ? 1 : 0]);
        $request->merge(['p_products' => !empty($request->p_products) ? 1 : 0]);
        $request->merge(['p_customer_quotations' => !empty($request->p_customer_quotations) ? 1 : 0]);
        $request->merge(['p_customer_remissions' => !empty($request->p_customer_remissions) ? 1 : 0]);
        $request->merge(['p_cfdis' => !empty($request->p_cfdis) ? 1 : 0]);
        $request->merge(['p_payrolls' => !empty($request->p_payrolls) ? 1 : 0]);
        $request->merge(['p_tax_mailbox' => !empty($request->p_tax_mailbox) ? 1 : 0]);
        $request->merge(['p_cfdi_downloads' => !empty($request->p_cfdi_downloads) ? 1 : 0]);
        $request->merge(['p_bank_reconciliations' => !empty($request->p_bank_reconciliations) ? 1 : 0]);
        $request->merge(['p_report_income_expenses' => !empty($request->p_report_income_expenses) ? 1 : 0]);
        $request->merge(['p_catalogs' => !empty($request->p_catalogs) ? 1 : 0]);
        $request->merge(['p_branch_offices' => !empty($request->p_branch_offices) ? 1 : 0]);
        $request->merge(['p_companies' => !empty($request->p_companies) ? 1 : 0]);
        $request->merge(['p_add_companies' => !empty($request->p_add_companies) ? 1 : 0]);
        $request->merge(['p_edit_companies' => !empty($request->p_edit_companies) ? 1 : 0]);
        $request->merge(['p_settings' => !empty($request->p_settings) ? 1 : 0]);
        $request->merge(['p_admin' => !empty($request->p_admin) ? 1 : 0]);
        $request->merge(['p_users' => !empty($request->p_users) ? 1 : 0]);
        $user->fill($request->only([
            'updated_uid',
            'name',
            'email',
            'expiry_date',
            'sort_order',
            'status',
            'sys_customer_id',
            'distributor',
            'available_folio',
            'p_customers',
            'p_products',
            'p_customer_quotations',
            'p_customer_remissions',
            'p_cfdis',
            'p_payrolls',
            'p_tax_mailbox',
            'p_cfdi_downloads',
            'p_bank_reconciliations',
            'p_report_income_expenses',
            'p_catalogs',
            'p_branch_offices',
            'p_companies',
            'p_add_companies',
            'p_edit_companies',
            'p_settings',
            'p_admin',
            'p_users'
        ]));

        //Si se ingresa una contraseña
        if(!empty($request->password)){
            $user->password = bcrypt($request->password);
        }

        $user->email_verified_at = \Date::now();
        $user->email_verified_token = null;

        //Guardar
        //Registro principal
        $user->save();

        //Envia correo con datos de acceso
        if(!empty($request->password)) {
            if (!empty($request->send_access)) {
                \Mail::to($user->email)->send(new SendAccessFactura16($user, $request->password));
            }
        }

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/users');

    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Auth\User  $user
     * @return \Illuminate\Http\Response
     */
    public function destroy(User $user)
    {
        //Logica
        $user->updated_uid = \Auth::user()->id;
        $user->status = 0;
        $user->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/users');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required',
            'expiry_date' => 'nullable|date|date_format:"'.setting('datetime_format', 'd-m-Y H:i:s').'"',
        ], [
            'name.*' => __('auth/user.error_name'),
            'expiry_date.*' => __('auth/user.error_expiry_date'),
            'expiry_date.required' => __('auth/user.error_expiry_date'),
            'expiry_date.date' => __('auth/user.error_expiry_date_format'),
            'expiry_date.date_format' => __('auth/user.error_expiry_date_format'),
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Auth\User  $user
     * @return \Illuminate\Http\Response
     */
    public function editProfile(User $user)
    {
        return view('auth.users.edit-profile',compact('user'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Auth\User  $user
     * @return \Illuminate\Http\Response
     */
    public function updateProfile(Request $request, User $user)
    {
        //Validacion
        $this->validate($request, [
            'name' => 'required',
            'password' => 'nullable|min:5|confirmed',
            'password_confirmation' => 'nullable|min:5',
            'file_avatar' => 'nullable|mimes:jpg,jpeg,png|max:2048'
        ], [
            'name.*' => __('auth/user.error_name'),
            'password.*' => __('auth/user.error_password'),
            'password_confirmation.*' => __('auth/user.error_password_confirmation'),
            'file_avatar.*' => __('auth/user.error_avatar'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $user->fill($request->only([
            'updated_uid',
            'name',
        ]));

        //Si se ingresa una contraseña
        if(!empty($request->password)){
            $user->password = bcrypt($request->password);
        }

        //Si suben una imagen
        if($request->hasFile('file_avatar')){
            //Si ya tenia un archivo lo eliminamos
            if(!empty($user->avatar)){
                \Storage::delete('images-users/' . User::PATH_AVATARS .'/'.$user->avatar);
            }
            $avatar = Helper::uploadFileImageUsers('file_avatar','images-users/' . User::PATH_AVATARS);
            $user->avatar = $avatar;
        }else{
            $user->avatar = $request->avatar; //si no tiene archivo sobreescribimos elque tenia
        }

        //Guardar
        //Registro principal
        $user->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/home');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Auth\User  $user
     * @return \Illuminate\Http\Response
     */
    public function sendResetLinkEmail(User $user)
    {
        //Logica
        $token = app(PasswordBroker::class)->createToken($user);
        $user->sendPasswordResetNotification($token);

        //Mensaje
        flash(__('auth/user.text_success_send_reset_link_email'))->success();

        //Redireccion
        return redirect('/users');
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Auth\User  $user
     * @return \Illuminate\Http\Response
     */
    public function updateSysCustomer(Request $request, User $user)
    {
        //Validacion

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['company_id' => null]);
        $user->fill($request->only([
            'updated_uid',
            'company_id',
            'sys_customer_id',
        ]));
        //Guardar
        //Registro principal
        $user->save();

        //Cierra variables de session
        $request->session()->forget('session_sat');
        $request->session()->forget('session_sat_fiel');

        //Mensaje

        //Redireccion
        return response()->json(['user' => $user]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Auth\User  $user
     * @return \Illuminate\Http\Response
     */
    public function updateCompany(Request $request, User $user)
    {
        //Validacion

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $user->fill($request->only([
            'updated_uid',
            'company_id',
        ]));
        //Guardar
        //Registro principal
        $user->save();

        //Cierra variables de session
        $request->session()->forget('session_sat');
        $request->session()->forget('session_sat_fiel');

        //Mensaje

        //Redireccion
        return response()->json(['user' => $user]);
    }
}
