<?php

namespace App\Http\Controllers\Base;

use App\Helpers\Helper;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Models\Base\Announcement;
use App\Http\Controllers\Controller;

class AnnouncementController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = Announcement::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('base.announcements.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $types = __('general.text_announcement_types');

        return view('base.announcements.create', compact('types'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['date_start' => Helper::convertDateToSql($request->date_start)]);
        $request->merge(['date_end' => Helper::convertDateToSql($request->date_end)]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $announcement = Announcement::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/base/announcements');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\Announcement  $announcement
     * @return \Illuminate\Http\Response
     */
    public function show(Announcement $announcement)
    {
        return redirect('/base/announcements');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\Announcement  $announcement
     * @return \Illuminate\Http\Response
     */
    public function edit(Announcement $announcement)
    {
        $types = __('general.text_announcement_types');

        return view('base.announcements.edit', compact('announcement', 'types'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\Announcement  $announcement
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Announcement $announcement)
    {
        //Validacion
        $this->validation($request);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['date_start' => Helper::convertDateToSql($request->date_start)]);
        $request->merge(['date_end' => Helper::convertDateToSql($request->date_end)]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $announcement->fill($request->only([
            'updated_uid',
            'sys_customer_id',
            'date_start',
            'date_end',
            'type',
            'description',
            'status',
        ]));

        //Guardar
        //Registro principal
        $announcement->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/base/announcements');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\Announcement  $announcement
     * @return \Illuminate\Http\Response
     */
    public function destroy(Announcement $announcement)
    {
        //Logica
        $announcement->updated_uid = \Auth::user()->id;
        $announcement->status = 0;
        $announcement->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/base/announcements');
    }

    public function validation(Request $request)
    {
        $this->validate($request, [
            'description' => 'required',
            'date_start' => 'required|date|date_format:"'.setting('date_format', 'd-m-Y').'"',
            'date_end' => 'required|date|date_format:"'.setting('date_format', 'd-m-Y').'"',
            'type' => 'required',
        ], [
            'description.*' => __('base/announcement.error_description'),
            'date_start.*' => __('base/announcement.error_date_start'),
            'date_start.date' => __('base/announcement.error_date_start_format'),
            'date_start.date_format' => __('base/announcement.error_date_start_format'),
            'date_end.*' => __('base/announcement.error_date_end'),
            'date_end.date' => __('base/announcement.error_date_end_format'),
            'date_end.date_format' => __('base/announcement.error_date_end_format'),
            'type.*' => __('base/announcement.error_type'),
        ]);
    }
}
