<?php

namespace App\Http\Controllers\Base;

use App\Exports\BankMoveTemplate;
use App\Exports\BankReconciliationsExport;
use App\Helpers\Helper;
use App\Imports\BankMoveTemplateImport;
use App\Models\Base\BankMove;
use App\Models\Base\BankMoveReconciled;
use App\Models\Base\BankMoveTemp;
use App\Models\Base\BankMoveTempReconciled;
use App\Models\Base\BankReconciliation;
use App\Models\Base\BankReconciliationLine;
use App\Models\Base\BankReconciliationLineTemp;
use App\Models\Base\BankReconciliationUnreconcile;
use App\Models\Base\CfdiDownload;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Facades\Excel;

class BankReconciliationController extends Controller
{
    private $list_status = [];


    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->list_status = [
            BankReconciliation::DRAFT => __('base/bank_reconciliation.text_status_draft'),
            BankReconciliation::OPEN => __('base/bank_reconciliation.text_status_open'),
            BankReconciliation::CANCEL => __('base/bank_reconciliation.text_status_cancel'),
        ];
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);
        $list_status = $this->list_status;
        if (empty($request->filter_date_from)) {
            $request->request->add([
                'filter_date_from' => Helper::date(\Date::parse('first day of this month'))
            ]);
        }
        if (empty($request->filter_date_to)) {
            $request->request->add([
                'filter_date_to' => Helper::date(\Date::parse('last day of this month'))
            ]);
        }

        //Consulta
        $results = BankReconciliation::filter($request->all())
            ->sortable(['created_at' => 'desc'])->paginate($limit);

        //Vista
        return view('base.bank_reconciliations.index',
            compact('results', 'list_status'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('base.bank_reconciliations.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        if(empty($request->name)){
            return response()->json(['error' => __('base/bank_reconciliation.error_name')], 422);
        }
        if(empty($request->key)){
            return response()->json(['error' => __('base/bank_reconciliation.error_key')], 422);
        }
        $bank_reconciliation_line_temps = BankReconciliationLineTemp::where('key', '=', $request->key)->get();
        if ($bank_reconciliation_line_temps->isEmpty()) {
            return response()->json(['error' => __('base/bank_reconciliation.error_key')], 422);
        }

        \DB::connection('tenant')->beginTransaction();
        try {
            //Logica
            $company = Helper::defaultCompany(); //Empresa
            $bank_move_temps = BankMoveTemp::where('key', '=', $request->key)->get();
            $hidden_cfdi_download = $request->hidden_cfdi_download;
            $hidden_bank_move_temp = $request->hidden_bank_move_temp;

            //Guardar
            //Registro principal
            $bank_reconciliation= BankReconciliation::create([
                'created_uid' => \Auth::user()->id,
                'updated_uid' => \Auth::user()->id,
                'name' => $request->name,
                'file_template' => $bank_move_temps->first()->file_template,
                'company_id' => $company->id,
                'status' => BankReconciliation::OPEN,
                'key' => $request->key,
            ]);

            foreach($bank_reconciliation_line_temps as $bank_reconciliation_line_temp){
                $bank_reconciliation_line = BankReconciliationLine::create([
                    'created_uid' => \Auth::user()->id,
                    'updated_uid' => \Auth::user()->id,
                    'bank_reconciliation_id' => $bank_reconciliation->id,
                    'key' => $request->key,
                    'amount_reconciled' => $bank_reconciliation_line_temp->amount_reconciled,
                    'sort_order' => $bank_reconciliation_line_temp->sort_order,
                    'status' => 1,
                ]);

                //Agregamos los movimientos del banco y su respectiva relacion
                $bank_move_temps = BankMoveTemp::where('bank_reconciliation_line_temp_id', '=', $bank_reconciliation_line_temp->id)->get();
                foreach($bank_move_temps as $bank_move_temp){
                    $bank_move = BankMove::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'key' => $request->key,
                        'date' => $bank_move_temp->date,
                        'reference' => $bank_move_temp->reference,
                        'amount' => $bank_move_temp->amount,
                        'currency_id' => $bank_move_temp->currency_id,
                        'company_id' => $company->id,
                        'file_template' => $bank_move_temp->file_template,
                        'bank_reconciliation_line_id' => $bank_reconciliation_line->id,
                        'status' => 1,
                        'conciled' => 1,
                    ]);
                    //Elimina el movimiento del banco
                    $tmp = array_search($bank_move_temp->id, $hidden_bank_move_temp);
                    if($tmp || $tmp == 0){
                        unset($hidden_bank_move_temp[$tmp]);
                    }
                    foreach($bank_move_temp->bankMoveTempReconcileds as $bank_move_temp_reconciled){
                        $bank_move_reconciled = BankMoveReconciled::create([
                            'created_uid' => \Auth::user()->id,
                            'updated_uid' => \Auth::user()->id,
                            'bank_move_id' => $bank_move->id,
                            'reconciled_id' => $bank_move_temp_reconciled->reconciled_id,
                            'key' => $request->key,
                            'amount_reconciled' => $bank_move_temp_reconciled->amount_reconciled,
                            'bank_reconciliation_line_id' => $bank_reconciliation_line->id,
                            'sort_order' => $bank_move_temp_reconciled->sort_order,
                            'status' => 1,
                        ]);
                        //Elimina el movimiento de los cfdis
                        $tmp = array_search($bank_move_temp_reconciled->reconciled_id, $hidden_cfdi_download);
                        if($tmp || $tmp == 0){
                            unset($hidden_cfdi_download[$tmp]);
                        }

                        //Actualiza el cfdi a conciliado
                        $cfdi_download = CfdiDownload::findOrFail($bank_move_temp_reconciled->reconciled_id);
                        $cfdi_download->conciled = 1;
                        $cfdi_download->balance -= $bank_move_temp_reconciled->amount_reconciled;
                        $cfdi_download->save();
                    }
                }
            }

            //Insertamos movimientos de banco sin conciliar
            if(!empty($hidden_bank_move_temp)){
                foreach($hidden_bank_move_temp as $tmp){
                    $bank_move_temp = BankMoveTemp::findOrFail($tmp);
                    $bank_move = BankMove::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'key' => $request->key,
                        'date' => $bank_move_temp->date,
                        'reference' => $bank_move_temp->reference,
                        'amount' => $bank_move_temp->amount,
                        'currency_id' => $bank_move_temp->currency_id,
                        'company_id' => $company->id,
                        'file_template' => $bank_move_temp->file_template,
                        'bank_reconciliation_line_id' => null,
                        'status' => 1,
                        'conciled' => 0,
                    ]);
                }
            }

            //Insertamos los cfdi's que no fueron conciliadas
            if(!empty($hidden_cfdi_download)) {
                foreach ($hidden_cfdi_download as $tmp) {
                    /*$bank_reconciliation_unreconcile = BankReconciliationUnreconcile::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'bank_reconciliation_id' => $bank_reconciliation->id,
                        'cfdi_download_id' => $tmp,
                        'key' => $request->key,
                        'sort_order' => $bank_reconciliation_line_temp->sort_order,
                        'status' => 1,
                    ]);*/
                }
            }

            //eliminar registros temporales
            BankMoveTempReconciled::where('key', '=', $request->key)->delete();
            BankMoveTemp::where('key', '=', $request->key)->delete();
            BankReconciliationLineTemp::where('key', $request->key)->delete();

            //
            \DB::connection('tenant')->commit();

            //Mensaje
            flash(__('general.text_form_success_add'))->success();
            //
            return response()->json(['success' => true]);

        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            return response()->json(['error' => $e->getMessage()], 422);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Base\BankReconciliation  $bank_reconciliation
     * @return \Illuminate\Http\Response
     */
    public function show(BankReconciliation $bank_reconciliation)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Base\BankReconciliation  $bank_reconciliation
     * @return \Illuminate\Http\Response
     */
    public function edit(BankReconciliation $bank_reconciliation)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Base\BankReconciliation  $bank_reconciliation
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, BankReconciliation $bank_reconciliation)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Base\BankReconciliation  $bank_reconciliation
     * @return \Illuminate\Http\Response
     */
    public function destroy(BankReconciliation $bank_reconciliation)
    {
        \DB::connection('tenant')->beginTransaction();
        try {
        //
        $bank_reconciliation->updated_uid = \Auth::user()->id;
        $bank_reconciliation->status = BankReconciliation::CANCEL;
        $bank_reconciliation->save();

        foreach($bank_reconciliation->bankReconciliationLines as $bank_reconciliation_line){
            foreach($bank_reconciliation_line->bankMoveReconcileds as $bank_move_reconciled){
                $cfdi_download = CfdiDownload::findOrFail($bank_move_reconciled->reconciled_id);
                $cfdi_download->conciled = 0;
                $cfdi_download->balance += $bank_move_reconciled->amount_reconciled;
                $cfdi_download->save();
            }
        }

        BankMoveTempReconciled::where('key', '=', $bank_reconciliation->key)->delete();
        BankMoveTemp::where('key', '=', $bank_reconciliation->key)->delete();
        BankReconciliationLineTemp::where('key', $bank_reconciliation->key)->delete();
        BankMoveReconciled::where('key', '=', $bank_reconciliation->key)->delete();
        BankMove::where('key', '=', $bank_reconciliation->key)->delete();
        BankReconciliationUnreconcile::where('key', $bank_reconciliation->key)->delete();
        $bank_reconciliation->bankReconciliationLines()->delete();
        $bank_reconciliation->delete();

        //
        \DB::connection('tenant')->commit();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/base/bank-reconciliations');
        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            flash($e->getMessage())->error();
            return redirect('/base/bank-reconciliations');
        }
    }

    /**
     * Descargar plantilla
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function downloadTemplate(Request $request)
    {
        //Datos
        $data [] = (object)[
            'date' => '15-08-2020',
            'credit' => '1500',
            'debit' => '',
            'reference' => '1234567',
            'currency_code' => 'MXN',
        ];
        $data [] = (object)[
            'date' => '16-08-2020',
            'credit' => '',
            'debit' => '1000',
            'reference' => 'A2345',
            'currency_code' => 'MXN',
        ];

        //Descargar archivo
        while (ob_get_level()) ob_end_clean();
        ob_start();

        return Excel::download(new BankMoveTemplate($request, $data), __('base/bank_reconciliation.text_file_template') . '-' . config('app.name') . '.xlsx',\Maatwebsite\Excel\Excel::XLSX);
    }

    /**
     * Importa movimientos de banco
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function bankMoveTempTemplateImport(Request $request)
    {
        //Validacion
        //Validacion de archivos por extension
        if ($request->hasFile('file_template_file')) {
            $request->merge(['file_template_ext' => request()->file('file_template_file')->getClientOriginalExtension()]);
        }
        $validator = \Validator::make($request->all(), [
            'file_template_file' => 'required|max:2048',
            'file_template_file_ext' => 'nullable|in:xls,xlsx'
        ], [
            'file_template_file.*' => __('base/bank_reconciliation.error_file_template'),
            'file_template_file_ext.*' => __('base/bank_reconciliation.error_file_template'),
        ]);

        //Errores
        if ($validator->fails()) {
            $errors = '';
            foreach ($validator->errors()->all() as $message) {
                $errors = $message;
            }
            return response()->json(['error' => $errors], 422);
        }

        //Variables

        //Logica
        if ($request->ajax() ) {
            try {
                //Subimos el archivo
                //Logica
                $company = Helper::defaultCompany(); //Empresa

                //Si suben una imagen
                if ($request->hasFile('file_template_file')) {
                    $file_template = Helper::uploadFileImage('file_template_file', BankReconciliation::PATH_FILES,$company->id);
                    $request->merge(['file_template' => $file_template]);
                }
                $request->merge(['key' => Str::random(40)]); //Key para identificar todas las partidas de este archivo

                //Borramos todos los registros del usuario
                $bank_move_temps = BankMoveTemp::where('created_uid','=',\Auth::user()->id)->get();
                if($bank_move_temps->isNotEmpty()) {
                    foreach($bank_move_temps as $bank_move_temp) {
                        $bank_move_temp->bankMoveTempReconcileds()->delete();
                        $bank_move_temp->delete();
                    }
                }

                //Borramos la numeracion de control
                $bank_reconciliation_line_temps = BankReconciliationLineTemp::where('created_uid','=',\Auth::user()->id)->delete();

                //Importar
                Excel::import(new BankMoveTemplateImport($request), request()->file('file_template_file'));

                //Obtenemos registros
                $results = BankMoveTemp::where('key','=',$request->key)->get();

                $html = view('layouts.partials.bank_reconciliations.bank_move_temp', compact('results'))->render();

                return response()->json(['html' => $html,'key' => $request->key]);
            } catch (\Illuminate\Validation\ValidationException $e ) {
                $errors = '';
                foreach ($e->errors() as $message) {
                    $errors = $message;
                }
                return response()->json(['error' => $errors], 422);
            } catch (\Exception $e) {
                return response()->json(['error' => $e->getMessage()], 422);
            }
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Realiza conciliacion manual
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function bankMoveTempManualReconciled(Request $request)
    {
        //Validacion
        if(empty($request->selected_cfdi_download)){
            return response()->json(['error' => __('base/bank_reconciliation.error_cfdi_download_empty')], 422);
        }
        if(empty($request->selected_bank_move_temp)){
            return response()->json(['error' => __('base/bank_reconciliation.error_bank_move_temp_empty')], 422);
        }

        $cfdi_download_total = 0;
        $cfdi_download_currency_code = '';
        $cfdi_download_type = '';
        foreach($request->selected_cfdi_download as $tmp){
            $cfdi_download = CfdiDownload::findOrFail($tmp);
            $cfdi_download_total += $cfdi_download->amount_total;
            $currency_code = $cfdi_download->currency->code ?? '';
            if($cfdi_download_currency_code == ''){
                $cfdi_download_currency_code = $currency_code;
            }elseif($cfdi_download_currency_code != $currency_code){ //Valida que sean de la misma moneda
                return response()->json(['error' => __('base/bank_reconciliation.error_cfdi_download_currency_code')], 422);
            }
            $type = $cfdi_download->type;
            if($cfdi_download_type == ''){
                $cfdi_download_type = $type;
            }elseif($cfdi_download_type != $type){ //Valida que sean del mismo tipo
                return response()->json(['error' => __('base/bank_reconciliation.error_cfdi_download_type')], 422);
            }
            $results = BankMoveTempReconciled::where('reconciled_id','=',$tmp)->where('key', '=', $request->key)->get();
            if($results->isNotEmpty()){
                return response()->json(['error' => __('base/bank_reconciliation.error_selected_reconciled')], 422);
            }
        }

        $bank_move_temp_total = 0;
        $bank_move_temp_currency_code = '';
        $bank_move_temp_type = '';
        foreach($request->selected_bank_move_temp as $tmp){
            $bank_move_temp = BankMoveTemp::findOrFail($tmp);
            $bank_move_temp_total += abs($bank_move_temp->amount);
            $currency_code = $bank_move_temp->currency->code ?? '';
            if($bank_move_temp_currency_code == ''){
                $bank_move_temp_currency_code = $currency_code;
            }elseif($bank_move_temp_currency_code != $currency_code){ //Valida que sean de la misma moneda
                return response()->json(['error' => __('base/bank_reconciliation.error_bank_move_temp_currency_code')], 422);
            }
            $type = $bank_move_temp->amount < 0 ? 1 : 2;
            if($bank_move_temp_type == ''){
                $bank_move_temp_type = $type;
            }elseif($bank_move_temp_type != $type){ //Valida que sean del mismo tipo
                return response()->json(['error' => __('base/bank_reconciliation.error_bank_move_temp_type')], 422);
            }
            $results = BankMoveTempReconciled::where('bank_move_temp_id','=',$tmp)->where('key', '=', $request->key)->get();
            if($results->isNotEmpty()){
                return response()->json(['error' => __('base/bank_reconciliation.error_selected_reconciled')], 422);
            }
        }

        if($cfdi_download_currency_code != $bank_move_temp_currency_code){
            return response()->json(['error' => __('base/bank_reconciliation.error_cfdi_download_vs_bank_move_temp_currency_code')], 422);
        }
        if($cfdi_download_total != $bank_move_temp_total){
            return response()->json(['error' => __('base/bank_reconciliation.error_cfdi_download_vs_bank_move_temp_total')], 422);
        }
        if($cfdi_download_type != $bank_move_temp_type){
            return response()->json(['error' => __('base/bank_reconciliation.error_selected')], 422);
        }

        \DB::connection('tenant')->beginTransaction();
        try {
            //Crea numeracion de control
            $sort_order = 1;
            $bank_reconciliation_line_temps = BankReconciliationLineTemp::where('key', '=', $request->key)->get();
            if($bank_reconciliation_line_temps->isNotEmpty()){
                $sort_order = $bank_reconciliation_line_temps->max('sort_order') + 1;
            }
            $bank_reconciliation_line_temp = BankReconciliationLineTemp::create([
                'created_uid' => \Auth::user()->id,
                'updated_uid' => \Auth::user()->id,
                'key' => $request->key,
                'amount_reconciled' => $bank_move_temp_total,
                'sort_order' => $sort_order,
                'status' => 1,
            ]);

            $selected_cfdi_download = [];
            foreach($request->selected_cfdi_download as $tmp2) {
                $cfdi_download = CfdiDownload::findOrFail($tmp2);
                $selected_cfdi_download[$tmp2] = $cfdi_download->amount_total;
            }
            foreach($request->selected_bank_move_temp as $tmp){
                $bank_move_temp = BankMoveTemp::findOrFail($tmp);
                $bank_move_temp->bank_reconciliation_line_temp_id = $bank_reconciliation_line_temp->id; //Aasocia el ID de partida
                $bank_move_temp->save();
                $total = abs($bank_move_temp->amount);
                foreach($selected_cfdi_download as $cfdi_download_id => $balance){
                    if($total > 0 && $balance > 0) {
                        $amount_reconciled = 0;
                        if ($balance >= $total) {
                            $amount_reconciled = $total;
                            $total = 0;
                        } elseif ($balance < $total) {
                            $amount_reconciled = $balance;
                            $total -= $amount_reconciled;
                        }
                        $bank_move_temp_reconciled = BankMoveTempReconciled::create([
                            'created_uid' => \Auth::user()->id,
                            'updated_uid' => \Auth::user()->id,
                            'bank_move_temp_id' => $bank_move_temp->id,
                            'reconciled_id' => $cfdi_download_id,
                            'key' => $request->key,
                            'amount_reconciled' => $amount_reconciled,
                            'bank_reconciliation_line_temp_id' => $bank_reconciliation_line_temp->id, //Aasocia el ID de partida
                            'sort_order' => 0,
                            'status' => 1,
                        ]);
                        //Reduce el saldo y si ya no tiene saldo ya no lo aplica
                        $selected_cfdi_download[$cfdi_download_id] -= $amount_reconciled;
                        if($selected_cfdi_download[$cfdi_download_id] <= 0){
                            unset($selected_cfdi_download[$cfdi_download_id]);
                        }
                    }
                }
            }
            //
            \DB::connection('tenant')->commit();
            //
            return response()->json(['selected_cfdi_download' => $request->selected_cfdi_download, 'selected_bank_move_temp' => $request->selected_bank_move_temp, 'sort_order' => $sort_order]);

        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            return response()->json(['error' => $e->getMessage()], 422);
        }
    }

    /**
     * Realiza conciliacion manual
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function bankMoveTempManualUnreconciled(Request $request)
    {
        //Validacion
        if(empty($request->selected_cfdi_download) && empty($request->selected_bank_move_temp)){
            return response()->json(['error' => __('base/bank_reconciliation.error_selected')], 422);
        }

        \DB::connection('tenant')->beginTransaction();
        try{
            $selected_cfdi_download = [];
            $selected_bank_move_temp = [];
            if(!empty($request->selected_cfdi_download)){
                foreach($request->selected_cfdi_download as $tmp){
                    $results = BankMoveTempReconciled::where('reconciled_id','=',$tmp)->where('key', '=', $request->key)->get();
                    if($results->isNotEmpty()){
                        $result = $results->first();
                        $bank_move_temp_reconcileds = BankMoveTempReconciled::where('bank_reconciliation_line_temp_id','=',$result->bank_reconciliation_line_temp_id)->where('key', '=', $request->key)->get();
                        if($bank_move_temp_reconcileds->isNotEmpty()){
                            foreach($bank_move_temp_reconcileds as $tmp2){
                                $selected_cfdi_download[$tmp2->reconciled_id] = $tmp2->reconciled_id;
                                $selected_bank_move_temp[$tmp2->bank_move_temp_id] = $tmp2->bank_move_temp_id;
                            }
                        }
                        BankMoveTempReconciled::where('bank_reconciliation_line_temp_id','=',$result->bank_reconciliation_line_temp_id)->where('key', '=', $request->key)->delete();
                        BankMoveTemp::where('bank_reconciliation_line_temp_id','=',$result->bank_reconciliation_line_temp_id)->where('key', '=', $request->key)->update(['bank_reconciliation_line_temp_id' => null]);
                        BankReconciliationLineTemp::where('id','=',$result->bank_reconciliation_line_temp_id)->where('key', '=', $request->key)->delete();
                    }
                }
            }
            if(!empty($request->selected_bank_move_temp)) {
                foreach ($request->selected_bank_move_temp as $tmp) {
                    $results = BankMoveTemp::where('id','=',$tmp)->where('key', '=', $request->key)->get();
                    if($results->isNotEmpty()) {
                        $result = $results->first();
                        $bank_move_temp_reconcileds = BankMoveTempReconciled::where('bank_reconciliation_line_temp_id','=',$result->bank_reconciliation_line_temp_id)->where('key', '=', $request->key)->get();
                        if($bank_move_temp_reconcileds->isNotEmpty()){
                            foreach($bank_move_temp_reconcileds as $tmp2){
                                $selected_cfdi_download[$tmp2->reconciled_id] = $tmp2->reconciled_id;
                                $selected_bank_move_temp[$tmp2->bank_move_temp_id] = $tmp2->bank_move_temp_id;
                            }
                        }
                        BankMoveTempReconciled::where('bank_reconciliation_line_temp_id','=',$result->bank_reconciliation_line_temp_id)->where('key', '=', $request->key)->delete();
                        BankMoveTemp::where('bank_reconciliation_line_temp_id','=',$result->bank_reconciliation_line_temp_id)->where('key', '=', $request->key)->update(['bank_reconciliation_line_temp_id' => null]);
                        BankReconciliationLineTemp::where('id','=',$result->bank_reconciliation_line_temp_id)->where('key', '=', $request->key)->delete();
                    }
                }
            }

            //
            \DB::connection('tenant')->commit();
            //
            return response()->json(['selected_cfdi_download' => $selected_cfdi_download, 'selected_bank_move_temp' => $selected_bank_move_temp]);
        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            return response()->json(['error' => $e->getMessage()], 422);
        }
    }
    /**
     * Realiza conciliacion manual
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function bankMoveTempAutoReconciled(Request $request)
    {
        //Validacion
        if(empty($request->hidden_cfdi_download)){
            return response()->json(['error' => __('base/bank_reconciliation.error_cfdi_download_empty')], 422);
        }
        if(empty($request->hidden_bank_move_temp)){
            return response()->json(['error' => __('base/bank_reconciliation.error_bank_move_temp_empty')], 422);
        }

        $reconciles = [];
        $hidden_cfdi_download = $request->hidden_cfdi_download;
        foreach($request->hidden_bank_move_temp as $tmp){
            $results = BankMoveTempReconciled::where('bank_move_temp_id','=',$tmp)->where('key', '=', $request->key)->get();
            if($results->isNotEmpty()) {
                continue; //Si ya esta conciliado lo omitimos
            }
            $bank_move_temp = BankMoveTemp::findOrFail($tmp);
            $currency_code = $bank_move_temp->currency->code ?? '';
            $type = $bank_move_temp->amount < 0 ? 1 : 2;
            foreach($hidden_cfdi_download as $keytmp2 => $tmp2){
                $results = BankMoveTempReconciled::where('reconciled_id','=',$tmp2)->where('key', '=', $request->key)->get();
                if($results->isNotEmpty()){
                    continue; //Si ya esta conciliado lo omitimos
                }
                $cfdi_download = CfdiDownload::findOrFail($tmp2);
                //Buscamos conciliacion solo si es de la misma moneda y tipo
                $cfdi_download_currency_code = $cfdi_download->currency->code ?? '';
                $cfdi_download_type = $cfdi_download->type;
                //Si son de la misma fecha y el mismo monto lo conciliamos
                if($cfdi_download_currency_code == $currency_code && $cfdi_download_type == $type && abs($bank_move_temp->amount) == $cfdi_download->amount_total && Helper::date(Helper::createDateTimeFromSql($cfdi_download->date)) == Helper::date(Helper::createDateTimeFromSql($bank_move_temp->date))){
                    $reconciles[$bank_move_temp->id] = $tmp2;
                    unset($hidden_cfdi_download[$keytmp2]);
                    break;
                }elseif($cfdi_download_currency_code == $currency_code && $cfdi_download_type == $type && abs($bank_move_temp->amount) == $cfdi_download->amount_total){
                    $amount_total_temp = $cfdi_download->amount_total;
                    $valid = true;
                    //Solo si son del mismo monto, pero valida que no exista otro movimiento
                    //con el mismo importe si es haci no lo concilia
                    foreach($hidden_cfdi_download as $keytmp3 => $tmp3){
                        if($tmp2 != $tmp3) { //Que sean id's diferentes
                            $results = BankMoveTempReconciled::where('reconciled_id', '=', $tmp3)->where('key', '=', $request->key)->get();
                            if ($results->isNotEmpty()) {
                                continue; //Si ya esta conciliado lo omitimos
                            }
                            $cfdi_download = CfdiDownload::findOrFail($tmp3);
                            //Buscamos conciliacion solo si es de la misma moneda y tipo
                            $cfdi_download_currency_code = $cfdi_download->currency->code ?? '';
                            $cfdi_download_type = $cfdi_download->type;
                            if($cfdi_download_currency_code == $currency_code && $cfdi_download_type == $type && $amount_total_temp == $cfdi_download->amount_total){
                                //si encuentra otro cfdi con el mismo importe lo descarta
                                $valid = false;
                                break;
                            }
                        }
                    }
                    if($valid){
                        $reconciles[$bank_move_temp->id] = $tmp2;
                        unset($hidden_cfdi_download[$keytmp2]);
                        break;
                    }
                }
            }
        }

        \DB::connection('tenant')->beginTransaction();
        try {
            $selected_reconciliation_line = [];
            foreach($reconciles as $tmp => $tmp2) {
                $selected_cfdi_download = [];
                $selected_bank_move_temp = [];
                $selected_cfdi_download[$tmp2] = $tmp2;
                $selected_bank_move_temp[$tmp] = $tmp;
                //Crea numeracion de control
                $sort_order = 1;
                $bank_reconciliation_line_temps = BankReconciliationLineTemp::where('key', '=', $request->key)->get();
                if ($bank_reconciliation_line_temps->isNotEmpty()) {
                    $sort_order = $bank_reconciliation_line_temps->max('sort_order') + 1;
                }
                $bank_move_temp = BankMoveTemp::findOrFail($tmp);
                $bank_reconciliation_line_temp = BankReconciliationLineTemp::create([
                    'created_uid' => \Auth::user()->id,
                    'updated_uid' => \Auth::user()->id,
                    'key' => $request->key,
                    'amount_reconciled' => abs($bank_move_temp->amount),
                    'sort_order' => $sort_order,
                    'status' => 1,
                ]);
                $bank_move_temp->bank_reconciliation_line_temp_id = $bank_reconciliation_line_temp->id; //Aasocia el ID de partida
                $bank_move_temp->save();
                $selected_reconciliation_line[$sort_order] = [
                    'selected_cfdi_download' => $selected_cfdi_download,
                    'selected_bank_move_temp' => $selected_bank_move_temp
                ];
                $cfdi_download = CfdiDownload::findOrFail($tmp2);
                $bank_move_temp_reconciled = BankMoveTempReconciled::create([
                    'created_uid' => \Auth::user()->id,
                    'updated_uid' => \Auth::user()->id,
                    'bank_move_temp_id' => $bank_move_temp->id,
                    'reconciled_id' => $cfdi_download->id,
                    'key' => $request->key,
                    'amount_reconciled' => $cfdi_download->amount_total,
                    'bank_reconciliation_line_temp_id' => $bank_reconciliation_line_temp->id,
                    //Aasocia el ID de partida
                    'sort_order' => 0,
                    'status' => 1,
                ]);
            }
            //
            \DB::connection('tenant')->commit();
            //
            return response()->json(['selected_reconciliation_line' => $selected_reconciliation_line]);

        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            return response()->json(['error' => $e->getMessage()], 422);
        }
    }

    /**
     * Clase generica de impresion
     *
     * @param BankReconciliation $bank_reconciliation
     * @return mixed
     */
    public function print(BankReconciliation $bank_reconciliation)
    {
        //PDF
        $pdf = \PDF::loadView('base.bank_reconciliations.pdf', compact('bank_reconciliation'));

        //Redireccion
        return $pdf->stream($bank_reconciliation->name . '.pdf');
    }

    /**
     * Exportar datos a excel
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function exportToExcel(Request $request)
    {
        while (ob_get_level()) ob_end_clean();
        ob_start();

        return Excel::download(new BankReconciliationsExport($request),
            __('base/bank_reconciliation.document_title') . '-' . config('app.name') . '.xlsx');
    }

}
