<?php

namespace App\Http\Controllers\Base;

use App\Helpers\Helper;
use App\Models\Base\BranchOffice;
use App\Models\Catalogs\Country;
use App\Models\Catalogs\Postcode;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;

class BranchOfficeController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = BranchOffice::filter($request->all())->sortable('name')->paginate($limit);

        //Vista
        return view('base.branch_offices.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $countries = Country::populateSelect()->get()->pluck('name_sat', 'id');

        return view('base.branch_offices.create', compact('countries'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);

        //Logica
        $company = Helper::defaultCompany(); //Empresa
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo
        $request->merge(['company_id' => $company->id]);

        //Guardar
        //Registro principal
        $branch_office = BranchOffice::create($request->input());

        //Si suben una imagen
        if ($request->hasFile('file_logo_docs')) {
            $logo_docs = Helper::uploadFileImage('file_logo_docs', BranchOffice::PATH_LOGO_DOCS,$company->id);
            $branch_office->logo_docs = $logo_docs;
            $branch_office->save();
        }

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/base/branch-offices');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Base\BranchOffice  $branchOffice
     * @return \Illuminate\Http\Response
     */
    public function show(BranchOffice $branch_office)
    {
        return redirect('/base/branch-offices');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Base\BranchOffice  $branch_office
     * @return \Illuminate\Http\Response
     */
    public function edit(BranchOffice $branch_office)
    {
        $countries = Country::populateSelect()->get()->pluck('name_sat', 'id');

        return view('base.branch_offices.edit', compact('branch_office',  'countries'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Base\BranchOffice  $branch_office
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, BranchOffice $branch_office)
    {
        //Validacion
        $this->validation($request);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $branch_office->fill($request->only([
            'updated_uid',
            'name',
            'email',
            'phone',
            'phone_mobile',
            'address_1',
            'address_2',
            'address_3',
            'address_4',
            'address_5',
            'address_6',
            'city_id',
            'state_id',
            'country_id',
            'postcode',
            'comment',
            'sort_order',
            'status',
            'logo_docs',
        ]));

        //Guardar
        //Registro principal
        $branch_office->save();

        //Si suben una imagen
        if ($request->hasFile('file_logo_docs')) {
            //Si ya tenia un archivo lo eliminamos
            if (!empty($branch_office->logo_docs)) {
                \Storage::delete(Helper::setDirectory(BranchOffice::PATH_LOGO_DOCS,$branch_office->company_id) . '/' . $branch_office->logo_docs);
            }
            $logo_docs = Helper::uploadFileImage('file_logo_docs', BranchOffice::PATH_LOGO_DOCS,$branch_office->company_id);
            $branch_office->logo_docs = $logo_docs;
            $branch_office->save();
        }

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/base/branch-offices');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Base\BranchOffice  $branch_office
     * @return \Illuminate\Http\Response
     */
    public function destroy(BranchOffice $branch_office)
    {
        //Logica
        $branch_office->updated_uid = \Auth::user()->id;
        $branch_office->status = 0;
        $branch_office->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/base/branch-offices');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
            'email' => 'nullable|email',
            'country_id' => 'required|integer',
            'postcode' => 'required',
            'file_logo_docs' => 'nullable|mimes:jpeg,jpg,png|max:2048',
        ], [
            'name.*' => __('base/branch_office.error_name'),
            'email.email' => __('base/branch_office.error_email_format'),
            'country_id.*' => __('base/branch_office.error_country_id'),
            'postcode.*' => __('base/branch_office.error_postcode'),
            'file_logo_docs.*' => __('base/branch_office.error_logo_docs'),
        ]);
    }

    /**
     * Obtener registro
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getBranchOffice(Request $request)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            $branch_office = BranchOffice::findOrFail($id);
            $postcode = Postcode::where('name','=',$branch_office->postcode)->first();
            $hours = 0;
            if(!empty($postcode) && !in_array(abs($postcode->diff2), [5, 6]) ){
                $hours = 6 - abs($postcode->diff2);
            }
            $cfdi_date = \Date::now();
            if($hours > 0){
                $cfdi_date->addHours($hours);
            }elseif($hours < 0){
                $cfdi_date->subHours(abs($hours));
            }
            $branch_office->cfdi_date_time = Helper::dateTime($cfdi_date);
            return response()->json($branch_office,200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }
}
