<?php

namespace App\Http\Controllers\Base;

use App\Helpers\Helper;
use App\Http\Middleware\CheckDemo;
use App\Http\Middleware\CheckSuperadmin;
use App\Models\Base\BranchOffice;
use App\Models\Base\Company;
use App\Models\Base\CompanyBankAccount;
use App\Models\Base\DocumentType;
use App\Models\Base\Pac;
use App\Models\Catalogs\Bank;
use App\Models\Catalogs\Country;
use App\Models\Catalogs\Currency;
use App\Models\Catalogs\TaxRegimen;
use Hyn\Tenancy\Environment;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use SoapClient;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Process\Process;

class CompanyController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        //$this->middleware(CheckSuperadmin::class,['only' => ['create','store','show','destroy']]);
        $this->middleware(CheckDemo::class,['only' => ['update']]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);
        if (!isset($request->filter_status) && $request->filter_status != 'all') {
            $request->request->add([
                'filter_status' => '1'
            ]);
        }

        // Redirecciona para crear empresa
        if(!empty(setting('multi_taxid'))){
            //Consulta
            $results = Company::filter($request->all());
            if(isset($request->filter_status) && $request->filter_status != 'all'){
                $results = $results->where('companies.status','=',(int)$request->filter_status);
            }
            $results = $results->sortable('name')
                ->paginate($limit);

            //Vista
            return view('base.companies.index', compact('results'));
        }

        //Redirecciona para editar
        $company = Company::get()->first();
        return redirect(route('companies.edit', [$company->id]));

    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $tax_regimens = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');
        $countries = Country::populateSelect()->get()->pluck('name_sat', 'id');
        $banks = Bank::populateSelect()->pluck('name', 'id');
        $currencies = Currency::populateSelect()->get()->pluck('name_sat', 'id');

        return view('base.companies.create', compact('tax_regimens', 'countries','banks','currencies'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     * @throws ValidationException
     */
    public function store(Request $request)
    {
        //Validacion
        $request->merge(['name' => trim(str_replace('  ', ' ', $request->name))]);
        $request->merge(['taxid' => strtoupper(trim($request->taxid))]);
        $request->merge(['curp' => strtoupper(trim($request->curp))]);
        $this->validation($request);
        $this->validate($request, [
            'taxid' => 'required',
        ], [
            'taxid.required' => __('base/company.error_taxid'),
            'taxid.unique' => __('base/company.error_taxid_unique'),
        ]);

        //Validaciones manuales
        $validator = \Validator::make([], []);

        //Valida que no exista un directorio con el mismo nombre
        /*if(\Storage::exists($request->taxid)) {
            $validator->after(function ($validator) {
                $validator->errors()->add('taxid', __('base/company.error_taxid_directory_exists'));
            });
        }
        if ($validator->fails()) {
            throw new ValidationException($validator);
        }*/

        \DB::connection('tenant')->beginTransaction();
        try {
            //Logica
            $request->merge(['created_uid' => \Auth::user()->id]);
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['curp' => strtoupper(trim($request->curp))]);
            $request->merge(['sort_order' => 0]);
            $request->merge(['status' => 1]); //Por default activo
            $request->merge(['module_payroll' => 1]); //Por default activo para nominas
            $request->merge(['path_files' => (string) Str::uuid()]);
            //Si suben una imagen
            /*if ($request->hasFile('file_image')) {
                $image = Helper::uploadFileImage('file_image', Company::PATH_IMAGES);
                $request->merge(['image' => $image]);
            }*/
            //Archivos SAT
            //Convertir en CER a PEM
            /*$path_file_cer_pem = '';
            if ($request->hasFile('file_file_cer')) {
                $tmp = $this->convertCerToPem(); //
                $request->merge(['file_cer' => !empty($tmp['file_cer']) ? $tmp['file_cer'] : null]);
                $request->merge(['certificate_number' => !empty($tmp['certificate_number']) ? $tmp['certificate_number'] : null]);
                $request->merge(['date_start' => !empty($tmp['date_start']) ? $tmp['date_start'] : null]);
                $request->merge(['date_end' => !empty($tmp['date_end']) ? $tmp['date_end'] : null]);
                $path_file_cer_pem = !empty($tmp['path_file_cer_pem']) ? $tmp['path_file_cer_pem'] : null;
            }
            //Convertir en KEY a PEM, debe contener la contraseña
            $path_file_key_pem = '';
            if ($request->hasFile('file_file_key')) {
                $tmp = $this->convertKeyToPem($request->password_key, $path_file_cer_pem); //
                $request->merge(['password_key' => !empty($tmp['password_key']) ? $tmp['password_key'] : null]);
                $request->merge(['file_key' => !empty($tmp['file_key']) ? $tmp['file_key'] : null]);
                $path_file_key_pem = !empty($tmp['path_file_key_pem']) ? $tmp['path_file_key_pem'] : null;
            }
            //Crear archivo PFX
            if (!empty($path_file_cer_pem) && !empty($path_file_key_pem)) {
                $tmp = $this->createPfx($path_file_key_pem, $request->password_key, $path_file_cer_pem); //
                $request->merge(['file_pfx' => !empty($tmp['file_pfx']) ? $tmp['file_pfx'] : null]);
            }*/

            //Guardar
            //Registro principal
            $company = Company::create($request->input());

            //Si suben una imagen
            if ($request->hasFile('file_image')) {
                $image = Helper::uploadFileImage('file_image', Company::PATH_IMAGES, $company->id);
                $company->image = $image;
                $company->save();
            }

            //Archivos SAT
            //Convertir en CER a PEM
            $path_file_cer_pem = '';
            if ($request->hasFile('file_file_cer')) {
                $tmp = $this->convertCerToPem($company); //
                $company->file_cer = !empty($tmp['file_cer']) ? $tmp['file_cer'] : null;
                $company->certificate_number = !empty($tmp['certificate_number']) ? $tmp['certificate_number'] : null;
                $company->date_start = !empty($tmp['date_start']) ? $tmp['date_start'] : null;
                $company->date_end = !empty($tmp['date_end']) ? $tmp['date_end'] : null;
                $company->titular = !empty($tmp['titular']) ? $tmp['titular'] : null;
                $path_file_cer_pem = !empty($tmp['path_file_cer_pem']) ? $tmp['path_file_cer_pem'] : null;
            }
            //Convertir en KEY a PEM, debe contener la contraseña
            $path_file_key_pem = '';
            if ($request->hasFile('file_file_key')) {
                $tmp = $this->convertKeyToPem($company,$request->password_key, $path_file_cer_pem); //
                $company->password_key = !empty($tmp['password_key']) ? $tmp['password_key'] : null;
                $company->file_key = !empty($tmp['file_key']) ? $tmp['file_key'] : null;
                $path_file_key_pem = !empty($tmp['path_file_key_pem']) ? $tmp['path_file_key_pem'] : null;
            }
            //Crear archivo PFX
            if (!empty($path_file_cer_pem) && !empty($path_file_key_pem)) {
                $tmp = $this->createPfx($company,$path_file_key_pem, $request->password_key, $path_file_cer_pem); //
                $company->file_pfx = !empty($tmp['file_pfx']) ? $tmp['file_pfx'] : null;
            }

            $company->save();

            //Cuentas bancarias
            //Guarda
            if (!empty($request->item_bank_account)) {
                foreach ($request->item_bank_account as $key => $result) {
                    $company_bank_account = CompanyBankAccount::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'company_id' => $company->id,
                        'name' => $result['name'],
                        'bank_id' => $result['bank_id'],
                        'currency_id' => $result['currency_id'],
                        'account_number' => $result['account_number'],
                        'sort_order' => $key,
                        'status' => 1,
                    ]);
                }
            }

            //Crear datos iniciales
            //---Tpos de documento
            DocumentType::create([
                'name' => 'Factura',
                'code' => 'customer.invoice',
                'prefix' => 'FAC',
                'current_number' => 0,
                'increment_number' => 1,
                'nature' => \App\Models\Base\DocumentType::DEBIT,
                'cfdi_type_id' => \App\Models\Catalogs\CfdiType::where('code','=','I')->get()->random()->id,
                'sort_order' => 1,
                'status' => 1,
                'company_id' => $company->id,
            ]);
            DocumentType::create([
                'name' => 'Nota de crédito',
                'code' => 'customer.credit_note',
                'prefix' => 'NCR',
                'current_number' => 0,
                'increment_number' => 1,
                'nature' => \App\Models\Base\DocumentType::CREDIT,
                'cfdi_type_id' => \App\Models\Catalogs\CfdiType::where('code','=','E')->get()->random()->id,
                'sort_order' => 2,
                'status' => 1,
                'company_id' => $company->id,
            ]);
            DocumentType::create([
                'name' => 'Recibo de Pago',
                'code' => 'customer.payment',
                'prefix' => 'PAG',
                'current_number' => 0,
                'increment_number' => 1,
                'nature' => \App\Models\Base\DocumentType::CREDIT,
                'cfdi_type_id' => \App\Models\Catalogs\CfdiType::where('code','=','P')->get()->random()->id,
                'sort_order' => 3,
                'status' => 1,
                'company_id' => $company->id,
            ]);
            DocumentType::create([
                'name' => 'Recibo de arrendamiento',
                'code' => 'customer.lease',
                'prefix' => 'ARR',
                'current_number' => 0,
                'increment_number' => 1,
                'nature' => \App\Models\Base\DocumentType::DEBIT,
                'cfdi_type_id' => \App\Models\Catalogs\CfdiType::where('code','=','I')->get()->random()->id,
                'sort_order' => 4,
                'status' => 1,
                'company_id' => $company->id,
            ]);
            DocumentType::create([
                'name' => 'Recibo de honorarios',
                'code' => 'customer.fee',
                'prefix' => 'HON',
                'current_number' => 0,
                'increment_number' => 1,
                'nature' => \App\Models\Base\DocumentType::DEBIT,
                'cfdi_type_id' => \App\Models\Catalogs\CfdiType::where('code','=','I')->get()->random()->id,
                'sort_order' => 5,
                'status' => 1,
                'company_id' => $company->id,
            ]);
            DocumentType::create([
                'name' => 'Carta porte',
                'code' => 'customer.transfer',
                'prefix' => 'TRA',
                'current_number' => 0,
                'increment_number' => 1,
                'nature' => \App\Models\Base\DocumentType::NO_NATURE,
                'cfdi_type_id' => \App\Models\Catalogs\CfdiType::where('code','=','T')->get()->random()->id,
                'sort_order' => 6,
                'status' => 1,
                'company_id' => $company->id,
            ]);
            DocumentType::create([
                'name' => 'Cotización',
                'code' => 'customer.quotation',
                'prefix' => 'CTZ',
                'current_number' => 0,
                'increment_number' => 1,
                'nature' => \App\Models\Base\DocumentType::NO_NATURE,
                'cfdi_type_id' => null,
                'sort_order' => 7,
                'status' => 1,
                'company_id' => $company->id,
            ]);
            DocumentType::create([
                'name' => 'Recibo de nómina',
                'code' => 'employee.payroll',
                'prefix' => 'NOM',
                'current_number' => 0,
                'increment_number' => 1,
                'nature' => \App\Models\Base\DocumentType::NO_NATURE,
                'cfdi_type_id' => \App\Models\Catalogs\CfdiType::where('code','=','N')->get()->random()->id,
                'sort_order' => 8,
                'status' => 1,
                'company_id' => $company->id,
            ]);
            DocumentType::create([
                'name' => 'Nota de venta',
                'code' => 'customer.remission',
                'prefix' => 'CVN',
                'current_number' => 0,
                'increment_number' => 1,
                'nature' => \App\Models\Base\DocumentType::NO_NATURE,
                'cfdi_type_id' => null,
                'sort_order' => 9,
                'status' => 1,
                'company_id' => $company->id,
            ]);
            //---Sucursales
            BranchOffice::create([
                'name' => 'MATRIZ',
                'email' => '',
                'phone' => '',
                'address_1' => '',
                'address_2' => '',
                'address_4' => '',
                'country_id' => $company->country->id,
                'postcode' => $company->postcode,
                'company_id' => $company->id,
            ]);

            //Crea un cliente en automatico para finkok de timbrado bajo demanda
            $pac = Pac::findOrFail(setting('default_pac_id')); //PAC
            if(!empty($pac)){
                if(preg_match('/finkok/i', $pac->code)) {
                    $client = new SoapClient(str_replace('cancel', 'registration', $pac->ws_url_cancel));
                    $params = [
                        'reseller_username' => $pac->username,
                        'reseller_password' => Crypt::decryptString($pac->password),
                        'taxpayer_id' => $company->taxid,
                        'type_user' => 'O',
                    ];
                    $response = $client->__soapCall('add', ['parameters' => $params]);
                }
            }

            \DB::connection('tenant')->commit();

            //Mensaje
            flash(__('general.text_form_success_add'))->success();

            //Redireccion
            if(!empty(setting('multi_taxid'))){
                return redirect(route('companies.index'));
            }else{
                return redirect(route('companies.edit', [$company->id]));
            }
        } catch (ProcessFailedException $e) {
            if(\Storage::exists($request->taxid)) {
                \Storage::deleteDirectory($request->taxid);
            }
            //Fix fecha
            if ($request->hasFile('file_file_cer')) {
                if (!empty($request->date_start)) {
                    $request->merge([
                        'date_start' => Helper::convertSqlToDateTime($request->date_start),
                    ]);
                }
                if (!empty($request->date_end)) {
                    $request->merge([
                        'date_end' => Helper::convertSqlToDateTime($request->date_end),
                    ]);
                }
            }
            $msg = $e->getMessage();
            if(strstr( $msg, 'rror decrypting ke')){
                $msg = __('general.error_password_file_key');
            }
            \DB::connection('tenant')->rollback();
            flash(nl2br($msg))->error();
            return back()->withInput();
        } catch (\Exception $e) {
            if(\Storage::exists($request->taxid)) {
                \Storage::deleteDirectory($request->taxid);
            }
            //Fix fecha
            if ($request->hasFile('file_file_cer')) {
                if (!empty($request->date_start)) {
                    $request->merge([
                        'date_start' => Helper::convertSqlToDateTime($request->date_start),
                    ]);
                }
                if (!empty($request->date_end)) {
                    $request->merge([
                        'date_end' => Helper::convertSqlToDateTime($request->date_end),
                    ]);
                }
            }

            \DB::connection('tenant')->rollback();
            flash($e->getMessage())->error();
            return back()->withInput();
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Base\Company $company
     * @return \Illuminate\Http\Response
     */
    public function show(Company $company)
    {
        return redirect('/base/companies');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Base\Company $company
     * @return \Illuminate\Http\Response
     */
    public function edit(Company $company)
    {
        $tax_regimens = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');
        $countries = Country::populateSelect()->get()->pluck('name_sat', 'id');
        $banks = Bank::populateSelect()->pluck('name', 'id');
        $currencies = Currency::populateSelect()->get()->pluck('name_sat', 'id');

        return view('base.companies.edit', compact('company', 'tax_regimens', 'countries','banks','currencies'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  \App\Models\Base\Company $company
     * @return \Illuminate\Http\Response
     * @throws ValidationException
     */
    public function update(Request $request, Company $company)
    {
        //Validacion
        $request->merge(['name' => trim(str_replace('  ', ' ', $request->name))]);
        $request->merge(['taxid' => strtoupper(trim($request->taxid))]);
        $request->merge(['curp' => strtoupper(trim($request->curp))]);
        $this->validation($request);
        $this->validate($request, [
            'taxid' => [
                'required',
            ],
        ], [
            'taxid.required' => __('base/company.error_taxid'),
            'taxid.unique' => __('base/company.error_taxid_unique'),
        ]);

        //Validaciones manuales
        $validator = \Validator::make([], []);

        //Valida que no exista un directorio con el mismo nombre
        /*if($company->taxid != $request->taxid && \Storage::exists($request->taxid)) {
            $validator->after(function ($validator) {
                $validator->errors()->add('taxid', __('base/company.error_taxid_directory_exists'));
            });
        }
        if ($validator->fails()) {
            throw new ValidationException($validator);
        }*/

        \DB::connection('tenant')->beginTransaction();
        try {
            //Rfc para validar cambio de directorio
            $taxid_old = $company->taxid;

            //Logica
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['curp' => strtoupper(trim($request->curp))]);
            $request->merge(['sort_order' => 0]);
            $request->merge(['status' => !empty($request->status) ? 1 : 0]);
            $company->fill($request->only([
                'updated_uid',
                'name',
                'image',
                'taxid',
                'tax_regimen_id',
                'email',
                'phone',
                'phone_mobile',
                'address_1',
                'address_2',
                'address_3',
                'address_4',
                'address_5',
                'address_6',
                'city_id',
                'state_id',
                'country_id',
                'postcode',
                'comment',
                'sort_order',
                'status',
                'curp',
                'employer_register',
                'comercial_name',
                'tax_regimen_id2',
            ]));

            //Si suben una imagen
            if ($request->hasFile('file_image')) {
                //Si ya tenia un archivo lo eliminamos
                if (!empty($company->image)) {
                    \Storage::delete(Helper::setDirectory(Company::PATH_IMAGES,$company->id) . '/' . $company->image);
                }
                $image = Helper::uploadFileImage('file_image', Company::PATH_IMAGES,$company->id);
                $company->image = $image;
            } else {
                $company->image = $request->image; //si no tiene archivo sobreescribimos elque tenia
            }

            //Archivos SAT
            //Convertir en CER a PEM
            $path_file_cer_pem = '';
            if ($request->hasFile('file_file_cer')) {
                $tmp = $this->convertCerToPem($company); //
                $company->file_cer = !empty($tmp['file_cer']) ? $tmp['file_cer'] : null;
                $company->certificate_number = !empty($tmp['certificate_number']) ? $tmp['certificate_number'] : null;
                $company->date_start = !empty($tmp['date_start']) ? $tmp['date_start'] : null;
                $company->date_end = !empty($tmp['date_end']) ? $tmp['date_end'] : null;
                $company->titular = !empty($tmp['titular']) ? $tmp['titular'] : null;
                $path_file_cer_pem = !empty($tmp['path_file_cer_pem']) ? $tmp['path_file_cer_pem'] : null;
            }
            //Convertir en KEY a PEM, debe contener la contraseña
            $path_file_key_pem = '';
            if ($request->hasFile('file_file_key')) {
                $tmp = $this->convertKeyToPem($company,$request->password_key, $path_file_cer_pem); //
                $company->password_key = !empty($tmp['password_key']) ? $tmp['password_key'] : null;
                $company->file_key = !empty($tmp['file_key']) ? $tmp['file_key'] : null;
                $path_file_key_pem = !empty($tmp['path_file_key_pem']) ? $tmp['path_file_key_pem'] : null;
            }
            //Crear archivo PFX
            if (!empty($path_file_cer_pem) && !empty($path_file_key_pem)) {
                $tmp = $this->createPfx($company,$path_file_key_pem, $request->password_key, $path_file_cer_pem); //
                $company->file_pfx = !empty($tmp['file_pfx']) ? $tmp['file_pfx'] : null;
            }

            //Guardar
            //Registro principal
            $company->save();

            //Cuentas bancarias
            //Elimina
            if (!empty($request->delete_item_bank_account)) {
                foreach ($request->delete_item_bank_account as $key => $result) {
                    //Actualizar status
                    $company_bank_account = CompanyBankAccount::findOrFail($result);
                    $company_bank_account->updated_uid = \Auth::user()->id;
                    $company_bank_account->status = 0;
                    $company_bank_account->save();
                }
            }
            //Guarda
            if (!empty($request->item_bank_account)) {
                foreach ($request->item_bank_account as $key => $result) {
                    //Datos
                    $data = [
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'company_id' => $company->id,
                        'name' => $result['name'],
                        'bank_id' => $result['bank_id'],
                        'currency_id' => $result['currency_id'],
                        'account_number' => $result['account_number'],
                        'sort_order' => $key,
                        'status' => 1,
                    ];
                    //Valida si es registro nuevo o actualizacion
                    if (!empty($result['id'])) {
                        $company_bank_account = CompanyBankAccount::findOrFail($result['id']);
                        $company_bank_account->fill(array_only($data,[
                            'updated_uid',
                            'company_id',
                            'name',
                            'bank_id',
                            'currency_id',
                            'account_number',
                            'sort_order',
                        ]));
                        $company_bank_account->save();
                    }else {
                        $company_bank_account = CompanyBankAccount::create($data);
                    }
                }
            }

            if($taxid_old != $company->taxid){
                //Crea un cliente en automatico para finkok de timbrado bajo demanda
                $pac = Pac::findOrFail(setting('default_pac_id')); //PAC
                if(!empty($pac)){
                    if(preg_match('/finkok/i', $pac->code)) {
                        $client = new SoapClient(str_replace('cancel', 'registration', $pac->ws_url_cancel));
                        $params = [
                            'reseller_username' => $pac->username,
                            'reseller_password' => Crypt::decryptString($pac->password),
                            'taxpayer_id' => $company->taxid,
                            'type_user' => 'O',
                        ];
                        $response = $client->__soapCall('add', ['parameters' => $params]);
                    }
                }
            }

            //Si el RFC cambia renombramos
            if(!empty($taxid_old) && \Storage::exists($taxid_old) && $taxid_old != $company->taxid){
                \Storage::move($taxid_old, $company->taxid);
            }

            \DB::connection('tenant')->commit();

            //Mensaje
            flash(__('general.text_form_success_edit'))->success();

            //Redireccion
            if(!empty(setting('multi_taxid'))){
                return redirect(route('companies.index'));
            }else{
                return redirect(route('companies.edit', [$company->id]));
            }
        } catch (ProcessFailedException $e) {
            //Fix fecha
            if ($request->hasFile('file_file_cer')) {
                if (!empty($request->date_start)) {
                    $request->merge([
                        'date_start' => Helper::convertSqlToDateTime($request->date_start),
                    ]);
                }
                if (!empty($request->date_end)) {
                    $request->merge([
                        'date_end' => Helper::convertSqlToDateTime($request->date_end),
                    ]);
                }
            }

            $msg = $e->getMessage();
            if(strstr( $msg, 'rror decrypting ke')){
                $msg = __('general.error_password_file_key');
            }
            \DB::connection('tenant')->rollback();
            flash(nl2br($msg))->error();
            return back()->withInput();
        } catch (\Exception $e) {
            //Fix fecha
            if ($request->hasFile('file_file_cer')) {
                if (!empty($request->date_start)) {
                    $request->merge([
                        'date_start' => Helper::convertSqlToDateTime($request->date_start),
                    ]);
                }
                if (!empty($request->date_end)) {
                    $request->merge([
                        'date_end' => Helper::convertSqlToDateTime($request->date_end),
                    ]);
                }
            }

            \DB::connection('tenant')->rollback();
            flash($e->getMessage())->error();
            return back()->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Base\Company $company
     * @return \Illuminate\Http\Response
     */
    public function destroy(Company $company)
    {
        //Logica
        $company->updated_uid = \Auth::user()->id;
        $company->status = 0;
        $company->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/base/companies');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        //Validacion de archivos por extension
        if ($request->hasFile('file_file_cer')) {
            $request->merge(['file_file_cer_ext' => strtolower(request()->file('file_file_cer')->getClientOriginalExtension())]);
        }
        if ($request->hasFile('file_file_key')) {
            $request->merge(['file_file_key_ext' => strtolower(request()->file('file_file_key')->getClientOriginalExtension())]);
        }

        $request->merge(['taxid' => strtoupper(trim($request->taxid))]);

        $this->validate($request, [
            'name' => 'required|string',
            'file_image' => 'nullable|mimes:jpeg,jpg,png|max:2048',
            'taxid' => [
                'required',
                'regex:/^[A-Z&Ñ]{3,4}[0-9]{2}(0[1-9]|1[012])(0[1-9]|[12][0-9]|3[01])[A-Z0-9]{2}[0-9A]$/i'
            ],
            'curp' => 'nullable|size:18',
            'tax_regimen_id' => 'required|integer',
            'email' => 'nullable|email',
            'country_id' => 'required|integer',
            'postcode' => 'required',
            'file_file_cer_ext' => 'nullable|in:cer',
            'file_file_key_ext' => 'nullable|in:key',
            'password_key' => 'required_with:file_file_key_ext',
            'item_bank_account.*.name' => 'required',
            'item_bank_account.*.bank_id' => 'required',
            'item_bank_account.*.currency_id' => 'required',
            'item_bank_account.*.account_number' => [
                'required',
                'regex:/^([0-9]{10,11}|[0-9]{15,16}|[0-9]{18})$/'
            ],
        ], [
            'name.*' => __('base/company.error_name'),
            'file_image.*' => __('base/company.error_image'),
            'taxid.required' => __('base/company.error_taxid'),
            'taxid.regex' => __('base/company.error_taxid_format'),
            'curp.*' => __('base/company.error_curp'),
            'tax_regimen_id.*' => __('base/company.error_tax_regimen_id'),
            'email.email' => __('base/company.error_email_format'),
            'country_id.*' => __('base/company.error_country_id'),
            'postcode.*' => __('base/company.error_postcode'),
            'file_file_cer_ext.*' => __('base/company.error_file_cer'),
            'file_file_key_ext.*' => __('base/company.error_file_key'),
            'password_key.*' => __('base/company.error_password_key'),
            'item_bank_account.*.name.*' => __('base/company.error_bank_account_name'),
            'item_bank_account.*.bank_id.*' => __('base/company.error_bank_account_bank_id'),
            'item_bank_account.*.currency_id.*' => __('base/company.error_bank_account_currency_id'),
            'item_bank_account.*.account_number.required' => __('base/company.error_bank_account_account_number'),
            'item_bank_account.*.account_number.regex' => __('base/company.error_bank_account_account_number_format'),
        ]);
    }

    /**
     * Funcion para convertir cer a pem y obtener la informacion
     *
     * @return array
     */
    private function convertCerToPem($company)
    {
        $data = [];
        $name_file_cer = Str::random(40) . '.' . request()->file('file_file_cer')->getClientOriginalExtension();
        $path_file_cer = request()->file('file_file_cer')->storeAs(Helper::setDirectory(Company::PATH_FILES,$company->id), $name_file_cer); //ruta temporal del archivo
        if (\Storage::exists($path_file_cer)) {
            //Convertir en PEM
            $path_file_cer_pem = $path_file_cer . '.pem';
            if(config('app.shared_hosting')){
                system('openssl x509 -inform DER -in "' . \Storage::path($path_file_cer) . '" -outform PEM -pubkey -out "' . \Storage::path($path_file_cer_pem) . '"');
            }else {
                $process = new Process('openssl x509 -inform DER -in "' . \Storage::path($path_file_cer) . '" -outform PEM -pubkey -out "' . \Storage::path($path_file_cer_pem). '"');
                $process->run();
                if (!$process->isSuccessful()) {
                    throw new ProcessFailedException($process);
                }
            }
            //Genera informacion
            if (\Storage::exists($path_file_cer_pem)) {
                $tmp = openssl_x509_parse(\Storage::get($path_file_cer_pem));
                if($tmp['extensions']['keyUsage'] != 'Digital Signature, Non Repudiation'){
                    throw new \Exception(__('base/company.error_file_cer_validate_csd'));
                }
                $data = [
                    'file_cer' => $name_file_cer . '.pem',
                    'certificate_number' => Helper::certificateNumber($tmp['serialNumberHex']),
                    'date_start' => date('Y-m-d H:i:s', $tmp['validFrom_time_t']),
                    'date_end' => date('Y-m-d H:i:s', $tmp['validTo_time_t']),
                    'path_file_cer_pem' => $path_file_cer_pem,
                    'titular' => $tmp['subject']['O'] ?? '',
                ];
            }
            //Eliminar archivo .cer
            //\Storage::delete(Company::PATH_FILES . '/' . $name_file_cer);
        }
        return $data;
    }

    /**
     * Funcion para convertir cer a pem y obtener la informacion
     *
     * @param $request
     * @param $path_file_cer_pem
     * @return array
     * @throws \Exception
     */
    private function convertKeyToPem($company,$password_key, $path_file_cer_pem)
    {
        $data = [];
        $name_file_key = Str::random(40) . '.' . request()->file('file_file_key')->getClientOriginalExtension();
        $path_file_key = request()->file('file_file_key')->storeAs(Helper::setDirectory(Company::PATH_FILES,$company->id), $name_file_key);
        if (\Storage::exists($path_file_key)) {
            //
            $pass = $password_key;
            if(PHP_OS_FAMILY !== "Windows"){
                $pass = str_replace('$','\$', $pass);
            }

            //Convertir en PEM
            $path_file_key_pem = $path_file_key . '.pem';
            if(config('app.shared_hosting')){
                system('openssl pkcs8 -inform DER -in "' . \Storage::path($path_file_key) . '" -passin pass:"' . $pass . '" -outform PEM -out "' . \Storage::path($path_file_key_pem) . '"');
            }else {
                $process = new Process('openssl pkcs8 -inform DER -in "' . \Storage::path($path_file_key) . '" -passin pass:"' . $pass . '" -outform PEM -out "' . \Storage::path($path_file_key_pem) . '"');
                $process->run();
                if (!$process->isSuccessful()) {
                    throw new ProcessFailedException($process);
                }
            }
            //Convertir en PEM con contraseña
            $path_file_key_pass_pem = $path_file_key . '.pass.pem';
            if(config('app.shared_hosting')){
                system('openssl rsa -in "' . \Storage::path($path_file_key_pem) . '" -des3 -out "' . \Storage::path($path_file_key_pass_pem) . '" -passout pass:"' . $pass. '"');
            }else {
                //$process           = new Process('openssl pkcs8 -inform DER -in ' . $path_file_key . ' -passin pass:"' . $request->pass . '" -outform PEM -out ' . $path_file_key_pass_pem .' -passout pass:"' . $request->password_key.'"');
                $process = new Process('openssl rsa -in "' . \Storage::path($path_file_key_pem) . '" -des3 -out "' . \Storage::path($path_file_key_pass_pem) . '" -passout pass:"' . $pass. '"');
                $process->run();
                if (!$process->isSuccessful()) {
                    throw new ProcessFailedException($process);
                }
            }
            //Valida que la llave privada este correcta con su contraseña
            $this->validateKeyPem($path_file_key_pass_pem, $password_key);
            //Valida que la llave privada pertenezca al certificado
            $this->validateKeyBelongToCer($path_file_key_pass_pem, $password_key, $path_file_cer_pem);
            //Genera informacion
            $data = [
                'password_key' => Crypt::encryptString($password_key),
                'file_key' => $name_file_key . '.pass.pem',
                'path_file_key_pem' => $path_file_key_pem,
            ];
            //Eliminar archivo
            //\Storage::delete(Company::PATH_FILES . '/' . $name_file_key);
        }
        return $data;
    }

    /**
     * Valida que la contraseña este correcta con la llave en formato PEM
     *
     * @param $path_file_key_pass_pem
     * @param $password_key
     * @return bool
     * @throws \Exception
     */
    public function validateKeyPem($path_file_key_pass_pem, $password_key)
    {
        if (\Storage::exists($path_file_key_pass_pem)) {
            $pkeyid = openssl_pkey_get_private(\Storage::get($path_file_key_pass_pem), $password_key);
            if ($pkeyid == false) {
                throw new \Exception(__('base/company.error_file_key_validate_password'));
            } else {
                openssl_free_key($pkeyid);
                return true;
            }
        }
    }

    /**
     * Valida que que el archivo cer pertenezca al archivo key
     *
     * @param $path_file_key_pass_pem
     * @param $password_key
     * @param $path_file_cer_pem
     * @return bool
     * @throws \Exception
     */
    public function validateKeyBelongToCer($path_file_key_pass_pem, $password_key, $path_file_cer_pem)
    {
        if (\Storage::exists($path_file_key_pass_pem) && \Storage::exists($path_file_cer_pem)) {
            $text_test = 'Test CFDI 3.3';
            $pkeyid = openssl_pkey_get_private(\Storage::get($path_file_key_pass_pem), $password_key);
            $pubkeyid = openssl_pkey_get_public(\Storage::get($path_file_cer_pem));
            openssl_sign($text_test, $crypttext, $pkeyid, OPENSSL_ALGO_SHA256);
            $ok = openssl_verify($text_test, $crypttext, $pubkeyid, OPENSSL_ALGO_SHA256);
            openssl_free_key($pkeyid);
            if ($ok == 1) {
                return true;
            } else {
                throw new \Exception(__('base/company.error_file_key_validate_belong_to_file_cer'));
            }
        }
    }

    /**
     * Crea archivo pfx
     *
     * @param $path_file_key_pass_pem
     * @param $password_key
     * @param $path_file_cer_pem
     * @return bool
     * @throws \Exception
     */
    public function createPfx($company,$path_file_key_pem, $password_key, $path_file_cer_pem)
    {
        $data = [];
        if (\Storage::exists($path_file_key_pem) && \Storage::exists($path_file_cer_pem)) {
            $name_file_pfx = Str::random(40) . '.pfx';
            $path_file_pfx = Helper::setDirectory(Company::PATH_FILES,$company->id). '/' . $name_file_pfx;
            //Crear archivo PFX
            if(config('app.shared_hosting')){
                system('openssl pkcs12 -export -inkey "' . \Storage::path($path_file_key_pem) . '" -in "' . \Storage::path($path_file_cer_pem) . '" -out "' . \Storage::path($path_file_pfx) . '" -passout pass:"' . $password_key.'"');
            }else {
                $process = new Process('openssl pkcs12 -export -inkey "' . \Storage::path($path_file_key_pem) . '" -in "' . \Storage::path($path_file_cer_pem) . '" -out "' . \Storage::path($path_file_pfx) . '" -passout pass:"' . $password_key.'"');
                $process->run();
                if (!$process->isSuccessful()) {
                    throw new ProcessFailedException($process);
                }
            }
            //Genera informacion
            $data = [
                'file_pfx' => $name_file_pfx,
            ];
        }
        return $data;
    }

    /**
     * Modal para buscar empresas
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalUserSearch(Request $request)
    {
        //Variables
        $active_btn_select = (!empty($request->active_btn_select) ? $request->active_btn_select : '');
        //En caso de estar vacio le asignamos un datos para mostrar los clientes con "a"
        $request->merge(['filter_search' => !empty($request->filter_search) ? $request->filter_search : '']);

        //Logica
        if ($request->ajax()) {
            //
            $results = Company::filter($request->all())->active()->sortable('name')->limit(10)->get();

            //Busca empresa
            $html_companies = view('layouts.partials.companies.search',
                compact('results', 'active_btn_select'))->render();

            //modal de buscar
            $html = view('layouts.partials.companies.modal_user_search',
                compact('html_companies', 'active_btn_select'))->render();


            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para buscar empresas
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function search(Request $request)
    {
        //Variables
        $active_btn_select = (!empty($request->active_btn_select) ? $request->active_btn_select : '');
        //En caso de estar vacio le asignamos un datos para mostrar los clientes con "a"
        $request->merge(['filter_search' => !empty($request->filter_search) ? $request->filter_search : '']);

        //Logica
        if ($request->ajax()) {
            //
            $results = Company::filter($request->all())->active()->sortable('name')->limit(10)->get();

            //Busca cliente
            $html = view('layouts.partials.companies.search',
                compact('results', 'active_btn_select'))->render();

            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }


}
