<?php

namespace App\Http\Controllers\Base;

use App\Exports\EmployeesExport;
use App\Exports\EmployeeTemplateImportExport;
use App\Helpers\Helper;
use App\Imports\EmployeeTemplateBeforeImportImport;
use App\Imports\EmployeeTemplateImportImport;
use App\Models\Base\Employee;
use App\Models\Base\EmployeeCategory;
use App\Models\Catalogs\Bank;
use App\Models\Catalogs\ContractType;
use App\Models\Catalogs\Country;
use App\Models\Catalogs\FrequencyPayment;
use App\Models\Catalogs\JobRiskClasse;
use App\Models\Catalogs\RecruitmentRegime;
use App\Models\Catalogs\WorkdayType;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Validation\ValidationException;
use App\Models\Catalogs\TaxRegimen;

class EmployeeController extends Controller
{
    private $import_results = [];

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {

        $this->import_results [] = (object)[
            'code' => '4572',
            'firstname' => 'Juan',
            'lastname' => 'Perez',
            'lastname2' => 'Cruz',
            'taxid' => 'XAXX010101000',
            'tax_regimen' => 'Sueldos y Salarios e Ingresos Asimilados a Salarios',
            'curp' => 'PECJ110313HCMLNS09',
            'postcode' => '76240',
            'nss' => '',
            'email' => 'juan@gmail.com',
            'recruitment_regime' => 'Sueldos',
            'date_start_work' => '',
            'unionized' => 'NO',
            'workday_type' => '',
            'contract_type' => 'Contrato de trabajo por tiempo indeterminado',
            'employee_category' => '',
            'department' => 'Finanzas',
            'job' => '',
            'job_risk_classe' => '',
            'frequency_payment' => 'Quincenal',
            'bank' => '',
            'bank_account' => '',
            'base_salary' => '',
            'sdi' => '',
            'country' => 'México',
            'state' => 'Queretaro',
        ];
        $this->import_results [] = (object)[
            'code' => '7842',
            'firstname' => 'Marcos',
            'lastname' => 'Hernandez',
            'lastname2' => 'Perez',
            'taxid' => 'XAXX010101000',
            'tax_regimen' => 'Sueldos y Salarios e Ingresos Asimilados a Salarios',
            'curp' => 'HEPM110313HCMLNS09',
            'postcode' => '',
            'nss' => '123456789',
            'email' => 'marcos@gmail.com',
            'recruitment_regime' => 'Sueldos',
            'date_start_work' => '01-01-2018',
            'unionized' => 'NO',
            'workday_type' => 'Diurna',
            'contract_type' => 'Contrato de trabajo por tiempo indeterminado',
            'employee_category' => '',
            'department' => 'Finanzas',
            'job' => 'Contador',
            'job_risk_classe' => 'Clase I',
            'frequency_payment' => 'Quincenal',
            'bank' => 'BBVA BANCOMER',
            'bank_account' => '012560025555707821',
            'base_salary' => '109.12',
            'sdi' => '105.12',
            'country' => 'México',
            'state' => 'Hidalgo',
        ];
        $this->import_results [] = (object)[
            'code' => '7842',
            'firstname' => 'Maria',
            'lastname' => 'Lugo',
            'lastname2' => 'Juarez',
            'taxid' => 'XAXX010101000',
            'tax_regimen' => '',
            'curp' => 'LUJM110313HCMLNS09',
            'postcode' => '',
            'nss' => '123456789',
            'email' => 'maria@gmail.com',
            'recruitment_regime' => 'Sueldos',
            'date_start_work' => '24-05-2000',
            'unionized' => 'NO',
            'workday_type' => 'Diurna',
            'contract_type' => 'Contrato de trabajo por tiempo indeterminado',
            'employee_category' => '',
            'department' => 'Finanzas',
            'job' => 'Administrativa',
            'job_risk_classe' => 'Clase I',
            'frequency_payment' => 'Quincenal',
            'bank' => 'BBVA BANCOMER',
            'bank_account' => '012560025555703845',
            'base_salary' => '99.12',
            'sdi' => '85.12',
            'country' => 'México',
            'state' => 'Querétaro',
        ];
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);
        if (!isset($request->filter_status) && $request->filter_status != 'all') {
            $request->request->add([
                'filter_status' => '1'
            ]);
        }

        //Consulta
        $results = Employee::filter($request->all());
        if(isset($request->filter_status) && $request->filter_status != 'all'){
            $results = $results->where('employees.status','=',(int)$request->filter_status);
        }
        $results = $results->with('frequencyPayment')
            ->with('recruitmentRegime')
            ->sortable('name')
            ->paginate($limit);

        //Vista
        return view('base.employees.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $recruitment_regimes = RecruitmentRegime::populateSelect()->get()->pluck('name_sat', 'id');
        $workday_types = WorkdayType::populateSelect()->get()->pluck('name_sat', 'id');
        $contract_types = ContractType::populateSelect()->get()->pluck('name_sat', 'id');
        $job_risk_classes = JobRiskClasse::populateSelect()->get()->pluck('name_sat', 'id');
        $frequency_payments = FrequencyPayment::populateSelect()->get()->pluck('name_sat', 'id');
        $banks = Bank::populateSelect()->get()->pluck('name_sat', 'id');
        $countries = Country::populateSelect()->get()->pluck('name_sat', 'id');
        $employee_categories = EmployeeCategory::populateSelect()->pluck('name', 'id');
        $tax_regimens = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');

        return view('base.employees.create', compact('recruitment_regimes', 'workday_types', 'contract_types', 'job_risk_classes', 'frequency_payments', 'banks', 'countries','employee_categories','tax_regimens'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     * @throws ValidationException
     */
    public function store(Request $request)
    {
        //Validacion
        $request->merge(['taxid' => strtoupper(trim($request->taxid))]);
        $request->merge(['curp' => strtoupper(trim($request->curp))]);
        $this->validation($request);

        //Logica
        $company = Helper::defaultCompany(); //Empresa

        //Validaciones manuales
        $validator = \Validator::make([], []);
        //Validamos que RFC no este registrado previamente
        if ($request->taxid != 'XAXX010101000' && $request->taxid != 'XEXX010101000') {
            $employee_taxid = Employee::where('taxid', '=', $request->taxid)->where('company_id', '=', $company->id)->first();
            if (!empty($employee_taxid)) {
                $validator->after(function ($validator) {
                    $validator->errors()->add('taxid', __('base/employee.error_taxid_unique'));
                });
            }
        }
        //Validacion de RFC en la lista de LCO insigna
        try{
            if(!empty(setting('validate_taxid_insigna')) && $request->taxid != 'XAXX010101000' && $request->taxid != 'XEXX010101000'){
                if(!Helper::validateInsignaLcoRfc($request->taxid)){
                    $validator->after(function ($validator) {
                        $validator->errors()->add('taxid', __('base/employee.error_taxid_lco'));
                    });
                }
            }
        } catch (\Exception $e) {
            flash($e->getMessage())->error();
            return back()->withInput();
        }

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        \DB::connection('tenant')->beginTransaction();
        try {

            //Logica
            $date_start_work_fix = $request->date_start_work;//Fix valida si la fecha de inicio esta vacia en caso de error

            $request->merge(['created_uid' => \Auth::user()->id]);
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['name' => strtoupper(trim($request->firstname .' ' . trim($request->lastname) . ' ' . trim($request->lastname2)))]);
            $request->merge(['firstname' => strtoupper(trim($request->firstname))]);
            $request->merge(['lastname' => strtoupper(trim($request->lastname))]);
            $request->merge(['lastname2' => strtoupper(trim($request->lastname2))]);
            $request->merge(['curp' => strtoupper(trim($request->curp))]);
            $request->merge(['date_start_work' => !empty($request->date_start_work) ? Helper::convertDateToSql($request->date_start_work) : null]);
            $request->merge(['unionized' => !empty($request->unionized) ? 1 : 0]);
            $request->merge(['base_salary' => (double)$request->base_salary]);
            $request->merge(['sdi' => (double)$request->sdi]);
            $request->merge(['status' => 1]); //Por default activo
            $request->merge(['company_id' => $company->id]);

            //Guardar
            //Registro principal
            $employee = Employee::create($request->input());

            \DB::connection('tenant')->commit();

            //Mensaje
            flash(__('general.text_form_success_add'))->success();

            //Redireccion
            return redirect('/base/employees');
        } catch (\Exception $e) {
            //Fix fechas
            if(!empty($date_start_work_fix)) {
                $request->merge([
                    'date_start_work' => Helper::convertSqlToDate($request->date_start_work),
                ]);
            }else{
                $request->merge([
                    'date_start_work' => '',
                ]);
            }
            \DB::connection('tenant')->rollback();
            flash($e->getMessage())->error();
            return back()->withInput();
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Sales\Employee $employee
     * @return \Illuminate\Http\Response
     */
    public function show(Employee $employee)
    {
        return redirect('/base/employees');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Sales\Employee $employee
     * @return \Illuminate\Http\Response
     */
    public function edit(Employee $employee)
    {
        $recruitment_regimes = RecruitmentRegime::populateSelect()->get()->pluck('name_sat', 'id');
        $workday_types = WorkdayType::populateSelect()->get()->pluck('name_sat', 'id');
        $contract_types = ContractType::populateSelect()->get()->pluck('name_sat', 'id');
        $job_risk_classes = JobRiskClasse::populateSelect()->get()->pluck('name_sat', 'id');
        $frequency_payments = FrequencyPayment::populateSelect()->get()->pluck('name_sat', 'id');
        $banks = Bank::populateSelect()->get()->pluck('name_sat', 'id');
        $countries = Country::populateSelect()->get()->pluck('name_sat', 'id');
        $employee_categories = EmployeeCategory::populateSelect()->pluck('name', 'id');
        $tax_regimens = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');

        return view('base.employees.edit', compact('employee', 'recruitment_regimes', 'workday_types', 'contract_types', 'job_risk_classes', 'frequency_payments', 'countries', 'banks','employee_categories','tax_regimens'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param Request $request
     * @param Employee $employee
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     * @throws ValidationException
     */
    public function update(Request $request, Employee $employee)
    {
        //Validacion
        $request->merge(['taxid' => strtoupper(trim($request->taxid))]);
        $request->merge(['curp' => strtoupper(trim($request->curp))]);
        $this->validation($request);

        //Validaciones manuales
        $validator = \Validator::make([], []);
        //Validamos que RFC no ste registrado previamente
        if ($request->taxid != 'XAXX010101000' && $request->taxid != 'XEXX010101000') {
            $employee_taxid = Employee::where('id', '<>', $employee->id)->where('taxid', '=', $request->taxid)->where('company_id', '=', $employee->company_id)->first();
            if (!empty($employee_taxid)) {
                $validator->after(function ($validator) {
                    $validator->errors()->add('taxid', __('base/employee.error_taxid_unique'));
                });
            }
        }
        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        \DB::connection('tenant')->beginTransaction();
        try {

            //Logica
            $date_start_work_fix = $request->date_start_work;//Fix valida si la fecha de inicio esta vacia en caso de error

            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['name' => strtoupper(trim($request->firstname .' ' . trim($request->lastname) . ' ' . trim($request->lastname2)))]);
            $request->merge(['firstname' => strtoupper(trim($request->firstname))]);
            $request->merge(['lastname' => strtoupper(trim($request->lastname))]);
            $request->merge(['lastname2' => strtoupper(trim($request->lastname2))]);
            $request->merge(['curp' => strtoupper(trim($request->curp))]);
            $request->merge(['date_start_work' => !empty($request->date_start_work) ? Helper::convertDateToSql($request->date_start_work) : null]);
            $request->merge(['unionized' => !empty($request->unionized) ? 1 : 0]);
            $request->merge(['base_salary' => (double)$request->base_salary]);
            $request->merge(['sdi' => (double)$request->sdi]);
            $request->merge(['status' => !empty($request->status) ? 1 : 0]);
            $employee->fill($request->only([
                'updated_uid',
                'name',
                'firstname',
                'lastname',
                'lastname2',
                'code',
                'taxid',
                'curp',
                'email',
                'phone',
                'phone_mobile',
                'recruitment_regime_id',
                'date_start_work',
                'nss',
                'unionized',
                'workday_type_id',
                'contract_type_id',
                'department',
                'job',
                'job_risk_classe_id',
                'frequency_payment_id',
                'bank_id',
                'bank_account',
                'base_salary',
                'sdi',
                'country_id',
                'state_id',
                'comment',
                'sort_order',
                'status',
                'employee_category_id',
                'postcode',
                'tax_regimen_id'
            ]));

            //Guardar
            //Registro principal
            $employee->save();

            \DB::connection('tenant')->commit();

            //Mensaje
            flash(__('general.text_form_success_edit'))->success();

            //Redireccion
            return redirect('/base/employees');
        } catch (\Exception $e) {
            //Fix fechas
            if(!empty($date_start_work_fix)) {
                $request->merge([
                    'date_start_work' => Helper::convertSqlToDate($request->date_start_work),
                ]);
            }else{
                $request->merge([
                    'date_start_work' => '',
                ]);
            }
            \DB::connection('tenant')->rollback();
            flash($e->getMessage())->error();
            return back()->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Sales\Employee $employee
     * @return \Illuminate\Http\Response
     */
    public function destroy(Employee $employee)
    {
        //Logica
        $employee->updated_uid = \Auth::user()->id;
        $employee->status = 0;
        $employee->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/base/employees');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     * @throws ValidationException
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'code' => 'required',
            'firstname' => 'required|string',
            'lastname' => 'required|string',
            'taxid' => [
                'required',
                'regex:/^[A-Z&Ñ]{3,4}[0-9]{2}(0[1-9]|1[012])(0[1-9]|[12][0-9]|3[01])[A-Z0-9]{2}[0-9A]$/i'
            ],
            'curp' => 'required|size:18',
            'email' => 'nullable|email',
            'recruitment_regime_id' => 'required|integer',
            'date_start_work' => 'nullable|date|date_format:"'.setting('date_format', 'd-m-Y').'"',
            /*'nss' => 'required',*/
            /*'workday_type_id' => 'required|integer',*/
            'contract_type_id' => 'required|integer',
            'frequency_payment_id' => 'required|integer',
            /*'job_risk_classe_id' => 'required|integer',*/
            'base_salary' => 'nullable|numeric|min:0',
            'sdi' => 'nullable|numeric|min:0',
            'state_id' => 'required|integer',
            'country_id' => 'required|integer',
        ], [
            'code.*' => __('base/employee.error_code'),
            'firstname.*' => __('base/employee.error_firstname'),
            'lastname.*' => __('base/employee.error_lastname'),
            'taxid.required' => __('base/employee.error_taxid'),
            'taxid.regex' => __('base/employee.error_taxid_format'),
            'curp.*' => __('base/employee.error_curp'),
            'email.email' => __('base/employee.error_email_format'),
            'recruitment_regime_id.*' => __('base/employee.error_recruitment_regime_id'),
            'date_start_work.required' => __('base/employee.error_date_start_work'),
            'date_start_work.date' => __('base/employee.error_date_start_work_format'),
            'date_start_work.date_format' => __('base/employee.error_date_start_work_format'),
            /*'nss.*' => __('base/employee.error_nss'),*/
            /*'workday_type_id.*' => __('base/employee.error_workday_type_id'),*/
            'contract_type_id.*' => __('base/employee.error_contract_type_id'),
            /*'job_risk_classe_id.*' => __('base/employee.error_job_risk_classe_id'),*/
            'frequency_payment_id.*' => __('base/employee.error_frequency_payment_id'),
            'base_salary.*' => __('base/employee.error_base_salary'),
            'sdi.*' => __('base/employee.error_sdi'),
            'state_id.*' => __('base/employee.error_state_id'),
            'country_id.*' => __('base/employee.error_country_id'),
        ]);

        $validator = \Validator::make([], []);
        if(setting('cfdi_version') == 'cfdi40'){
            if (empty($request->tax_regimen_id)) {
                $validator->after(function ($validator) {
                    $validator->errors()->add('tax_regimen_id', __('base/employee.error_tax_regimen_id'));
                });
            }
            if (empty($request->postcode)) {
                $validator->after(function ($validator) {
                    $validator->errors()->add('postcode', __('base/employee.error_postcode'));
                });
            }
        }
        if ($validator->fails()) {
            throw new ValidationException($validator);
        }
    }

    /**
     * Obtener registro
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getEmployee(Request $request)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            $employee = Employee::findOrFail($id);
            return response()->json($employee, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Autoacompletar select2 de clientes
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function autocomplete(Request $request)
    {
        //Variables
        $term = $request->term;

        //Logica
        if ($request->ajax() && !empty($term)) {
            $tmp = Employee::filter(['filter_search_select2' => $term])->active()->sortable('name')->limit(16)->get();
            $results = [];
            if ($tmp->isNotEmpty()) {
                foreach ($tmp as $result) {
                    $results[] = [
                        'id' => $result->id,
                        'text' => $result->text_select2,
                        'description' => $result->description_select2
                    ];
                }
            }
            return response()->json($results, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para buscar clientes
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalSearch(Request $request)
    {
        //Variables
        $active_btn_select = (!empty($request->active_btn_select) ? $request->active_btn_select : '');
        //En caso de estar vacio le asignamos un datos para mostrar los clientes con "a"
        $request->merge(['filter_search' => !empty($request->filter_search) ? $request->filter_search : '']);

        //Logica
        if ($request->ajax()) {
            $results = Employee::filter($request->all())->active()->sortable('name')->limit(10)->get();

            //Busca cliente
            $html_employees = view('layouts.partials.employees.search',
                compact('results', 'active_btn_select'))->render();

            //modal de buscar
            $html = view('layouts.partials.employees.modal_search',
                compact('html_employees', 'active_btn_select'))->render();


            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para buscar clientes
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function search(Request $request)
    {
        //Variables
        $active_btn_select = (!empty($request->active_btn_select) ? $request->active_btn_select : '');
        //En caso de estar vacio le asignamos un datos para mostrar los clientes con "a"
        $request->merge(['filter_search' => !empty($request->filter_search) ? $request->filter_search : '']);

        //Logica
        if ($request->ajax()) {

            $results = Employee::filter($request->all())->active()->sortable('name')->limit(10)->get();

            //Busca cliente
            $html = view('layouts.partials.employees.search',
                compact('results', 'active_btn_select'))->render();

            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Exportar datos a excel
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function exportToExcel(Request $request)
    {
        while (ob_get_level()) ob_end_clean();
        ob_start();

        return Excel::download(new EmployeesExport($request),
            __('base/employee.document_title') . '-' . config('app.name') . '.xlsx');
    }

    /**
     * Importar viviendas desde archivo
     *
     * @return \Illuminate\Http\Response
     */
    public function createImport()
    {
        //Datos
        $recruitment_regimes = RecruitmentRegime::populateSelect()->get()->pluck('name_sat', 'id');
        $workday_types = WorkdayType::populateSelect()->get()->pluck('name_sat', 'id');
        $contract_types = ContractType::populateSelect()->get()->pluck('name_sat', 'id');
        $job_risk_classes = JobRiskClasse::populateSelect()->get()->pluck('name_sat', 'id');
        $frequency_payments = FrequencyPayment::populateSelect()->get()->pluck('name_sat', 'id');
        $banks = Bank::populateSelect()->get()->pluck('name_sat', 'id');
        $countries = Country::populateSelect()->get()->pluck('name_sat', 'id');
        $employee_categories = EmployeeCategory::populateSelect()->pluck('name', 'id');
        $import_results = $this->import_results;
        $tax_regimens = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');

        //Vista
        return view('base.employees.import',compact('recruitment_regimes','workday_types','contract_types','job_risk_classes','frequency_payments','banks','import_results','countries','employee_categories', 'tax_regimens'));
    }

    /**
     * Descargar plantilla
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function downloadTemplateImport(Request $request)
    {
        //Datos
        $import_results = $this->import_results;

        if(!empty($request->update)){
            $import_results = [];
            $results = Employee::filter([])
            ->with('frequencyPayment')
            ->with('recruitmentRegime')
            ->sortable('name')
            ->get();

            foreach($results as $result){
                $import_results[] = (object)[
                    'code' => $result->code,
                    'firstname' => $result->firstname,
                    'lastname' => $result->lastname,
                    'lastname2' => $result->lastname2,
                    'taxid' => $result->taxid,
                    'tax_regimen' => $result->taxRegimen->name ?? null,
                    'curp' => $result->curp,
                    'postcode' => $result->postcode,
                    'nss' => $result->nss,
                    'email' => $result->email,
                    'recruitment_regime' => $result->recruitmentRegime->name ?? null,
                    'date_start_work' => $result->date_start_work ? Helper::convertSqlToDate($result->date_start_work) : null,
                    'unionized' => $result->unionized ? __('general.text_yes') : __('general.text_no'),
                    'workday_type' => $result->workdayType->name ?? null,
                    'contract_type' => $result->contractType->name ?? null,
                    'employee_category' => $result->employeeCategory->name ?? null,
                    'department' => $result->department,
                    'job' => $result->job,
                    'job_risk_classe' => $result->jobRiskClasse->name ?? '',
                    'frequency_payment' => $result->frequencyPayment->name ?? '',
                    'bank' => $result->bank->name ?? '',
                    'bank_account' => $result->bank_account,
                    'base_salary' => $result->base_salary,
                    'sdi' => $result->sdi,
                    'country' => $result->country->name ?? '',
                    'state' => $result->state->name ?? '',
                ];
            }
        }

        //Descargar archivo
        while (ob_get_level()) ob_end_clean();
        ob_start();

        return Excel::download(new EmployeeTemplateImportExport($request, $import_results),
            __('base/employee.text_template_import') . '-' . config('app.name') . '.xlsx',\Maatwebsite\Excel\Excel::XLSX);
    }

    /**
     * Obtener informacion antes de procesar el archivo
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function beforeImport(Request $request)
    {
        //Validaciones
        //Validacion de archivos por extension
        if ($request->hasFile('file_employees_import')) {
            $request->merge(['file_employees_import_ext' => request()->file('file_employees_import')->getClientOriginalExtension()]);
        }
        $validator = \Validator::make($request->all(), [
            'file_employees_import' => 'required|max:2048',
            'file_employees_import_ext' => 'nullable|in:xls,xlsx'
        ], [
            'file_employees_import.*' => __('base/employee.error_file_employees_import'),
            'file_employees_import_ext.*' => __('base/employee.error_file_employees_import'),
        ]);
        //Validaciones manuales
        //$validator = \Validator::make([], []);

        //Errores
        if ($validator->fails()) {
            $errors = '';
            foreach ($validator->errors()->all() as $message) {
                $errors = $message;
            }
            return response()->json(['error' => $errors], 422);
        }

        try {
            //Lógica

            //Importar
            $import = new EmployeeTemplateBeforeImportImport;
            Excel::import($import, request()->file('file_employees_import'));

            //Mensaje
            return response()->json([
                'success' => 'ok',
                'total_import_employees' => sprintf(__('base/employee.help_import_sweet_alert_1'),Helper::numberFormat($import->total_employees)),
            ]);
        }catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 422);
        }
    }

    /**
     * Importar viviendas desde archivo
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     * @throws \Illuminate\Validation\ValidationException
     */
    public function storeImport(Request $request)
    {
        //Validacion
        //Validacion de archivos por extension
        if ($request->hasFile('file_employees_import')) {
            $request->merge(['file_employees_import_ext' => request()->file('file_employees_import')->getClientOriginalExtension()]);
        }
        $this->validate($request, [
            'file_employees_import' => 'required||max:2048',
            'file_employees_import_ext' => 'nullable|in:xls,xlsx'
        ], [
            'file_employees_import.*' => __('base/employee.error_file_employees_import'),
            'file_employees_import_ext.*' => __('base/employee.error_file_employees_import'),
        ]);

        try {
            //Lógica

            //Importar
            Excel::import(new EmployeeTemplateImportImport(), request()->file('file_employees_import'));

            //Mensaje
            flash(__('general.text_form_success_import'))->success();

            //Redireccion
            return redirect()->route('employees.index');

        } catch (\Illuminate\Validation\ValidationException $e ) {
            return back()->withErrors($e->errors());
        } catch (\Exception $e) {
            flash($e->getMessage())->error();
            return back()->withInput();
        }
    }
}
