<?php

namespace App\Http\Controllers\Base;

use App\Helpers\Helper;
use App\Models\Base\Pac;
use App\Helpers\PacHelper;
use App\Helpers\BaseHelper;
use Illuminate\Support\Str;
use App\Models\Base\Payroll;
use Illuminate\Http\Request;
use App\Helpers\Cfdi33Helper;
use App\Models\Base\Employee;
use App\Models\Catalogs\CfdiUse;
use App\Mail\SendEmployeePayroll;
use App\Models\Base\BranchOffice;
use App\Models\Base\CfdiDownload;
use App\Models\Catalogs\Currency;
use App\Models\Catalogs\Deduction;
use App\Models\Catalogs\PaymentWay;
use App\Models\Catalogs\Perception;
use App\Models\Catalogs\TaxRegimen;
use App\Http\Controllers\Controller;
use App\Models\Base\EmployeePayroll;
use App\Models\Catalogs\PaymentTerm;
use App\Models\Catalogs\PayrollType;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\Catalogs\CfdiRelation;
use Illuminate\Support\Facades\Crypt;
use App\Models\Catalogs\PaymentMethod;
use App\Exports\EmployeePayrollsExport;
use App\Models\Catalogs\DisabilityType;
use App\Models\Catalogs\SourceResource;
use App\Models\Base\EmployeePayrollCfdi;
use App\Models\Base\EmployeePayrollLine;
use App\Models\Catalogs\OtherPaymentType;
use App\Models\Catalogs\ReasonCancellation;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use App\Models\Base\EmployeePayrollRelation;
use Illuminate\Validation\ValidationException;
use App\Models\Base\EmployeePayrollOutsourcing;

class EmployeePayrollController extends Controller
{
    private $list_status = [];
    private $document_type_code = 'employee.payroll';

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->list_status = [
            EmployeePayroll::DRAFT => __('base/employee_payroll.text_status_draft'),
            EmployeePayroll::OPEN => __('base/employee_payroll.text_status_open'),
            EmployeePayroll::CANCEL => __('base/employee_payroll.text_status_cancel'),
        ];
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);
        if(!empty($request->download_xmlpdf)){
            $request->request->add([
                'page' => 1
            ]);
            $limit = 1000000;
        }
        $payroll_types = PayrollType::populateSelect()->pluck('name', 'id');
        $list_status = $this->list_status;
        if (empty($request->filter_date_payment_from)) {
            $request->request->add([
                'filter_date_payment_from' => Helper::date(\Date::parse('first day of this month')->subMonths(2))
            ]);
        }
        if (empty($request->filter_date_payment_to)) {
            $request->request->add([
                'filter_date_to' => Helper::date(\Date::parse('last day of this month'))
            ]);
        }
        $request->request->add(['filter_document_type_code' => $this->document_type_code]); //Filtra tipo de documento
        $payrolls = Payroll::populateSelect()->get()->pluck('name_date_payment', 'id');

        //Consulta
        $results = EmployeePayroll::filter($request->all())
            ->with('employeePayrollCfdi')
            ->with('employee')
            ->with('currency')
            ->sortable(['date_payment' => 'desc'])->paginate($limit);

        //Descar zip de XML´s y PDF´s
        if(!empty($request->download_xmlpdf)){
            if($results->isNotEmpty()) {
                $company = Helper::defaultCompany();
                $file_zip_name = __('base/employee_payroll.document_title_download_xmlpdf').'_' . $company->taxid .'_'. str_replace(['-','/'],['',''],$request->filter_date_payment_from).'_'. str_replace(['-','/'],['',''],$request->filter_date_payment_to) .'_'.random_int(10,99).'.zip';
                if(\Storage::exists('temp/' . $file_zip_name)) {
                    @\Storage::delete('temp/' . $file_zip_name);
                }
                $zip =\Zipper::make(\Storage::path('temp/' . $file_zip_name));

                foreach ($results as $result) {

                    //Ruta y validacion del XML
                    $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $result->company_id) . '/';
                    $file_xml_pac = $path_xml . $result->employeePayrollCfdi->file_xml_pac;
                    $file_xml = $path_xml . $result->employeePayrollCfdi->file_xml;
                    if (!empty($result->employeePayrollCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {
                        $zip->add(\Storage::path($file_xml_pac), str_replace('/','',$result->name) . '.xml');
                    }elseif(!empty($result->employeePayrollCfdi->file_xml) && \Storage::exists($file_xml)){
                        $zip->add(\Storage::path($file_xml), str_replace('/','',$result->name) . '.xml');
                    }

                    //Ruta y validacion del pdf
                    $pdf = $this->print($result, false, true);
                    $file_pdf_name = Str::random().'.pdf';
                    //Guardamos en directorio temporal
                    \Storage::put('temp/' . $file_pdf_name, $pdf);
                    $zip->add(\Storage::path('temp/' . $file_pdf_name), str_replace('/','',$result->name) . '.pdf');

                }
                $zip->close();
                if(\Storage::exists('temp/' . $file_zip_name)) {
                    while (ob_get_level()) ob_end_clean();
                    ob_start();

                    return response()->download(\Storage::path('temp/' . $file_zip_name));
                }else{
                    flash(__('base/employee_payroll.error_download_xmlpdf'))->warning();
                }
            }
        }

        //Vista
        return view('base.employee_payrolls.index', compact('results','payroll_types', 'payrolls', 'list_status'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $branch_offices = BranchOffice::populateSelect()->pluck('name', 'id');
        $payroll_types = PayrollType::populateSelect()->get()->pluck('name_sat', 'id');
        $cfdi_relations = CfdiRelation::populateSelect()->where('code','=','04')->get()->pluck('name_sat', 'id');
        $source_resources = SourceResource::populateSelect()->get()->pluck('name_sat', 'id');
        $company = Helper::defaultCompany(); //Empresa
        if(!empty($company->tax_regimen_id2)){
            $tax_regimens = TaxRegimen::populateSelect()->whereIn('id',[$company->tax_regimen_id, $company->tax_regimen_id2])->get()->pluck('name_sat', 'id');
        }else{
            $tax_regimens = collect([]);
        }
        $tax_regimen_employees = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');

        $duplicate_ep = null;
        if($request->duplicate_id){
            $duplicate_ep = EmployeePayroll::findOrFail($request->duplicate_id);
        }

        return view('base.employee_payrolls.create',
            compact( 'branch_offices', 'payroll_types','duplicate_ep', 'cfdi_relations','source_resources', 'tax_regimens', 'tax_regimen_employees'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);

        //Almacenamiento de archivos
        $employee_payroll = null;

        \DB::connection('tenant')->beginTransaction();
        try {
            //Logica
            $company = Helper::defaultCompany(); //Empresa
            $request->merge(['created_uid' => \Auth::user()->id]);
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['status' => isset($request->pre_payroll) ? EmployeePayroll::DRAFT : EmployeePayroll::OPEN]);
            $request->merge(['company_id' => $company->id]);
            //Ajusta fecha
            $request->merge(['date' => Helper::dateTimeToSql(BaseHelper::getDateTimeBranchOffice($request->branch_office_id))]);
            $request->merge(['date_payment' => Helper::convertDateToSql($request->date_payment)]);
            $request->merge(['date_start_payment' => Helper::convertDateToSql($request->date_start_payment)]);
            $request->merge(['date_end_payment' => Helper::convertDateToSql($request->date_end_payment)]);
            $request->merge(['rt_amount_an_exhibition' => $request->rt_amount_an_exhibition ?? 0]);
            $request->merge(['rt_partiality_amount' => $request->rt_partiality_amount ?? 0]);
            $request->merge(['rt_daily_amount' => $request->rt_daily_amount ?? 0]);
            $request->merge(['rt_cumulative_income' => $request->rt_cumulative_income ?? 0]);
            $request->merge(['rt_non_cumulative_income' => $request->rt_non_cumulative_income ?? 0]);
            $request->merge(['sp_total_amount' => $request->sp_total_amount ?? 0]);
            $request->merge(['sp_years_of_service' => $request->sp_years_of_service ?? 0]);
            $request->merge(['sp_last_salary' => $request->sp_last_salary ?? 0]);
            $request->merge(['sp_cumulative_income' => $request->sp_cumulative_income ?? 0]);
            $request->merge(['sp_non_cumulative_income' => $request->sp_non_cumulative_income ?? 0]);
            $request->merge(['sncf_amount_source_resource' => $request->sncf_amount_source_resource ?? 0]);

            //Datos default
            $payment_way =PaymentWay::where('code','=','99')->first();
            $request->merge(['payment_way_id' => !empty($payment_way) ? $payment_way->id : null]);
            $currency =Currency::where('code','=','MXN')->first();
            $request->merge(['currency_id' => !empty($currency) ? $currency->id : null]);
            $request->merge(['currency_value' => 1]);
            $payment_method =PaymentMethod::where('code','=','PUE')->first();
            $request->merge(['payment_method_id' => !empty($payment_method) ? $payment_method->id : null]);
            $cfdi_use =CfdiUse::where('code','=','P01')->first();
            if(setting('cfdi_version') == 'cfdi40'){
                $cfdi_use =CfdiUse::where('code','=','CN01')->first();
            }
            $request->merge(['cfdi_use_id' => !empty($cfdi_use) ? $cfdi_use->id : null]);

            //Datos del empleado
            $employee = Employee::findOrFail($request->employee_id);
            $request->merge(['base_salary' => !empty($employee) ? $employee->base_salary : 0]);
            $request->merge(['sdi' => !empty($employee) ? $employee->sdi : 0]);

            //Obtiene folio
            $document_type = Helper::getNextDocumentTypeByCode($this->document_type_code,$company->id,isset($request->pre_payroll) ? true : false, $request->branch_office_id);
            $request->merge(['document_type_id' => $document_type['id']]);
            $request->merge(['name' => $document_type['name']]);
            $request->merge(['serie' => $document_type['serie']]);
            $request->merge(['folio' => $document_type['folio']]);

            //Valida que tenga folios disponibles
            if(BaseHelper::getAvailableFolios()<=0 && !isset($request->pre_payroll)){
                throw new \Exception(__('general.error_available_folios'));
            }

            //Guardar
            //Registro principal
            $employee_payroll = EmployeePayroll::create($request->input());

            //Registro de lineas
            $amount_discount = 0;
            $amount_untaxed = 0;
            $amount_isr = 0;
            $amount_tax = 0;
            $amount_tax_ret = 0;
            $amount_total = 0;
            $amount_deductions = 0;
            $amount_perceptions = 0;
            $amount_other_payment_types = 0;

            //Percepciones
            if (!empty($request->item_perception)) {
                foreach ($request->item_perception as $key => $item) {
                    $perception = null;
                    if(!empty($item['perception_id'])){
                        $perception = Perception::findOrFail($item['perception_id']);
                    }

                    //Logica
                    $item_amount_taxed = (double)$item['amount_taxed'];
                    $item_amount_exempt = (double)$item['amount_exempt'];
                    $item_amount_total = $item_amount_taxed + $item_amount_exempt;
                    $item_amount_caused = 0;

                    //Sumatoria totales
                    $amount_discount += 0;
                    $amount_untaxed += $item_amount_total;
                    $amount_isr += 0;
                    $amount_tax += 0;
                    $amount_tax_ret += 0;
                    $amount_perceptions += $item_amount_total;
                    $amount_deductions += 0;
                    $amount_other_payment_types += 0;

                    //Guardar linea
                    $employee_payroll_line = EmployeePayrollLine::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'employee_payroll_id' => $employee_payroll->id,
                        'name' => $perception->name,
                        'code' => (!empty($item['code']) ? $item['code'] : (!empty($perception->code2) ? $perception->code2 : $perception->code)),
                        'perception_id' => $item['perception_id'],
                        'amount_taxed' => $item_amount_taxed,
                        'amount_exempt' => $item_amount_exempt,
                        'amount_total' => $item_amount_total,
                        'amount_caused' => $item_amount_caused,
                        'sort_order' => $key,
                        'status' => 1,
                        'ac_market_value' => (double)($item['ac_market_value'] ?? 0),
                        'ac_price_when_granted' => (double)($item['ac_price_when_granted'] ?? 0),
                        'hr_days_1' => (int)($item['hr_days_1'] ?? 0),
                        'hr_overtimes_type_id_1' => $item['hr_overtimes_type_id_1'] ?? null,
                        'hr_hours_1' => (int)($item['hr_hours_1'] ?? 0),
                        'hr_amount_paid_1' => (double)($item['hr_amount_paid_1'] ?? 0),
                        'hr_days_2' => (int)($item['hr_days_2'] ?? 0),
                        'hr_overtimes_type_id_2' => $item['hr_overtimes_type_id_2'] ?? null,
                        'hr_hours_2' => (int)($item['hr_hours_2'] ?? 0),
                        'hr_amount_paid_2' => (double)($item['hr_amount_paid_2'] ?? 0),
                        'hr_days_3' => (int)($item['hr_days_3'] ?? 0),
                        'hr_overtimes_type_id_3' => $item['hr_overtimes_type_id_3'] ?? null,
                        'hr_hours_3' => (int)($item['hr_hours_3'] ?? 0),
                        'hr_amount_paid_3' => (double)($item['hr_amount_paid_3'] ?? 0)
                    ]);
                }
            }

            //Deducciones
            if (!empty($request->item_deduction)) {
                foreach ($request->item_deduction as $key => $item) {
                    $deduction = null;
                    if(!empty($item['deduction_id'])){
                        $deduction = Deduction::findOrFail($item['deduction_id']);
                    }

                    //Logica
                    $item_amount_taxed = (double)$item['amount_total'];
                    $item_amount_exempt = 0;
                    $item_amount_total = $item_amount_taxed + $item_amount_exempt;
                    $item_amount_caused = 0;

                    //Sumatoria totales
                    if(!empty($deduction) && in_array($deduction->code,['002','101'])){ //Mostrar el ISR separado
                        $amount_discount += 0;
                        $amount_untaxed += 0;
                        $amount_isr += $item_amount_total;
                        $amount_tax += 0;
                        $amount_tax_ret += 0;
                    }else {
                        $amount_discount += $item_amount_total;
                        $amount_untaxed += 0;
                        $amount_isr += 0;
                        $amount_tax += 0;
                        $amount_tax_ret += 0;
                    }
                    $amount_perceptions += 0;
                    $amount_deductions += $item_amount_total;
                    $amount_other_payment_types += 0;

                    //Guardar linea
                    $employee_payroll_line = EmployeePayrollLine::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'employee_payroll_id' => $employee_payroll->id,
                        'name' => $deduction->name,
                        'code' => (!empty($item['code']) ? $item['code'] : (!empty($deduction->code2) ? $deduction->code2 : $deduction->code)),
                        'deduction_id' => $item['deduction_id'],
                        'amount_taxed' => $item_amount_taxed,
                        'amount_exempt' => $item_amount_exempt,
                        'amount_total' => $item_amount_total,
                        'amount_caused' => $item_amount_caused,
                        'sort_order' => $key,
                        'status' => 1,
                    ]);
                }
            }

            //Otros pagos
            if (!empty($request->item_other_payment_type)) {
                foreach ($request->item_other_payment_type as $key => $item) {
                    $other_payment_type = null;
                    if(!empty($item['other_payment_type_id'])){
                        $other_payment_type = OtherPaymentType::findOrFail($item['other_payment_type_id']);
                    }

                    //Logica
                    $item_amount_taxed = (double)$item['amount_total'];
                    $item_amount_exempt = 0;
                    $item_amount_total = $item_amount_taxed + $item_amount_exempt;
                    $item_amount_caused = (double)$item['amount_caused'];

                    //Sumatoria totales
                    $amount_discount += 0;
                    $amount_untaxed += $item_amount_total;
                    $amount_isr += 0;
                    $amount_tax += 0;
                    $amount_tax_ret += 0;
                    $amount_perceptions += 0;
                    $amount_deductions += 0;
                    $amount_other_payment_types += $item_amount_total;

                    //Guardar linea
                    $employee_payroll_line = EmployeePayrollLine::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'employee_payroll_id' => $employee_payroll->id,
                        'name' => $other_payment_type->name,
                        'code' => (!empty($item['code']) ? $item['code'] : (!empty($other_payment_type->code2) ? $other_payment_type->code2 : $other_payment_type->code)),
                        'other_payment_type_id' => $item['other_payment_type_id'],
                        'amount_taxed' => $item_amount_taxed,
                        'amount_exempt' => $item_amount_exempt,
                        'amount_total' => $item_amount_total,
                        'amount_caused' => $item_amount_caused,
                        'sort_order' => $key,
                        'status' => 1,
                        'cm_positive_balance' => (int)($item['cm_positive_balance'] ?? 0),
                        'cm_year' => (double)($item['cm_year'] ?? 0),
                        'cm_remainder' => (double)($item['cm_remainder'] ?? 0)
                    ]);
                }
            }

            //Incapacidades
            if (!empty($request->item_disability)) {
                foreach ($request->item_disability as $key => $item) {
                    $disability_type = null;
                    if(!empty($item['disability_type_id'])){
                        $disability_type = DisabilityType::findOrFail($item['disability_type_id']);
                    }

                    //Logica

                    //Guardar linea
                    $employee_payroll_line = EmployeePayrollLine::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'employee_payroll_id' => $employee_payroll->id,
                        'name' => $disability_type->name,
                        'disability_type_id' => $item['disability_type_id'],
                        'disability_days' => (int)$item['disability_days'],
                        'disability_amount' => (double)$item['disability_amount'],
                        'sort_order' => $key,
                        'status' => 1,
                    ]);
                }
            }

            //Subcontratacion
            if (!empty($request->item_outsourcing)) {
                foreach ($request->item_outsourcing as $key => $item) {
                    //Guardar
                    $employee_payroll_outsourcing = EmployeePayrollOutsourcing::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'employee_payroll_id' => $employee_payroll->id,
                        'taxid' => $item['taxid'],
                        'percent_time' => (double)$item['percent_time'],
                        'sort_order' => $key,
                        'status' => 1,
                    ]);
                }
            }

            //Cfdi relacionados
            if (!empty($request->item_relation)) {
                foreach ($request->item_relation as $key => $result) {
                    $employee_payroll_cfdi_relation = EmployeePayrollCfdi::where('employee_payroll_id','=',$result['relation_id'])->first();

                    //Guardar
                    $employee_payroll_relation = EmployeePayrollRelation::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'employee_payroll_id' => $employee_payroll->id,
                        'relation_id' => $result['relation_id'],
                        'sort_order' => $key,
                        'status' => 1,
                        'uuid_related' => !empty($employee_payroll_cfdi_relation) ? $employee_payroll_cfdi_relation->uuid : '',
                    ]);
                }
            }

            //Valida que tenga exista la clase de facturacion
            $class_cfdi = setting('cfdi_version');
            if (empty($class_cfdi)) {
                throw new \Exception(__('general.error_cfdi_version'));
            }
            if (!method_exists($this, $class_cfdi)) {
                throw new \Exception(__('general.error_cfdi_class_exists'));
            }

            //Registros de cfdi
            $employee_payroll_cfdi = EmployeePayrollCfdi::create([
                'created_uid' => \Auth::user()->id,
                'updated_uid' => \Auth::user()->id,
                'employee_payroll_id' => $employee_payroll->id,
                'name' => $employee_payroll->name,
                'cfdi_version' => $class_cfdi,
                'status' => 1,
            ]);

            //Gran total
            $amount_total = $amount_untaxed-$amount_discount-$amount_isr;

            //Actualiza registro principal con totales
            $employee_payroll->amount_discount = $amount_discount;
            $employee_payroll->amount_untaxed = $amount_untaxed;
            $employee_payroll->amount_isr = $amount_isr;
            $employee_payroll->amount_tax = $amount_tax;
            $employee_payroll->amount_tax_ret = $amount_tax_ret;
            $employee_payroll->amount_total = $amount_total;
            $employee_payroll->amount_perceptions = $amount_perceptions;
            $employee_payroll->amount_deductions = $amount_deductions;
            $employee_payroll->amount_other_payment_types = $amount_other_payment_types;
            $employee_payroll->update();

            //Solo guarda los datos sin timbrar en caso de pre-factura
            if(isset($request->pre_payroll)) {
                //Mensaje
                flash(__('general.text_success_pre_employee_payroll'))->success();

                //Genera XML sin timbrar
                $tmp = $this->$class_cfdi($employee_payroll, true);

                //Guardar registros de CFDI
                $employee_payroll_cfdi->fill(array_only($tmp, [
                    'pac_id',
                    'cfdi_version',
                    'uuid',
                    'date',
                    'file_xml',
                ]));
                $employee_payroll_cfdi->save();

            }else{ //Crear CFDI
                //Valida Empresa y PAC para timbrado
                PacHelper::validateSatActions($company);

                //Crear XML y timbra
                $tmp = $this->$class_cfdi($employee_payroll);

                //Guardar registros de CFDI
                $employee_payroll_cfdi->fill(array_only($tmp, [
                    'pac_id',
                    'cfdi_version',
                    'uuid',
                    'date',
                    'file_xml',
                    'file_xml_pac',
                ]));
                $employee_payroll_cfdi->save();

                //Disminuye folios
                BaseHelper::decrementFolios();

                //Mensaje
                flash(__('general.text_success_employee_payroll_cfdi'))->success();
            }

            \DB::connection('tenant')->commit();

            if(!isset($request->pre_payroll)) {
                $this->saveCfdiDownloads($employee_payroll, $employee_payroll_cfdi);
            }

        } catch (\Exception $e) {
            //Fix fechas
            $request->merge([
                'date_payment' => Helper::convertSqlToDate($request->date_payment),
            ]);
            $request->merge([
                'date_start_payment' => Helper::convertSqlToDate($request->date_start_payment),
            ]);
            $request->merge([
                'date_end_payment' => Helper::convertSqlToDate($request->date_end_payment),
            ]);

            \DB::connection('tenant')->rollback();
            $company = Helper::defaultCompany(); //Empresa
            \Log::error('(' . $company->taxid . ') ' . $e->getMessage());
            flash((isset($request->pre_payroll) ? '' : __('general.error_cfdi_pac').'<br>').$e->getMessage())->error();
            return back()->withInput();
        }


        //Almacenamiento dropbox
        self::dropboxBackup($employee_payroll);

        //Redireccion
        return redirect('/base/employee-payrolls');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Base\EmployeePayroll  $employee_payroll
     * @return \Illuminate\Http\Response
     */
    public function show(EmployeePayroll $employee_payroll)
    {
        //Datos
        $data = [];

        //Si tiene CFDI obtiene la informacion de los nodos
        if(!empty($employee_payroll->employeePayrollCfdi->file_xml_pac) && !empty($employee_payroll->employeePayrollCfdi->uuid)){

            $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
            $file_xml_pac = $path_xml . $employee_payroll->employeePayrollCfdi->file_xml_pac;

            //Valida que el archivo exista
            if(!empty($employee_payroll->employeePayrollCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {
                $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
                $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

                //Genera codigo QR
                $image = QrCode::format('png')->size(150)->margin(0)->generate($data['qr_cadena']);
                $data['qr'] = 'data:image/png;base64,' . base64_encode($image);

                //Regimen fiscal
                $data['tax_regimen'] = TaxRegimen::where('code', $data['cfdi33']->emisor['RegimenFiscal'])->first()->name_sat;
                $data['tax_regimen_employee'] = !empty($data['cfdi33']->receptor['RegimenFiscalReceptor']) ? TaxRegimen::where('code', $data['cfdi33']->receptor['RegimenFiscalReceptor'])->first()->name_sat : '';
            }
        }

        return view('base.employee_payrolls.show', compact('employee_payroll','data'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Base\EmployeePayroll  $employee_payroll
     * @return \Illuminate\Http\Response
     */
    public function edit(EmployeePayroll $employee_payroll)
    {
        $branch_offices = BranchOffice::populateSelect()->pluck('name', 'id');
        $payroll_types = PayrollType::populateSelect()->get()->pluck('name_sat', 'id');
        $cfdi_relations = CfdiRelation::populateSelect()->where('code','=','04')->get()->pluck('name_sat', 'id');
        $source_resources = SourceResource::populateSelect()->get()->pluck('name_sat', 'id');
        $company = $employee_payroll->company; //Empresa
        if(!empty($company->tax_regimen_id2)){
            $tax_regimens = TaxRegimen::populateSelect()->whereIn('id',[$company->tax_regimen_id, $company->tax_regimen_id2])->get()->pluck('name_sat', 'id');
        }else{
            $tax_regimens = collect([]);
        }
        $tax_regimen_employees = TaxRegimen::populateSelect()->get()->pluck('name_sat', 'id');

        return view('base.employee_payrolls.edit',
            compact( 'employee_payroll','branch_offices', 'payroll_types', 'cfdi_relations','source_resources', 'tax_regimens', 'tax_regimen_employees'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Base\EmployeePayroll  $employee_payroll
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, EmployeePayroll $employee_payroll)
    {
        //Validacion
        $this->validation($request);

        \DB::connection('tenant')->beginTransaction();
        try {

            //Logica
            $request->merge(['created_uid' => \Auth::user()->id]);
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['status' => isset($request->pre_payroll) ? EmployeePayroll::DRAFT : EmployeePayroll::OPEN]);
            //Ajusta fecha
            $request->merge(['date' => Helper::dateTimeToSql(BaseHelper::getDateTimeBranchOffice($request->branch_office_id))]);
            $request->merge(['date_payment' => Helper::convertDateToSql($request->date_payment)]);
            $request->merge(['date_start_payment' => Helper::convertDateToSql($request->date_start_payment)]);
            $request->merge(['date_end_payment' => Helper::convertDateToSql($request->date_end_payment)]);
            $request->merge(['rt_amount_an_exhibition' => $request->rt_amount_an_exhibition ?? 0]);
            $request->merge(['rt_partiality_amount' => $request->rt_partiality_amount ?? 0]);
            $request->merge(['rt_daily_amount' => $request->rt_daily_amount ?? 0]);
            $request->merge(['rt_cumulative_income' => $request->rt_cumulative_income ?? 0]);
            $request->merge(['rt_non_cumulative_income' => $request->rt_non_cumulative_income ?? 0]);
            $request->merge(['sp_total_amount' => $request->sp_total_amount ?? 0]);
            $request->merge(['sp_years_of_service' => $request->sp_years_of_service ?? 0]);
            $request->merge(['sp_last_salary' => $request->sp_last_salary ?? 0]);
            $request->merge(['sp_cumulative_income' => $request->sp_cumulative_income ?? 0]);
            $request->merge(['sp_non_cumulative_income' => $request->sp_non_cumulative_income ?? 0]);
            $request->merge(['sncf_amount_source_resource' => $request->sncf_amount_source_resource ?? 0]);

            //Datos del empleado
            $employee = Employee::findOrFail($request->employee_id);
            $request->merge(['base_salary' => !empty($employee) ? $employee->base_salary : 0]);
            $request->merge(['sdi' => !empty($employee) ? $employee->sdi : 0]);

            $employee_payroll->fill($request->only([
                'updated_uid',
                'date',
                'date_payment',
                'date_start_payment',
                'date_end_payment',
                'employee_id',
                'branch_office_id',
                'cfdi_relation_id',
                'base_salary',
                'sdi',
                'comment',
                'status',
                'reference',
                'taxid_origin',
                'rt_amount_an_exhibition',
                'rt_partiality_amount',
                'rt_daily_amount',
                'rt_cumulative_income',
                'rt_non_cumulative_income',
                'sp_total_amount',
                'sp_years_of_service',
                'sp_last_salary',
                'sp_cumulative_income',
                'sp_non_cumulative_income',
                'source_resource_id',
                'sncf_amount_source_resource',
                'tax_regimen_id',
                'tax_regimen_employee_id'
            ]));

            //Guardar
            //Registro principal
            $employee_payroll->save();

            //Actualiza folios
            if(!isset($request->pre_payroll)) {
                //Valida que tenga folios disponibles
                if(BaseHelper::getAvailableFolios()<=0){
                    throw new \Exception(__('general.error_available_folios'));
                }

                //Obtiene folio
                $document_type = Helper::getNextDocumentTypeByCode($this->document_type_code, $employee_payroll->company->id, false, $employee_payroll->branch_office_id);
                $employee_payroll->draft = $employee_payroll->name;
                $employee_payroll->name = $document_type['name'];
                $employee_payroll->serie = $document_type['serie'];
                $employee_payroll->folio = $document_type['folio'];
                $employee_payroll->save();
            }

            //Registro de lineas
            $amount_discount = 0;
            $amount_untaxed = 0;
            $amount_isr = 0;
            $amount_tax = 0;
            $amount_tax_ret = 0;
            $amount_total = 0;
            $amount_deductions = 0;
            $amount_perceptions = 0;
            $amount_other_payment_types = 0;


            //Elimina partidas Percepciones
            if (!empty($request->delete_item_perception)) {
                foreach ($request->delete_item_perception as $key => $result) {
                    //Actualizar status
                    $employee_payroll_line = EmployeePayrollLine::findOrFail($result);
                    $employee_payroll_line->updated_uid = \Auth::user()->id;
                    $employee_payroll_line->status = 0;
                    $employee_payroll_line->save();
                }
            }

            //Percepciones
            if (!empty($request->item_perception)) {
                foreach ($request->item_perception as $key => $item) {
                    $perception = null;
                    if(!empty($item['perception_id'])){
                        $perception = Perception::findOrFail($item['perception_id']);
                    }

                    //Logica
                    $item_amount_taxed = (double)$item['amount_taxed'];
                    $item_amount_exempt = (double)$item['amount_exempt'];
                    $item_amount_total = $item_amount_taxed + $item_amount_exempt;
                    $item_amount_caused = 0;

                    //Sumatoria totales
                    $amount_discount += 0;
                    $amount_untaxed += $item_amount_total;
                    $amount_isr += 0;
                    $amount_tax += 0;
                    $amount_tax_ret += 0;
                    $amount_perceptions += $item_amount_total;
                    $amount_deductions += 0;
                    $amount_other_payment_types += 0;

                    //Guardar linea
                    $data = [
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'employee_payroll_id' => $employee_payroll->id,
                        'name' => $perception->name,
                        'code' => (!empty($item['code']) ? $item['code'] : (!empty($perception->code2) ? $perception->code2 : $perception->code)),
                        'perception_id' => $item['perception_id'],
                        'amount_taxed' => $item_amount_taxed,
                        'amount_exempt' => $item_amount_exempt,
                        'amount_total' => $item_amount_total,
                        'amount_caused' => $item_amount_caused,
                        'sort_order' => $key,
                        'status' => 1,
                        'ac_market_value' => (double)($item['ac_market_value'] ?? 0),
                        'ac_price_when_granted' => (double)($item['ac_price_when_granted'] ?? 0),
                        'hr_days_1' => (int)($item['hr_days_1'] ?? 0),
                        'hr_overtimes_type_id_1' => $item['hr_overtimes_type_id_1'] ?? null,
                        'hr_hours_1' => (int)($item['hr_hours_1'] ?? 0),
                        'hr_amount_paid_1' => (double)($item['hr_amount_paid_1'] ?? 0),
                        'hr_days_2' => (int)($item['hr_days_2'] ?? 0),
                        'hr_overtimes_type_id_2' => $item['hr_overtimes_type_id_2'] ?? null,
                        'hr_hours_2' => (int)($item['hr_hours_2'] ?? 0),
                        'hr_amount_paid_2' => (double)($item['hr_amount_paid_2'] ?? 0),
                        'hr_days_3' => (int)($item['hr_days_3'] ?? 0),
                        'hr_overtimes_type_id_3' => $item['hr_overtimes_type_id_3'] ?? null,
                        'hr_hours_3' => (int)($item['hr_hours_3'] ?? 0),
                        'hr_amount_paid_3' => (double)($item['hr_amount_paid_3'] ?? 0)
                    ];
                    if (!empty($item['id'])) {
                        $employee_payroll_line = EmployeePayrollLine::findOrFail($item['id']);
                        $employee_payroll_line->fill(array_only($data, [
                            'updated_uid',
                            'name',
                            'code',
                            'perception_id',
                            'amount_taxed',
                            'amount_exempt',
                            'amount_total',
                            'amount_caused',
                            'sort_order',
                            'ac_market_value',
                            'ac_price_when_granted',
                            'hr_days_1',
                            'hr_overtimes_type_id_1',
                            'hr_hours_1',
                            'hr_amount_paid_1',
                            'hr_days_2',
                            'hr_overtimes_type_id_2',
                            'hr_hours_2',
                            'hr_amount_paid_2',
                            'hr_days_3',
                            'hr_overtimes_type_id_3',
                            'hr_hours_3',
                            'hr_amount_paid_3'
                        ]));
                        $employee_payroll_line->save();
                    }else{
                        $employee_payroll_line = EmployeePayrollLine::create($data);
                    }
                }
            }

            //Elimina partidas Deducciones
            if (!empty($request->delete_item_deduction)) {
                foreach ($request->delete_item_deduction as $key => $result) {
                    //Actualizar status
                    $employee_payroll_line = EmployeePayrollLine::findOrFail($result);
                    $employee_payroll_line->updated_uid = \Auth::user()->id;
                    $employee_payroll_line->status = 0;
                    $employee_payroll_line->save();
                }
            }

            //Deducciones
            if (!empty($request->item_deduction)) {
                foreach ($request->item_deduction as $key => $item) {
                    $deduction = null;
                    if(!empty($item['deduction_id'])){
                        $deduction = Deduction::findOrFail($item['deduction_id']);
                    }

                    //Logica
                    $item_amount_taxed = (double)$item['amount_total'];
                    $item_amount_exempt = 0;
                    $item_amount_total = $item_amount_taxed + $item_amount_exempt;
                    $item_amount_caused = 0;

                    //Sumatoria totales
                    if(!empty($deduction) && in_array($deduction->code,['002','101'])){ //Mostrar el ISR separado
                        $amount_discount += 0;
                        $amount_untaxed += 0;
                        $amount_isr += $item_amount_total;
                        $amount_tax += 0;
                        $amount_tax_ret += 0;
                    }else {
                        $amount_discount += $item_amount_total;
                        $amount_untaxed += 0;
                        $amount_isr += 0;
                        $amount_tax += 0;
                        $amount_tax_ret += 0;
                    }
                    $amount_perceptions += 0;
                    $amount_deductions += $item_amount_total;
                    $amount_other_payment_types += 0;

                    //Guardar linea
                    $data = [
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'employee_payroll_id' => $employee_payroll->id,
                        'name' => $deduction->name,
                        'code' => (!empty($item['code']) ? $item['code'] : (!empty($deduction->code2) ? $deduction->code2 : $deduction->code)),
                        'deduction_id' => $item['deduction_id'],
                        'amount_taxed' => $item_amount_taxed,
                        'amount_exempt' => $item_amount_exempt,
                        'amount_total' => $item_amount_total,
                        'amount_caused' => $item_amount_caused,
                        'sort_order' => $key,
                        'status' => 1,
                    ];
                    if (!empty($item['id'])) {
                        $employee_payroll_line = EmployeePayrollLine::findOrFail($item['id']);
                        $employee_payroll_line->fill(array_only($data, [
                            'updated_uid',
                            'name',
                            'code',
                            'deduction_id',
                            'amount_taxed',
                            'amount_exempt',
                            'amount_total',
                            'amount_caused',
                            'sort_order',
                        ]));
                        $employee_payroll_line->save();
                    }else{
                        $employee_payroll_line = EmployeePayrollLine::create($data);
                    }
                }
            }

            //Elimina partidas Otros pagos
            if (!empty($request->delete_item_other_payment_type)) {
                foreach ($request->delete_item_other_payment_type as $key => $result) {
                    //Actualizar status
                    $employee_payroll_line = EmployeePayrollLine::findOrFail($result);
                    $employee_payroll_line->updated_uid = \Auth::user()->id;
                    $employee_payroll_line->status = 0;
                    $employee_payroll_line->save();
                }
            }

            //Otros pagos
            if (!empty($request->item_other_payment_type)) {
                foreach ($request->item_other_payment_type as $key => $item) {
                    $other_payment_type = null;
                    if(!empty($item['other_payment_type_id'])){
                        $other_payment_type = OtherPaymentType::findOrFail($item['other_payment_type_id']);
                    }

                    //Logica
                    $item_amount_taxed = (double)$item['amount_total'];
                    $item_amount_exempt = 0;
                    $item_amount_total = $item_amount_taxed + $item_amount_exempt;
                    $item_amount_caused = (double)$item['amount_caused'];

                    //Sumatoria totales
                    $amount_discount += 0;
                    $amount_untaxed += $item_amount_total;
                    $amount_isr += 0;
                    $amount_tax += 0;
                    $amount_tax_ret += 0;
                    $amount_perceptions += 0;
                    $amount_deductions += 0;
                    $amount_other_payment_types += $item_amount_total;

                    //Guardar linea
                    $data = [
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'employee_payroll_id' => $employee_payroll->id,
                        'name' => $other_payment_type->name,
                        'code' => (!empty($item['code']) ? $item['code'] : (!empty($other_payment_type->code2) ? $other_payment_type->code2 : $other_payment_type->code)),
                        'other_payment_type_id' => $item['other_payment_type_id'],
                        'amount_taxed' => $item_amount_taxed,
                        'amount_exempt' => $item_amount_exempt,
                        'amount_total' => $item_amount_total,
                        'amount_caused' => $item_amount_caused,
                        'sort_order' => $key,
                        'status' => 1,
                        'cm_positive_balance' => (int)($item['cm_positive_balance'] ?? 0),
                        'cm_year' => (double)($item['cm_year'] ?? 0),
                        'cm_remainder' => (double)($item['cm_remainder'] ?? 0)
                    ];
                    if (!empty($item['id'])) {
                        $employee_payroll_line = EmployeePayrollLine::findOrFail($item['id']);
                        $employee_payroll_line->fill(array_only($data, [
                            'updated_uid',
                            'name',
                            'code',
                            'other_payment_type_id',
                            'amount_taxed',
                            'amount_exempt',
                            'amount_total',
                            'amount_caused',
                            'sort_order',
                            'cm_positive_balance',
                            'cm_year',
                            'cm_remainder'
                        ]));
                        $employee_payroll_line->save();
                    }else{
                        $employee_payroll_line = EmployeePayrollLine::create($data);
                    }
                }
            }

            //Elimina partidas Incapacidades
            if (!empty($request->delete_item_disability)) {
                foreach ($request->delete_item_disability as $key => $result) {
                    //Actualizar status
                    $employee_payroll_line = EmployeePayrollLine::findOrFail($result);
                    $employee_payroll_line->updated_uid = \Auth::user()->id;
                    $employee_payroll_line->status = 0;
                    $employee_payroll_line->save();
                }
            }

            //Incapacidades
            if (!empty($request->item_disability)) {
                foreach ($request->item_disability as $key => $item) {
                    $disability_type = null;
                    if(!empty($item['disability_type_id'])){
                        $disability_type = DisabilityType::findOrFail($item['disability_type_id']);
                    }

                    //Logica

                    //Guardar linea
                    $data = [
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'employee_payroll_id' => $employee_payroll->id,
                        'name' => $disability_type->name,
                        'disability_type_id' => $item['disability_type_id'],
                        'disability_days' => (int)$item['disability_days'],
                        'disability_amount' => (double)$item['disability_amount'],
                        'sort_order' => $key,
                        'status' => 1,
                    ];
                    if (!empty($item['id'])) {
                        $employee_payroll_line = EmployeePayrollLine::findOrFail($item['id']);
                        $employee_payroll_line->fill(array_only($data, [
                            'updated_uid',
                            'name',
                            'disability_type_id',
                            'disability_days',
                            'disability_amount',
                            'sort_order',
                        ]));
                        $employee_payroll_line->save();
                    }else{
                        $employee_payroll_line = EmployeePayrollLine::create($data);
                    }
                }
            }

            //Elimina partidas Subcontratacion
            if (!empty($request->delete_item_outsourcing)) {
                foreach ($request->delete_item_outsourcing as $key => $result) {
                    //Actualizar status
                    $employee_payroll_outsourcing = EmployeePayrollOutsourcing::findOrFail($result);
                    $employee_payroll_outsourcing->updated_uid = \Auth::user()->id;
                    $employee_payroll_outsourcing->status = 0;
                    $employee_payroll_outsourcing->save();
                }
            }

            //Subcontratacion
            if (!empty($request->item_outsourcing)) {
                foreach ($request->item_outsourcing as $key => $item) {
                    //Guardar
                    $data = [
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'employee_payroll_id' => $employee_payroll->id,
                        'taxid' => $item['taxid'],
                        'percent_time' => (double)$item['percent_time'],
                        'sort_order' => $key,
                        'status' => 1,
                    ];
                    if (!empty($item['id'])) {
                        $employee_payroll_outsourcing = EmployeePayrollOutsourcing::findOrFail($item['id']);
                        $employee_payroll_outsourcing->fill(array_only($data, [
                            'updated_uid',
                            'taxid',
                            'percent_time',
                            'sort_order',
                        ]));
                        $employee_payroll_outsourcing->save();
                    }else{
                        $employee_payroll_outsourcing = EmployeePayrollOutsourcing::create($data);
                    }
                }
            }

            //Cfdi relacionados
            EmployeePayrollRelation::where('employee_payroll_id','=',$employee_payroll->id)->delete(); //Borra todo e inserta nuevamente
            if (!empty($request->item_relation)) {
                foreach ($request->item_relation as $key => $result) {
                    $employee_payroll_cfdi_relation = EmployeePayrollCfdi::where('employee_payroll_id','=',$result['relation_id'])->first();

                    //Guardar
                    $employee_payroll_relation = EmployeePayrollRelation::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'employee_payroll_id' => $employee_payroll->id,
                        'relation_id' => $result['relation_id'],
                        'sort_order' => $key,
                        'status' => 1,
                        'uuid_related' => !empty($employee_payroll_cfdi_relation) ? $employee_payroll_cfdi_relation->uuid : '',
                    ]);
                }
            }

            //Valida que tenga exista la clase de facturacion
            $class_cfdi = setting('cfdi_version');
            if (empty($class_cfdi)) {
                throw new \Exception(__('general.error_cfdi_version'));
            }
            if (!method_exists($this, $class_cfdi)) {
                throw new \Exception(__('general.error_cfdi_class_exists'));
            }

            //Registros de cfdi
            EmployeePayrollCfdi::where('employee_payroll_id','=',$employee_payroll->id)->delete(); //Borra todo e inserta nuevamente
            $employee_payroll_cfdi = EmployeePayrollCfdi::create([
                'created_uid' => \Auth::user()->id,
                'updated_uid' => \Auth::user()->id,
                'employee_payroll_id' => $employee_payroll->id,
                'name' => $employee_payroll->name,
                'cfdi_version' => $class_cfdi,
                'status' => 1,
            ]);

            //Gran total
            $amount_total = $amount_untaxed-$amount_discount-$amount_isr;

            //Actualiza registro principal con totales
            $employee_payroll->amount_discount = $amount_discount;
            $employee_payroll->amount_untaxed = $amount_untaxed;
            $employee_payroll->amount_isr = $amount_isr;
            $employee_payroll->amount_tax = $amount_tax;
            $employee_payroll->amount_tax_ret = $amount_tax_ret;
            $employee_payroll->amount_total = $amount_total;
            $employee_payroll->amount_perceptions = $amount_perceptions;
            $employee_payroll->amount_deductions = $amount_deductions;
            $employee_payroll->amount_other_payment_types = $amount_other_payment_types;
            $employee_payroll->update();

            //Solo guarda los datos sin timbrar en caso de pre-factura
            if(isset($request->pre_payroll)) {
                //Mensaje
                flash(__('general.text_success_pre_employee_payroll'))->success();

                //Crear XML y timbra
                $tmp = $this->$class_cfdi($employee_payroll, true);

                //Guardar registros de CFDI
                $employee_payroll_cfdi->fill(array_only($tmp, [
                    'pac_id',
                    'cfdi_version',
                    'uuid',
                    'date',
                    'file_xml',
                    'file_xml_pac',
                ]));
                $employee_payroll_cfdi->save();

            }else{ //Crear CFDI
                //Valida Empresa y PAC para timbrado
                PacHelper::validateSatActions($employee_payroll->company);

                //Crear XML y timbra
                $tmp = $this->$class_cfdi($employee_payroll);

                //Guardar registros de CFDI
                $employee_payroll_cfdi->fill(array_only($tmp, [
                    'pac_id',
                    'cfdi_version',
                    'uuid',
                    'date',
                    'file_xml',
                    'file_xml_pac',
                ]));
                $employee_payroll_cfdi->save();

                //Disminuye folios
                BaseHelper::decrementFolios();

                //Mensaje
                flash(__('general.text_success_employee_payroll_cfdi'))->success();
            }

            \DB::connection('tenant')->commit();

            if(!isset($request->pre_payroll)) {
                $this->saveCfdiDownloads($employee_payroll, $employee_payroll_cfdi);
            }


        } catch (\Exception $e) {
            //Fix fechas
            $request->merge([
                'date_payment' => Helper::convertSqlToDate($request->date_payment),
            ]);
            $request->merge([
                'date_start_payment' => Helper::convertSqlToDate($request->date_start_payment),
            ]);
            $request->merge([
                'date_end_payment' => Helper::convertSqlToDate($request->date_end_payment),
            ]);

            \DB::connection('tenant')->rollback();
            \Log::error('(' . $employee_payroll->company->taxid . ') ' . $e->getMessage());
            flash((isset($request->pre_payroll) ? '' : __('general.error_cfdi_pac').'<br>').$e->getMessage())->error();
            return back()->withInput();
        }

        //Almacenamiento dropbox
        self::dropboxBackup($employee_payroll);

        //Redireccion
        return redirect('/base/employee-payrolls');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Base\EmployeePayroll  $employee_payroll
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, EmployeePayroll $employee_payroll)
    {
        \DB::connection('tenant')->beginTransaction();
        try {
            //Logica
            if ((int)$employee_payroll->status != EmployeePayroll::CANCEL) {
                //Actualiza status
                $employee_payroll->updated_uid = \Auth::user()->id;
                $employee_payroll->status = EmployeePayroll::CANCEL;
                $employee_payroll->save();

                //Actualiza status CFDI
                $employee_payroll->employeePayrollCfdi->status = 0;
                $employee_payroll->employeePayrollCfdi->reason_cancellation_id = $request->reason_cancellation_id;
                $employee_payroll->employeePayrollCfdi->reason_cancellation_uuid = $request->reason_cancellation_uuid;
                $employee_payroll->employeePayrollCfdi->save();

                //Cancelacion del timbre fiscal
                if (!empty($employee_payroll->employeePayrollCfdi->cfdi_version) && !empty($employee_payroll->employeePayrollCfdi->uuid)) {
                    //Valida Empresa y PAC para cancelar timbrado
                    PacHelper::validateSatCancelActions($employee_payroll->company,$employee_payroll->employeePayrollCfdi->pac);

                    //Obtener el sellos del CFDI
                    $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
                    $file_xml_pac = $path_xml . $employee_payroll->employeePayrollCfdi->file_xml_pac;
                    $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
                    $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

                    //Arreglo temporal para actualizar Customer Invoice CFDI
                    $tmp = [
                        'cancel_date' => Helper::dateTimeToSql(\Date::now()),
                        'cancel_response' => '',
                        'cancel_state' => $request->cancel_state,
                        'rfcR' => $employee_payroll->employee->taxid,
                        'uuid' => $employee_payroll->employeePayrollCfdi->uuid,
                        'total' => Helper::numberFormat($employee_payroll->amount_total,
                            $employee_payroll->currency->decimal_place, false),
                        'cfdi_type' => $employee_payroll->documentType->cfdiType->code,
                        'cfdi_version' => 'cfdi3.3',
                        'fe' => substr($data['cfdi33']->complemento->timbreFiscalDigital['SelloCFD'],-8), //Los últimos 8 caracteres del Atributo Sello del CFDI
                        'file_xml_pac' => $file_xml_pac,
                        'reason_cancellation_code' => $employee_payroll->employeePayrollCfdi->reasonCancellation->code ?? '',
                        'reason_cancellation_uuid' => $employee_payroll->employeePayrollCfdi->reason_cancellation_uuid ?? '',
                    ];

                    //Cancelar Timbrado de XML
                    $class_pac = $employee_payroll->employeePayrollCfdi->pac->code . 'Cancel';
                    $tmp = PacHelper::$class_pac($tmp,$employee_payroll->company,$employee_payroll->employeePayrollCfdi->pac);

                    //Guardar registros de CFDI
                    $employee_payroll->employeePayrollCfdi->fill(array_only($tmp,[
                        'cancel_date',
                        'cancel_response',
                        'cancel_state',
                    ]));
                    $employee_payroll->employeePayrollCfdi->save();

                    $cfdi_download = CfdiDownload::where('uuid', $employee_payroll->employeePayrollCfdi->uuid)->where('type', 2)->first();
                    if(!empty($cfdi_download)){
                        $cfdi_download->status = 'Cancelado';
                        $cfdi_download->save();
                    }

                    //Disminuye folios
                    BaseHelper::decrementFolios();

                }
            }
            \DB::connection('tenant')->commit();

            //Mensaje
            flash(__('general.text_form_success_cancel'))->success();

        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            \Log::error('(' . $employee_payroll->company->taxid . ') ' . $e->getMessage());
            flash($e->getMessage())->error();
            return redirect('/base/employee-payrolls');
        }

        //Almacenamiento dropbox
        if ((int)$employee_payroll->status == EmployeePayroll::CANCEL) {
            self::dropboxBackup($employee_payroll,false,true);
        }

        //Redireccion
        return redirect('/base/employee-payrolls');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {

        $this->validate($request, [
            'employee_id' => 'required|integer',
            'payroll_type_id' => 'required|integer',
            'branch_office_id' => 'required|integer',
            'date_payment' => 'required|date|date_format:"'.setting('date_format', 'd-m-Y').'"',
            'date_start_payment' => 'required|date|date_format:"'.setting('date_format', 'd-m-Y').'"',
            'date_end_payment' => 'required|date|date_format:"'.setting('date_format', 'd-m-Y').'"|after_or_equal:' . Helper::date(Helper::createDate($request->date_start_payment)),
            'payment_days' => 'required|numeric|min:0.1',
            'cfdi_relation_id' => 'nullable|integer|required_with:item_relation',
            'item_perception.*.perception_id' => 'required',
            'item_perception.*.amount_taxed' => 'nullable|numeric|min:0',
            'item_perception.*.amount_exempt' => 'nullable|numeric|min:0',
            'item_deduction.*.deduction_id' => 'required',
            'item_deduction.*.amount_total' => 'required|numeric|min:0',
            'item_other_payment_type.*.other_payment_type_id' => 'required',
            'item_other_payment_type.*.amount_total' => 'required|numeric|min:0',
            'item_outsourcing.*.taxid' => 'required',
            'item_outsourcing.*.percent_time' => 'required|numeric|min:0.001',
            'item_relation.*.relation_id' => 'required',

        ], [
            'employee_id.*' => __('base/employee_payroll.error_employee_id'),
            'payroll_type_id.*' => __('base/employee_payroll.error_payroll_type_id'),
            'branch_office_id.*' => __('base/employee_payroll.error_branch_office_id'),
            'date_payment.required' => __('base/employee_payroll.error_date_payment'),
            'date_payment.date' => __('base/employee_payroll.error_date_payment_format'),
            'date_payment.date_format' => __('base/employee_payroll.error_date_payment_format'),
            'date_start_payment.required' => __('base/employee_payroll.error_date_start_payment'),
            'date_start_payment.date' => __('base/employee_payroll.error_date_start_payment_format'),
            'date_start_payment.date_format' => __('base/employee_payroll.error_date_start_payment_format'),
            'date_end_payment.required' => __('base/employee_payroll.error_date_end_payment'),
            'date_end_payment.date' => __('base/employee_payroll.error_date_end_payment_format'),
            'date_end_payment.date_format' => __('base/employee_payroll.error_date_end_payment_format'),
            'date_end_payment.after_or_equal' => __('base/employee_payroll.error_date_end_payment_after'),
            'payment_days.*' => __('base/employee_payroll.error_payment_days'),
            'cfdi_relation_id.*' => __('base/employee_payroll.error_cfdi_relation_id'),
            'item_perception.*.perception_id.*' => __('base/employee_payroll.error_line_perception_id'),
            'item_perception.*.amount_taxed.*' => __('base/employee_payroll.error_line_amount_taxed'),
            'item_perception.*.amount_exempt.*' => __('base/employee_payroll.error_line_amount_exempt'),
            'item_deduction.*.deduction_id.*' => __('base/employee_payroll.error_line_deduction_id'),
            'item_deduction.*.amount_total.*' => __('base/employee_payroll.error_line_amount_total'),
            'item_other_payment_type.*.other_payment_type_id.*' => __('base/employee_payroll.error_line_other_payment_type_id'),
            'item_other_payment_type.*.amount_total.*' => __('base/employee_payroll.error_line_amount_total'),
            'item_outsourcing.*.taxid.*' => __('base/employee_payroll.error_line_taxid'),
            'item_outsourcing.*.percent_time.*' => __('base/employee_payroll.error_line_percent_time'),
            'item_relation.*.relation_id.*' => __('base/employee_payroll.error_relation_relation_id'),
        ]);

        //Validaciones manuales
        $validator = \Validator::make([], []);
        if(setting('cfdi_version') == 'cfdi40'){
            if (empty($request->tax_regimen_employee_id)) {
                $validator->after(function ($validator) {
                    $validator->errors()->add('tax_regimen_employee_id', __('base/employee_payroll.error_tax_regimen_employee_id'));
                });
            }
            $employee = Employee::find($request->employee_id);
            if (empty($employee->postcode) && !in_array($employee->taxid, ['XAXX010101000', 'XEXX010101000'])) {
                $validator->after(function ($validator) {
                    $validator->errors()->add('employee_id', __('base/employee_payroll.error_postcode_employee'));
                });
            }
        }

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }
    }

    /**
     * Crear XML y enviar a timbrar CFDI 3.3
     *
     * @param EmployeePayroll $employee_payroll
     * @return array|\CfdiUtils\Elements\Cfdi33\Concepto|float|int
     * @throws \Exception
     */
    private function cfdi33(EmployeePayroll $employee_payroll, $draft = false)
    {
        try {
            //Logica
            $pac = Pac::findOrFail(setting('default_pac_id')); //PAC

            //Arreglo CFDI 3.3
            $cfdi33 = [];
            if (!empty($employee_payroll->serie)) {
                $cfdi33['Serie'] = $employee_payroll->serie;
            }
            $cfdi33['Folio'] = $employee_payroll->folio;
            $cfdi33['Fecha'] = \Date::parse($employee_payroll->date)->format('Y-m-d\TH:i:s');
            //$cfdi33['Sello']
            $cfdi33['FormaPago'] = $employee_payroll->paymentWay->code;
            $cfdi33['NoCertificado'] = $employee_payroll->company->certificate_number;
            //$cfdi33['Certificado']
            //$cfdi33['CondicionesDePago']
            $cfdi33['SubTotal'] = Helper::numberFormat($employee_payroll->amount_untaxed /*+ $employee_payroll->amount_discount*/, $employee_payroll->currency->decimal_place, false);
            if(($employee_payroll->amount_discount + $employee_payroll->amount_isr)>0) {
                $cfdi33['Descuento'] = Helper::numberFormat($employee_payroll->amount_discount + $employee_payroll->amount_isr, $employee_payroll->currency->decimal_place, false);
            }
            $cfdi33['Moneda'] = $employee_payroll->currency->code;
            //$cfdi33['TipoCambio']
            $cfdi33['Total'] = Helper::numberFormat($employee_payroll->amount_total, $employee_payroll->currency->decimal_place, false);
            $cfdi33['TipoDeComprobante'] = $employee_payroll->documentType->cfdiType->code;
            $cfdi33['MetodoPago'] = $employee_payroll->paymentMethod->code;
            $cfdi33['LugarExpedicion'] = $employee_payroll->branchOffice->postcode;
            if (!empty($employee_payroll->confirmacion)) {
                $cfdi33['Confirmacion'] = $employee_payroll->confirmacion;
            }
            //---Cfdi Relacionados
            $cfdi33_relacionados = [];
            $cfdi33_relacionado = [];
            if (!empty($employee_payroll->cfdi_relation_id)) {
                $cfdi33_relacionados['TipoRelacion'] = $employee_payroll->cfdiRelation->code;
                if ($employee_payroll->employeePayrollRelations->isNotEmpty()) {
                    foreach ($employee_payroll->employeePayrollRelations as $key => $result) {
                        $cfdi33_relacionado[$key] = [];
                        $cfdi33_relacionado[$key]['UUID'] = $result->uuid_related;
                    }
                }
            }
            //---Emisor
            $cfdi33_emisor = [];
            $cfdi33_emisor['Rfc'] = $employee_payroll->company->taxid;
            $cfdi33_emisor['Nombre'] = trim($employee_payroll->company->name);
            $cfdi33_emisor['RegimenFiscal'] = $employee_payroll->company->taxRegimen->code;
            //---Receptor
            $cfdi33_receptor = [];
            $cfdi33_receptor['Rfc'] = $employee_payroll->employee->taxid;
            $cfdi33_receptor['Nombre'] = trim($employee_payroll->employee->name);
            //$cfdi33_receptor['ResidenciaFiscal']
            //$cfdi33_receptor['NumRegIdTrib']
            $cfdi33_receptor['UsoCFDI'] = $employee_payroll->cfdiUse->code;
            //---Conceptos
            $cfdi33_conceptos = [];
            $cfdi33_conceptos_traslados = [];
            $cfdi33_conceptos_retenciones = [];
            $key = 0;
            $cfdi33_conceptos [$key]['ClaveProdServ'] = '84111505';
            //$cfdi33_conceptos [$key]['NoIdentificacion']
            $cfdi33_conceptos [$key]['Cantidad'] = '1';
            $cfdi33_conceptos [$key]['ClaveUnidad'] = 'ACT';
            //$cfdi33_conceptos [$key]['Unidad'];
            $cfdi33_conceptos [$key]['Descripcion'] = 'Pago de nómina';
            $cfdi33_conceptos [$key]['ValorUnitario'] = Helper::numberFormat($employee_payroll->amount_untaxed, 6, false);
            $cfdi33_conceptos [$key]['Importe'] = Helper::numberFormat($employee_payroll->amount_untaxed, 2, false);
            if($employee_payroll->amount_discount + $employee_payroll->amount_isr > 0){
                $cfdi33_conceptos [$key]['Descuento'] = Helper::numberFormat($employee_payroll->amount_discount + $employee_payroll->amount_isr, 2, false);
            }
            //['InformacionAduanera']
            //['CuentaPredial']
            //['ComplementoConcepto']
            //['Parte']
            //Complemento
            //Impuestos por concepto


            //Impuestos
            $cfdi33_retenciones = [];
            $cfdi33_traslados = [];

            $cfdi33_impuestos = [];
            //$cfdi33_impuestos['TotalImpuestosRetenidos']
            //$cfdi33_impuestos['TotalImpuestosTrasladados']

            //Genera XML
            $certificado = new \CfdiUtils\Certificado\Certificado(\Storage::path($employee_payroll->company->pathFileCer()));
            $creator = new \CfdiUtils\CfdiCreator33($cfdi33, $certificado);
            $creator->setXmlResolver(PacHelper::resourcePathCfdiUtils()); //Almacenamiento local
            $comprobante = $creator->comprobante();
            //Complemento de nomina
            $comprobante->addAttributes([
                'xsi:schemaLocation' => 'http://www.sat.gob.mx/cfd/3 http://www.sat.gob.mx/sitio_internet/cfd/3/cfdv33.xsd http://www.sat.gob.mx/nomina12 http://www.sat.gob.mx/sitio_internet/cfd/nomina/nomina12.xsd',
                'xmlns:nomina12' => 'http://www.sat.gob.mx/nomina12'
            ]);

            if (!empty($cfdi33_relacionados)) {
                $comprobante->addCfdiRelacionados($cfdi33_relacionados);
            }
            if (!empty($cfdi33_relacionado)) {
                foreach ($cfdi33_relacionado as $key => $result) {
                    $comprobante->addCfdiRelacionado($result);
                }
            }
            $comprobante->addEmisor($cfdi33_emisor);
            $comprobante->addReceptor($cfdi33_receptor);
            //Conceptos
            foreach ($cfdi33_conceptos as $key => $result) {
                $concepto = $comprobante->addConcepto($result);
                if (!empty($cfdi33_conceptos_traslados[$key])) {
                    foreach ($cfdi33_conceptos_traslados[$key] as $result2) {
                        $concepto->multiTraslado($result2);
                    }
                }
                if (!empty($cfdi33_conceptos_retenciones[$key])) {
                    foreach ($cfdi33_conceptos_retenciones[$key] as $result2) {
                        $concepto->multiRetencion($result2);
                    }
                }
            }
            //Impuestos
            if(!empty($cfdi33_impuestos)) {
                $comprobante->addImpuestos($cfdi33_impuestos);
                if (!empty($cfdi33_retenciones)) {
                    foreach ($cfdi33_retenciones as $result2) {
                        $comprobante->multiRetencion($result2);
                    }
                }
                if (!empty($cfdi33_traslados)) {
                    foreach ($cfdi33_traslados as $result2) {
                        $comprobante->multiTraslado($result2);
                    }
                }
            }

            //Complemento de nomina
            //Calculo de antiguedad
            $antiquity = '';
            if(!empty($employee_payroll->employee->date_start_work)){
                if(!empty($employee_payroll->date_end_payment)){
                    $date1 = \Date::parse($employee_payroll->employee->date_start_work);
                    $date2 = \Date::parse($employee_payroll->date_end_payment);
                    $diff_weeks = $date1->diffInWeeks($date2);
                    $diff_weeks = floor($diff_weeks);
                    if($diff_weeks>=1){
                        $antiquity = 'P' . $diff_weeks .'W';
                    }else{
                        $diff_weeks = $date1->diffInDays($date2);
                        $diff_weeks = floor($diff_weeks);
                        if($diff_weeks == 0){
                            $diff_weeks = 1;
                        }
                        if($diff_weeks>=1){
                            $antiquity = 'P' . $diff_weeks .'D';
                        }
                    }
                }
            }

            $nomina12 = [];
            $nomina12['xsi:schemaLocation'] = 'http://www.sat.gob.mx/nomina12 http://www.sat.gob.mx/sitio_internet/cfd/nomina/nomina12.xsd';
            $nomina12['xmlns:xsi'] = 'http://www.w3.org/2001/XMLSchema-instance';
            $nomina12['xmlns:nomina12'] = 'http://www.sat.gob.mx/nomina12';
            $nomina12['Version'] = '1.2';
            $nomina12['TipoNomina'] = $employee_payroll->payrollType->code;
            $nomina12['FechaPago'] = \Date::parse($employee_payroll->date_payment)->format('Y-m-d');
            $nomina12['FechaInicialPago'] = \Date::parse($employee_payroll->date_start_payment)->format('Y-m-d');
            $nomina12['FechaFinalPago'] = \Date::parse($employee_payroll->date_end_payment)->format('Y-m-d');
            $nomina12['NumDiasPagados'] = Helper::numberFormat($employee_payroll->payment_days, 3, false);
            if($employee_payroll->amount_perceptions>0 || $employee_payroll->employeeActivePerceptionsPayrollLines->isNotEmpty()) {
                $nomina12['TotalPercepciones'] = Helper::numberFormat($employee_payroll->amount_perceptions, $employee_payroll->currency->decimal_place, false);
            }
            if($employee_payroll->amount_deductions>0 || $employee_payroll->employeeActiveDeductionsPayrollLines->isNotEmpty()) {
                $nomina12['TotalDeducciones'] = Helper::numberFormat($employee_payroll->amount_deductions, $employee_payroll->currency->decimal_place, false);
            }
            if($employee_payroll->amount_other_payment_types>0 || $employee_payroll->employeeActiveOtherPaymentTypesPayrollLines->isNotEmpty()) {
                $nomina12['TotalOtrosPagos'] = Helper::numberFormat($employee_payroll->amount_other_payment_types, $employee_payroll->currency->decimal_place, false);
            }
            $nodo_payroll = new \CfdiUtils\Nodes\Node(
                'nomina12:Nomina', // nombre del elemento raíz
                $nomina12
            );
            //Nodo emisor
            $nomina12_emisor = [];
            if(!empty($employee_payroll->company->curp)){
                $nomina12_emisor['Curp'] = $employee_payroll->company->curp;
            }
            if(!empty($employee_payroll->company->employer_register) && !empty($employee_payroll->employee->nss) && !empty($employee_payroll->employee->date_start_work) && !empty($antiquity) && $employee_payroll->employee->sdi > 0) {
                $nomina12_emisor['RegistroPatronal'] = $employee_payroll->company->employer_register;
            }
            if(!empty($employee_payroll->taxid_origin)){
                $nomina12_emisor['RfcPatronOrigen'] = $employee_payroll->taxid_origin;
            }
            $nodo_nomina12_emisor = new \CfdiUtils\Nodes\Node(
                'nomina12:Emisor', // nombre del elemento raíz
                $nomina12_emisor
            );

            //EntidadSNCF
            if(!empty($employee_payroll->source_resource_id)){
                $nomina12_emisor_sncf = [];
                $nomina12_emisor_sncf['OrigenRecurso'] = $employee_payroll->sourceResource->code;
                if($employee_payroll->sourceResource->code == 'IM'){
                    $nomina12_emisor_sncf['MontoRecursoPropio'] = Helper::numberFormat($employee_payroll->sncf_amount_source_resource, $employee_payroll->currency->decimal_place, false);
                }
                $nodo_nomina12_emisor_sncf = new \CfdiUtils\Nodes\Node(
                    'nomina12:EntidadSNCF', // nombre del elemento raíz
                    $nomina12_emisor_sncf
                );
                $nodo_nomina12_emisor->addChild($nodo_nomina12_emisor_sncf);
            }


            $nodo_payroll->addChild($nodo_nomina12_emisor);

            //Nodo receptor
            $nomina12_receptor = [];
            $nomina12_receptor['Curp'] = $employee_payroll->employee->curp;
            if(!empty($employee_payroll->employee->nss)) {
                $nomina12_receptor['NumSeguridadSocial'] = $employee_payroll->employee->nss;
            }
            if(!empty($employee_payroll->employee->date_start_work)) {
                $nomina12_receptor['FechaInicioRelLaboral'] = \Date::parse($employee_payroll->employee->date_start_work)->format('Y-m-d');
            }
            if(!empty($antiquity)) {
                $nomina12_receptor['Antigüedad'] = $antiquity;
            }
            $nomina12_receptor['TipoContrato'] = $employee_payroll->employee->contractType->code;
            $nomina12_receptor['Sindicalizado'] = !empty($employee_payroll->employee->unionized) ? 'Sí' : 'No';
            if(!empty($employee_payroll->employee->workdayType->code)) {
                $nomina12_receptor['TipoJornada'] = $employee_payroll->employee->workdayType->code;
            }
            $nomina12_receptor['TipoRegimen'] = $employee_payroll->employee->recruitmentRegime->code;
            $nomina12_receptor['NumEmpleado'] = $employee_payroll->employee->code;
            if(!empty($employee_payroll->employee->department)) {
                $nomina12_receptor['Departamento'] = $employee_payroll->employee->department;
            }
            if(!empty($employee_payroll->employee->job)) {
                $nomina12_receptor['Puesto'] = $employee_payroll->employee->job;
            }
            if(!empty($employee_payroll->employee->jobRiskClasse->code)) {
                $nomina12_receptor['RiesgoPuesto'] = $employee_payroll->employee->jobRiskClasse->code;
            }
            $nomina12_receptor['PeriodicidadPago'] = $employee_payroll->employee->frequencyPayment->code;
            if(!empty($employee_payroll->employee->bank->code)) {
                $nomina12_receptor['Banco'] = $employee_payroll->employee->bank->code;
            }
            if(!empty($employee_payroll->employee->bank_account)) {
                $nomina12_receptor['CuentaBancaria'] = $employee_payroll->employee->bank_account;
            }
            if($employee_payroll->employee->base_salary > 0) {
                $nomina12_receptor['SalarioBaseCotApor'] = Helper::numberFormat($employee_payroll->base_salary, $employee_payroll->currency->decimal_place, false);
            }
            if($employee_payroll->employee->sdi > 0) {
                $nomina12_receptor['SalarioDiarioIntegrado'] = Helper::numberFormat($employee_payroll->sdi, $employee_payroll->currency->decimal_place, false);
            }
            $nomina12_receptor['ClaveEntFed'] = $employee_payroll->employee->state->code;
            $nodo_nomina12_receptor = new \CfdiUtils\Nodes\Node(
                'nomina12:Receptor', // nombre del elemento raíz
                $nomina12_receptor
            );

            //SubContratacion
            if($employee_payroll->employeeActivePayrollOutsourcings->isNotEmpty()) {
                foreach ($employee_payroll->employeeActivePayrollOutsourcings as $key => $result) {
                    $nomina12_subcontratacion = [];
                    $nomina12_subcontratacion['RfcLabora'] = $result->taxid;
                    $nomina12_subcontratacion['PorcentajeTiempo'] = round($result->percent_time,3);
                    $nodo_nomina12_subcontratacion = new \CfdiUtils\Nodes\Node(
                        'nomina12:SubContratacion', // nombre del elemento raíz
                        $nomina12_subcontratacion
                    );
                    $nodo_nomina12_receptor->addChild($nodo_nomina12_subcontratacion);
                }
            }

            $nodo_payroll->addChild($nodo_nomina12_receptor);

            //Nodo percepciones
            if($employee_payroll->employeeActivePerceptionsPayrollLines->isNotEmpty()) {
                $nomina12_percepciones = [];
                if($employee_payroll->employeeActivePerceptions1PayrollLines->isNotEmpty()) {
                    $nomina12_percepciones['TotalSueldos'] = Helper::numberFormat($employee_payroll->employeeActivePerceptions1PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->employeeActivePerceptions2PayrollLines->isNotEmpty()) {
                    $nomina12_percepciones['TotalSeparacionIndemnizacion'] = Helper::numberFormat($employee_payroll->employeeActivePerceptions2PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->employeeActivePerceptions3PayrollLines->isNotEmpty()) {
                    $nomina12_percepciones['TotalJubilacionPensionRetiro'] = Helper::numberFormat($employee_payroll->employeeActivePerceptions3PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                //if($employee_payroll->employeeActivePerceptionsPayrollLines->sum('amount_taxed') > 0) {
                    $nomina12_percepciones['TotalGravado'] = Helper::numberFormat($employee_payroll->employeeActivePerceptionsPayrollLines->sum('amount_taxed'),
                        $employee_payroll->currency->decimal_place, false);
                //}
                //if($employee_payroll->employeeActivePerceptionsPayrollLines->sum('amount_exempt') > 0) {
                    $nomina12_percepciones['TotalExento'] = Helper::numberFormat($employee_payroll->employeeActivePerceptionsPayrollLines->sum('amount_exempt'),
                        $employee_payroll->currency->decimal_place, false);
                //}
                $nodo_nomina12_percepciones = new \CfdiUtils\Nodes\Node(
                    'nomina12:Percepciones', // nombre del elemento raíz
                    $nomina12_percepciones
                );
                //Percepciones
                foreach ($employee_payroll->employeeActivePerceptionsPayrollLines as $key => $result) {
                    $nomina12_percepcion = [];
                    $nomina12_percepcion['TipoPercepcion'] = $result->perception->code;
                    $nomina12_percepcion['Clave'] = !empty($result->code) ? $result->code : $result->perception->code;
                    $nomina12_percepcion['Concepto'] =str_limit($result->name,99,'');
                    //if($result->amount_taxed>0) {
                        $nomina12_percepcion['ImporteGravado'] = Helper::numberFormat($result->amount_taxed,
                            $employee_payroll->currency->decimal_place, false);
                    //}
                    //if($result->amount_exempt>0) {
                        $nomina12_percepcion['ImporteExento'] = Helper::numberFormat($result->amount_exempt,
                            $employee_payroll->currency->decimal_place, false);
                    //}

                    $nodo_nomina12_percepcion = new \CfdiUtils\Nodes\Node(
                        'nomina12:Percepcion', // nombre del elemento raíz
                        $nomina12_percepcion
                    );

                    //AccionesOTitulos
                    if($result->perception->code == '045'){
                        $nomina12_acciones_o_titulos['ValorMercado'] = Helper::numberFormat($result->ac_market_value,
                            $employee_payroll->currency->decimal_place, false);
                        $nomina12_acciones_o_titulos['PrecioAlOtorgarse'] = Helper::numberFormat($result->ac_price_when_granted,
                            $employee_payroll->currency->decimal_place, false);
                        $nodo_nomina12_acciones_o_titulos = new \CfdiUtils\Nodes\Node(
                            'nomina12:AccionesOTitulos', // nombre del elemento raíz
                            $nomina12_acciones_o_titulos
                        );
                        $nodo_nomina12_percepcion->addChild($nodo_nomina12_acciones_o_titulos);
                    }
                    //HorasExtra
                    if($result->perception->code == '019'){
                        if($result->hr_days_1 > 0 && $result->hr_hours_1>0 && $result->hr_amount_paid_1 > 0) {
                            $nomina12_horas_extra['Dias'] = Helper::numberFormat($result->hr_days_1, 0, false);
                            $nomina12_horas_extra['TipoHoras'] = $result->overtimesType1->code;
                            $nomina12_horas_extra['HorasExtra'] = Helper::numberFormat($result->hr_hours_1, 0, false);
                            $nomina12_horas_extra['ImportePagado'] = Helper::numberFormat($result->hr_amount_paid_1, $employee_payroll->currency->decimal_place, false);
                            $nodo_nomina12_horas_extra = new \CfdiUtils\Nodes\Node(
                                'nomina12:HorasExtra', // nombre del elemento raíz
                                $nomina12_horas_extra
                            );
                            $nodo_nomina12_percepcion->addChild($nodo_nomina12_horas_extra);
                        }
                        if($result->hr_days_2 > 0 && $result->hr_hours_2>0 && $result->hr_amount_paid_2 > 0) {
                            $nomina12_horas_extra['Dias'] = Helper::numberFormat($result->hr_days_2, 0, false);
                            $nomina12_horas_extra['TipoHoras'] = $result->overtimesType2->code;
                            $nomina12_horas_extra['HorasExtra'] = Helper::numberFormat($result->hr_hours_2, 0, false);
                            $nomina12_horas_extra['ImportePagado'] = Helper::numberFormat($result->hr_amount_paid_2, $employee_payroll->currency->decimal_place, false);
                            $nodo_nomina12_horas_extra = new \CfdiUtils\Nodes\Node(
                                'nomina12:HorasExtra', // nombre del elemento raíz
                                $nomina12_horas_extra
                            );
                            $nodo_nomina12_percepcion->addChild($nodo_nomina12_horas_extra);
                        }
                        if($result->hr_days_3 > 0 && $result->hr_hours_3>0 && $result->hr_amount_paid_3 > 0) {
                            $nomina12_horas_extra['Dias'] = Helper::numberFormat($result->hr_days_3, 0, false);
                            $nomina12_horas_extra['TipoHoras'] = $result->overtimesType3->code;
                            $nomina12_horas_extra['HorasExtra'] = Helper::numberFormat($result->hr_hours_3, 0, false);
                            $nomina12_horas_extra['ImportePagado'] = Helper::numberFormat($result->hr_amount_paid_3, $employee_payroll->currency->decimal_place, false);
                            $nodo_nomina12_horas_extra = new \CfdiUtils\Nodes\Node(
                                'nomina12:HorasExtra', // nombre del elemento raíz
                                $nomina12_horas_extra
                            );
                            $nodo_nomina12_percepcion->addChild($nodo_nomina12_horas_extra);
                        }
                    }

                    $nodo_nomina12_percepciones->addChild($nodo_nomina12_percepcion);
                }
                //JubilacionPensionRetiro
                $nomina12_jubilacion = [];
                if($employee_payroll->rt_amount_an_exhibition > 0) {
                    $nomina12_jubilacion['TotalUnaExhibicion'] = Helper::numberFormat($employee_payroll->rt_amount_an_exhibition,
                        $employee_payroll->currency->decimal_place, false);
                }elseif($employee_payroll->rt_partiality_amount > 0 || $employee_payroll->rt_daily_amount>0){
                    $nomina12_jubilacion['TotalParcialidad'] = Helper::numberFormat($employee_payroll->rt_partiality_amount,
                        $employee_payroll->currency->decimal_place, false);
                    $nomina12_jubilacion['MontoDiario'] = Helper::numberFormat($employee_payroll->rt_daily_amount,
                        $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->rt_cumulative_income > 0) {
                    $nomina12_jubilacion['IngresoAcumulable'] = Helper::numberFormat($employee_payroll->rt_cumulative_income,
                        $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->rt_non_cumulative_income > 0) {
                    $nomina12_jubilacion['IngresoNoAcumulable'] = Helper::numberFormat($employee_payroll->rt_non_cumulative_income,
                        $employee_payroll->currency->decimal_place, false);
                }
                if(!empty($nomina12_jubilacion)) {
                    $nodo_nomina12_jubilacion = new \CfdiUtils\Nodes\Node(
                        'nomina12:JubilacionPensionRetiro', // nombre del elemento raíz
                        $nomina12_jubilacion
                    );
                    $nodo_nomina12_percepciones->addChild($nodo_nomina12_jubilacion);
                }

                //SeparacionIndemnizacion
                $nomina12_seperacion = [];
                if($employee_payroll->sp_total_amount > 0) {
                    $nomina12_seperacion['TotalPagado'] = Helper::numberFormat($employee_payroll->sp_total_amount,
                        $employee_payroll->currency->decimal_place, false);
                    $nomina12_seperacion['NumAñosServicio'] = Helper::numberFormat($employee_payroll->sp_years_of_service,
                        0, false);
                    $nomina12_seperacion['UltimoSueldoMensOrd'] = Helper::numberFormat($employee_payroll->sp_last_salary,
                        $employee_payroll->currency->decimal_place, false);
                    $nomina12_seperacion['IngresoAcumulable'] = Helper::numberFormat($employee_payroll->sp_cumulative_income,
                        $employee_payroll->currency->decimal_place, false);
                    $nomina12_seperacion['IngresoNoAcumulable'] = Helper::numberFormat($employee_payroll->sp_non_cumulative_income,
                        $employee_payroll->currency->decimal_place, false);

                    if(!empty($nomina12_seperacion)) {
                        $nodo_nomina12_separacion = new \CfdiUtils\Nodes\Node(
                            'nomina12:SeparacionIndemnizacion', // nombre del elemento raíz
                            $nomina12_seperacion
                        );
                        $nodo_nomina12_percepciones->addChild($nodo_nomina12_separacion);
                    }
                }

                $nodo_payroll->addChild($nodo_nomina12_percepciones);
            }

            //Nodo deducciones
            if($employee_payroll->employeeActiveDeductionsPayrollLines->isNotEmpty()) {
                $nomina12_deducciones = [];
                if($employee_payroll->employeeActiveDeductions1PayrollLines->isNotEmpty()) {
                    $nomina12_deducciones['TotalOtrasDeducciones'] = Helper::numberFormat($employee_payroll->employeeActiveDeductions1PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->employeeActiveDeductions2PayrollLines->isNotEmpty()) {
                    $nomina12_deducciones['TotalImpuestosRetenidos'] = Helper::numberFormat($employee_payroll->employeeActiveDeductions2PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                $nodo_nomina12_deducciones = new \CfdiUtils\Nodes\Node(
                    'nomina12:Deducciones', // nombre del elemento raíz
                    $nomina12_deducciones
                );
                //Deducciones
                foreach ($employee_payroll->employeeActiveDeductionsPayrollLines as $key => $result) {
                    $nomina12_deduccion = [];
                    $nomina12_deduccion['TipoDeduccion'] = $result->deduction->code;
                    $nomina12_deduccion['Clave'] = !empty($result->code) ? $result->code : $result->deduction->code;
                    $nomina12_deduccion['Concepto'] =str_limit($result->name,99,'');
                    $nomina12_deduccion['Importe'] = Helper::numberFormat($result->amount_total,
                            $employee_payroll->currency->decimal_place, false);
                    $nodo_nomina12_deduccion = new \CfdiUtils\Nodes\Node(
                        'nomina12:Deduccion', // nombre del elemento raíz
                        $nomina12_deduccion
                    );
                    $nodo_nomina12_deducciones->addChild($nodo_nomina12_deduccion);
                }

                $nodo_payroll->addChild($nodo_nomina12_deducciones);
            }

            //Nodo Otros pagos
            if($employee_payroll->employeeActiveOtherPaymentTypesPayrollLines->isNotEmpty()) {
                $nomina12_otros_pagos = [];
                $nodo_nomina12_otros_pagos = new \CfdiUtils\Nodes\Node(
                    'nomina12:OtrosPagos', // nombre del elemento raíz
                    $nomina12_otros_pagos
                );
                //Otros pagos
                foreach ($employee_payroll->employeeActiveOtherPaymentTypesPayrollLines as $key => $result) {
                    $nomina12_otro_pago = [];
                    $nomina12_otro_pago['TipoOtroPago'] = $result->otherPaymentType->code;
                    $nomina12_otro_pago['Clave'] = !empty($result->code) ? $result->code : $result->otherPaymentType->code;
                    $nomina12_otro_pago['Concepto'] =str_limit($result->name,99,'');
                    $nomina12_otro_pago['Importe'] = Helper::numberFormat($result->amount_total,
                        $employee_payroll->currency->decimal_place, false);
                    $nodo_nomina12_otro_pago = new \CfdiUtils\Nodes\Node(
                        'nomina12:OtroPago', // nombre del elemento raíz
                        $nomina12_otro_pago
                    );
                    //SubsidioAlEmpleo
                    if($result->otherPaymentType->code == '002') {
                        $nomina12_otros_pago_subsidio_empleo = [];
                        $nomina12_otros_pago_subsidio_empleo['SubsidioCausado'] = Helper::numberFormat($result->amount_caused, $employee_payroll->currency->decimal_place, false);
                        $nodo_nomina12_otro_subsidio_empleo = new \CfdiUtils\Nodes\Node(
                            'nomina12:SubsidioAlEmpleo', // nombre del elemento raíz
                            $nomina12_otros_pago_subsidio_empleo
                        );
                        $nodo_nomina12_otro_pago->addChild($nodo_nomina12_otro_subsidio_empleo);
                    }
                    //CompensacionSaldosAFavor
                    if($result->otherPaymentType->code == '004'){
                        $nomina12_otros_pago_compensacion['SaldoAFavor'] = Helper::numberFormat($result->cm_positive_balance, $employee_payroll->currency->decimal_place, false);
                        $nomina12_otros_pago_compensacion['Año'] = Helper::numberFormat($result->cm_year, 0, false);
                        $nomina12_otros_pago_compensacion['RemanenteSalFav'] = Helper::numberFormat($result->cm_remainder, $employee_payroll->currency->decimal_place, false);
                        $nodo_nomina12_otros_pago_compensacion = new \CfdiUtils\Nodes\Node(
                            'nomina12:CompensacionSaldosAFavor', // nombre del elemento raíz
                            $nomina12_otros_pago_compensacion
                        );
                        $nodo_nomina12_otro_pago->addChild($nodo_nomina12_otros_pago_compensacion);
                    }

                    $nodo_nomina12_otros_pagos->addChild($nodo_nomina12_otro_pago);
                }

                $nodo_payroll->addChild($nodo_nomina12_otros_pagos);
            }

            //Incapacidades
            if($employee_payroll->employeeActiveDisabilitiesPayrollLines->isNotEmpty()) {
                $nomina12_incapacidades = [];
                $nodo_nomina12_incapacidades = new \CfdiUtils\Nodes\Node(
                    'nomina12:Incapacidades', // nombre del elemento raíz
                    $nomina12_incapacidades
                );
                //Deducciones
                foreach ($employee_payroll->employeeActiveDisabilitiesPayrollLines as $key => $result) {
                    $nomina12_incapacidad = [];
                    $nomina12_incapacidad['DiasIncapacidad'] = Helper::numberFormat($result->disability_days, 0, false);
                    $nomina12_incapacidad['TipoIncapacidad'] = $result->disabilityType->code;
                    $nomina12_incapacidad['ImporteMonetario'] = Helper::numberFormat($result->disability_amount,
                        $employee_payroll->currency->decimal_place, false);
                    $nodo_nomina12_incapacidad = new \CfdiUtils\Nodes\Node(
                        'nomina12:Incapacidad', // nombre del elemento raíz
                        $nomina12_incapacidad
                    );
                    $nodo_nomina12_incapacidades->addChild($nodo_nomina12_incapacidad);
                }

                $nodo_payroll->addChild($nodo_nomina12_incapacidades);
            }

            //Agregamos al complemento
            $comprobante->addComplemento($nodo_payroll);


            //Método de ayuda para establecer las sumas del comprobante e impuestos con base en la suma de los conceptos y la agrupación de sus impuestos
            //$creator->addSumasConceptos(null, 2);
            //Método de ayuda para generar el sello (obtener la cadena de origen y firmar con la llave privada)
            $creator->addSello('file://' . \Storage::path($employee_payroll->company->pathFileKeyPassPem()), Crypt::decryptString($employee_payroll->company->password_key));
            //Valida la estructura
            //$creator->validate();

            //Guarda XML
            //dd($creator->asXml());
            $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
            $file_xml = Helper::makeDirectoryCfdi($path_xml) . '/' . (!$draft ? Str::random(40) : str_replace('/','',$employee_payroll->name)) . '.xml';
            $creator->saveXml(\Storage::path($path_xml . $file_xml));


            //Arreglo temporal para actualizar Customer Invoice CFDI
            $tmp = [
                'pac_id' => $pac->id,
                'cfdi_version' => setting('cfdi_version'),
                'uuid' => '',
                'date' => null,
                'path_xml' => $path_xml,
                'file_xml' => $file_xml,
                'file_xml_pac' => '',
                'pac' => $pac,
            ];

            //Timbrado de XML
            if(!$draft) {
                $class_pac = $pac->code;
                $tmp = PacHelper::$class_pac($tmp, $creator);
            }

            return $tmp;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Crear XML y enviar a timbrar CFDI 4.0
     *
     * @param EmployeePayroll $employee_payroll
     * @return array|\CfdiUtils\Elements\Cfdi33\Concepto|float|int
     * @throws \Exception
     */
    private function cfdi40(EmployeePayroll $employee_payroll, $draft = false)
    {
        try {
            //Logica
            $pac = Pac::findOrFail(setting('default_pac_id')); //PAC

            //Arreglo CFDI 3.3
            $cfdi33 = [];
            if (!empty($employee_payroll->serie)) {
                $cfdi33['Serie'] = $employee_payroll->serie;
            }
            $cfdi33['Folio'] = $employee_payroll->folio;
            $cfdi33['Fecha'] = \Date::parse($employee_payroll->date)->format('Y-m-d\TH:i:s');
            //$cfdi33['Sello']
            //$cfdi33['FormaPago'] = $employee_payroll->paymentWay->code;
            $cfdi33['NoCertificado'] = $employee_payroll->company->certificate_number;
            //$cfdi33['Certificado']
            //$cfdi33['CondicionesDePago']
            $cfdi33['SubTotal'] = Helper::numberFormat($employee_payroll->amount_untaxed /*+ $employee_payroll->amount_discount*/, $employee_payroll->currency->decimal_place, false);
            if(($employee_payroll->amount_discount + $employee_payroll->amount_isr)>0) {
                $cfdi33['Descuento'] = Helper::numberFormat($employee_payroll->amount_discount + $employee_payroll->amount_isr, $employee_payroll->currency->decimal_place, false);
            }
            $cfdi33['Moneda'] = $employee_payroll->currency->code;
            //$cfdi33['TipoCambio']
            $cfdi33['Total'] = Helper::numberFormat($employee_payroll->amount_total, $employee_payroll->currency->decimal_place, false);
            $cfdi33['TipoDeComprobante'] = $employee_payroll->documentType->cfdiType->code;
            $cfdi33['Exportacion'] = '01';
            $cfdi33['MetodoPago'] = $employee_payroll->paymentMethod->code;
            $cfdi33['LugarExpedicion'] = $employee_payroll->branchOffice->postcode;
            if (!empty($employee_payroll->confirmacion)) {
                $cfdi33['Confirmacion'] = $employee_payroll->confirmacion;
            }
            //---Informacion global
            //---Cfdi Relacionados
            $cfdi33_relacionados = [];
            $cfdi33_relacionado = [];
            if (!empty($employee_payroll->cfdi_relation_id)) {
                $cfdi33_relacionados['TipoRelacion'] = $employee_payroll->cfdiRelation->code;
                if ($employee_payroll->employeePayrollRelations->isNotEmpty()) {
                    foreach ($employee_payroll->employeePayrollRelations as $key => $result) {
                        $cfdi33_relacionado[$key] = [];
                        $cfdi33_relacionado[$key]['UUID'] = $result->uuid_related;
                    }
                }
            }
            //---Emisor
            $cfdi33_emisor = [];
            $cfdi33_emisor['Rfc'] = $employee_payroll->company->taxid;
            $cfdi33_emisor['Nombre'] = trim($employee_payroll->company->name);
            $cfdi33_emisor['RegimenFiscal'] = !empty($employee_payroll->tax_regimen_id) ? $employee_payroll->taxRegimen->code : $employee_payroll->company->taxRegimen->code;
            $cfdi33_emisor['FacAtrAdquirente'] = null;
            //---Receptor
            $cfdi33_receptor = [];
            $cfdi33_receptor['Rfc'] = $employee_payroll->employee->taxid;
            $cfdi33_receptor['Nombre'] = trim($employee_payroll->employee->name);
            $cfdi33_receptor['DomicilioFiscalReceptor'] = $employee_payroll->employee->postcode;
            //$cfdi33_receptor['ResidenciaFiscal']
            //$cfdi33_receptor['NumRegIdTrib']
            $cfdi33_receptor['RegimenFiscalReceptor'] = !empty($employee_payroll->tax_regimen_employee_id) ? $employee_payroll->taxRegimenEmployee->code : $employee_payroll->employee->taxRegimen->code;
            $cfdi33_receptor['UsoCFDI'] = $employee_payroll->cfdiUse->code;
            //---Conceptos
            $cfdi33_conceptos = [];
            $cfdi33_conceptos_traslados = [];
            $cfdi33_conceptos_retenciones = [];
            $key = 0;
            $cfdi33_conceptos [$key]['ClaveProdServ'] = '84111505';
            //$cfdi33_conceptos [$key]['NoIdentificacion']
            $cfdi33_conceptos [$key]['Cantidad'] = '1';
            $cfdi33_conceptos [$key]['ClaveUnidad'] = 'ACT';
            //$cfdi33_conceptos [$key]['Unidad'];
            $cfdi33_conceptos [$key]['Descripcion'] = 'Pago de nómina';
            $cfdi33_conceptos [$key]['ValorUnitario'] = Helper::numberFormat($employee_payroll->amount_untaxed, 6, false);
            $cfdi33_conceptos [$key]['Importe'] = Helper::numberFormat($employee_payroll->amount_untaxed, 2, false);
            if($employee_payroll->amount_discount + $employee_payroll->amount_isr > 0){
                $cfdi33_conceptos [$key]['Descuento'] = Helper::numberFormat($employee_payroll->amount_discount + $employee_payroll->amount_isr, 2, false);
            }
            //['InformacionAduanera']
            //['CuentaPredial']
            //['ComplementoConcepto']
            //['Parte']
            //Complemento
            //Impuestos por concepto
            $cfdi33_conceptos [$key]['ObjetoImp'] = '01';


            //Impuestos
            $cfdi33_retenciones = [];
            $cfdi33_traslados = [];

            $cfdi33_impuestos = [];
            //$cfdi33_impuestos['TotalImpuestosRetenidos']
            //$cfdi33_impuestos['TotalImpuestosTrasladados']

            //Genera XML
            $certificado = new \CfdiUtils\Certificado\Certificado(\Storage::path($employee_payroll->company->pathFileCer()));
            $creator = new \CfdiUtils\CfdiCreator40($cfdi33, $certificado);
            $creator->setXmlResolver(PacHelper::resourcePathCfdiUtils()); //Almacenamiento local
            $comprobante = $creator->comprobante();
            //Complemento de nomina
            $comprobante->addAttributes([
                'xsi:schemaLocation' => 'http://www.sat.gob.mx/cfd/4 http://www.sat.gob.mx/sitio_internet/cfd/4/cfdv40.xsd http://www.sat.gob.mx/nomina12 http://www.sat.gob.mx/sitio_internet/cfd/nomina/nomina12.xsd',
                'xmlns:nomina12' => 'http://www.sat.gob.mx/nomina12'
            ]);

            if (!empty($cfdi33_relacionados)) {
                $relacionados = $comprobante->addCfdiRelacionados($cfdi33_relacionados);
                if (!empty($cfdi33_relacionado)) {
                    foreach ($cfdi33_relacionado as $key => $result) {
                        $relacionados->addCfdiRelacionado($result);
                    }
                }
            }
            $comprobante->addEmisor($cfdi33_emisor);
            $comprobante->addReceptor($cfdi33_receptor);
            //Conceptos
            foreach ($cfdi33_conceptos as $key => $result) {
                $concepto = $comprobante->addConcepto($result);
                if (!empty($cfdi33_conceptos_traslados[$key])) {
                    foreach ($cfdi33_conceptos_traslados[$key] as $result2) {
                        $concepto->multiTraslado($result2);
                    }
                }
                if (!empty($cfdi33_conceptos_retenciones[$key])) {
                    foreach ($cfdi33_conceptos_retenciones[$key] as $result2) {
                        $concepto->multiRetencion($result2);
                    }
                }
            }
            //Impuestos
            if(!empty($cfdi33_impuestos)) {
                $comprobante->addImpuestos($cfdi33_impuestos);
                if (!empty($cfdi33_retenciones)) {
                    foreach ($cfdi33_retenciones as $result2) {
                        $comprobante->multiRetencion($result2);
                    }
                }
                if (!empty($cfdi33_traslados)) {
                    foreach ($cfdi33_traslados as $result2) {
                        $comprobante->multiTraslado($result2);
                    }
                }
            }

            //Complemento de nomina
            //Calculo de antiguedad
            $antiquity = '';
            if(!empty($employee_payroll->employee->date_start_work)){
                if(!empty($employee_payroll->date_end_payment)){
                    $date1 = \Date::parse($employee_payroll->employee->date_start_work);
                    $date2 = \Date::parse($employee_payroll->date_end_payment);
                    $diff_weeks = $date1->diffInWeeks($date2);
                    $diff_weeks = floor($diff_weeks);
                    if($diff_weeks>=1){
                        $antiquity = 'P' . $diff_weeks .'W';
                    }else{
                        $diff_weeks = $date1->diffInDays($date2);
                        $diff_weeks = floor($diff_weeks);
                        if($diff_weeks == 0){
                            $diff_weeks = 1;
                        }
                        if($diff_weeks>=1){
                            $antiquity = 'P' . $diff_weeks .'D';
                        }
                    }
                }
            }

            $nomina12 = [];
            $nomina12['xsi:schemaLocation'] = 'http://www.sat.gob.mx/nomina12 http://www.sat.gob.mx/sitio_internet/cfd/nomina/nomina12.xsd';
            $nomina12['xmlns:xsi'] = 'http://www.w3.org/2001/XMLSchema-instance';
            $nomina12['xmlns:nomina12'] = 'http://www.sat.gob.mx/nomina12';
            $nomina12['Version'] = '1.2';
            $nomina12['TipoNomina'] = $employee_payroll->payrollType->code;
            $nomina12['FechaPago'] = \Date::parse($employee_payroll->date_payment)->format('Y-m-d');
            $nomina12['FechaInicialPago'] = \Date::parse($employee_payroll->date_start_payment)->format('Y-m-d');
            $nomina12['FechaFinalPago'] = \Date::parse($employee_payroll->date_end_payment)->format('Y-m-d');
            $nomina12['NumDiasPagados'] = Helper::numberFormat($employee_payroll->payment_days, 3, false);
            if($employee_payroll->amount_perceptions>0 || $employee_payroll->employeeActivePerceptionsPayrollLines->isNotEmpty()) {
                $nomina12['TotalPercepciones'] = Helper::numberFormat($employee_payroll->amount_perceptions, $employee_payroll->currency->decimal_place, false);
            }
            if($employee_payroll->amount_deductions>0 || $employee_payroll->employeeActiveDeductionsPayrollLines->isNotEmpty()) {
                $nomina12['TotalDeducciones'] = Helper::numberFormat($employee_payroll->amount_deductions, $employee_payroll->currency->decimal_place, false);
            }
            if($employee_payroll->amount_other_payment_types>0 || $employee_payroll->employeeActiveOtherPaymentTypesPayrollLines->isNotEmpty()) {
                $nomina12['TotalOtrosPagos'] = Helper::numberFormat($employee_payroll->amount_other_payment_types, $employee_payroll->currency->decimal_place, false);
            }
            $nodo_payroll = new \CfdiUtils\Nodes\Node(
                'nomina12:Nomina', // nombre del elemento raíz
                $nomina12
            );
            //Nodo emisor
            $nomina12_emisor = [];
            if(!empty($employee_payroll->company->curp)){
                $nomina12_emisor['Curp'] = $employee_payroll->company->curp;
            }
            if(!empty($employee_payroll->company->employer_register) && !empty($employee_payroll->employee->nss) && !empty($employee_payroll->employee->date_start_work) && !empty($antiquity) && $employee_payroll->employee->sdi > 0) {
                $nomina12_emisor['RegistroPatronal'] = $employee_payroll->company->employer_register;
            }
            if(!empty($employee_payroll->taxid_origin)){
                $nomina12_emisor['RfcPatronOrigen'] = $employee_payroll->taxid_origin;
            }
            $nodo_nomina12_emisor = new \CfdiUtils\Nodes\Node(
                'nomina12:Emisor', // nombre del elemento raíz
                $nomina12_emisor
            );

            //EntidadSNCF
            if(!empty($employee_payroll->source_resource_id)){
                $nomina12_emisor_sncf = [];
                $nomina12_emisor_sncf['OrigenRecurso'] = $employee_payroll->sourceResource->code;
                if($employee_payroll->sourceResource->code == 'IM'){
                    $nomina12_emisor_sncf['MontoRecursoPropio'] = Helper::numberFormat($employee_payroll->sncf_amount_source_resource, $employee_payroll->currency->decimal_place, false);
                }
                $nodo_nomina12_emisor_sncf = new \CfdiUtils\Nodes\Node(
                    'nomina12:EntidadSNCF', // nombre del elemento raíz
                    $nomina12_emisor_sncf
                );
                $nodo_nomina12_emisor->addChild($nodo_nomina12_emisor_sncf);
            }


            $nodo_payroll->addChild($nodo_nomina12_emisor);

            //Nodo receptor
            $nomina12_receptor = [];
            $nomina12_receptor['Curp'] = $employee_payroll->employee->curp;
            if(!empty($employee_payroll->employee->nss)) {
                $nomina12_receptor['NumSeguridadSocial'] = $employee_payroll->employee->nss;
            }
            if(!empty($employee_payroll->employee->date_start_work)) {
                $nomina12_receptor['FechaInicioRelLaboral'] = \Date::parse($employee_payroll->employee->date_start_work)->format('Y-m-d');
            }
            if(!empty($antiquity)) {
                $nomina12_receptor['Antigüedad'] = $antiquity;
            }
            $nomina12_receptor['TipoContrato'] = $employee_payroll->employee->contractType->code;
            $nomina12_receptor['Sindicalizado'] = !empty($employee_payroll->employee->unionized) ? 'Sí' : 'No';
            if(!empty($employee_payroll->employee->workdayType->code)) {
                $nomina12_receptor['TipoJornada'] = $employee_payroll->employee->workdayType->code;
            }
            $nomina12_receptor['TipoRegimen'] = $employee_payroll->employee->recruitmentRegime->code;
            $nomina12_receptor['NumEmpleado'] = $employee_payroll->employee->code;
            if(!empty($employee_payroll->employee->department)) {
                $nomina12_receptor['Departamento'] = $employee_payroll->employee->department;
            }
            if(!empty($employee_payroll->employee->job)) {
                $nomina12_receptor['Puesto'] = $employee_payroll->employee->job;
            }
            if(!empty($employee_payroll->employee->jobRiskClasse->code)) {
                $nomina12_receptor['RiesgoPuesto'] = $employee_payroll->employee->jobRiskClasse->code;
            }
            $nomina12_receptor['PeriodicidadPago'] = $employee_payroll->employee->frequencyPayment->code;
            if(!empty($employee_payroll->employee->bank->code)) {
                $nomina12_receptor['Banco'] = $employee_payroll->employee->bank->code;
            }
            if(!empty($employee_payroll->employee->bank_account)) {
                $nomina12_receptor['CuentaBancaria'] = $employee_payroll->employee->bank_account;
            }
            if($employee_payroll->employee->base_salary > 0) {
                $nomina12_receptor['SalarioBaseCotApor'] = Helper::numberFormat($employee_payroll->base_salary, $employee_payroll->currency->decimal_place, false);
            }
            if($employee_payroll->employee->sdi > 0) {
                $nomina12_receptor['SalarioDiarioIntegrado'] = Helper::numberFormat($employee_payroll->sdi, $employee_payroll->currency->decimal_place, false);
            }
            $nomina12_receptor['ClaveEntFed'] = $employee_payroll->employee->state->code;
            $nodo_nomina12_receptor = new \CfdiUtils\Nodes\Node(
                'nomina12:Receptor', // nombre del elemento raíz
                $nomina12_receptor
            );

            //SubContratacion
            if($employee_payroll->employeeActivePayrollOutsourcings->isNotEmpty()) {
                foreach ($employee_payroll->employeeActivePayrollOutsourcings as $key => $result) {
                    $nomina12_subcontratacion = [];
                    $nomina12_subcontratacion['RfcLabora'] = $result->taxid;
                    $nomina12_subcontratacion['PorcentajeTiempo'] = round($result->percent_time,3);
                    $nodo_nomina12_subcontratacion = new \CfdiUtils\Nodes\Node(
                        'nomina12:SubContratacion', // nombre del elemento raíz
                        $nomina12_subcontratacion
                    );
                    $nodo_nomina12_receptor->addChild($nodo_nomina12_subcontratacion);
                }
            }

            $nodo_payroll->addChild($nodo_nomina12_receptor);

            //Nodo percepciones
            if($employee_payroll->employeeActivePerceptionsPayrollLines->isNotEmpty()) {
                $nomina12_percepciones = [];
                if($employee_payroll->employeeActivePerceptions1PayrollLines->isNotEmpty()) {
                    $nomina12_percepciones['TotalSueldos'] = Helper::numberFormat($employee_payroll->employeeActivePerceptions1PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->employeeActivePerceptions2PayrollLines->isNotEmpty()) {
                    $nomina12_percepciones['TotalSeparacionIndemnizacion'] = Helper::numberFormat($employee_payroll->employeeActivePerceptions2PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->employeeActivePerceptions3PayrollLines->isNotEmpty()) {
                    $nomina12_percepciones['TotalJubilacionPensionRetiro'] = Helper::numberFormat($employee_payroll->employeeActivePerceptions3PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                //if($employee_payroll->employeeActivePerceptionsPayrollLines->sum('amount_taxed') > 0) {
                    $nomina12_percepciones['TotalGravado'] = Helper::numberFormat($employee_payroll->employeeActivePerceptionsPayrollLines->sum('amount_taxed'),
                        $employee_payroll->currency->decimal_place, false);
                //}
                //if($employee_payroll->employeeActivePerceptionsPayrollLines->sum('amount_exempt') > 0) {
                    $nomina12_percepciones['TotalExento'] = Helper::numberFormat($employee_payroll->employeeActivePerceptionsPayrollLines->sum('amount_exempt'),
                        $employee_payroll->currency->decimal_place, false);
                //}
                $nodo_nomina12_percepciones = new \CfdiUtils\Nodes\Node(
                    'nomina12:Percepciones', // nombre del elemento raíz
                    $nomina12_percepciones
                );
                //Percepciones
                foreach ($employee_payroll->employeeActivePerceptionsPayrollLines as $key => $result) {
                    $nomina12_percepcion = [];
                    $nomina12_percepcion['TipoPercepcion'] = $result->perception->code;
                    $nomina12_percepcion['Clave'] = !empty($result->code) ? $result->code : $result->perception->code;
                    $nomina12_percepcion['Concepto'] =str_limit($result->name,99,'');
                    //if($result->amount_taxed>0) {
                        $nomina12_percepcion['ImporteGravado'] = Helper::numberFormat($result->amount_taxed,
                            $employee_payroll->currency->decimal_place, false);
                    //}
                    //if($result->amount_exempt>0) {
                        $nomina12_percepcion['ImporteExento'] = Helper::numberFormat($result->amount_exempt,
                            $employee_payroll->currency->decimal_place, false);
                    //}

                    $nodo_nomina12_percepcion = new \CfdiUtils\Nodes\Node(
                        'nomina12:Percepcion', // nombre del elemento raíz
                        $nomina12_percepcion
                    );

                    //AccionesOTitulos
                    if($result->perception->code == '045'){
                        $nomina12_acciones_o_titulos['ValorMercado'] = Helper::numberFormat($result->ac_market_value,
                            $employee_payroll->currency->decimal_place, false);
                        $nomina12_acciones_o_titulos['PrecioAlOtorgarse'] = Helper::numberFormat($result->ac_price_when_granted,
                            $employee_payroll->currency->decimal_place, false);
                        $nodo_nomina12_acciones_o_titulos = new \CfdiUtils\Nodes\Node(
                            'nomina12:AccionesOTitulos', // nombre del elemento raíz
                            $nomina12_acciones_o_titulos
                        );
                        $nodo_nomina12_percepcion->addChild($nodo_nomina12_acciones_o_titulos);
                    }
                    //HorasExtra
                    if($result->perception->code == '019'){
                        if($result->hr_days_1 > 0 && $result->hr_hours_1>0 && $result->hr_amount_paid_1 > 0) {
                            $nomina12_horas_extra['Dias'] = Helper::numberFormat($result->hr_days_1, 0, false);
                            $nomina12_horas_extra['TipoHoras'] = $result->overtimesType1->code;
                            $nomina12_horas_extra['HorasExtra'] = Helper::numberFormat($result->hr_hours_1, 0, false);
                            $nomina12_horas_extra['ImportePagado'] = Helper::numberFormat($result->hr_amount_paid_1, $employee_payroll->currency->decimal_place, false);
                            $nodo_nomina12_horas_extra = new \CfdiUtils\Nodes\Node(
                                'nomina12:HorasExtra', // nombre del elemento raíz
                                $nomina12_horas_extra
                            );
                            $nodo_nomina12_percepcion->addChild($nodo_nomina12_horas_extra);
                        }
                        if($result->hr_days_2 > 0 && $result->hr_hours_2>0 && $result->hr_amount_paid_2 > 0) {
                            $nomina12_horas_extra['Dias'] = Helper::numberFormat($result->hr_days_2, 0, false);
                            $nomina12_horas_extra['TipoHoras'] = $result->overtimesType2->code;
                            $nomina12_horas_extra['HorasExtra'] = Helper::numberFormat($result->hr_hours_2, 0, false);
                            $nomina12_horas_extra['ImportePagado'] = Helper::numberFormat($result->hr_amount_paid_2, $employee_payroll->currency->decimal_place, false);
                            $nodo_nomina12_horas_extra = new \CfdiUtils\Nodes\Node(
                                'nomina12:HorasExtra', // nombre del elemento raíz
                                $nomina12_horas_extra
                            );
                            $nodo_nomina12_percepcion->addChild($nodo_nomina12_horas_extra);
                        }
                        if($result->hr_days_3 > 0 && $result->hr_hours_3>0 && $result->hr_amount_paid_3 > 0) {
                            $nomina12_horas_extra['Dias'] = Helper::numberFormat($result->hr_days_3, 0, false);
                            $nomina12_horas_extra['TipoHoras'] = $result->overtimesType3->code;
                            $nomina12_horas_extra['HorasExtra'] = Helper::numberFormat($result->hr_hours_3, 0, false);
                            $nomina12_horas_extra['ImportePagado'] = Helper::numberFormat($result->hr_amount_paid_3, $employee_payroll->currency->decimal_place, false);
                            $nodo_nomina12_horas_extra = new \CfdiUtils\Nodes\Node(
                                'nomina12:HorasExtra', // nombre del elemento raíz
                                $nomina12_horas_extra
                            );
                            $nodo_nomina12_percepcion->addChild($nodo_nomina12_horas_extra);
                        }
                    }

                    $nodo_nomina12_percepciones->addChild($nodo_nomina12_percepcion);
                }
                //JubilacionPensionRetiro
                $nomina12_jubilacion = [];
                if($employee_payroll->rt_amount_an_exhibition > 0) {
                    $nomina12_jubilacion['TotalUnaExhibicion'] = Helper::numberFormat($employee_payroll->rt_amount_an_exhibition,
                        $employee_payroll->currency->decimal_place, false);
                }elseif($employee_payroll->rt_partiality_amount > 0 || $employee_payroll->rt_daily_amount>0){
                    $nomina12_jubilacion['TotalParcialidad'] = Helper::numberFormat($employee_payroll->rt_partiality_amount,
                        $employee_payroll->currency->decimal_place, false);
                    $nomina12_jubilacion['MontoDiario'] = Helper::numberFormat($employee_payroll->rt_daily_amount,
                        $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->rt_cumulative_income > 0) {
                    $nomina12_jubilacion['IngresoAcumulable'] = Helper::numberFormat($employee_payroll->rt_cumulative_income,
                        $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->rt_non_cumulative_income > 0) {
                    $nomina12_jubilacion['IngresoNoAcumulable'] = Helper::numberFormat($employee_payroll->rt_non_cumulative_income,
                        $employee_payroll->currency->decimal_place, false);
                }
                if(!empty($nomina12_jubilacion)) {
                    $nodo_nomina12_jubilacion = new \CfdiUtils\Nodes\Node(
                        'nomina12:JubilacionPensionRetiro', // nombre del elemento raíz
                        $nomina12_jubilacion
                    );
                    $nodo_nomina12_percepciones->addChild($nodo_nomina12_jubilacion);
                }

                //SeparacionIndemnizacion
                $nomina12_seperacion = [];
                if($employee_payroll->sp_total_amount > 0) {
                    $nomina12_seperacion['TotalPagado'] = Helper::numberFormat($employee_payroll->sp_total_amount,
                        $employee_payroll->currency->decimal_place, false);
                    $nomina12_seperacion['NumAñosServicio'] = Helper::numberFormat($employee_payroll->sp_years_of_service,
                        0, false);
                    $nomina12_seperacion['UltimoSueldoMensOrd'] = Helper::numberFormat($employee_payroll->sp_last_salary,
                        $employee_payroll->currency->decimal_place, false);
                    $nomina12_seperacion['IngresoAcumulable'] = Helper::numberFormat($employee_payroll->sp_cumulative_income,
                        $employee_payroll->currency->decimal_place, false);
                    $nomina12_seperacion['IngresoNoAcumulable'] = Helper::numberFormat($employee_payroll->sp_non_cumulative_income,
                        $employee_payroll->currency->decimal_place, false);

                    if(!empty($nomina12_seperacion)) {
                        $nodo_nomina12_separacion = new \CfdiUtils\Nodes\Node(
                            'nomina12:SeparacionIndemnizacion', // nombre del elemento raíz
                            $nomina12_seperacion
                        );
                        $nodo_nomina12_percepciones->addChild($nodo_nomina12_separacion);
                    }
                }

                $nodo_payroll->addChild($nodo_nomina12_percepciones);
            }

            //Nodo deducciones
            if($employee_payroll->employeeActiveDeductionsPayrollLines->isNotEmpty()) {
                $nomina12_deducciones = [];
                if($employee_payroll->employeeActiveDeductions1PayrollLines->isNotEmpty()) {
                    $nomina12_deducciones['TotalOtrasDeducciones'] = Helper::numberFormat($employee_payroll->employeeActiveDeductions1PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->employeeActiveDeductions2PayrollLines->isNotEmpty()) {
                    $nomina12_deducciones['TotalImpuestosRetenidos'] = Helper::numberFormat($employee_payroll->employeeActiveDeductions2PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                $nodo_nomina12_deducciones = new \CfdiUtils\Nodes\Node(
                    'nomina12:Deducciones', // nombre del elemento raíz
                    $nomina12_deducciones
                );
                //Deducciones
                foreach ($employee_payroll->employeeActiveDeductionsPayrollLines as $key => $result) {
                    $nomina12_deduccion = [];
                    $nomina12_deduccion['TipoDeduccion'] = $result->deduction->code;
                    $nomina12_deduccion['Clave'] = !empty($result->code) ? $result->code : $result->deduction->code;
                    $nomina12_deduccion['Concepto'] =str_limit($result->name,99,'');
                    $nomina12_deduccion['Importe'] = Helper::numberFormat($result->amount_total,
                            $employee_payroll->currency->decimal_place, false);
                    $nodo_nomina12_deduccion = new \CfdiUtils\Nodes\Node(
                        'nomina12:Deduccion', // nombre del elemento raíz
                        $nomina12_deduccion
                    );
                    $nodo_nomina12_deducciones->addChild($nodo_nomina12_deduccion);
                }

                $nodo_payroll->addChild($nodo_nomina12_deducciones);
            }

            //Nodo Otros pagos
            if($employee_payroll->employeeActiveOtherPaymentTypesPayrollLines->isNotEmpty()) {
                $nomina12_otros_pagos = [];
                $nodo_nomina12_otros_pagos = new \CfdiUtils\Nodes\Node(
                    'nomina12:OtrosPagos', // nombre del elemento raíz
                    $nomina12_otros_pagos
                );
                //Otros pagos
                foreach ($employee_payroll->employeeActiveOtherPaymentTypesPayrollLines as $key => $result) {
                    $nomina12_otro_pago = [];
                    $nomina12_otro_pago['TipoOtroPago'] = $result->otherPaymentType->code;
                    $nomina12_otro_pago['Clave'] = !empty($result->code) ? $result->code : $result->otherPaymentType->code;
                    $nomina12_otro_pago['Concepto'] =str_limit($result->name,99,'');
                    $nomina12_otro_pago['Importe'] = Helper::numberFormat($result->amount_total,
                        $employee_payroll->currency->decimal_place, false);
                    $nodo_nomina12_otro_pago = new \CfdiUtils\Nodes\Node(
                        'nomina12:OtroPago', // nombre del elemento raíz
                        $nomina12_otro_pago
                    );
                    //SubsidioAlEmpleo
                    if($result->otherPaymentType->code == '002') {
                        $nomina12_otros_pago_subsidio_empleo = [];
                        $nomina12_otros_pago_subsidio_empleo['SubsidioCausado'] = Helper::numberFormat($result->amount_caused, $employee_payroll->currency->decimal_place, false);
                        $nodo_nomina12_otro_subsidio_empleo = new \CfdiUtils\Nodes\Node(
                            'nomina12:SubsidioAlEmpleo', // nombre del elemento raíz
                            $nomina12_otros_pago_subsidio_empleo
                        );
                        $nodo_nomina12_otro_pago->addChild($nodo_nomina12_otro_subsidio_empleo);
                    }
                    //CompensacionSaldosAFavor
                    if($result->otherPaymentType->code == '004'){
                        $nomina12_otros_pago_compensacion['SaldoAFavor'] = Helper::numberFormat($result->cm_positive_balance, $employee_payroll->currency->decimal_place, false);
                        $nomina12_otros_pago_compensacion['Año'] = Helper::numberFormat($result->cm_year, 0, false);
                        $nomina12_otros_pago_compensacion['RemanenteSalFav'] = Helper::numberFormat($result->cm_remainder, $employee_payroll->currency->decimal_place, false);
                        $nodo_nomina12_otros_pago_compensacion = new \CfdiUtils\Nodes\Node(
                            'nomina12:CompensacionSaldosAFavor', // nombre del elemento raíz
                            $nomina12_otros_pago_compensacion
                        );
                        $nodo_nomina12_otro_pago->addChild($nodo_nomina12_otros_pago_compensacion);
                    }

                    $nodo_nomina12_otros_pagos->addChild($nodo_nomina12_otro_pago);
                }

                $nodo_payroll->addChild($nodo_nomina12_otros_pagos);
            }

            //Incapacidades
            if($employee_payroll->employeeActiveDisabilitiesPayrollLines->isNotEmpty()) {
                $nomina12_incapacidades = [];
                $nodo_nomina12_incapacidades = new \CfdiUtils\Nodes\Node(
                    'nomina12:Incapacidades', // nombre del elemento raíz
                    $nomina12_incapacidades
                );
                //Deducciones
                foreach ($employee_payroll->employeeActiveDisabilitiesPayrollLines as $key => $result) {
                    $nomina12_incapacidad = [];
                    $nomina12_incapacidad['DiasIncapacidad'] = Helper::numberFormat($result->disability_days, 0, false);
                    $nomina12_incapacidad['TipoIncapacidad'] = $result->disabilityType->code;
                    $nomina12_incapacidad['ImporteMonetario'] = Helper::numberFormat($result->disability_amount,
                        $employee_payroll->currency->decimal_place, false);
                    $nodo_nomina12_incapacidad = new \CfdiUtils\Nodes\Node(
                        'nomina12:Incapacidad', // nombre del elemento raíz
                        $nomina12_incapacidad
                    );
                    $nodo_nomina12_incapacidades->addChild($nodo_nomina12_incapacidad);
                }

                $nodo_payroll->addChild($nodo_nomina12_incapacidades);
            }

            //Agregamos al complemento
            $comprobante->addComplemento($nodo_payroll);


            //Método de ayuda para establecer las sumas del comprobante e impuestos con base en la suma de los conceptos y la agrupación de sus impuestos
            //$creator->addSumasConceptos(null, 2);
            //Método de ayuda para generar el sello (obtener la cadena de origen y firmar con la llave privada)
            $creator->addSello('file://' . \Storage::path($employee_payroll->company->pathFileKeyPassPem()), Crypt::decryptString($employee_payroll->company->password_key));
            //Valida la estructura
            //$creator->validate();

            //Guarda XML
            //dd($creator->asXml());
            $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
            $file_xml = Helper::makeDirectoryCfdi($path_xml) . '/' . (!$draft ? Str::random(40) : str_replace('/','',$employee_payroll->name)) . '.xml';
            $creator->saveXml(\Storage::path($path_xml . $file_xml));


            //Arreglo temporal para actualizar Customer Invoice CFDI
            $tmp = [
                'pac_id' => $pac->id,
                'cfdi_version' => setting('cfdi_version'),
                'uuid' => '',
                'date' => null,
                'path_xml' => $path_xml,
                'file_xml' => $file_xml,
                'file_xml_pac' => '',
                'pac' => $pac,
            ];

            //Timbrado de XML
            if(!$draft) {
                $class_pac = $pac->code;
                $tmp = PacHelper::$class_pac($tmp, $creator);
            }

            return $tmp;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    /**
     * Descarga de archivo XML
     *
     * @param Request $request
     * @param EmployeePayroll $employee_payroll
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|\Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function downloadXml(Request $request, EmployeePayroll $employee_payroll)
    {
        //Ruta y validacion del XML
        $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
        $file_xml_pac = $path_xml . $employee_payroll->employeePayrollCfdi->file_xml_pac;
        $file_xml = $path_xml . $employee_payroll->employeePayrollCfdi->file_xml;
        if (!empty($employee_payroll->employeePayrollCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {
            while (ob_get_level()) ob_end_clean();
            ob_start();

            return response()->download(\Storage::path($file_xml_pac), str_replace('/','',$employee_payroll->name) . '.xml',['Cache-Control' => 'no-cache, must-revalidate']);
        }elseif (!empty($employee_payroll->employeePayrollCfdi->file_xml) && \Storage::exists($file_xml)) {
            while (ob_get_level()) ob_end_clean();
            ob_start();

            return response()->download(\Storage::path($file_xml), str_replace('/','',$employee_payroll->name) . '.xml',['Cache-Control' => 'no-cache, must-revalidate']);
        }

        //Mensaje
        flash(__('base/employee_payroll.error_download_xml'))->error();

        //Redireccion
        return redirect('/base/employee-payrolls');
    }

    /**
     * Descarga de archivo PDF
     *
     * @param Request $request
     * @param EmployeePayroll $employee_payroll
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|\Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function downloadPdf(Request $request, EmployeePayroll $employee_payroll)
    {
        //Descarga archivo
        return $this->print($employee_payroll, true);
    }

    /**
     * Cambiar estatus a enviada
     *
     * @param EmployeePayroll $employee_payroll
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function markSent(EmployeePayroll $employee_payroll)
    {
        //Logica
        $employee_payroll->updated_uid = \Auth::user()->id;
        $employee_payroll->mail_sent = 1;
        $employee_payroll->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/base/employee-payrolls');
    }

    /**
     * Clase generica de impresion
     *
     * @param EmployeePayroll $employee_payroll
     * @return mixed
     */
    public function print(EmployeePayroll $employee_payroll, $download = false, $save = false)
    {
        try {
            //Logica
            $tmp = 'default';
            if (!empty($employee_payroll->employeePayrollCfdi->cfdi_version)) {
                $tmp = $employee_payroll->employeePayrollCfdi->cfdi_version;
            }
            $class_print = 'print' . ucfirst($tmp);
            return $this->$class_print($employee_payroll, $download, $save);

        } catch (\Exception $e) {
            flash($e->getMessage())->error();
            return redirect('/base/employee-payrolls');
        }
    }

    /**
     * Impresion default
     *
     * @param $employee_payroll
     * @return mixed
     */
    private function printDefault($employee_payroll, $download = false, $save = false)
    {
        //PDF
        $pdf = \PDF::loadView('base.employee_payrolls.pdf_default', compact('employee_payroll'));

        //Guardar
        if ($save) {
            return $pdf->output();
        }

        //Descarga
        if($download){
            return $pdf->download($employee_payroll->documentType->name . '_' . str_replace('/','',$employee_payroll->name) . '.pdf');
        }

        //Redireccion
        return $pdf->stream($employee_payroll->documentType->name . '_' . str_replace('/','',$employee_payroll->name) . '.pdf');
    }

    /**
     * Impresion CFDI 3.3
     *
     * @param $employee_payroll
     * @param bool $download
     * @return mixed
     * @throws \Exception
     */
    private function printCfdi33($employee_payroll, $download = false, $save = false)
    {
        //Datos del XML timbrado
        $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
        $file_xml_pac = $path_xml . $employee_payroll->employeePayrollCfdi->file_xml_pac;
        $data = [];


        //Valida que el archivo exista
        if (!empty($employee_payroll->employeePayrollCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {

            //Crea arreglo
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
            $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

            //Genera codigo QR
            $image = QrCode::format('png')->size(150)->margin(0)->generate($data['qr_cadena']);
            $data['qr'] = 'data:image/png;base64,' . base64_encode($image);

            //Regimen fiscal
            $data['tax_regimen'] = TaxRegimen::where('code', $data['cfdi33']->emisor['RegimenFiscal'])->first()->name_sat;
        }

        //
        $subsidio_causado = $employee_payroll->employeeActiveOtherPaymentTypesPayrollLines()->whereHas('otherPaymentType', function ($q) {
            $q->whereIn('other_payment_types.code', ['002']);
        })->sum('amount_caused');
        $data['subsidio_causado'] = $subsidio_causado;

        //PDF
        $pdf_template = \App\Helpers\Helper::companyPdfTemplate($employee_payroll->company_id);//Plantilla
        $pdf_template = 'default'; //Fix
        if($employee_payroll->status == EmployeePayroll::DRAFT && $employee_payroll->employeeActivePerceptions4PayrollLines->isNotEmpty()){
            $pdf_template = 'default_draft'; //Fix
        }
        $pdf = \PDF::loadView('base.employee_payrolls.pdf_cfdi33_' . $pdf_template, compact('employee_payroll', 'data'));

        //Guardar
        if ($save) {
            return $pdf->output();
        }

        //Descargar
        if ($download) {
            return $pdf->download($employee_payroll->documentType->name . '_' . str_replace('/','',$employee_payroll->name) . '.pdf');
        }

        //Redireccion
        return $pdf->stream($employee_payroll->documentType->name . '_' . str_replace('/','',$employee_payroll->name) . '.pdf');
    }

    /**
     * Impresion CFDI 4.0
     *
     * @param $employee_payroll
     * @param bool $download
     * @return mixed
     * @throws \Exception
     */
    private function printCfdi40($employee_payroll, $download = false, $save = false)
    {
        //Datos del XML timbrado
        $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
        $file_xml_pac = $path_xml . $employee_payroll->employeePayrollCfdi->file_xml_pac;
        $data = [];

        //Valida que el archivo exista
        if (!empty($employee_payroll->employeePayrollCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {

            //Crea arreglo
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
            $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

            //Genera codigo QR
            $image = QrCode::format('png')->size(150)->margin(0)->generate($data['qr_cadena']);
            $data['qr'] = 'data:image/png;base64,' . base64_encode($image);

            //Regimen fiscal
            $data['tax_regimen'] = TaxRegimen::where('code', $data['cfdi33']->emisor['RegimenFiscal'])->first()->name_sat;
            $data['tax_regimen_employee'] = TaxRegimen::where('code', $data['cfdi33']->receptor['RegimenFiscalReceptor'])->first()->name_sat;
        }

        //
        $subsidio_causado = $employee_payroll->employeeActiveOtherPaymentTypesPayrollLines()->whereHas('otherPaymentType', function ($q) {
            $q->whereIn('other_payment_types.code', ['002']);
        })->sum('amount_caused');
        $data['subsidio_causado'] = $subsidio_causado;

        //PDF
        $pdf_template = \App\Helpers\Helper::companyPdfTemplate($employee_payroll->company_id);//Plantilla
        $pdf_template = 'default'; //Fix
        if($employee_payroll->status == EmployeePayroll::DRAFT && $employee_payroll->employeeActivePerceptions4PayrollLines->isNotEmpty()){
            $pdf_template = 'default_draft'; //Fix
        }
        $pdf = \PDF::loadView('base.employee_payrolls.pdf_cfdi40_' . $pdf_template, compact('employee_payroll', 'data'));

        //Guardar
        if ($save) {
            return $pdf->output();
        }

        //Descargar
        if ($download) {
            return $pdf->download($employee_payroll->documentType->name . '_' . str_replace('/','',$employee_payroll->name) . '.pdf');
        }

        //Redireccion
        return $pdf->stream($employee_payroll->documentType->name . '_' . str_replace('/','',$employee_payroll->name) . '.pdf');
    }

    /**
     * Modal para envio de correo de factura
     *
     * @param Request $request
     * @param EmployeePayroll $employee_payroll
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalSendMail(Request $request, EmployeePayroll $employee_payroll)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            //Correo default del cliente
            $reply = Helper::replyMails();
            $to = [];
            $to_selected = [];
            if (!empty($employee_payroll->employee->email)) {
                $email = $employee_payroll->employee->email;
                $to[$email] = $email;
                $to_selected [] = $email;
            }
            //Etiquetas solo son demostrativas
            $files = [
                'pdf' => str_replace('/','',$employee_payroll->name) . '.pdf',
                'xml' => str_replace('/','',$employee_payroll->name) . '.xml'
            ];
            $files_selected = array_keys($files);

            //modal de mensaje
            $html = view('layouts.partials.employee_payrolls.modal_send_mail',
                compact('employee_payroll', 'to', 'to_selected', 'files', 'files_selected', 'reply'))->render();

            //Mensaje predefinido
            $custom_message = view('layouts.partials.employee_payrolls.message_send_mail',
                compact('employee_payroll'))->render();

            return response()->json(['html' => $html, 'custom_message' => $custom_message]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Envio de factura por correo
     *
     * @param Request $request
     * @param EmployeePayroll $employee_payroll
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function sendMail(Request $request, EmployeePayroll $employee_payroll)
    {
        //Validaciones
        $validator = \Validator::make($request->all(), [
            'subject' => 'required',
            'to' => 'required',
            'to.*' => 'email',
            'message' => 'required'
        ], [
            'subject.*' => __('general.error_mail_subject'),
            'to.required' => __('general.error_mail_to'),
            'to.*.email' => __('general.error_mail_to_format'),
            'message.*' => __('general.error_mail_message'),
        ]);
        if ($validator->fails()) {
            $errors = '<ul>';
            foreach ($validator->errors()->all() as $message) {
                $errors .= '<li>'.$message . '</li>';
            }
            $errors .= '</ul>';
            return response()->json(['error' => $errors], 422);
        }

        //Creamos PDF en stream
        $pdf = $this->print($employee_payroll);
        //Ruta del XML
        $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
        $file_xml_pac = $path_xml . $employee_payroll->employeePayrollCfdi->file_xml_pac;
        if(empty($employee_payroll->employeePayrollCfdi->file_xml_pac)){
            $file_xml_pac = $path_xml . $employee_payroll->employeePayrollCfdi->file_xml;
        }

        //Envio de correo
        $to = $request->to;
        $reply = $request->reply;
        \Mail::to($to)->send(new SendEmployeePayroll($employee_payroll, $request->subject, $request->message, $pdf,
            $file_xml_pac, $reply));

        //Actualiza campo de enviado
        $employee_payroll->updated_uid = \Auth::user()->id;
        $employee_payroll->mail_sent = 1;
        $employee_payroll->save();

        //Mensaje
        return response()->json([
            'success' => sprintf(__('base/employee_payroll.text_success_send_mail'), $employee_payroll->name)
        ]);
    }

    /**
     * Obtener registro
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getEmployeePayroll(Request $request)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            $employee_payroll = EmployeePayroll::findOrFail($id);
            $employee_payroll->uuid = $employee_payroll->employeePayrollCfdi->uuid ?? '';
            return response()->json($employee_payroll, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Autoacompletar select2 de facturas solo activas del SAT
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function autocompleteCfdi(Request $request)
    {
        //Variables
        $term = $request->term;
        $employee_id = !empty($request->employee_id) ? $request->employee_id : '111111111111'; //Filtra las facturas por cliente

        //Logica
        if ($request->ajax() && !empty($term)) {
            $tmp = EmployeePayroll::filter([
                'filter_search_cfdi_select2' => $term,
                'filter_employee_id' => $employee_id,
                'filter_document_type_code' => $this->document_type_code,
            ])->sortable('date')->limit(16)->get();
            $results = [];
            if ($tmp->isNotEmpty()) {
                foreach ($tmp as $result) {
                    $results[] = [
                        'id' => $result->id,
                        'text' => $result->text_select2,
                        'description' => $result->description_select2
                    ];
                }
            }
            return response()->json($results, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Calcula el total de las lineas
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function totalLines(Request $request)
    {

        //Variables
        $json = new \stdClass;
        $input_items_perception = $request->item_perception;
        $input_items_deduction = $request->item_deduction;
        $input_items_other_payment_type = $request->item_other_payment_type;
        $currency_id = $request->currency_id;
        $currency_code = 'MXN';

        if ($request->ajax()) {
            //Datos de moneda
            if (!empty($currency_id)) {
                $currency = Currency::findOrFail($currency_id);
                $currency_code = $currency->code;
            }
            //Variables de totales
            $amount_discount = 0;
            $amount_untaxed = 0;
            $amount_isr = 0;
            $amount_tax = 0;
            $amount_tax_ret = 0;
            $amount_total = 0;
            $balance = 0;

            //Percepciones
            if (!empty($input_items_perception)) {
                foreach ($input_items_perception as $key => $item) {
                    $perception = null;
                    if(!empty($item['perception_id'])){
                        $perception = Perception::findOrFail($item['perception_id']);
                    }
                    //Logica
                    $item_amount_taxed = (double)$item['amount_taxed'];
                    $item_amount_exempt = (double)$item['amount_exempt'];
                    $item_amount_total = $item_amount_taxed + $item_amount_exempt;

                    //Sumatoria totales
                    $amount_discount += 0;
                    $amount_untaxed += $item_amount_total;
                    $amount_isr += 0;
                    $amount_tax += 0;
                    $amount_tax_ret += 0;
                }
            }
            //Deducciones
            if (!empty($input_items_deduction)) {
                foreach ($input_items_deduction as $key => $item) {
                    $deduction = null;
                    if(!empty($item['deduction_id'])){
                        $deduction = Deduction::findOrFail($item['deduction_id']);
                    }
                    //Logica
                    $item_amount_taxed = (double)$item['amount_total'];
                    $item_amount_exempt = 0;
                    $item_amount_total = $item_amount_taxed + $item_amount_exempt;

                    //Sumatoria totales
                    if(!empty($deduction) && in_array($deduction->code,['002','101'])){ //Mostrar el ISR separado
                        $amount_discount += 0;
                        $amount_untaxed += 0;
                        $amount_isr += $item_amount_total;
                        $amount_tax += 0;
                        $amount_tax_ret += 0;
                    }else {
                        $amount_discount += $item_amount_total;
                        $amount_untaxed += 0;
                        $amount_isr += 0;
                        $amount_tax += 0;
                        $amount_tax_ret += 0;
                    }
                }
            }
            //Otros pagos
            if (!empty($input_items_other_payment_type)) {
                foreach ($input_items_other_payment_type as $key => $item) {
                    $other_payment_type = null;
                    if(!empty($item['other_payment_type_id'])){
                        $other_payment_type = OtherPaymentType::findOrFail($item['other_payment_type_id']);
                    }
                    //Logica
                    $item_amount_taxed = (double)($item['amount_total'] ?? '');
                    $item_amount_exempt = 0;
                    $item_amount_total = $item_amount_taxed + $item_amount_exempt;

                    //Sumatoria totales
                    $amount_discount += 0;
                    $amount_untaxed += $item_amount_total;
                    $amount_isr += 0;
                    $amount_tax += 0;
                    $amount_tax_ret += 0;
                }
            }
            //Gran total
            $amount_total = $amount_untaxed-$amount_discount-$amount_isr;
            //Respuesta
            $json->amount_discount = money($amount_discount, $currency_code, true)->format();
            $json->amount_untaxed = money($amount_untaxed, $currency_code, true)->format();
            $json->amount_isr = money($amount_isr, $currency_code, true)->format();
            $json->amount_tax = money($amount_tax + $amount_tax_ret, $currency_code, true)->format();
            $json->amount_total = money($amount_total, $currency_code, true)->format();
            return response()->json($json);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Calcula los dias de pago
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getPaymentDays(Request $request)
    {

        //Variables
        $json = new \stdClass;

        if ($request->ajax()) {

            if(!empty($request->date_start_payment) && !empty($request->date_end_payment)) {
                //Respuesta
                $date_start_payment = Helper::createDate($request->date_start_payment);
                $date_end_payment = Helper::createDate($request->date_end_payment);
                $payment_days = $date_start_payment->diffInDays($date_end_payment,false);
                $json->payroll_name = 'Nómina ' . (!empty($request->date_start_payment) ? (ucfirst(Helper::createDate($request->date_start_payment)->format('M')) . ', ' . Helper::createDate($request->date_start_payment)->format('Y')) : '');
                $json->payment_days = Helper::numberFormat($payment_days >= 0 ? $payment_days + 1 : 0,0,false);
            }
            return response()->json($json);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para cancelar factura
     *
     * @param Request $request
     * @param EmployeePayroll $employee_payroll
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalCancel(Request $request, EmployeePayroll $employee_payroll)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            //Obtener informacion de estatus
            $data_status_sat = [
                'cancelable' => 1,
                'pac_is_cancelable' => ''
            ];
            $reason_cancellations = ReasonCancellation::populateSelect()->get()->pluck('name_sat', 'id');
            if (!empty($employee_payroll->employeePayrollCfdi->cfdi_version) && !empty($employee_payroll->employeePayrollCfdi->uuid)) {

                //Obtener el sellos del CFDI
                $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
                $file_xml_pac = $path_xml . $employee_payroll->employeePayrollCfdi->file_xml_pac;
                $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
                $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

                $tmp = [
                    'rfcR' => $employee_payroll->employee->taxid,
                    'uuid' => $employee_payroll->employeePayrollCfdi->uuid,
                    'total' => Helper::numberFormat($employee_payroll->amount_total, $employee_payroll->currency->decimal_place, false),
                    'fe' => substr($data['cfdi33']->complemento->timbreFiscalDigital['SelloCFD'],-8), //Los últimos 8 caracteres del Atributo Sello del CFDI
                    'file_xml_pac' => $file_xml_pac
                ];
                $class_pac = $employee_payroll->employeePayrollCfdi->pac->code . 'Status';
                $data_status_sat = PacHelper::$class_pac($tmp,$employee_payroll->company,$employee_payroll->employeePayrollCfdi->pac);
            }

            $is_cancelable = true;
            if($data_status_sat['cancelable'] == 3){
                $is_cancelable = false;
            }

            //modal de cancelar
            $html = view('layouts.partials.employee_payrolls.modal_cancel', compact('employee_payroll','data_status_sat','is_cancelable', 'reason_cancellations'))->render();

            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Exportar datos a excel
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function exportToExcel(Request $request)
    {
        while (ob_get_level()) ob_end_clean();
        ob_start();

        return Excel::download(new EmployeePayrollsExport($request),
            __('base/employee_payroll.document_title') . '-' . config('app.name') . '.xlsx');
    }

    /**
     * Funcion para timbrar factura con estatus de prefactura
     *
     * @param EmployeePayroll $employee_payroll
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function stamp(EmployeePayroll $employee_payroll){

        \DB::connection('tenant')->beginTransaction();
        try {
            $invoiced = false;
            if((int) $employee_payroll->status == EmployeePayroll::DRAFT) {
                //Valida que tenga exista la clase de facturacion
                $class_cfdi = setting('cfdi_version');
                if (empty($class_cfdi)) {
                    throw new \Exception(__('general.error_cfdi_version'));
                }
                if (!method_exists($this, $class_cfdi)) {
                    throw new \Exception(__('general.error_cfdi_class_exists'));
                }

                //Valida que tenga folios disponibles
                if(BaseHelper::getAvailableFolios()<=0){
                    throw new \Exception(__('general.error_available_folios'));
                }

                //CFDI
                $employee_payroll_cfdi = $employee_payroll->EmployeePayrollCfdi;

                //Actualiza registr principal
                $employee_payroll->updated_uid = \Auth::user()->id;
                $employee_payroll->date = Helper::dateTimeToSql(BaseHelper::getDateTimeBranchOffice($employee_payroll->branch_office_id));
                $employee_payroll->status = EmployeePayroll::OPEN;
                $employee_payroll->save();

                //Actualiza folios
                //Obtiene folio
                $document_type = Helper::getNextDocumentTypeByCode($this->document_type_code, $employee_payroll->company->id, false, $employee_payroll->branch_office_id);
                $employee_payroll->draft = $employee_payroll->name;
                $employee_payroll->name = $document_type['name'];
                $employee_payroll->serie = $document_type['serie'];
                $employee_payroll->folio = $document_type['folio'];
                $employee_payroll->save();
                $employee_payroll_cfdi->name = $employee_payroll->name;
                $employee_payroll_cfdi->save();

                //Valida Empresa y PAC para timbrado
                PacHelper::validateSatActions($employee_payroll->company);

                //Crear XML y timbra
                $tmp = $this->$class_cfdi($employee_payroll);

                //Guardar registros de CFDI
                $employee_payroll_cfdi->fill(array_only($tmp, [
                    'pac_id',
                    'cfdi_version',
                    'uuid',
                    'date',
                    'file_xml',
                    'file_xml_pac',
                ]));
                $employee_payroll_cfdi->save();

                //Disminuye folios
                BaseHelper::decrementFolios();

                //Mensaje
                flash(__('general.text_success_employee_payroll_cfdi'))->success();

                $invoiced = true;
            }

            \DB::connection('tenant')->commit();

            if($invoiced) {
                $this->saveCfdiDownloads($employee_payroll, $employee_payroll_cfdi);
            }

        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            flash( __('general.error_cfdi_pac').'<br>'.$e->getMessage())->error();
            return back()->withInput();
        }

        //Almacenamiento dropbox
        self::dropboxBackup($employee_payroll);

        //Redireccion
        return redirect('/base/employee-payrolls');
    }

    /**
     * Modal para estatus de CFDI
     *
     * @param Request $request
     * @param EmployeePayroll $employee_payroll
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalStatusSat(Request $request, EmployeePayroll $employee_payroll)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            //Obtener informacion de estatus
            $data_status_sat = [
                'cancelable' => 1,
                'pac_is_cancelable' => ''
            ];
            if (!empty($employee_payroll->employeePayrollCfdi->cfdi_version) && !empty($employee_payroll->employeePayrollCfdi->uuid)) {

                //Obtener el sellos del CFDI
                $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
                $file_xml_pac = $path_xml . $employee_payroll->employeePayrollCfdi->file_xml_pac;
                $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
                $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

                $tmp = [
                    'rfcR' => $employee_payroll->employee->taxid,
                    'uuid' => $employee_payroll->employeePayrollCfdi->uuid,
                    'total' => Helper::numberFormat($employee_payroll->amount_total, $employee_payroll->currency->decimal_place, false),
                    'fe' => substr($data['cfdi33']->complemento->timbreFiscalDigital['SelloCFD'],-8), //Los últimos 8 caracteres del Atributo Sello del CFDI
                    'file_xml_pac' => $file_xml_pac
                ];
                $class_pac = $employee_payroll->employeePayrollCfdi->pac->code . 'Status';
                $data_status_sat = PacHelper::$class_pac($tmp,$employee_payroll->company,$employee_payroll->employeePayrollCfdi->pac);
            }
            $is_cancelable = true;
            if($data_status_sat['cancelable'] == 3){
                $is_cancelable = false;
            }

            //modal de visualizar estatus en el SAT
            $html = view('layouts.partials.employee_payrolls.modal_status_sat', compact('employee_payroll','data_status_sat','is_cancelable'))->render();

            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Respaldo en dropbox si esta activo
     *
     * @param $employee_payroll
     */
    public function dropboxBackup($employee_payroll, $save_xml = true, $save_pdf = true){
        try{
            if(!empty($employee_payroll) && !empty(setting('dropbox_backup')) && !empty(setting('dropbox_access_token'))){
                $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
                $file_xml_pac = $path_xml . $employee_payroll->employeePayrollCfdi->file_xml_pac;
                if (!empty($employee_payroll->employeePayrollCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {

                    $folder = !empty(setting('dropbox_folder','')) ? setting('dropbox_folder','') . '/' : setting('dropbox_folder','');
                    $folder .= strtoupper(\Date::parse($employee_payroll->date)->format('F Y'));
                    $folder .= '/' .preg_replace("/[^a-zA-Z0-9\s]/", '', $employee_payroll->company->name);

                    //Guarda XML
                    if($save_xml) {
                        \Storage::disk('dropbox')->putFileAs($folder, new \Illuminate\Http\File(\Storage::path($file_xml_pac)), str_replace('/','',$employee_payroll->name) . '.xml');
                    }

                    //Guarda PDF
                    if($save_pdf) {
                        $pdf = $this->print($employee_payroll, false, true);
                        \Storage::disk('dropbox')->put($folder . '/' . $employee_payroll->name . '.pdf', $pdf);
                    }
                }

            }
        }catch (\Exception $e){
            flash($e->getMessage())->error();
        }
    }

    private function saveCfdiDownloads($employee_payroll, $employee_payroll_cfdi){
        //Directorio de XML
        $path_files = Helper::setDirectory(CfdiDownload::PATH_FILES, $employee_payroll->company->id);
        $tmp_path = str_replace(['files/', '/xml'], '', EmployeePayroll::PATH_XML_FILES);
        if (!\Storage::exists($path_files . '/' . $tmp_path)) {
            \Storage::makeDirectory($path_files . '/' . $tmp_path, 0777, true, true);
        }

        //Genera PDF
        $pdf = $this->print($employee_payroll, false, true);
        $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
        $file_xml = $tmp_path . '/' . $employee_payroll_cfdi->uuid . '.xml';
        $file_pdf = str_replace('.xml', '.pdf', $file_xml);
        \Storage::put($path_files . '/' . $file_pdf, $pdf);

        if(!empty($employee_payroll_cfdi->file_xml_pac)){
            \Storage::copy($path_xml . $employee_payroll_cfdi->file_xml_pac, $path_files . '/' . $file_xml);
        }

        //Guarda registro en CFDIS descargados
        $cfdi_download = CfdiDownload::create([
            'created_uid' => \Auth::user()->id,
            'updated_uid' => \Auth::user()->id,
            'type' => 2,
            'uuid' => $employee_payroll_cfdi->uuid,
            'file_xml' => $file_xml,
            'file_pdf' => $file_pdf,
            'file_acuse' => null,
            'status' => 'Vigente',
            'is_cancelable' => 'Cancelable sin aceptación',
            'date_cancel' => null,
            'company_id' => $employee_payroll->company_id,
        ]);
        $data_xml = Helper::parseXmlToArrayCfdi33($path_xml . '/' . $employee_payroll_cfdi->file_xml_pac);
        $data_xml['data']['customer_id'] = $employee_payroll->customer_id;
        $data_xml['data']['caccounting_type_id'] = $employee_payroll->customer->caccounting_type_sale_id ?? null;
        $cfdi_download->fill($data_xml['data']);
        $cfdi_download->save();
    }
}
