<?php

namespace App\Http\Controllers\Base;

use App\Helpers\Helper;
use App\Http\Middleware\CheckDemo;
use App\Http\Middleware\CheckSuperadmin;
use App\Http\Middleware\CheckSuperadminOrDistributor;
use App\Mail\SendActiveFolios;
use App\Models\Base\Folio;
use App\Models\Catalogs\Currency;
use App\Models\System\SysCustomer;
use App\Models\System\SysSalesOrder;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;

class FolioController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware(CheckSuperadmin::class,
            ['only' => ['show', 'edit', 'update']]);
        $this->middleware(CheckSuperadminOrDistributor::class,
            ['only' => ['create','store', 'activeFolios']]);
        $this->middleware(CheckDemo::class);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = Folio::filter($request->all())->sortable(['created_at' => 'desc'])->paginate($limit);

        //Vista
        return view('base.folios.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('base.folios.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);

        //Logica
        $currency = Currency::where('code', '=', 'MXN')->get()->first();
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['name' => 'XXX' . random_int(1000, 9999)]); //Folio temporal
        $request->merge(['payment_platform' => __('base/folio.text_folio_assignments')]);
        $request->merge(['currency_id' => !empty($currency) ? $currency->id : null]);
        $request->merge(['currency_value' => !empty($currency) ? $currency->rate : null]);
        $request->merge(['quantity' => 1]);
        $request->merge(['price_unit' => $request->amount_total]);
        $request->merge(['price_reduce' => $request->amount_total]);
        $request->merge(['amount_untaxed' => $request->amount_total]);
        $request->merge(['amount_total' => $request->amount_total]);
        $request->merge(['quantity_folio' => (int)$request->quantity_folio]);
        $request->merge(['available_folio' => (int)$request->quantity_folio]);
        $request->merge(['expiry_date' => Helper::convertDateToSql($request->expiry_date)]);
        $request->merge(['sort_order' => 0]);
        $request->merge(['status' => !empty($request->status) ? $request->status : Folio::PER_ACTIVE]);

        //Guardar
        //Registro principal
        $folio = Folio::create($request->input());
        $folio->name = 'CFL' . $folio->id; //Actualizamos el folio
        $folio->save();

        //Guardamos en pedidos del sistema
        $company = Helper::firstCompany();
        $sys_customer = \Auth::user()->sysCustomer;
        $sys_sales_order = SysSalesOrder::create([
            'created_uid' => \Auth::user()->id,
            'updated_uid' => \Auth::user()->id,
            'name' => 'XXX' . random_int(1000, 9999),
            'sys_customer_id' => !empty($sys_customer) ? $sys_customer->id : null,
            'currency_id' => $folio->currency_id,
            'currency_value' => $folio->currency_value,
            'product' => $request->product,
            'quantity' => $folio->quantity,
            'price_unit' => $folio->price_unit,
            'price_reduce' => $folio->price_reduce,
            'amount_untaxed' => $folio->amount_untaxed,
            'amount_total' => $folio->amount_total,
            'folio_id' => $folio->id,
            'sys_customer_folio' => $folio->name,
            'quantity_folio' => $folio->quantity_folio,
            'payment_platform' => __('base/folio.text_folio_assignments'),
            'status' => SysSalesOrder::PER_ACTIVE
        ]);
        $sys_sales_order->name = 'PV'.$sys_sales_order->id; //Actualizamos el folio
        $sys_sales_order->save();

        //Mensaje
        flash(__('general.text_form_success_add_buy_folios'))->success();

        //Redireccion
        return redirect('/base/folios');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Base\Folio $folio
     * @return \Illuminate\Http\Response
     */
    public function show(Folio $folio)
    {
        return redirect('/base/folios');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Base\Folio $folio
     * @return \Illuminate\Http\Response
     */
    public function edit(Folio $folio)
    {
        return redirect('/base/folios');
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  \App\Models\Base\Folio $folio
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Folio $folio)
    {
        return redirect('/base/folios');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Base\Folio $folio
     * @return \Illuminate\Http\Response
     */
    public function destroy(Folio $folio)
    {
        //Logica
        if ($folio->status == Folio::PER_ACTIVE || \Auth::user()->superadmin) {
            $folio->updated_uid = \Auth::user()->id;
            $folio->status = Folio::CANCELED;
            $folio->save();

            //Activa el pedido del sistemas
            if (\Auth::user()->sysCustomer){
                $sys_sales_order = SysSalesOrder::where('sys_customer_id', '=', \Auth::user()->sysCustomer->id)
                    ->where('folio_id', '=', $folio->id)
                    ->update([
                        'updated_uid' => \Auth::user()->id,
                        'status' => SysSalesOrder::CANCELED,
                    ]);
            }
        }

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/base/folios');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Base\Folio $folio
     * @return \Illuminate\Http\Response
     */
    public function activeFolios(Folio $folio)
    {
        $company = Helper::firstCompany();
        $sys_customer = \Auth::user()->sysCustomer;

        //validacion el distribuidor del cliente debe tener los suficientes folios para asignarle
        if(!empty($sys_customer->distributor)){
            $distributor = $sys_customer->distributor;
            if($distributor->available_folio < $folio->quantity_folio){
                //Mensaje
                flash(sprintf(__('auth/user.error_available_folio'),$distributor->name,$distributor->available_folio))->error();
                //Redireccion
                return redirect('/base/folios');
            }
        }

        //Logica
        if ($folio->status == Folio::PER_ACTIVE) {
            $folio->updated_uid = \Auth::user()->id;
            $folio->activation_date = \Date::now();
            $folio->status = Folio::ACTIVE;
            $folio->save();

            //Activa el pedido del sistemas
            if (!empty($sys_customer)) {
                $sys_sales_order = SysSalesOrder::where('sys_customer_id', '=', $sys_customer->id)
                    ->where('folio_id', '=', $folio->id)
                    ->update([
                        'updated_uid' => \Auth::user()->id,
                        'activation_date' => \Date::now(),
                        'status' => SysSalesOrder::ACTIVE,
                    ]);
            }

            //Descuenta de los folios del distribuidor
            if(!empty($sys_customer->distributor)) {
                $distributor = $sys_customer->distributor;
                $distributor->available_folio -= $folio->quantity_folio;
                $distributor->save();
            }

            //Correo de notificacion de activacion de folios
            \Mail::to($folio->createdUser)->send(new SendActiveFolios($folio));

            //Mensaje
            flash(__('general.text_form_success_add_active_folios'))->success();
            flash(__('general.text_form_success_add_active_folios_notification'))->success();

        }

        //Redireccion
        return redirect('/base/folios');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'product' => 'required|string',
            'amount_total' => 'required|numeric',
            'quantity_folio' => 'required|integer',
            'expiry_date' => 'required|date|date_format:"' . setting('date_format', 'd-m-Y') . '"',
        ], [
            'product.*' => __('base/folio.error_product'),
            'amount_total.*' => __('base/folio.error_amount_total'),
            'quantity_folio.required' => __('base/folio.error_quantity_folio'),
            'quantity_folio.integer' => __('base/folio.error_quantity_folio_integer'),
            'expiry_date.date_format' => __('base/folio.error_expiry_date_format'),
        ]);
    }
}
