<?php

namespace App\Http\Controllers\Base;

use App\Http\Middleware\CheckSuperadmin;
use App\Models\Base\Pac;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Validation\Rule;

class PacController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware(CheckSuperadmin::class);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = Pac::filter($request->all())->sortable('name')->paginate($limit);

        //Vista
        return view('base.pacs.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('base.pacs.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.pacs',
        ], [
            'code.required' => __('base/pac.error_code'),
            'code.unique' => __('base/pac.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['password' => !empty($request->password) ? Crypt::encryptString($request->password) : '']);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['test' => !empty($request->test) ? 1 : 0]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $pac = Pac::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/base/pacs');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Base\Pac  $pac
     * @return \Illuminate\Http\Response
     */
    public function show(Pac $pac)
    {
        return redirect('/base/pacs');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Base\Pac  $pac
     * @return \Illuminate\Http\Response
     */
    public function edit(Pac $pac)
    {
        return view('base.pacs.edit', compact('pac'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Base\Pac  $pac
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Pac $pac)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.pacs')->ignore($pac->id)
            ],
        ], [
            'code.required' => __('base/pac.error_code'),
            'code.unique' => __('base/pac.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['test' => !empty($request->test) ? 1 : 0]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $pac->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'ws_url',
            'ws_url_cancel',
            'username',
            'test',
            'comment',
            'sort_order',
            'status',
            'team_id',
            'token'
        ]));

        //Si se ingresa una contraseña
        if(!empty($request->password)){
            $pac->password = Crypt::encryptString($request->password);
        }

        //Guardar
        //Registro principal
        $pac->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/base/pacs');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Base\Pac  $pac
     * @return \Illuminate\Http\Response
     */
    public function destroy(Pac $pac)
    {
        //Logica
        $pac->updated_uid = \Auth::user()->id;
        $pac->status = 0;
        $pac->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/base/pacs');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
            'ws_url' => 'required',
        ], [
            'name.*' => __('base/pac.error_name'),
            'ws_url.*' => __('base/pac.error_ws_url'),
        ]);
    }
}
