<?php

namespace App\Http\Controllers\Base;

use App\Helpers\Helper;
use App\Models\Base\Pac;
use App\Helpers\PacHelper;
use App\Helpers\BaseHelper;
use Illuminate\Support\Str;
use App\Models\Base\Payroll;
use Illuminate\Http\Request;
use App\Helpers\Cfdi33Helper;
use App\Models\Base\Employee;
use App\Models\Catalogs\CfdiUse;
use App\Models\Base\BranchOffice;
use App\Models\Base\CfdiDownload;
use App\Models\Catalogs\Currency;
use App\Models\Catalogs\Deduction;
use App\Models\Catalogs\PaymentWay;
use App\Models\Catalogs\Perception;
use App\Models\Catalogs\TaxRegimen;
use App\Http\Controllers\Controller;
use App\Models\Base\EmployeePayroll;
use App\Models\Catalogs\PayrollType;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\Base\EmployeeCategory;
use Illuminate\Support\Facades\Crypt;
use App\Models\Catalogs\OvertimesType;
use App\Models\Catalogs\PaymentMethod;
use App\Models\Catalogs\DisabilityType;
use App\Models\Base\EmployeePayrollCfdi;
use App\Models\Base\EmployeePayrollLine;
use App\Models\Catalogs\OtherPaymentType;
use App\Models\Catalogs\ReasonCancellation;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use App\Exports\PayrollsTemplateImportExport;
use App\Imports\PayrollsTemplateImportImport;
use Illuminate\Validation\ValidationException;
use App\Imports\PayrollsTemplateBeforeImportImport;
use App\Exports\PayrollAssimilatedTemplateImportExport;
use App\Imports\PayrollAssimilatedTemplateImportImport;
use App\Imports\PayrollAssimilatedTemplateBeforeImportImport;

class PayrollController extends Controller
{
    private $list_status = [];
    private $document_type_code = 'employee.payroll';

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->list_status = [
            Payroll::DRAFT => __('base/payroll.text_status_draft'),
            Payroll::OPEN => __('base/payroll.text_status_open'),
            Payroll::CANCEL => __('base/payroll.text_status_cancel'),
        ];
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);
        $payroll_types = PayrollType::populateSelect()->pluck('name', 'id');
        $list_status = $this->list_status;
        if (empty($request->filter_date_payment_from)) {
            $request->request->add([
                'filter_date_payment_from' => Helper::date(\Date::parse('first day of this month')->subMonths(2))
            ]);
        }
        if (empty($request->filter_date_payment_to)) {
            $request->request->add([
                'filter_date_to' => Helper::date(\Date::parse('last day of this month'))
            ]);
        }

        //Consulta
        $results = Payroll::filter($request->all())
            ->with('currency')
            ->sortable(['date_payment' => 'desc'])->paginate($limit);

        //Vista
        return view('base.payrolls.index', compact('results','payroll_types', 'list_status'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Base\Payroll  $payroll
     * @return \Illuminate\Http\Response
     */
    public function show(Payroll $payroll)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Base\Payroll  $payroll
     * @return \Illuminate\Http\Response
     */
    public function edit(Payroll $payroll)
    {
        $branch_offices = BranchOffice::populateSelect()->pluck('name', 'id');
        $payroll_types = PayrollType::populateSelect()->get()->pluck('name_sat', 'id');

        return view('base.payrolls.edit', compact( 'payroll','branch_offices', 'payroll_types'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Base\Payroll  $payroll
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Payroll $payroll)
    {
        //Validacion
        $this->validation($request);

        \DB::connection('tenant')->beginTransaction();
        try {

            //Logica
            $request->merge(['created_uid' => \Auth::user()->id]);
            $request->merge(['updated_uid' => \Auth::user()->id]);

            $payroll->fill($request->only([
                'updated_uid',
                'name',
                'payroll_type_id',
                'branch_office_id'
            ]));

            //Guardar
            //Registro principal
            $payroll->save();

            //Actualiza los registros de la nomina
            $payroll->employeePayrolls()->update([
                'payroll_type_id' => $request->payroll_type_id,
                'branch_office_id' => $request->branch_office_id,
                'reference' => $request->reference,
            ]);

            //Mensaje
            if(empty($request->save_and_stamp)) {
                flash(__('general.text_form_success_add'))->success();
            }

            \DB::connection('tenant')->commit();

            if(!empty($request->save_and_stamp)){
                return redirect()->route('payrolls/stamp',['id' => $payroll->id]);
            }

        } catch (\Exception $e) {

            \DB::connection('tenant')->rollback();
            \Log::error($e->getMessage());
            flash($e->getMessage())->error();
            return back()->withInput();
        }

        //Redireccion
        return redirect('/base/payrolls');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Base\Payroll  $payroll
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, Payroll $payroll)
    {
        if($payroll->employeePayrolls->isNotEmpty()) {

            foreach ($payroll->employeePayrolls as $employee_payroll) {
                \DB::connection('tenant')->beginTransaction();
                try {
                    //Logica
                    if ((int)$employee_payroll->status != EmployeePayroll::CANCEL) {
                        //Actualiza status
                        $employee_payroll->updated_uid = \Auth::user()->id;
                        $employee_payroll->status = EmployeePayroll::CANCEL;
                        $employee_payroll->save();

                        //Actualiza status CFDI
                        $employee_payroll->employeePayrollCfdi->status = 0;
                        $employee_payroll->employeePayrollCfdi->reason_cancellation_id = $request->reason_cancellation_id;
                        $employee_payroll->employeePayrollCfdi->reason_cancellation_uuid = $request->reason_cancellation_uuid;
                        $employee_payroll->employeePayrollCfdi->save();

                        //Cancelacion del timbre fiscal
                        if (!empty($employee_payroll->employeePayrollCfdi->cfdi_version) && !empty($employee_payroll->employeePayrollCfdi->uuid)) {
                            //Valida Empresa y PAC para cancelar timbrado
                            PacHelper::validateSatCancelActions($employee_payroll->company, $employee_payroll->employeePayrollCfdi->pac);

                            //Obtener el sellos del CFDI
                            $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
                            $file_xml_pac = $path_xml . $employee_payroll->employeePayrollCfdi->file_xml_pac;
                            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
                            $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

                            //Arreglo temporal para actualizar Customer Invoice CFDI
                            $tmp = [
                                'cancel_date' => Helper::dateTimeToSql(\Date::now()),
                                'cancel_response' => '',
                                'cancel_state' => $request->cancel_state,
                                'rfcR' => $employee_payroll->employee->taxid,
                                'uuid' => $employee_payroll->employeePayrollCfdi->uuid,
                                'total' => Helper::numberFormat($employee_payroll->amount_total,
                                    $employee_payroll->currency->decimal_place, false),
                                'cfdi_type' => $employee_payroll->documentType->cfdiType->code,
                                'cfdi_version' => 'cfdi3.3',
                                'fe' => substr($data['cfdi33']->complemento->timbreFiscalDigital['SelloCFD'],-8), //Los últimos 8 caracteres del Atributo Sello del CFDI
                                'file_xml_pac' => $file_xml_pac,
                                'reason_cancellation_code' => $employee_payroll->employeePayrollCfdi->reasonCancellation->code ?? '',
                                'reason_cancellation_uuid' => $employee_payroll->employeePayrollCfdi->reason_cancellation_uuid ?? '',
                            ];

                            //Cancelar Timbrado de XML
                            $class_pac = $employee_payroll->employeePayrollCfdi->pac->code . 'Cancel';
                            $tmp = PacHelper::$class_pac($tmp, $employee_payroll->company,
                                $employee_payroll->employeePayrollCfdi->pac);

                            //Guardar registros de CFDI
                            $employee_payroll->employeePayrollCfdi->fill(array_only($tmp, [
                                'cancel_date',
                                'cancel_response',
                                'cancel_state',
                            ]));
                            $employee_payroll->employeePayrollCfdi->save();

                            $cfdi_download = CfdiDownload::where('uuid', $employee_payroll->employeePayrollCfdi->uuid)->where('type', 2)->first();
                            if(!empty($cfdi_download)){
                                $cfdi_download->status = 'Cancelado';
                                $cfdi_download->save();
                            }

                            //Disminuye folios
                            BaseHelper::decrementFolios();

                        }
                    }
                    \DB::connection('tenant')->commit();


                } catch (\Exception $e) {
                    \DB::connection('tenant')->rollback();
                    \Log::error('(' . $employee_payroll->company->taxid . ') ' . $e->getMessage());
                    flash($e->getMessage())->error();
                    return redirect('/base/payrolls');
                }

                //Almacenamiento dropbox
                if ((int)$employee_payroll->status == EmployeePayroll::CANCEL) {
                    self::dropboxBackup($employee_payroll, false, true);
                }
            }
        }

        //Mensaje
        flash(__('general.text_form_success_cancel'))->success();

        $payroll->status = Payroll::CANCEL;
        $payroll->save();

        //Redireccion
        return redirect('/base/payrolls');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {

        $this->validate($request, [
            'name' => 'required',
            'payroll_type_id' => 'required',
            'branch_office_id' => 'required',

        ], [
            'name.*' => __('base/payroll.error_name'),
            'payroll_type_id.*' => __('base/payroll.error_payroll_type_id'),
            'branch_office_id.*' => __('base/payroll.error_branch_office_id'),
        ]);
    }

    /**
     * Crear XML y enviar a timbrar CFDI 3.3
     *
     * @param EmployeePayroll $employee_payroll
     * @return array|\CfdiUtils\Elements\Cfdi33\Concepto|float|int
     * @throws \Exception
     */
    private function cfdi33(EmployeePayroll $employee_payroll, $draft = false)
    {
        $current_entity_loader = @libxml_disable_entity_loader(false);
        try {
            //Logica
            $pac = Pac::findOrFail(setting('default_pac_id')); //PAC

            //Arreglo CFDI 3.3
            $cfdi33 = [];
            if (!empty($employee_payroll->serie)) {
                $cfdi33['Serie'] = $employee_payroll->serie;
            }
            $cfdi33['Folio'] = $employee_payroll->folio;
            $cfdi33['Fecha'] = \Date::parse($employee_payroll->date)->format('Y-m-d\TH:i:s');
            //$cfdi33['Sello']
            $cfdi33['FormaPago'] = $employee_payroll->paymentWay->code;
            $cfdi33['NoCertificado'] = $employee_payroll->company->certificate_number;
            //$cfdi33['Certificado']
            //$cfdi33['CondicionesDePago']
            $cfdi33['SubTotal'] = Helper::numberFormat($employee_payroll->amount_untaxed /*+ $employee_payroll->amount_discount*/, $employee_payroll->currency->decimal_place, false);
            if(($employee_payroll->amount_discount + $employee_payroll->amount_isr)>0) {
                $cfdi33['Descuento'] = Helper::numberFormat($employee_payroll->amount_discount + $employee_payroll->amount_isr, $employee_payroll->currency->decimal_place, false);
            }
            $cfdi33['Moneda'] = $employee_payroll->currency->code;
            //$cfdi33['TipoCambio']
            $cfdi33['Total'] = Helper::numberFormat($employee_payroll->amount_total, $employee_payroll->currency->decimal_place, false);
            $cfdi33['TipoDeComprobante'] = $employee_payroll->documentType->cfdiType->code;
            $cfdi33['MetodoPago'] = $employee_payroll->paymentMethod->code;
            $cfdi33['LugarExpedicion'] = $employee_payroll->branchOffice->postcode;
            if (!empty($employee_payroll->confirmacion)) {
                $cfdi33['Confirmacion'] = $employee_payroll->confirmacion;
            }
            //---Cfdi Relacionados
            $cfdi33_relacionados = [];
            $cfdi33_relacionado = [];
            if (!empty($employee_payroll->cfdi_relation_id)) {
                $cfdi33_relacionados['TipoRelacion'] = $employee_payroll->cfdiRelation->code;
                if ($employee_payroll->employeePayrollRelations->isNotEmpty()) {
                    foreach ($employee_payroll->employeePayrollRelations as $key => $result) {
                        $cfdi33_relacionado[$key] = [];
                        $cfdi33_relacionado[$key]['UUID'] = $result->uuid_related;
                    }
                }
            }
            //---Emisor
            $cfdi33_emisor = [];
            $cfdi33_emisor['Rfc'] = $employee_payroll->company->taxid;
            $cfdi33_emisor['Nombre'] = trim($employee_payroll->company->name);
            $cfdi33_emisor['RegimenFiscal'] = $employee_payroll->company->taxRegimen->code;
            //---Receptor
            $cfdi33_receptor = [];
            $cfdi33_receptor['Rfc'] = $employee_payroll->employee->taxid;
            $cfdi33_receptor['Nombre'] = trim($employee_payroll->employee->name);
            //$cfdi33_receptor['ResidenciaFiscal']
            //$cfdi33_receptor['NumRegIdTrib']
            $cfdi33_receptor['UsoCFDI'] = $employee_payroll->cfdiUse->code;
            //---Conceptos
            $cfdi33_conceptos = [];
            $cfdi33_conceptos_traslados = [];
            $cfdi33_conceptos_retenciones = [];
            $key = 0;
            $cfdi33_conceptos [$key]['ClaveProdServ'] = '84111505';
            //$cfdi33_conceptos [$key]['NoIdentificacion']
            $cfdi33_conceptos [$key]['Cantidad'] = '1';
            $cfdi33_conceptos [$key]['ClaveUnidad'] = 'ACT';
            //$cfdi33_conceptos [$key]['Unidad'];
            $cfdi33_conceptos [$key]['Descripcion'] = 'Pago de nómina';
            $cfdi33_conceptos [$key]['ValorUnitario'] = Helper::numberFormat($employee_payroll->amount_untaxed, 6, false);
            $cfdi33_conceptos [$key]['Importe'] = Helper::numberFormat($employee_payroll->amount_untaxed, 2, false);
            if($employee_payroll->amount_discount + $employee_payroll->amount_isr > 0){
                $cfdi33_conceptos [$key]['Descuento'] = Helper::numberFormat($employee_payroll->amount_discount + $employee_payroll->amount_isr, 2, false);
            }
            //['InformacionAduanera']
            //['CuentaPredial']
            //['ComplementoConcepto']
            //['Parte']
            //Complemento
            //Impuestos por concepto


            //Impuestos
            $cfdi33_retenciones = [];
            $cfdi33_traslados = [];

            $cfdi33_impuestos = [];
            //$cfdi33_impuestos['TotalImpuestosRetenidos']
            //$cfdi33_impuestos['TotalImpuestosTrasladados']

            //Genera XML
            $certificado = new \CfdiUtils\Certificado\Certificado(\Storage::path($employee_payroll->company->pathFileCer()));
            $creator = new \CfdiUtils\CfdiCreator33($cfdi33, $certificado);
            $creator->setXmlResolver(PacHelper::resourcePathCfdiUtils()); //Almacenamiento local
            $comprobante = $creator->comprobante();
            //Complemento de nomina
            $comprobante->addAttributes([
                'xsi:schemaLocation' => 'http://www.sat.gob.mx/cfd/3 http://www.sat.gob.mx/sitio_internet/cfd/3/cfdv33.xsd http://www.sat.gob.mx/nomina12 http://www.sat.gob.mx/sitio_internet/cfd/nomina/nomina12.xsd',
                'xmlns:nomina12' => 'http://www.sat.gob.mx/nomina12'
            ]);

            if (!empty($cfdi33_relacionados)) {
                $comprobante->addCfdiRelacionados($cfdi33_relacionados);
            }
            if (!empty($cfdi33_relacionado)) {
                foreach ($cfdi33_relacionado as $key => $result) {
                    $comprobante->addCfdiRelacionado($result);
                }
            }
            $comprobante->addEmisor($cfdi33_emisor);
            $comprobante->addReceptor($cfdi33_receptor);
            //Conceptos
            foreach ($cfdi33_conceptos as $key => $result) {
                $concepto = $comprobante->addConcepto($result);
                if (!empty($cfdi33_conceptos_traslados[$key])) {
                    foreach ($cfdi33_conceptos_traslados[$key] as $result2) {
                        $concepto->multiTraslado($result2);
                    }
                }
                if (!empty($cfdi33_conceptos_retenciones[$key])) {
                    foreach ($cfdi33_conceptos_retenciones[$key] as $result2) {
                        $concepto->multiRetencion($result2);
                    }
                }
            }
            //Impuestos
            if(!empty($cfdi33_impuestos)) {
                $comprobante->addImpuestos($cfdi33_impuestos);
                if (!empty($cfdi33_retenciones)) {
                    foreach ($cfdi33_retenciones as $result2) {
                        $comprobante->multiRetencion($result2);
                    }
                }
                if (!empty($cfdi33_traslados)) {
                    foreach ($cfdi33_traslados as $result2) {
                        $comprobante->multiTraslado($result2);
                    }
                }
            }

            //Complemento de nomina
            //Calculo de antiguedad
            $antiquity = '';
            if(!empty($employee_payroll->employee->date_start_work)){
                if(!empty($employee_payroll->date_end_payment)){
                    $date1 = \Date::parse($employee_payroll->employee->date_start_work);
                    $date2 = \Date::parse($employee_payroll->date_end_payment);
                    $diff_weeks = $date1->diffInWeeks($date2);
                    $diff_weeks = floor($diff_weeks);
                    if($diff_weeks>=1){
                        $antiquity = 'P' . $diff_weeks .'W';
                    }else{
                        $diff_weeks = $date1->diffInDays($date2);
                        $diff_weeks = floor($diff_weeks);
                        if($diff_weeks == 0){
                            $diff_weeks = 1;
                        }
                        if($diff_weeks>=1){
                            $antiquity = 'P' . $diff_weeks .'D';
                        }
                    }
                }
            }

            $nomina12 = [];
            $nomina12['xsi:schemaLocation'] = 'http://www.sat.gob.mx/nomina12 http://www.sat.gob.mx/sitio_internet/cfd/nomina/nomina12.xsd';
            $nomina12['xmlns:xsi'] = 'http://www.w3.org/2001/XMLSchema-instance';
            $nomina12['xmlns:nomina12'] = 'http://www.sat.gob.mx/nomina12';
            $nomina12['Version'] = '1.2';
            $nomina12['TipoNomina'] = $employee_payroll->payrollType->code;
            $nomina12['FechaPago'] = \Date::parse($employee_payroll->date_payment)->format('Y-m-d');
            $nomina12['FechaInicialPago'] = \Date::parse($employee_payroll->date_start_payment)->format('Y-m-d');
            $nomina12['FechaFinalPago'] = \Date::parse($employee_payroll->date_end_payment)->format('Y-m-d');
            $nomina12['NumDiasPagados'] = Helper::numberFormat($employee_payroll->payment_days, 3, false);
            if($employee_payroll->amount_perceptions>0 || $employee_payroll->employeeActivePerceptionsPayrollLines->isNotEmpty()) {
                $nomina12['TotalPercepciones'] = Helper::numberFormat($employee_payroll->amount_perceptions, $employee_payroll->currency->decimal_place, false);
            }
            if($employee_payroll->amount_deductions>0 || $employee_payroll->employeeActiveDeductionsPayrollLines->isNotEmpty()) {
                $nomina12['TotalDeducciones'] = Helper::numberFormat($employee_payroll->amount_deductions, $employee_payroll->currency->decimal_place, false);
            }
            if($employee_payroll->amount_other_payment_types>0 || $employee_payroll->employeeActiveOtherPaymentTypesPayrollLines->isNotEmpty()) {
                $nomina12['TotalOtrosPagos'] = Helper::numberFormat($employee_payroll->amount_other_payment_types, $employee_payroll->currency->decimal_place, false);
            }
            $nodo_payroll = new \CfdiUtils\Nodes\Node(
                'nomina12:Nomina', // nombre del elemento raíz
                $nomina12
            );
            //Nodo emisor
            $nomina12_emisor = [];
            if(!empty($employee_payroll->company->curp)){
                $nomina12_emisor['Curp'] = $employee_payroll->company->curp;
            }
            if(!empty($employee_payroll->company->employer_register) && !empty($employee_payroll->employee->nss) && !empty($employee_payroll->employee->date_start_work) && !empty($antiquity) && $employee_payroll->employee->sdi > 0) {
                $nomina12_emisor['RegistroPatronal'] = $employee_payroll->company->employer_register;
            }
            if(!empty($employee_payroll->taxid_origin)){
                $nomina12_emisor['RfcPatronOrigen'] = $employee_payroll->taxid_origin;
            }
            //EntidadSNCF
            $nodo_nomina12_emisor = new \CfdiUtils\Nodes\Node(
                'nomina12:Emisor', // nombre del elemento raíz
                $nomina12_emisor
            );
            $nodo_payroll->addChild($nodo_nomina12_emisor);

            //Nodo receptor
            $nomina12_receptor = [];
            $nomina12_receptor['Curp'] = $employee_payroll->employee->curp;
            if(!empty($employee_payroll->employee->nss)) {
                $nomina12_receptor['NumSeguridadSocial'] = $employee_payroll->employee->nss;
            }
            if(!empty($employee_payroll->employee->date_start_work)) {
                $nomina12_receptor['FechaInicioRelLaboral'] = \Date::parse($employee_payroll->employee->date_start_work)->format('Y-m-d');
            }
            if(!empty($antiquity)) {
                $nomina12_receptor['Antigüedad'] = $antiquity;
            }
            $nomina12_receptor['TipoContrato'] = $employee_payroll->employee->contractType->code;
            $nomina12_receptor['Sindicalizado'] = !empty($employee_payroll->employee->unionized) ? 'Sí' : 'No';
            if(!empty($employee_payroll->employee->workdayType->code)) {
                $nomina12_receptor['TipoJornada'] = $employee_payroll->employee->workdayType->code;
            }
            $nomina12_receptor['TipoRegimen'] = $employee_payroll->employee->recruitmentRegime->code;
            $nomina12_receptor['NumEmpleado'] = $employee_payroll->employee->code;
            if(!empty($employee_payroll->employee->department)) {
                $nomina12_receptor['Departamento'] = $employee_payroll->employee->department;
            }
            if(!empty($employee_payroll->employee->job)) {
                $nomina12_receptor['Puesto'] = $employee_payroll->employee->job;
            }
            if(!empty($employee_payroll->employee->jobRiskClasse->code)) {
                $nomina12_receptor['RiesgoPuesto'] = $employee_payroll->employee->jobRiskClasse->code;
            }
            $nomina12_receptor['PeriodicidadPago'] = $employee_payroll->employee->frequencyPayment->code;
            if(!empty($employee_payroll->employee->bank->code)) {
                $nomina12_receptor['Banco'] = $employee_payroll->employee->bank->code;
            }
            if(!empty($employee_payroll->employee->bank_account)) {
                $nomina12_receptor['CuentaBancaria'] = $employee_payroll->employee->bank_account;
            }
            if($employee_payroll->employee->base_salary > 0) {
                $nomina12_receptor['SalarioBaseCotApor'] = Helper::numberFormat($employee_payroll->base_salary, $employee_payroll->currency->decimal_place, false);
            }
            if($employee_payroll->employee->sdi > 0) {
                $nomina12_receptor['SalarioDiarioIntegrado'] = Helper::numberFormat($employee_payroll->sdi, $employee_payroll->currency->decimal_place, false);
            }
            $nomina12_receptor['ClaveEntFed'] = $employee_payroll->employee->state->code;
            $nodo_nomina12_receptor = new \CfdiUtils\Nodes\Node(
                'nomina12:Receptor', // nombre del elemento raíz
                $nomina12_receptor
            );

            //SubContratacion
            if($employee_payroll->employeeActivePayrollOutsourcings->isNotEmpty()) {
                foreach ($employee_payroll->employeeActivePayrollOutsourcings as $key => $result) {
                    $nomina12_subcontratacion = [];
                    $nomina12_subcontratacion['RfcLabora'] = $result->taxid;
                    $nomina12_subcontratacion['PorcentajeTiempo'] = round($result->percent_time,3);
                    $nodo_nomina12_subcontratacion = new \CfdiUtils\Nodes\Node(
                        'nomina12:SubContratacion', // nombre del elemento raíz
                        $nomina12_subcontratacion
                    );
                    $nodo_nomina12_receptor->addChild($nodo_nomina12_subcontratacion);
                }
            }

            $nodo_payroll->addChild($nodo_nomina12_receptor);

            //Nodo percepciones
            if($employee_payroll->employeeActivePerceptionsPayrollLines->isNotEmpty()) {
                $nomina12_percepciones = [];
                if($employee_payroll->employeeActivePerceptions1PayrollLines->isNotEmpty()) {
                    $nomina12_percepciones['TotalSueldos'] = Helper::numberFormat($employee_payroll->employeeActivePerceptions1PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->employeeActivePerceptions2PayrollLines->isNotEmpty()) {
                    $nomina12_percepciones['TotalSeparacionIndemnizacion'] = Helper::numberFormat($employee_payroll->employeeActivePerceptions2PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->employeeActivePerceptions3PayrollLines->isNotEmpty()) {
                    $nomina12_percepciones['TotalJubilacionPensionRetiro'] = Helper::numberFormat($employee_payroll->employeeActivePerceptions3PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                //if($employee_payroll->employeeActivePerceptionsPayrollLines->sum('amount_taxed') > 0) {
                    $nomina12_percepciones['TotalGravado'] = Helper::numberFormat($employee_payroll->employeeActivePerceptionsPayrollLines->sum('amount_taxed'),
                        $employee_payroll->currency->decimal_place, false);
                //}
                //if($employee_payroll->employeeActivePerceptionsPayrollLines->sum('amount_exempt') > 0) {
                    $nomina12_percepciones['TotalExento'] = Helper::numberFormat($employee_payroll->employeeActivePerceptionsPayrollLines->sum('amount_exempt'),
                        $employee_payroll->currency->decimal_place, false);
                //}
                $nodo_nomina12_percepciones = new \CfdiUtils\Nodes\Node(
                    'nomina12:Percepciones', // nombre del elemento raíz
                    $nomina12_percepciones
                );
                //Percepciones
                foreach ($employee_payroll->employeeActivePerceptionsPayrollLines as $key => $result) {
                    $nomina12_percepcion = [];
                    $nomina12_percepcion['TipoPercepcion'] = $result->perception->code;
                    $nomina12_percepcion['Clave'] = !empty($result->code) ? $result->code : $result->perception->code;
                    $nomina12_percepcion['Concepto'] =str_limit($result->name,99,'');
                    //if($result->amount_taxed>0) {
                        $nomina12_percepcion['ImporteGravado'] = Helper::numberFormat($result->amount_taxed,
                            $employee_payroll->currency->decimal_place, false);
                    //}
                    //if($result->amount_exempt>0) {
                        $nomina12_percepcion['ImporteExento'] = Helper::numberFormat($result->amount_exempt,
                            $employee_payroll->currency->decimal_place, false);
                    //}

                    $nodo_nomina12_percepcion = new \CfdiUtils\Nodes\Node(
                        'nomina12:Percepcion', // nombre del elemento raíz
                        $nomina12_percepcion
                    );

                    //AccionesOTitulos
                    if($result->perception->code == '045'){
                        $nomina12_acciones_o_titulos['ValorMercado'] = Helper::numberFormat($result->ac_market_value,
                            $employee_payroll->currency->decimal_place, false);
                        $nomina12_acciones_o_titulos['PrecioAlOtorgarse'] = Helper::numberFormat($result->ac_price_when_granted,
                            $employee_payroll->currency->decimal_place, false);
                        $nodo_nomina12_acciones_o_titulos = new \CfdiUtils\Nodes\Node(
                            'nomina12:AccionesOTitulos', // nombre del elemento raíz
                            $nomina12_acciones_o_titulos
                        );
                        $nodo_nomina12_percepcion->addChild($nodo_nomina12_acciones_o_titulos);
                    }
                    //HorasExtra
                    if($result->perception->code == '019'){
                        if($result->hr_days_1 > 0 && $result->hr_hours_1>0 && $result->hr_amount_paid_1 > 0) {
                            $nomina12_horas_extra['Dias'] = Helper::numberFormat($result->hr_days_1, 0, false);
                            $nomina12_horas_extra['TipoHoras'] = $result->overtimesType1->code;
                            $nomina12_horas_extra['HorasExtra'] = Helper::numberFormat($result->hr_hours_1, 0, false);
                            $nomina12_horas_extra['ImportePagado'] = Helper::numberFormat($result->hr_amount_paid_1, $employee_payroll->currency->decimal_place, false);
                            $nodo_nomina12_horas_extra = new \CfdiUtils\Nodes\Node(
                                'nomina12:HorasExtra', // nombre del elemento raíz
                                $nomina12_horas_extra
                            );
                            $nodo_nomina12_percepcion->addChild($nodo_nomina12_horas_extra);
                        }
                        if($result->hr_days_2 > 0 && $result->hr_hours_2>0 && $result->hr_amount_paid_2 > 0) {
                            $nomina12_horas_extra['Dias'] = Helper::numberFormat($result->hr_days_2, 0, false);
                            $nomina12_horas_extra['TipoHoras'] = $result->overtimesType2->code;
                            $nomina12_horas_extra['HorasExtra'] = Helper::numberFormat($result->hr_hours_2, 0, false);
                            $nomina12_horas_extra['ImportePagado'] = Helper::numberFormat($result->hr_amount_paid_2, $employee_payroll->currency->decimal_place, false);
                            $nodo_nomina12_horas_extra = new \CfdiUtils\Nodes\Node(
                                'nomina12:HorasExtra', // nombre del elemento raíz
                                $nomina12_horas_extra
                            );
                            $nodo_nomina12_percepcion->addChild($nodo_nomina12_horas_extra);
                        }
                        if($result->hr_days_3 > 0 && $result->hr_hours_3>0 && $result->hr_amount_paid_3 > 0) {
                            $nomina12_horas_extra['Dias'] = Helper::numberFormat($result->hr_days_3, 0, false);
                            $nomina12_horas_extra['TipoHoras'] = $result->overtimesType3->code;
                            $nomina12_horas_extra['HorasExtra'] = Helper::numberFormat($result->hr_hours_3, 0, false);
                            $nomina12_horas_extra['ImportePagado'] = Helper::numberFormat($result->hr_amount_paid_3, $employee_payroll->currency->decimal_place, false);
                            $nodo_nomina12_horas_extra = new \CfdiUtils\Nodes\Node(
                                'nomina12:HorasExtra', // nombre del elemento raíz
                                $nomina12_horas_extra
                            );
                            $nodo_nomina12_percepcion->addChild($nodo_nomina12_horas_extra);
                        }
                    }

                    $nodo_nomina12_percepciones->addChild($nodo_nomina12_percepcion);
                }
                //JubilacionPensionRetiro
                $nomina12_jubilacion = [];
                if($employee_payroll->rt_amount_an_exhibition > 0) {
                    $nomina12_jubilacion['TotalUnaExhibicion'] = Helper::numberFormat($employee_payroll->rt_amount_an_exhibition,
                        $employee_payroll->currency->decimal_place, false);
                }elseif($employee_payroll->rt_partiality_amount > 0 || $employee_payroll->rt_daily_amount>0){
                    $nomina12_jubilacion['TotalParcialidad'] = Helper::numberFormat($employee_payroll->rt_partiality_amount,
                        $employee_payroll->currency->decimal_place, false);
                    $nomina12_jubilacion['MontoDiario'] = Helper::numberFormat($employee_payroll->rt_daily_amount,
                        $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->rt_cumulative_income > 0) {
                    $nomina12_jubilacion['IngresoAcumulable'] = Helper::numberFormat($employee_payroll->rt_cumulative_income,
                        $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->rt_non_cumulative_income > 0) {
                    $nomina12_jubilacion['IngresoNoAcumulable'] = Helper::numberFormat($employee_payroll->rt_non_cumulative_income,
                        $employee_payroll->currency->decimal_place, false);
                }
                if(!empty($nomina12_jubilacion)) {
                    $nodo_nomina12_jubilacion = new \CfdiUtils\Nodes\Node(
                        'nomina12:JubilacionPensionRetiro', // nombre del elemento raíz
                        $nomina12_jubilacion
                    );
                    $nodo_nomina12_percepciones->addChild($nodo_nomina12_jubilacion);
                }

                //SeparacionIndemnizacion
                $nomina12_seperacion = [];
                if($employee_payroll->sp_total_amount > 0) {
                    $nomina12_seperacion['TotalPagado'] = Helper::numberFormat($employee_payroll->sp_total_amount,
                        $employee_payroll->currency->decimal_place, false);
                    $nomina12_seperacion['NumAñosServicio'] = Helper::numberFormat($employee_payroll->sp_years_of_service,
                        0, false);
                    $nomina12_seperacion['UltimoSueldoMensOrd'] = Helper::numberFormat($employee_payroll->sp_last_salary,
                        $employee_payroll->currency->decimal_place, false);
                    $nomina12_seperacion['IngresoAcumulable'] = Helper::numberFormat($employee_payroll->sp_cumulative_income,
                        $employee_payroll->currency->decimal_place, false);
                    $nomina12_seperacion['IngresoNoAcumulable'] = Helper::numberFormat($employee_payroll->sp_non_cumulative_income,
                        $employee_payroll->currency->decimal_place, false);

                    if(!empty($nomina12_seperacion)) {
                        $nodo_nomina12_separacion = new \CfdiUtils\Nodes\Node(
                            'nomina12:SeparacionIndemnizacion', // nombre del elemento raíz
                            $nomina12_seperacion
                        );
                        $nodo_nomina12_percepciones->addChild($nodo_nomina12_separacion);
                    }
                }

                $nodo_payroll->addChild($nodo_nomina12_percepciones);
            }

            //Nodo deducciones
            if($employee_payroll->employeeActiveDeductionsPayrollLines->isNotEmpty()) {
                $nomina12_deducciones = [];
                if($employee_payroll->employeeActiveDeductions1PayrollLines->isNotEmpty()) {
                    $nomina12_deducciones['TotalOtrasDeducciones'] = Helper::numberFormat($employee_payroll->employeeActiveDeductions1PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->employeeActiveDeductions2PayrollLines->isNotEmpty()) {
                    $nomina12_deducciones['TotalImpuestosRetenidos'] = Helper::numberFormat($employee_payroll->employeeActiveDeductions2PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                $nodo_nomina12_deducciones = new \CfdiUtils\Nodes\Node(
                    'nomina12:Deducciones', // nombre del elemento raíz
                    $nomina12_deducciones
                );
                //Deducciones
                foreach ($employee_payroll->employeeActiveDeductionsPayrollLines as $key => $result) {
                    $nomina12_deduccion = [];
                    $nomina12_deduccion['TipoDeduccion'] = $result->deduction->code;
                    $nomina12_deduccion['Clave'] = !empty($result->code) ? $result->code : $result->deduction->code;
                    $nomina12_deduccion['Concepto'] =str_limit($result->name,99,'');
                    $nomina12_deduccion['Importe'] = Helper::numberFormat($result->amount_total,
                            $employee_payroll->currency->decimal_place, false);
                    $nodo_nomina12_deduccion = new \CfdiUtils\Nodes\Node(
                        'nomina12:Deduccion', // nombre del elemento raíz
                        $nomina12_deduccion
                    );
                    $nodo_nomina12_deducciones->addChild($nodo_nomina12_deduccion);
                }

                $nodo_payroll->addChild($nodo_nomina12_deducciones);
            }

            //Nodo Otros pagos
            if($employee_payroll->employeeActiveOtherPaymentTypesPayrollLines->isNotEmpty()) {
                $nomina12_otros_pagos = [];
                $nodo_nomina12_otros_pagos = new \CfdiUtils\Nodes\Node(
                    'nomina12:OtrosPagos', // nombre del elemento raíz
                    $nomina12_otros_pagos
                );
                //Otros pagos
                foreach ($employee_payroll->employeeActiveOtherPaymentTypesPayrollLines as $key => $result) {
                    $nomina12_otro_pago = [];
                    $nomina12_otro_pago['TipoOtroPago'] = $result->otherPaymentType->code;
                    $nomina12_otro_pago['Clave'] = !empty($result->code) ? $result->code : $result->otherPaymentType->code;
                    $nomina12_otro_pago['Concepto'] =str_limit($result->name,99,'');
                    $nomina12_otro_pago['Importe'] = Helper::numberFormat($result->amount_total,
                        $employee_payroll->currency->decimal_place, false);
                    $nodo_nomina12_otro_pago = new \CfdiUtils\Nodes\Node(
                        'nomina12:OtroPago', // nombre del elemento raíz
                        $nomina12_otro_pago
                    );
                    //SubsidioAlEmpleo
                    if($result->otherPaymentType->code == '002') {
                        $nomina12_otros_pago_subsidio_empleo = [];
                        $nomina12_otros_pago_subsidio_empleo['SubsidioCausado'] = Helper::numberFormat($result->amount_caused, $employee_payroll->currency->decimal_place, false);
                        $nodo_nomina12_otro_subsidio_empleo = new \CfdiUtils\Nodes\Node(
                            'nomina12:SubsidioAlEmpleo', // nombre del elemento raíz
                            $nomina12_otros_pago_subsidio_empleo
                        );
                        $nodo_nomina12_otro_pago->addChild($nodo_nomina12_otro_subsidio_empleo);
                    }
                    //CompensacionSaldosAFavor
                    if($result->otherPaymentType->code == '004'){
                        $nomina12_otros_pago_compensacion['SaldoAFavor'] = Helper::numberFormat($result->cm_positive_balance, $employee_payroll->currency->decimal_place, false);
                        $nomina12_otros_pago_compensacion['Año'] = Helper::numberFormat($result->cm_year, 0, false);
                        $nomina12_otros_pago_compensacion['RemanenteSalFav'] = Helper::numberFormat($result->cm_remainder, $employee_payroll->currency->decimal_place, false);
                        $nodo_nomina12_otros_pago_compensacion = new \CfdiUtils\Nodes\Node(
                            'nomina12:CompensacionSaldosAFavor', // nombre del elemento raíz
                            $nomina12_otros_pago_compensacion
                        );
                        $nodo_nomina12_otro_pago->addChild($nodo_nomina12_otros_pago_compensacion);
                    }

                    $nodo_nomina12_otros_pagos->addChild($nodo_nomina12_otro_pago);
                }

                $nodo_payroll->addChild($nodo_nomina12_otros_pagos);
            }

            //Incapacidades
            if($employee_payroll->employeeActiveDisabilitiesPayrollLines->isNotEmpty()) {
                $nomina12_incapacidades = [];
                $nodo_nomina12_incapacidades = new \CfdiUtils\Nodes\Node(
                    'nomina12:Incapacidades', // nombre del elemento raíz
                    $nomina12_incapacidades
                );
                //Deducciones
                foreach ($employee_payroll->employeeActiveDisabilitiesPayrollLines as $key => $result) {
                    $nomina12_incapacidad = [];
                    $nomina12_incapacidad['DiasIncapacidad'] = Helper::numberFormat($result->disability_days, 0, false);
                    $nomina12_incapacidad['TipoIncapacidad'] = $result->disabilityType->code;
                    $nomina12_incapacidad['ImporteMonetario'] = Helper::numberFormat($result->disability_amount,
                        $employee_payroll->currency->decimal_place, false);
                    $nodo_nomina12_incapacidad = new \CfdiUtils\Nodes\Node(
                        'nomina12:Incapacidad', // nombre del elemento raíz
                        $nomina12_incapacidad
                    );
                    $nodo_nomina12_incapacidades->addChild($nodo_nomina12_incapacidad);
                }

                $nodo_payroll->addChild($nodo_nomina12_incapacidades);
            }

            //Agregamos al complemento
            $comprobante->addComplemento($nodo_payroll);


            //Método de ayuda para establecer las sumas del comprobante e impuestos con base en la suma de los conceptos y la agrupación de sus impuestos
            //$creator->addSumasConceptos(null, 2);
            //Método de ayuda para generar el sello (obtener la cadena de origen y firmar con la llave privada)
            $creator->addSello('file://' . \Storage::path($employee_payroll->company->pathFileKeyPassPem()), Crypt::decryptString($employee_payroll->company->password_key));
            //Valida la estructura
            //$creator->validate();

            //Guarda XML
            //dd($creator->asXml());
            $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
            $file_xml = Helper::makeDirectoryCfdi($path_xml) . '/' . (!$draft ? Str::random(40) : str_replace('/','',$employee_payroll->name)) . '.xml';
            $creator->saveXml(\Storage::path($path_xml . $file_xml));


            //Arreglo temporal para actualizar Customer Invoice CFDI
            $tmp = [
                'pac_id' => $pac->id,
                'cfdi_version' => setting('cfdi_version'),
                'uuid' => '',
                'date' => null,
                'path_xml' => $path_xml,
                'file_xml' => $file_xml,
                'file_xml_pac' => '',
                'pac' => $pac,
            ];

            //Timbrado de XML
            if(!$draft) {
                $class_pac = $pac->code;
                $tmp = PacHelper::$class_pac($tmp, $creator);
            }

            return $tmp;
        } catch (\Exception $e) {
            throw $e;
        }

        @libxml_disable_entity_loader($current_entity_loader);
    }

    /**
     * Crear XML y enviar a timbrar CFDI 4.0
     *
     * @param EmployeePayroll $employee_payroll
     * @return array|\CfdiUtils\Elements\Cfdi33\Concepto|float|int
     * @throws \Exception
     */
    private function cfdi40(EmployeePayroll $employee_payroll, $draft = false)
    {
        $current_entity_loader = @libxml_disable_entity_loader(false);
        try {
            //Logica
            $pac = Pac::findOrFail(setting('default_pac_id')); //PAC

            //Arreglo CFDI 3.3
            $cfdi33 = [];
            if (!empty($employee_payroll->serie)) {
                $cfdi33['Serie'] = $employee_payroll->serie;
            }
            $cfdi33['Folio'] = $employee_payroll->folio;
            $cfdi33['Fecha'] = \Date::parse($employee_payroll->date)->format('Y-m-d\TH:i:s');
            //$cfdi33['Sello']
            //$cfdi33['FormaPago'] = $employee_payroll->paymentWay->code;
            $cfdi33['NoCertificado'] = $employee_payroll->company->certificate_number;
            //$cfdi33['Certificado']
            //$cfdi33['CondicionesDePago']
            $cfdi33['SubTotal'] = Helper::numberFormat($employee_payroll->amount_untaxed /*+ $employee_payroll->amount_discount*/, $employee_payroll->currency->decimal_place, false);
            if(($employee_payroll->amount_discount + $employee_payroll->amount_isr)>0) {
                $cfdi33['Descuento'] = Helper::numberFormat($employee_payroll->amount_discount + $employee_payroll->amount_isr, $employee_payroll->currency->decimal_place, false);
            }
            $cfdi33['Moneda'] = $employee_payroll->currency->code;
            //$cfdi33['TipoCambio']
            $cfdi33['Total'] = Helper::numberFormat($employee_payroll->amount_total, $employee_payroll->currency->decimal_place, false);
            $cfdi33['TipoDeComprobante'] = $employee_payroll->documentType->cfdiType->code;
            $cfdi33['Exportacion'] = '01';
            $cfdi33['MetodoPago'] = $employee_payroll->paymentMethod->code;
            $cfdi33['LugarExpedicion'] = $employee_payroll->branchOffice->postcode;
            if (!empty($employee_payroll->confirmacion)) {
                $cfdi33['Confirmacion'] = $employee_payroll->confirmacion;
            }
            //---Cfdi Relacionados
            $cfdi33_relacionados = [];
            $cfdi33_relacionado = [];
            if (!empty($employee_payroll->cfdi_relation_id)) {
                $cfdi33_relacionados['TipoRelacion'] = $employee_payroll->cfdiRelation->code;
                if ($employee_payroll->employeePayrollRelations->isNotEmpty()) {
                    foreach ($employee_payroll->employeePayrollRelations as $key => $result) {
                        $cfdi33_relacionado[$key] = [];
                        $cfdi33_relacionado[$key]['UUID'] = $result->uuid_related;
                    }
                }
            }
            //---Emisor
            $cfdi33_emisor = [];
            $cfdi33_emisor['Rfc'] = $employee_payroll->company->taxid;
            $cfdi33_emisor['Nombre'] = trim($employee_payroll->company->name);
            $cfdi33_emisor['RegimenFiscal'] = !empty($employee_payroll->tax_regimen_id) ? $employee_payroll->taxRegimen->code : $employee_payroll->company->taxRegimen->code;
            $cfdi33_emisor['FacAtrAdquirente'] = null;
            //---Receptor
            $cfdi33_receptor = [];
            $cfdi33_receptor['Rfc'] = $employee_payroll->employee->taxid;
            $cfdi33_receptor['Nombre'] = trim($employee_payroll->employee->name);
            $cfdi33_receptor['DomicilioFiscalReceptor'] = $employee_payroll->employee->postcode;
            //$cfdi33_receptor['ResidenciaFiscal']
            //$cfdi33_receptor['NumRegIdTrib']
            $cfdi33_receptor['RegimenFiscalReceptor'] = !empty($employee_payroll->tax_regimen_employee_id) ? $employee_payroll->taxRegimenEmployee->code : $employee_payroll->employee->taxRegimen->code;
            $cfdi33_receptor['UsoCFDI'] = $employee_payroll->cfdiUse->code;
            //---Conceptos
            $cfdi33_conceptos = [];
            $cfdi33_conceptos_traslados = [];
            $cfdi33_conceptos_retenciones = [];
            $key = 0;
            $cfdi33_conceptos [$key]['ClaveProdServ'] = '84111505';
            //$cfdi33_conceptos [$key]['NoIdentificacion']
            $cfdi33_conceptos [$key]['Cantidad'] = '1';
            $cfdi33_conceptos [$key]['ClaveUnidad'] = 'ACT';
            //$cfdi33_conceptos [$key]['Unidad'];
            $cfdi33_conceptos [$key]['Descripcion'] = 'Pago de nómina';
            $cfdi33_conceptos [$key]['ValorUnitario'] = Helper::numberFormat($employee_payroll->amount_untaxed, 6, false);
            $cfdi33_conceptos [$key]['Importe'] = Helper::numberFormat($employee_payroll->amount_untaxed, 2, false);
            if($employee_payroll->amount_discount + $employee_payroll->amount_isr > 0){
                $cfdi33_conceptos [$key]['Descuento'] = Helper::numberFormat($employee_payroll->amount_discount + $employee_payroll->amount_isr, 2, false);
            }
            //['InformacionAduanera']
            //['CuentaPredial']
            //['ComplementoConcepto']
            //['Parte']
            //Complemento
            //Impuestos por concepto
            $cfdi33_conceptos [$key]['ObjetoImp'] = '01';

            //Impuestos
            $cfdi33_retenciones = [];
            $cfdi33_traslados = [];

            $cfdi33_impuestos = [];
            //$cfdi33_impuestos['TotalImpuestosRetenidos']
            //$cfdi33_impuestos['TotalImpuestosTrasladados']

            //Genera XML
            $certificado = new \CfdiUtils\Certificado\Certificado(\Storage::path($employee_payroll->company->pathFileCer()));
            $creator = new \CfdiUtils\CfdiCreator40($cfdi33, $certificado);
            $creator->setXmlResolver(PacHelper::resourcePathCfdiUtils()); //Almacenamiento local
            $comprobante = $creator->comprobante();
            //Complemento de nomina
            $comprobante->addAttributes([
                'xsi:schemaLocation' => 'http://www.sat.gob.mx/cfd/4 http://www.sat.gob.mx/sitio_internet/cfd/4/cfdv40.xsd http://www.sat.gob.mx/nomina12 http://www.sat.gob.mx/sitio_internet/cfd/nomina/nomina12.xsd',
                'xmlns:nomina12' => 'http://www.sat.gob.mx/nomina12'
            ]);

            if (!empty($cfdi33_relacionados)) {
                $relacionados = $comprobante->addCfdiRelacionados($cfdi33_relacionados);
                if (!empty($cfdi33_relacionado)) {
                    foreach ($cfdi33_relacionado as $key => $result) {
                        $relacionados->addCfdiRelacionado($result);
                    }
                }
            }
            $comprobante->addEmisor($cfdi33_emisor);
            $comprobante->addReceptor($cfdi33_receptor);
            //Conceptos
            foreach ($cfdi33_conceptos as $key => $result) {
                $concepto = $comprobante->addConcepto($result);
                if (!empty($cfdi33_conceptos_traslados[$key])) {
                    foreach ($cfdi33_conceptos_traslados[$key] as $result2) {
                        $concepto->multiTraslado($result2);
                    }
                }
                if (!empty($cfdi33_conceptos_retenciones[$key])) {
                    foreach ($cfdi33_conceptos_retenciones[$key] as $result2) {
                        $concepto->multiRetencion($result2);
                    }
                }
            }
            //Impuestos
            if(!empty($cfdi33_impuestos)) {
                $comprobante->addImpuestos($cfdi33_impuestos);
                if (!empty($cfdi33_retenciones)) {
                    foreach ($cfdi33_retenciones as $result2) {
                        $comprobante->multiRetencion($result2);
                    }
                }
                if (!empty($cfdi33_traslados)) {
                    foreach ($cfdi33_traslados as $result2) {
                        $comprobante->multiTraslado($result2);
                    }
                }
            }

            //Complemento de nomina
            //Calculo de antiguedad
            $antiquity = '';
            if(!empty($employee_payroll->employee->date_start_work)){
                if(!empty($employee_payroll->date_end_payment)){
                    $date1 = \Date::parse($employee_payroll->employee->date_start_work);
                    $date2 = \Date::parse($employee_payroll->date_end_payment);
                    $diff_weeks = $date1->diffInWeeks($date2);
                    $diff_weeks = floor($diff_weeks);
                    if($diff_weeks>=1){
                        $antiquity = 'P' . $diff_weeks .'W';
                    }else{
                        $diff_weeks = $date1->diffInDays($date2);
                        $diff_weeks = floor($diff_weeks);
                        if($diff_weeks == 0){
                            $diff_weeks = 1;
                        }
                        if($diff_weeks>=1){
                            $antiquity = 'P' . $diff_weeks .'D';
                        }
                    }
                }
            }

            $nomina12 = [];
            $nomina12['xsi:schemaLocation'] = 'http://www.sat.gob.mx/nomina12 http://www.sat.gob.mx/sitio_internet/cfd/nomina/nomina12.xsd';
            $nomina12['xmlns:xsi'] = 'http://www.w3.org/2001/XMLSchema-instance';
            $nomina12['xmlns:nomina12'] = 'http://www.sat.gob.mx/nomina12';
            $nomina12['Version'] = '1.2';
            $nomina12['TipoNomina'] = $employee_payroll->payrollType->code;
            $nomina12['FechaPago'] = \Date::parse($employee_payroll->date_payment)->format('Y-m-d');
            $nomina12['FechaInicialPago'] = \Date::parse($employee_payroll->date_start_payment)->format('Y-m-d');
            $nomina12['FechaFinalPago'] = \Date::parse($employee_payroll->date_end_payment)->format('Y-m-d');
            $nomina12['NumDiasPagados'] = Helper::numberFormat($employee_payroll->payment_days, 3, false);
            if($employee_payroll->amount_perceptions>0 || $employee_payroll->employeeActivePerceptionsPayrollLines->isNotEmpty()) {
                $nomina12['TotalPercepciones'] = Helper::numberFormat($employee_payroll->amount_perceptions, $employee_payroll->currency->decimal_place, false);
            }
            if($employee_payroll->amount_deductions>0 || $employee_payroll->employeeActiveDeductionsPayrollLines->isNotEmpty()) {
                $nomina12['TotalDeducciones'] = Helper::numberFormat($employee_payroll->amount_deductions, $employee_payroll->currency->decimal_place, false);
            }
            if($employee_payroll->amount_other_payment_types>0 || $employee_payroll->employeeActiveOtherPaymentTypesPayrollLines->isNotEmpty()) {
                $nomina12['TotalOtrosPagos'] = Helper::numberFormat($employee_payroll->amount_other_payment_types, $employee_payroll->currency->decimal_place, false);
            }
            $nodo_payroll = new \CfdiUtils\Nodes\Node(
                'nomina12:Nomina', // nombre del elemento raíz
                $nomina12
            );
            //Nodo emisor
            $nomina12_emisor = [];
            if(!empty($employee_payroll->company->curp)){
                $nomina12_emisor['Curp'] = $employee_payroll->company->curp;
            }

            if(!empty($employee_payroll->company->employer_register) && !empty($employee_payroll->employee->nss) && !empty($employee_payroll->employee->date_start_work) && !empty($antiquity) && $employee_payroll->employee->sdi > 0) {
                $nomina12_emisor['RegistroPatronal'] = $employee_payroll->company->employer_register;
            }
            if(!empty($employee_payroll->taxid_origin)){
                $nomina12_emisor['RfcPatronOrigen'] = $employee_payroll->taxid_origin;
            }
            $nodo_nomina12_emisor = new \CfdiUtils\Nodes\Node(
                'nomina12:Emisor', // nombre del elemento raíz
                $nomina12_emisor
            );

            //EntidadSNCF

            $nodo_payroll->addChild($nodo_nomina12_emisor);

            //Nodo receptor
            $nomina12_receptor = [];
            $nomina12_receptor['Curp'] = $employee_payroll->employee->curp;
            if(!empty($employee_payroll->employee->nss)) {
                $nomina12_receptor['NumSeguridadSocial'] = $employee_payroll->employee->nss;
            }
            if(!empty($employee_payroll->employee->date_start_work)) {
                $nomina12_receptor['FechaInicioRelLaboral'] = \Date::parse($employee_payroll->employee->date_start_work)->format('Y-m-d');
            }
            if(!empty($antiquity)) {
                $nomina12_receptor['Antigüedad'] = $antiquity;
            }
            $nomina12_receptor['TipoContrato'] = $employee_payroll->employee->contractType->code;
            $nomina12_receptor['Sindicalizado'] = !empty($employee_payroll->employee->unionized) ? 'Sí' : 'No';
            if(!empty($employee_payroll->employee->workdayType->code)) {
                $nomina12_receptor['TipoJornada'] = $employee_payroll->employee->workdayType->code;
            }
            $nomina12_receptor['TipoRegimen'] = $employee_payroll->employee->recruitmentRegime->code;
            $nomina12_receptor['NumEmpleado'] = $employee_payroll->employee->code;
            if(!empty($employee_payroll->employee->department)) {
                $nomina12_receptor['Departamento'] = $employee_payroll->employee->department;
            }
            if(!empty($employee_payroll->employee->job)) {
                $nomina12_receptor['Puesto'] = $employee_payroll->employee->job;
            }
            if(!empty($employee_payroll->employee->jobRiskClasse->code)) {
                $nomina12_receptor['RiesgoPuesto'] = $employee_payroll->employee->jobRiskClasse->code;
            }
            $nomina12_receptor['PeriodicidadPago'] = $employee_payroll->employee->frequencyPayment->code;
            if(!empty($employee_payroll->employee->bank->code)) {
                $nomina12_receptor['Banco'] = $employee_payroll->employee->bank->code;
            }
            if(!empty($employee_payroll->employee->bank_account)) {
                $nomina12_receptor['CuentaBancaria'] = $employee_payroll->employee->bank_account;
            }
            if($employee_payroll->employee->base_salary > 0) {
                $nomina12_receptor['SalarioBaseCotApor'] = Helper::numberFormat($employee_payroll->base_salary, $employee_payroll->currency->decimal_place, false);
            }
            if($employee_payroll->employee->sdi > 0) {
                $nomina12_receptor['SalarioDiarioIntegrado'] = Helper::numberFormat($employee_payroll->sdi, $employee_payroll->currency->decimal_place, false);
            }
            $nomina12_receptor['ClaveEntFed'] = $employee_payroll->employee->state->code;
            $nodo_nomina12_receptor = new \CfdiUtils\Nodes\Node(
                'nomina12:Receptor', // nombre del elemento raíz
                $nomina12_receptor
            );

            //SubContratacion
            if($employee_payroll->employeeActivePayrollOutsourcings->isNotEmpty()) {
                foreach ($employee_payroll->employeeActivePayrollOutsourcings as $key => $result) {
                    $nomina12_subcontratacion = [];
                    $nomina12_subcontratacion['RfcLabora'] = $result->taxid;
                    $nomina12_subcontratacion['PorcentajeTiempo'] = round($result->percent_time,3);
                    $nodo_nomina12_subcontratacion = new \CfdiUtils\Nodes\Node(
                        'nomina12:SubContratacion', // nombre del elemento raíz
                        $nomina12_subcontratacion
                    );
                    $nodo_nomina12_receptor->addChild($nodo_nomina12_subcontratacion);
                }
            }

            $nodo_payroll->addChild($nodo_nomina12_receptor);

            //Nodo percepciones
            if($employee_payroll->employeeActivePerceptionsPayrollLines->isNotEmpty()) {
                $nomina12_percepciones = [];
                if($employee_payroll->employeeActivePerceptions1PayrollLines->isNotEmpty()) {
                    $nomina12_percepciones['TotalSueldos'] = Helper::numberFormat($employee_payroll->employeeActivePerceptions1PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->employeeActivePerceptions2PayrollLines->isNotEmpty()) {
                    $nomina12_percepciones['TotalSeparacionIndemnizacion'] = Helper::numberFormat($employee_payroll->employeeActivePerceptions2PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->employeeActivePerceptions3PayrollLines->isNotEmpty()) {
                    $nomina12_percepciones['TotalJubilacionPensionRetiro'] = Helper::numberFormat($employee_payroll->employeeActivePerceptions3PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                //if($employee_payroll->employeeActivePerceptionsPayrollLines->sum('amount_taxed') > 0) {
                    $nomina12_percepciones['TotalGravado'] = Helper::numberFormat($employee_payroll->employeeActivePerceptionsPayrollLines->sum('amount_taxed'),
                        $employee_payroll->currency->decimal_place, false);
                //}
                //if($employee_payroll->employeeActivePerceptionsPayrollLines->sum('amount_exempt') > 0) {
                    $nomina12_percepciones['TotalExento'] = Helper::numberFormat($employee_payroll->employeeActivePerceptionsPayrollLines->sum('amount_exempt'),
                        $employee_payroll->currency->decimal_place, false);
                //}
                $nodo_nomina12_percepciones = new \CfdiUtils\Nodes\Node(
                    'nomina12:Percepciones', // nombre del elemento raíz
                    $nomina12_percepciones
                );
                //Percepciones
                foreach ($employee_payroll->employeeActivePerceptionsPayrollLines as $key => $result) {
                    $nomina12_percepcion = [];
                    $nomina12_percepcion['TipoPercepcion'] = $result->perception->code;
                    $nomina12_percepcion['Clave'] = !empty($result->code) ? $result->code : $result->perception->code;
                    $nomina12_percepcion['Concepto'] =str_limit($result->name,99,'');
                    //if($result->amount_taxed>0) {
                        $nomina12_percepcion['ImporteGravado'] = Helper::numberFormat($result->amount_taxed,
                            $employee_payroll->currency->decimal_place, false);
                    //}
                    //if($result->amount_exempt>0) {
                        $nomina12_percepcion['ImporteExento'] = Helper::numberFormat($result->amount_exempt,
                            $employee_payroll->currency->decimal_place, false);
                    //}

                    $nodo_nomina12_percepcion = new \CfdiUtils\Nodes\Node(
                        'nomina12:Percepcion', // nombre del elemento raíz
                        $nomina12_percepcion
                    );

                    //AccionesOTitulos
                    if($result->perception->code == '045'){
                        $nomina12_acciones_o_titulos['ValorMercado'] = Helper::numberFormat($result->ac_market_value,
                            $employee_payroll->currency->decimal_place, false);
                        $nomina12_acciones_o_titulos['PrecioAlOtorgarse'] = Helper::numberFormat($result->ac_price_when_granted,
                            $employee_payroll->currency->decimal_place, false);
                        $nodo_nomina12_acciones_o_titulos = new \CfdiUtils\Nodes\Node(
                            'nomina12:AccionesOTitulos', // nombre del elemento raíz
                            $nomina12_acciones_o_titulos
                        );
                        $nodo_nomina12_percepcion->addChild($nodo_nomina12_acciones_o_titulos);
                    }
                    //HorasExtra
                    if($result->perception->code == '019'){
                        if($result->hr_days_1 > 0 && $result->hr_hours_1>0 && $result->hr_amount_paid_1 > 0) {
                            $nomina12_horas_extra['Dias'] = Helper::numberFormat($result->hr_days_1, 0, false);
                            $nomina12_horas_extra['TipoHoras'] = $result->overtimesType1->code;
                            $nomina12_horas_extra['HorasExtra'] = Helper::numberFormat($result->hr_hours_1, 0, false);
                            $nomina12_horas_extra['ImportePagado'] = Helper::numberFormat($result->hr_amount_paid_1, $employee_payroll->currency->decimal_place, false);
                            $nodo_nomina12_horas_extra = new \CfdiUtils\Nodes\Node(
                                'nomina12:HorasExtra', // nombre del elemento raíz
                                $nomina12_horas_extra
                            );
                            $nodo_nomina12_percepcion->addChild($nodo_nomina12_horas_extra);
                        }
                        if($result->hr_days_2 > 0 && $result->hr_hours_2>0 && $result->hr_amount_paid_2 > 0) {
                            $nomina12_horas_extra['Dias'] = Helper::numberFormat($result->hr_days_2, 0, false);
                            $nomina12_horas_extra['TipoHoras'] = $result->overtimesType2->code;
                            $nomina12_horas_extra['HorasExtra'] = Helper::numberFormat($result->hr_hours_2, 0, false);
                            $nomina12_horas_extra['ImportePagado'] = Helper::numberFormat($result->hr_amount_paid_2, $employee_payroll->currency->decimal_place, false);
                            $nodo_nomina12_horas_extra = new \CfdiUtils\Nodes\Node(
                                'nomina12:HorasExtra', // nombre del elemento raíz
                                $nomina12_horas_extra
                            );
                            $nodo_nomina12_percepcion->addChild($nodo_nomina12_horas_extra);
                        }
                        if($result->hr_days_3 > 0 && $result->hr_hours_3>0 && $result->hr_amount_paid_3 > 0) {
                            $nomina12_horas_extra['Dias'] = Helper::numberFormat($result->hr_days_3, 0, false);
                            $nomina12_horas_extra['TipoHoras'] = $result->overtimesType3->code;
                            $nomina12_horas_extra['HorasExtra'] = Helper::numberFormat($result->hr_hours_3, 0, false);
                            $nomina12_horas_extra['ImportePagado'] = Helper::numberFormat($result->hr_amount_paid_3, $employee_payroll->currency->decimal_place, false);
                            $nodo_nomina12_horas_extra = new \CfdiUtils\Nodes\Node(
                                'nomina12:HorasExtra', // nombre del elemento raíz
                                $nomina12_horas_extra
                            );
                            $nodo_nomina12_percepcion->addChild($nodo_nomina12_horas_extra);
                        }
                    }

                    $nodo_nomina12_percepciones->addChild($nodo_nomina12_percepcion);
                }
                //JubilacionPensionRetiro
                $nomina12_jubilacion = [];
                if($employee_payroll->rt_amount_an_exhibition > 0) {
                    $nomina12_jubilacion['TotalUnaExhibicion'] = Helper::numberFormat($employee_payroll->rt_amount_an_exhibition,
                        $employee_payroll->currency->decimal_place, false);
                }elseif($employee_payroll->rt_partiality_amount > 0 || $employee_payroll->rt_daily_amount>0){
                    $nomina12_jubilacion['TotalParcialidad'] = Helper::numberFormat($employee_payroll->rt_partiality_amount,
                        $employee_payroll->currency->decimal_place, false);
                    $nomina12_jubilacion['MontoDiario'] = Helper::numberFormat($employee_payroll->rt_daily_amount,
                        $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->rt_cumulative_income > 0) {
                    $nomina12_jubilacion['IngresoAcumulable'] = Helper::numberFormat($employee_payroll->rt_cumulative_income,
                        $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->rt_non_cumulative_income > 0) {
                    $nomina12_jubilacion['IngresoNoAcumulable'] = Helper::numberFormat($employee_payroll->rt_non_cumulative_income,
                        $employee_payroll->currency->decimal_place, false);
                }
                if(!empty($nomina12_jubilacion)) {
                    $nodo_nomina12_jubilacion = new \CfdiUtils\Nodes\Node(
                        'nomina12:JubilacionPensionRetiro', // nombre del elemento raíz
                        $nomina12_jubilacion
                    );
                    $nodo_nomina12_percepciones->addChild($nodo_nomina12_jubilacion);
                }

                //SeparacionIndemnizacion
                $nomina12_seperacion = [];
                if($employee_payroll->sp_total_amount > 0) {
                    $nomina12_seperacion['TotalPagado'] = Helper::numberFormat($employee_payroll->sp_total_amount,
                        $employee_payroll->currency->decimal_place, false);
                    $nomina12_seperacion['NumAñosServicio'] = Helper::numberFormat($employee_payroll->sp_years_of_service,
                        0, false);
                    $nomina12_seperacion['UltimoSueldoMensOrd'] = Helper::numberFormat($employee_payroll->sp_last_salary,
                        $employee_payroll->currency->decimal_place, false);
                    $nomina12_seperacion['IngresoAcumulable'] = Helper::numberFormat($employee_payroll->sp_cumulative_income,
                        $employee_payroll->currency->decimal_place, false);
                    $nomina12_seperacion['IngresoNoAcumulable'] = Helper::numberFormat($employee_payroll->sp_non_cumulative_income,
                        $employee_payroll->currency->decimal_place, false);

                    if(!empty($nomina12_seperacion)) {
                        $nodo_nomina12_separacion = new \CfdiUtils\Nodes\Node(
                            'nomina12:SeparacionIndemnizacion', // nombre del elemento raíz
                            $nomina12_seperacion
                        );
                        $nodo_nomina12_percepciones->addChild($nodo_nomina12_separacion);
                    }
                }

                $nodo_payroll->addChild($nodo_nomina12_percepciones);
            }

            //Nodo deducciones
            if($employee_payroll->employeeActiveDeductionsPayrollLines->isNotEmpty()) {
                $nomina12_deducciones = [];
                if($employee_payroll->employeeActiveDeductions1PayrollLines->isNotEmpty()) {
                    $nomina12_deducciones['TotalOtrasDeducciones'] = Helper::numberFormat($employee_payroll->employeeActiveDeductions1PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                if($employee_payroll->employeeActiveDeductions2PayrollLines->isNotEmpty()) {
                    $nomina12_deducciones['TotalImpuestosRetenidos'] = Helper::numberFormat($employee_payroll->employeeActiveDeductions2PayrollLines->sum('amount_total'), $employee_payroll->currency->decimal_place, false);
                }
                $nodo_nomina12_deducciones = new \CfdiUtils\Nodes\Node(
                    'nomina12:Deducciones', // nombre del elemento raíz
                    $nomina12_deducciones
                );
                //Deducciones
                foreach ($employee_payroll->employeeActiveDeductionsPayrollLines as $key => $result) {
                    $nomina12_deduccion = [];
                    $nomina12_deduccion['TipoDeduccion'] = $result->deduction->code;
                    $nomina12_deduccion['Clave'] = !empty($result->code) ? $result->code : $result->deduction->code;
                    $nomina12_deduccion['Concepto'] =str_limit($result->name,99,'');
                    $nomina12_deduccion['Importe'] = Helper::numberFormat($result->amount_total,
                            $employee_payroll->currency->decimal_place, false);
                    $nodo_nomina12_deduccion = new \CfdiUtils\Nodes\Node(
                        'nomina12:Deduccion', // nombre del elemento raíz
                        $nomina12_deduccion
                    );
                    $nodo_nomina12_deducciones->addChild($nodo_nomina12_deduccion);
                }

                $nodo_payroll->addChild($nodo_nomina12_deducciones);
            }

            //Nodo Otros pagos
            if($employee_payroll->employeeActiveOtherPaymentTypesPayrollLines->isNotEmpty()) {
                $nomina12_otros_pagos = [];
                $nodo_nomina12_otros_pagos = new \CfdiUtils\Nodes\Node(
                    'nomina12:OtrosPagos', // nombre del elemento raíz
                    $nomina12_otros_pagos
                );
                //Otros pagos
                foreach ($employee_payroll->employeeActiveOtherPaymentTypesPayrollLines as $key => $result) {
                    $nomina12_otro_pago = [];
                    $nomina12_otro_pago['TipoOtroPago'] = $result->otherPaymentType->code;
                    $nomina12_otro_pago['Clave'] = !empty($result->code) ? $result->code : $result->otherPaymentType->code;
                    $nomina12_otro_pago['Concepto'] =str_limit($result->name,99,'');
                    $nomina12_otro_pago['Importe'] = Helper::numberFormat($result->amount_total,
                        $employee_payroll->currency->decimal_place, false);
                    $nodo_nomina12_otro_pago = new \CfdiUtils\Nodes\Node(
                        'nomina12:OtroPago', // nombre del elemento raíz
                        $nomina12_otro_pago
                    );
                    //SubsidioAlEmpleo
                    if($result->otherPaymentType->code == '002') {
                        $nomina12_otros_pago_subsidio_empleo = [];
                        $nomina12_otros_pago_subsidio_empleo['SubsidioCausado'] = Helper::numberFormat($result->amount_caused, $employee_payroll->currency->decimal_place, false);
                        $nodo_nomina12_otro_subsidio_empleo = new \CfdiUtils\Nodes\Node(
                            'nomina12:SubsidioAlEmpleo', // nombre del elemento raíz
                            $nomina12_otros_pago_subsidio_empleo
                        );
                        $nodo_nomina12_otro_pago->addChild($nodo_nomina12_otro_subsidio_empleo);
                    }
                    //CompensacionSaldosAFavor
                    if($result->otherPaymentType->code == '004'){
                        $nomina12_otros_pago_compensacion['SaldoAFavor'] = Helper::numberFormat($result->cm_positive_balance, $employee_payroll->currency->decimal_place, false);
                        $nomina12_otros_pago_compensacion['Año'] = Helper::numberFormat($result->cm_year, 0, false);
                        $nomina12_otros_pago_compensacion['RemanenteSalFav'] = Helper::numberFormat($result->cm_remainder, $employee_payroll->currency->decimal_place, false);
                        $nodo_nomina12_otros_pago_compensacion = new \CfdiUtils\Nodes\Node(
                            'nomina12:CompensacionSaldosAFavor', // nombre del elemento raíz
                            $nomina12_otros_pago_compensacion
                        );
                        $nodo_nomina12_otro_pago->addChild($nodo_nomina12_otros_pago_compensacion);
                    }

                    $nodo_nomina12_otros_pagos->addChild($nodo_nomina12_otro_pago);
                }

                $nodo_payroll->addChild($nodo_nomina12_otros_pagos);
            }

            //Incapacidades
            if($employee_payroll->employeeActiveDisabilitiesPayrollLines->isNotEmpty()) {
                $nomina12_incapacidades = [];
                $nodo_nomina12_incapacidades = new \CfdiUtils\Nodes\Node(
                    'nomina12:Incapacidades', // nombre del elemento raíz
                    $nomina12_incapacidades
                );
                //Deducciones
                foreach ($employee_payroll->employeeActiveDisabilitiesPayrollLines as $key => $result) {
                    $nomina12_incapacidad = [];
                    $nomina12_incapacidad['DiasIncapacidad'] = Helper::numberFormat($result->disability_days, 0, false);
                    $nomina12_incapacidad['TipoIncapacidad'] = $result->disabilityType->code;
                    $nomina12_incapacidad['ImporteMonetario'] = Helper::numberFormat($result->disability_amount,
                        $employee_payroll->currency->decimal_place, false);
                    $nodo_nomina12_incapacidad = new \CfdiUtils\Nodes\Node(
                        'nomina12:Incapacidad', // nombre del elemento raíz
                        $nomina12_incapacidad
                    );
                    $nodo_nomina12_incapacidades->addChild($nodo_nomina12_incapacidad);
                }

                $nodo_payroll->addChild($nodo_nomina12_incapacidades);
            }

            //Agregamos al complemento
            $comprobante->addComplemento($nodo_payroll);

            //Método de ayuda para establecer las sumas del comprobante e impuestos con base en la suma de los conceptos y la agrupación de sus impuestos
            //$creator->addSumasConceptos(null, 2);
            //Método de ayuda para generar el sello (obtener la cadena de origen y firmar con la llave privada)
            $creator->addSello('file://' . \Storage::path($employee_payroll->company->pathFileKeyPassPem()), Crypt::decryptString($employee_payroll->company->password_key));
            //Valida la estructura
            //$creator->validate();

            //Guarda XML
            //dd($creator->asXml());
            $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
            $file_xml = Helper::makeDirectoryCfdi($path_xml) . '/' . (!$draft ? Str::random(40) : str_replace('/','',$employee_payroll->name)) . '.xml';
            $creator->saveXml(\Storage::path($path_xml . $file_xml));


            //Arreglo temporal para actualizar Customer Invoice CFDI
            $tmp = [
                'pac_id' => $pac->id,
                'cfdi_version' => setting('cfdi_version'),
                'uuid' => '',
                'date' => null,
                'path_xml' => $path_xml,
                'file_xml' => $file_xml,
                'file_xml_pac' => '',
                'pac' => $pac,
            ];

            //Timbrado de XML
            if(!$draft) {
                $class_pac = $pac->code;
                $tmp = PacHelper::$class_pac($tmp, $creator);
            }

            return $tmp;
        } catch (\Exception $e) {
            throw $e;
        }

        @libxml_disable_entity_loader($current_entity_loader);
    }

    /**
     * Ddescarga de archivos XML
     *
     * @param Request $request
     * @param Payroll $payroll
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|\Symfony\Component\HttpFoundation\BinaryFileResponse
     * @throws \Exception
     */
    public function downloadXmlPdf(Payroll $payroll)
    {
        //Ruta y validacion del XML
        $results = $payroll->employeePayrolls;

        if($results->isNotEmpty()) {
            $file_zip_name = $payroll->name .'_'.random_int(100,999).'.zip';
            $zip =\Zipper::make(\Storage::path('temp/' . $file_zip_name));

            foreach ($results as $result) {

                //Ruta y validacion del XML
                $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $result->company_id) . '/';
                $file_xml_pac = $path_xml . $result->employeePayrollCfdi->file_xml_pac;
                $file_xml = $path_xml . $result->employeePayrollCfdi->file_xml;
                if (!empty($result->employeePayrollCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {
                    $zip->add(\Storage::path($file_xml_pac), str_replace('/','',$result->name) . '.xml');
                }elseif(!empty($result->employeePayrollCfdi->file_xml) && \Storage::exists($file_xml)){
                    $zip->add(\Storage::path($file_xml), str_replace('/','',$result->name) . '.xml');
                }

                //Ruta y validacion del pdf
                $pdf = $this->print($result, false, true);
                $file_pdf_name = Str::random().'.pdf';
                //Guardamos en directorio temporal
                \Storage::put('temp/' . $file_pdf_name, $pdf);
                $zip->add(\Storage::path('temp/' . $file_pdf_name), str_replace('/','',$result->name) . '.pdf');

            }
            $zip->close();
            while (ob_get_level()) ob_end_clean();
            ob_start();

            return response()->download(\Storage::path('temp/' . $file_zip_name));
        }

        //Mensaje
        flash(__('base/payroll.error_download_xml'))->error();

        //Redireccion
        return redirect('/base/payrolls');
    }

    /**
     * Clase generica de impresion
     *
     * @param EmployeePayroll $employee_payroll
     * @return mixed
     */
    public function print(EmployeePayroll $employee_payroll, $download = false, $save = false)
    {
        try {
            //Logica
            $tmp = 'default';
            if (!empty($employee_payroll->employeePayrollCfdi->cfdi_version)) {
                $tmp = $employee_payroll->employeePayrollCfdi->cfdi_version;
            }
            $class_print = 'print' . ucfirst($tmp);
            return $this->$class_print($employee_payroll, $download, $save);

        } catch (\Exception $e) {
            flash($e->getMessage())->error();
            return redirect('/base/employee-payrolls');
        }
    }

    /**
     * Impresion default
     *
     * @param $employee_payroll
     * @return mixed
     */
    private function printDefault($employee_payroll, $download = false, $save = false)
    {
        //PDF
        $pdf = \PDF::loadView('base.employee_payrolls.pdf_default', compact('employee_payroll'));

        //Guardar
        if ($save) {
            return $pdf->output();
        }

        //Descarga
        if($download){
            return $pdf->download($employee_payroll->documentType->name . '_' . str_replace('/','',$employee_payroll->name) . '.pdf');
        }

        //Redireccion
        return $pdf->stream($employee_payroll->documentType->name . '_' . str_replace('/','',$employee_payroll->name) . '.pdf');
    }

    /**
     * Impresion CFDI 3.3
     *
     * @param $employee_payroll
     * @param bool $download
     * @return mixed
     * @throws \Exception
     */
    private function printCfdi33($employee_payroll, $download = false, $save = false)
    {
        //Datos del XML timbrado
        $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
        $file_xml_pac = $path_xml . $employee_payroll->employeePayrollCfdi->file_xml_pac;
        $data = [];


        //Valida que el archivo exista
        if (!empty($employee_payroll->employeePayrollCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {

            //Crea arreglo
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
            $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

            //Genera codigo QR
            $image = QrCode::format('png')->size(150)->margin(0)->generate($data['qr_cadena']);
            $data['qr'] = 'data:image/png;base64,' . base64_encode($image);

            //Regimen fiscal
            $data['tax_regimen'] = TaxRegimen::where('code', $data['cfdi33']->emisor['RegimenFiscal'])->first()->name_sat;
        }

        //
        $subsidio_causado = $employee_payroll->employeeActiveOtherPaymentTypesPayrollLines()->whereHas('otherPaymentType', function ($q) {
            $q->whereIn('other_payment_types.code', ['002']);
        })->sum('amount_caused');
        $data['subsidio_causado'] = $subsidio_causado;

        //PDF
        $pdf_template = \App\Helpers\Helper::companyPdfTemplate($employee_payroll->company_id);//Plantilla
        $pdf_template = 'default'; //Fix
        if($employee_payroll->status == EmployeePayroll::DRAFT && $employee_payroll->employeeActivePerceptions4PayrollLines->isNotEmpty()){
            $pdf_template = 'default_draft'; //Fix
        }
        $pdf = \PDF::loadView('base.employee_payrolls.pdf_cfdi33_' . $pdf_template,
            compact('employee_payroll', 'data'));

        //Guardar
        if ($save) {
            return $pdf->output();
        }

        //Descargar
        if ($download) {
            return $pdf->download($employee_payroll->documentType->name . '_' . str_replace('/','',$employee_payroll->name) . '.pdf');
        }

        //Redireccion
        return $pdf->stream($employee_payroll->documentType->name . '_' . str_replace('/','',$employee_payroll->name) . '.pdf');
    }

    /**
     * Impresion CFDI 4.0
     *
     * @param $employee_payroll
     * @param bool $download
     * @return mixed
     * @throws \Exception
     */
    private function printCfdi40($employee_payroll, $download = false, $save = false)
    {
        //Datos del XML timbrado
        $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
        $file_xml_pac = $path_xml . $employee_payroll->employeePayrollCfdi->file_xml_pac;
        $data = [];


        //Valida que el archivo exista
        if (!empty($employee_payroll->employeePayrollCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {

            //Crea arreglo
            $cfdi = \CfdiUtils\Cfdi::newFromString(\Storage::get($file_xml_pac));
            $data = Cfdi33Helper::getQuickArrayCfdi($cfdi);

            //Genera codigo QR
            $image = QrCode::format('png')->size(150)->margin(0)->generate($data['qr_cadena']);
            $data['qr'] = 'data:image/png;base64,' . base64_encode($image);

            //Regimen fiscal
            $data['tax_regimen'] = TaxRegimen::where('code', $data['cfdi33']->emisor['RegimenFiscal'])->first()->name_sat;
            $data['tax_regimen_employee'] = TaxRegimen::where('code', $data['cfdi33']->receptor['RegimenFiscalReceptor'])->first()->name_sat;
        }

        //
        $subsidio_causado = $employee_payroll->employeeActiveOtherPaymentTypesPayrollLines()->whereHas('otherPaymentType', function ($q) {
            $q->whereIn('other_payment_types.code', ['002']);
        })->sum('amount_caused');
        $data['subsidio_causado'] = $subsidio_causado;

        //PDF
        $pdf_template = \App\Helpers\Helper::companyPdfTemplate($employee_payroll->company_id);//Plantilla
        $pdf_template = 'default'; //Fix
        if($employee_payroll->status == EmployeePayroll::DRAFT && $employee_payroll->employeeActivePerceptions4PayrollLines->isNotEmpty()){
            $pdf_template = 'default_draft'; //Fix
        }
        $pdf = \PDF::loadView('base.employee_payrolls.pdf_cfdi40_' . $pdf_template,
            compact('employee_payroll', 'data'));

        //Guardar
        if ($save) {
            return $pdf->output();
        }

        //Descargar
        if ($download) {
            return $pdf->download($employee_payroll->documentType->name . '_' . str_replace('/','',$employee_payroll->name) . '.pdf');
        }

        //Redireccion
        return $pdf->stream($employee_payroll->documentType->name . '_' . str_replace('/','',$employee_payroll->name) . '.pdf');
    }

    /**
     * Modal para cancelar factura
     *
     * @param Request $request
     * @param EmployeePayroll $employee_payroll
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalCancel(Request $request, Payroll $payroll)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            $reason_cancellations = ReasonCancellation::populateSelect()->get()->pluck('name_sat', 'id');

            //modal de cancelar
            $html = view('layouts.partials.payrolls.modal_cancel', compact('payroll', 'reason_cancellations'))->render();

            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Funcion para timbrar factura con estatus de prefactura
     *
     * @param EmployeePayroll $employee_payroll
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function stamp(Payroll $payroll){

        if($payroll->employeePayrolls->isNotEmpty()) {

            //Valida los folios a utilizar
            if(BaseHelper::getAvailableFolios()<$payroll->employeePayrolls->count()){
                flash(__('general.error_available_folios'))->error();
                return back()->withInput();
            }

            foreach ($payroll->employeePayrolls as $employee_payroll) {

                \DB::connection('tenant')->beginTransaction();
                $rfc_error = '';
                try {
                    $invoiced = true;
                    if ((int)$employee_payroll->status == EmployeePayroll::DRAFT) {
                        $rfc_error = ($employee_payroll->employee->curp ?? '') . ' ' . ($employee_payroll->employee->name ?? '');
                        //Valida que tenga exista la clase de facturacion
                        $class_cfdi = setting('cfdi_version');
                        if (empty($class_cfdi)) {
                            throw new \Exception(__('general.error_cfdi_version'));
                        }
                        if (!method_exists($this, $class_cfdi)) {
                            throw new \Exception(__('general.error_cfdi_class_exists'));
                        }

                        //Valida que tenga folios disponibles
                        if (BaseHelper::getAvailableFolios() <= 0) {
                            throw new \Exception(__('general.error_available_folios'));
                        }

                        //CFDI
                        $employee_payroll_cfdi = $employee_payroll->EmployeePayrollCfdi;

                        //Actualiza registr principal
                        $employee_payroll->updated_uid = \Auth::user()->id;
                        $employee_payroll->date = Helper::dateTimeToSql(BaseHelper::getDateTimeBranchOffice($employee_payroll->branch_office_id));
                        $employee_payroll->status = EmployeePayroll::OPEN;
                        $employee_payroll->save();

                        //Actualiza folios
                        //Obtiene folio
                        $document_type = Helper::getNextDocumentTypeByCode($this->document_type_code, $employee_payroll->company->id,false,$employee_payroll->branch_office_id);
                        $employee_payroll->draft = $employee_payroll->name;
                        $employee_payroll->name = $document_type['name'];
                        $employee_payroll->serie = $document_type['serie'];
                        $employee_payroll->folio = $document_type['folio'];
                        $employee_payroll->save();
                        $employee_payroll_cfdi->name = $employee_payroll->name;
                        $employee_payroll_cfdi->save();

                        //Valida Empresa y PAC para timbrado
                        PacHelper::validateSatActions($employee_payroll->company);

                        //Crear XML y timbra
                        $tmp = $this->$class_cfdi($employee_payroll);

                        //Guardar registros de CFDI
                        $employee_payroll_cfdi->fill(array_only($tmp, [
                            'pac_id',
                            'cfdi_version',
                            'uuid',
                            'date',
                            'file_xml',
                            'file_xml_pac',
                        ]));
                        $employee_payroll_cfdi->save();

                        //Disminuye folios
                        BaseHelper::decrementFolios();

                        $invoiced = true;
                    }

                    \DB::connection('tenant')->commit();

                    if($invoiced){
                        $this->saveCfdiDownloads($employee_payroll, $employee_payroll_cfdi);
                    }

                } catch (\Exception $e) {
                    \DB::connection('tenant')->rollback();
                    if(!empty($rfc_error))
                        flash($rfc_error)->error();
                    flash(__('general.error_cfdi_pac') . '<br>' . $e->getMessage())->error();
                    return back()->withInput();
                }

                //Almacenamiento dropbox
                self::dropboxBackup($employee_payroll);

            }

            //Mensaje
            flash(__('general.text_success_payroll_cfdi'))->success();

            $payroll->status = Payroll::OPEN;
            $payroll->save();

        }

        //Redireccion
        return redirect('/base/payrolls');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function createImportAssimilated()
    {
        $branch_offices = BranchOffice::populateSelect()->pluck('name', 'id');
        $payroll_types = PayrollType::populateSelect()->get()->pluck('name_sat', 'id');
        $employee_categories = EmployeeCategory::populateSelect()->pluck('name', 'id');
        $company = Helper::defaultCompany(); //Empresa
        if(!empty($company->tax_regimen_id2)){
            $tax_regimens = TaxRegimen::populateSelect()->whereIn('id',[$company->tax_regimen_id, $company->tax_regimen_id2])->get()->pluck('name_sat', 'id');
        }else{
            $tax_regimens = collect([]);
        }

        $import_results [] = (object)[
            'employee_curp' => 'PECJ110313HCMLNS09',
            'employee' => 'Juan Perez Cruz',
            'date_payment' => '30-11-2019',
            'date_start_payment' => '01-11-2019',
            'date_end_payment' => '30-11-2019',
            'payment_days' => '30',
            'amount_assimilated' => '740.92',
            'amount_isr' => '21.5'
        ];
        $import_results [] = (object)[
            'employee_curp' => 'HEPM110313HCMLNS09',
            'employee' => 'Marcos Hernandez Perez',
            'date_payment' => '30-11-2019',
            'date_start_payment' => '01-11-2019',
            'date_end_payment' => '30-11-2019',
            'payment_days' => '30',
            'amount_assimilated' => '564.25',
            'amount_isr' => '10.83'
        ];
        $import_results [] = (object)[
            'employee_curp' => 'LUJM110313HCMLNS09',
            'employee' => 'Maria Lugo Juarez',
            'date_payment' => '30-11-2019',
            'date_start_payment' => '01-11-2019',
            'date_end_payment' => '30-11-2019',
            'payment_days' => '30',
            'amount_assimilated' => '1488.78',
            'amount_isr' => '69.37'
        ];

        return view('base.payrolls.import_assimilated',
            compact('branch_offices', 'payroll_types','import_results','employee_categories','tax_regimens'));
    }

    /**
     * Descargar plantilla
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function downloadTemplateImportAssimilated(Request $request)
    {
        //Datos
        $employees = Employee::filter($request->all())->active()->orderBy('sort_order')->orderBy('name')->get();

        //Descargar archivo
        while (ob_get_level()) ob_end_clean();
        ob_start();

        return Excel::download(new PayrollAssimilatedTemplateImportExport($request, $employees), __('base/payroll.text_template_import') . '-' . config('app.name') . '.xlsx',\Maatwebsite\Excel\Excel::XLSX);
    }

    /**
     * Obtener informacion antes de procesar el archivo
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function beforeImportAssimilated(Request $request)
    {
        //Validaciones
        //Validacion de archivos por extension
        if ($request->hasFile('file_payrolls_import')) {
            $request->merge(['file_payrolls_import_ext' => request()->file('file_payrolls_import')->getClientOriginalExtension()]);
        }
        $validator = \Validator::make($request->all(), [
            'file_payrolls_import' => 'required|max:2048',
            'file_payrolls_import_ext' => 'nullable|in:xls,xlsx'
        ], [
            'file_payrolls_import.*' => __('base/payroll.error_file_payrolls_import'),
            'file_payrolls_import_ext.*' => __('base/payroll.error_file_payrolls_import'),
        ]);
        //Validaciones manuales
        //$validator = \Validator::make([], []);

        //Errores
        if ($validator->fails()) {
            $errors = '';
            foreach ($validator->errors()->all() as $message) {
                $errors = $message;
            }
            return response()->json(['error' => $errors], 422);
        }

        try {
            //Lógica

            //Importar
            $import = new PayrollAssimilatedTemplateBeforeImportImport;
            Excel::import($import, request()->file('file_payrolls_import'));

            //Mensaje
            return response()->json([
                'success' => 'ok',
                'total_import_payrolls' => sprintf(__('base/payroll.help_import_sweet_alert_1'),Helper::numberFormat($import->total_payrolls),Helper::numberFormatMoney($import->total_amount_assimilated_payrolls),Helper::numberFormatMoney($import->total_amount_isr_payrolls)),
            ]);
        }catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 422);
        }
    }

    /**
     * Importar cuotas desde archivo
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     * @throws \Illuminate\Validation\ValidationException
     */
    public function storeImportAssimilated(Request $request)
    {
        //Validacion
        //Validacion de archivos por extension
        if ($request->hasFile('file_payrolls_import')) {
            $request->merge(['file_payrolls_import_ext' => request()->file('file_payrolls_import')->getClientOriginalExtension()]);
        }
        $this->validate($request, [
            'name' => 'required',
            'payroll_type_id' => 'required',
            'branch_office_id' => 'required',
            'file_payrolls_import' => 'required|max:2048',
            'file_payrolls_import_ext' => 'nullable|in:xls,xlsx'
        ], [
            'name.*' => __('base/payroll.error_name'),
            'payroll_type_id.*' => __('base/payroll.error_payroll_type_id'),
            'branch_office_id.*' => __('base/payroll.error_branch_office_id'),
            'file_payrolls_import.*' => __('base/payroll.error_file_payrolls_import'),
            'file_payrolls_import_ext.*' => __('base/payroll.error_file_payrolls_import'),
        ]);

        //filas
        $rows = null;
        $data_payroll = [];
        $data_payrolls = [];

        //Validaciones y arreglo de valores
        try {
            //Lógica

            //Importar
            $import = new PayrollAssimilatedTemplateImportImport;
            Excel::import($import, request()->file('file_payrolls_import'));

            $rows = $import->rows;

            //Validacion
            foreach ($rows as $key => $row) {
                $num_row = $key + 2;

                if (strpos($row['fecha_de_pago'], "'") || strpos($row['fecha_de_pago'], "-") || strpos($row['fecha_de_pago'], "/")) {
                    $rows[$key]['fecha_de_pago'] = $row['fecha_de_pago'] = str_replace('\'', '', str_replace('/', '-', $row['fecha_de_pago']));
                } else {
                    $rows[$key]['fecha_de_pago'] = \Date::instance(\PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($row['fecha_de_pago']))->format(setting('date_format', 'd-m-Y'));
                }
                if (strpos($row['fecha_inicial_de_pago'], "'") || strpos($row['fecha_inicial_de_pago'], "-") || strpos($row['fecha_inicial_de_pago'], "/")) {
                    $rows[$key]['fecha_inicial_de_pago'] = $row['fecha_inicial_de_pago'] = str_replace('\'', '', str_replace('/', '-', $row['fecha_inicial_de_pago']));
                } else {
                    $rows[$key]['fecha_inicial_de_pago'] = \Date::instance(\PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($row['fecha_inicial_de_pago']))->format(setting('date_format', 'd-m-Y'));
                }
                if (strpos($row['fecha_final_de_pago'], "'") || strpos($row['fecha_final_de_pago'], "-") || strpos($row['fecha_final_de_pago'], "/")) {
                    $rows[$key]['fecha_final_de_pago'] = $row['fecha_final_de_pago'] = str_replace('\'', '', str_replace('/', '-', $row['fecha_final_de_pago']));
                } else {
                    $rows[$key]['fecha_final_de_pago'] = \Date::instance(\PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($row['fecha_final_de_pago']))->format(setting('date_format', 'd-m-Y'));
                }

                \Validator::make($row->toArray(), [
                    'curp' => 'required|size:18',
                    'fecha_de_pago' => 'required|date|date_format:"'.setting('date_format', 'd-m-Y').'"',
                    'fecha_inicial_de_pago' => 'required|date|date_format:"'.setting('date_format', 'd-m-Y').'"',
                    'fecha_final_de_pago' => 'required|date|date_format:"'.setting('date_format', 'd-m-Y').'"|after_or_equal:' . Helper::date(Helper::createDate($row['fecha_inicial_de_pago'])),
                    'dias_pagados' => 'required|numeric|min:0.001',
                    'importe_asimilados' => 'required|numeric|min:0.001',
                    'importe_isr' => 'required|numeric|min:0.001',
                ], [
                    'curp.*' => __('base/employee.error_curp') . sprintf(__('base/payroll.error_row'), $num_row,$row['curp']),
                    'fecha_de_pago.required' => __('base/employee_payroll.error_date_payment') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_de_pago']),
                    'fecha_de_pago.date' => __('base/employee_payroll.error_date_payment_format') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_de_pago']),
                    'fecha_de_pago.date_format' => __('base/employee_payroll.error_date_payment_format') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_de_pago']),
                    'fecha_inicial_de_pago.required' => __('base/employee_payroll.error_date_start_payment') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_inicial_de_pago']),
                    'fecha_inicial_de_pago.date' => __('base/employee_payroll.error_date_start_payment_format') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_inicial_de_pago']),
                    'fecha_inicial_de_pago.date_format' => __('base/employee_payroll.error_date_start_payment_format') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_inicial_de_pago']),
                    'fecha_final_de_pago.required' => __('base/employee_payroll.error_date_end_payment') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_final_de_pago']),
                    'fecha_final_de_pago.date' => __('base/employee_payroll.error_date_end_payment_format') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_final_de_pago']),
                    'fecha_final_de_pago.date_format' => __('base/employee_payroll.error_date_end_payment_format') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_final_de_pago']),
                    'fecha_final_de_pago.after_or_equal' => __('base/employee_payroll.error_date_end_payment_after') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_final_de_pago']),
                    'dias_pagados.*' => __('base/payroll.error_payment_days') . sprintf(__('base/payroll.error_row'), $num_row,$row['dias_pagados']),
                    'importe_asimilados.*' => __('base/payroll.error_amount_assimilated') . sprintf(__('base/payroll.error_row'), $num_row,$row['importe_asimilados']),
                    'importe_isr.*' => __('base/payroll.error_amount_isr') . sprintf(__('base/payroll.error_row'), $num_row,$row['importe_isr']),
                ])->validate();
            }

            //Logica
            $company = Helper::defaultCompany(); //Empresa
            $payment_way =PaymentWay::where('code','=','99')->first();
            $currency =Currency::where('code','=','MXN')->first();
            $payment_method =PaymentMethod::where('code','=','PUE')->first();
            $cfdi_use =CfdiUse::where('code','=','P01')->first();
            if(setting('cfdi_version') == 'cfdi40'){
                $cfdi_use =CfdiUse::where('code','=','CN01')->first();
            }

            //Agrupamos los datos correctamente
            $count=0;
            $amount_discount = 0;
            $amount_untaxed = 0;
            $amount_isr = 0;
            $amount_tax = 0;
            $amount_tax_ret = 0;
            $amount_total = 0;
            $amount_deductions = 0;
            $amount_perceptions = 0;
            $amount_other_payment_types = 0;
            foreach ($rows as $key => $row) {
                $num_row = $key + 2;

                //Empleado
                $employee = Employee::where('curp', '=', $row['curp'])->where('company_id', '=', Helper::defaultCompany()->id)->first();
                if (empty($employee)) {
                    throw new \Exception(__('base/payroll.error_employee_id2') . sprintf(__('base/payroll.error_row'), $num_row, $row['empleado']));
                }
                if(setting('cfdi_version') == 'cfdi40'){
                    if (empty($employee->tax_regimen_id)) {
                        throw new \Exception(__('base/payroll.error_tax_regimen_employee_id2') . sprintf(__('base/payroll.error_row'), $num_row, $row['empleado']));
                    }
                    if (empty($employee->postcode) && !in_array($employee->taxid, ['XAXX010101000', 'XEXX010101000'])) {
                        throw new \Exception(__('base/payroll.error_postcode_employee') . sprintf(__('base/payroll.error_row'), $num_row, $row['empleado']));
                    }
                }

                if(empty($data_payroll)){
                    $data_payroll = [
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'name' => $request->name,
                        'generated_type' => 'Asimilados',
                        'file' => '',
                        'payroll_type_id' => $request->payroll_type_id,
                        'branch_office_id' => $request->branch_office_id,
                        'date_payment' => Helper::convertDateToSql($row['fecha_de_pago']),
                        'date_start_payment' => Helper::convertDateToSql($row['fecha_inicial_de_pago']),
                        'date_end_payment' => Helper::convertDateToSql($row['fecha_final_de_pago']),
                        'payment_days' => $row['dias_pagados'],
                        'currency_id' => $currency->id,
                        'currency_value' => 1,
                        'amount_discount' => 0,
                        'amount_untaxed' => 0,
                        'amount_isr' => 0,
                        'amount_tax' => 0,
                        'amount_tax_ret' => 0,
                        'amount_total' => 0,
                        'amount_perceptions' => 0,
                        'amount_deductions' => 0,
                        'amount_other_payment_types' => 0,
                        'comment' => '',
                        'company_id' => $company->id,
                        'status' => Payroll::DRAFT,
                        'tax_regimen_id' => $request->tax_regimen_id,
                    ];
                }
                $amount_untaxed += (double)$row['importe_asimilados'];
                $amount_isr += (double)$row['importe_isr'];
                $amount_perceptions += (double)$row['importe_asimilados'];
                $amount_deductions += (double)$row['importe_isr'];
            }

        } catch (\Illuminate\Validation\ValidationException $e ) {
            return back()->withErrors($e->errors());
        } catch (\Exception $e) {
            flash($e->getMessage())->error();
            return back()->withInput();
        }

        //Gran total
        $amount_total = $amount_untaxed-$amount_discount-$amount_isr;
        $data_payroll['amount_untaxed'] = $amount_untaxed;
        $data_payroll['amount_isr'] = $amount_isr;
        $data_payroll['amount_total'] = $amount_total;
        $data_payroll['amount_perceptions'] = $amount_perceptions;
        $data_payroll['amount_deductions'] = $amount_deductions;

        //Genera facturas
        $rfc_error = '';
        try {

            if(!empty($data_payroll)) {
                //Registro principal
                $payroll = Payroll::create($data_payroll);
                $perception_assimilated =Perception::where('code','=','046')->first();
                $deduction_isr =Deduction::where('code','=','002')->first();


                foreach ($rows as $key => $row) {

                    \DB::connection('tenant')->beginTransaction();
                    $rfc_error = ($row['curp'] ?? '') . ' ' . ($row['empleado'] ?? '');

                    //Logica
                    $date = BaseHelper::getDateTimeBranchOffice($request->branch_office_id);
                    $employee = Employee::where('curp', '=', $row['curp'])->where('company_id', '=', Helper::defaultCompany()->id)->first();

                    //Obtiene folio
                    $document_type = Helper::getNextDocumentTypeByCode($this->document_type_code,$company->id, true, $request->branch_office_id);

                    //Guardar
                    //Registro principal
                    $employee_payroll = EmployeePayroll::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'name' => $document_type['name'],
                        'serie' => $document_type['serie'],
                        'folio' => $document_type['folio'],
                        'date' => Helper::dateTimeToSql($date),
                        'date_payment' => Helper::convertDateToSql($row['fecha_de_pago']),
                        'date_start_payment' => Helper::convertDateToSql($row['fecha_inicial_de_pago']),
                        'date_end_payment' => Helper::convertDateToSql($row['fecha_final_de_pago']),
                        'employee_id' => $employee->id,
                        'payroll_type_id' => $request->payroll_type_id,
                        'branch_office_id' => $request->branch_office_id,
                        'payment_way_id' => $payment_way->id,
                        'payment_method_id' => $payment_method->id,
                        'cfdi_use_id' => $cfdi_use->id,
                        'currency_id' => $currency->id,
                        'currency_value' => 1,
                        'base_salary' => $employee->base_salary,
                        'sdi' => $employee->sdi,
                        'amount_discount' => 0,
                        'amount_untaxed' => (double)$row['importe_asimilados'],
                        'amount_isr' => (double)$row['importe_isr'],
                        'amount_tax' => 0,
                        'amount_tax_ret' => 0,
                        'amount_total' => (double)$row['importe_asimilados'] - (double)$row['importe_isr'],
                        'amount_perceptions' => (double)$row['importe_asimilados'],
                        'amount_deductions' => (double)$row['importe_isr'],
                        'amount_other_payment_types' => 0,
                        'document_type_id' => $document_type['id'],
                        'cfdi_relation_id' => null,
                        'company_id' => $company->id,
                        'payment_days' => $row['dias_pagados'],
                        'status' => EmployeePayroll::DRAFT,
                        'payroll_id' => $payroll->id,
                        'reference' => !empty($request->reference) ? $request->reference : __('base/employee_payroll.text_reference'),
                        'tax_regimen_id' => $data_payroll['tax_regimen_id'],
                        'tax_regimen_employee_id' => $employee->tax_regimen_id,
                    ]);

                    //Registro de lineas
                    $amount_discount = 0;
                    $amount_untaxed = 0;
                    $amount_isr = 0;
                    $amount_tax = 0;
                    $amount_tax_ret = 0;
                    $amount_total = 0;
                    $amount_deductions = 0;
                    $amount_perceptions = 0;
                    $amount_other_payment_types = 0;

                    //Percepciones
                    //Logica
                    $item_amount_taxed = (double)$row['importe_asimilados'];
                    $item_amount_exempt = 0;
                    $item_amount_total = $item_amount_taxed + $item_amount_exempt;
                    $item_amount_caused = 0;

                    //Sumatoria totales
                    $amount_discount += 0;
                    $amount_untaxed += $item_amount_total;
                    $amount_isr += 0;
                    $amount_tax += 0;
                    $amount_tax_ret += 0;
                    $amount_perceptions += $item_amount_total;
                    $amount_deductions += 0;
                    $amount_other_payment_types += 0;

                    //Guardar linea
                    $employee_payroll_line = EmployeePayrollLine::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'employee_payroll_id' => $employee_payroll->id,
                        'name' => $perception_assimilated->name,
                        'code' => !empty($perception_assimilated->code2) ? $perception_assimilated->code2 : $perception_assimilated->code,
                        'perception_id' => $perception_assimilated->id,
                        'amount_taxed' => $item_amount_taxed,
                        'amount_exempt' => $item_amount_exempt,
                        'amount_total' => $item_amount_total,
                        'amount_caused' => $item_amount_caused,
                        'sort_order' => $key,
                        'status' => 1,
                    ]);

                    //Deducciones
                    //Logica
                    $item_amount_taxed = (double)$row['importe_isr'];
                    $item_amount_exempt = 0;
                    $item_amount_total = $item_amount_taxed + $item_amount_exempt;
                    $item_amount_caused = 0;

                    //Sumatoria totales
                    $amount_discount += 0;
                    $amount_untaxed += 0;
                    $amount_isr += $item_amount_total;
                    $amount_tax += 0;
                    $amount_tax_ret += 0;
                    $amount_perceptions += 0;
                    $amount_deductions += $item_amount_total;
                    $amount_other_payment_types += 0;

                    //Guardar linea
                    $employee_payroll_line = EmployeePayrollLine::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'employee_payroll_id' => $employee_payroll->id,
                        'name' => $deduction_isr->name,
                        'code' => !empty($deduction_isr->code2) ? $deduction_isr->code2 : $deduction_isr->code,
                        'deduction_id' => $deduction_isr->id,
                        'amount_taxed' => $item_amount_taxed,
                        'amount_exempt' => $item_amount_exempt,
                        'amount_total' => $item_amount_total,
                        'amount_caused' => $item_amount_caused,
                        'sort_order' => $key,
                        'status' => 1,
                    ]);


                    //Valida que tenga exista la clase de facturacion
                    $class_cfdi = setting('cfdi_version');
                    if (empty($class_cfdi)) {
                        throw new \Exception(__('general.error_cfdi_version'));
                    }
                    if (!method_exists($this, $class_cfdi)) {
                        throw new \Exception(__('general.error_cfdi_class_exists'));
                    }

                    //Registros de cfdi
                    $employee_payroll_cfdi = EmployeePayrollCfdi::create([
                        'created_uid' => \Auth::user()->id,
                        'updated_uid' => \Auth::user()->id,
                        'employee_payroll_id' => $employee_payroll->id,
                        'name' => $employee_payroll->name,
                        'cfdi_version' => $class_cfdi,
                        'status' => 1,
                    ]);

                    //Gran total
                    $amount_total = $amount_untaxed-$amount_discount-$amount_isr;

                    //Actualiza registro principal con totales
                    $employee_payroll->amount_discount = $amount_discount;
                    $employee_payroll->amount_untaxed = $amount_untaxed;
                    $employee_payroll->amount_isr = $amount_isr;
                    $employee_payroll->amount_tax = $amount_tax;
                    $employee_payroll->amount_tax_ret = $amount_tax_ret;
                    $employee_payroll->amount_total = $amount_total;
                    $employee_payroll->amount_perceptions = $amount_perceptions;
                    $employee_payroll->amount_deductions = $amount_deductions;
                    $employee_payroll->amount_other_payment_types = $amount_other_payment_types;
                    $employee_payroll->update();

                    //Valida Empresa y PAC para timbrado
                    /*PacHelper::validateSatActions($company);*/

                    //Crear XML y timbra
                    $tmp = $this->$class_cfdi($employee_payroll, true);

                    //Guardar registros de CFDI
                    $employee_payroll_cfdi->fill(array_only($tmp, [
                        'pac_id',
                        'cfdi_version',
                        'uuid',
                        'date',
                        'file_xml',
                        'file_xml_pac',
                    ]));
                    $employee_payroll_cfdi->save();

                    //Disminuye folios
                    /*BaseHelper::decrementFolios();*/

                    \DB::connection('tenant')->commit();

                    /*$this->saveCfdiDownloads($employee_payroll);*/

                    //Guardamos facturas creadas
                    $employee_payrolls[] = $employee_payroll;

                }
            }


            //Mensaje
            flash(__('base/payroll.text_form_success_import'))->success();

        } catch (\Illuminate\Validation\ValidationException $e ) {
            if(!empty($payroll) && $key == 0){
                $payroll->forceDelete();
            }
            \DB::connection('tenant')->rollback();
            if(!empty($rfc_error))
                flash($rfc_error)->error();
            return back()->withErrors($e->errors());
        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            if(!empty($rfc_error))
                flash($rfc_error)->error();
            flash($e->getMessage())->error();
            return back()->withInput();
        }

        //Almacenamiento dropbox
        /*if(!empty($employee_payrolls)){
            foreach($employee_payrolls as $employee_payroll){
                self::dropboxBackup($employee_payroll);
            }
        }*/

        //Redireccion
        return redirect()->route('payrolls.index');
    }

    /**
     * Respaldo en dropbox si esta activo
     *
     * @param $employee_payroll
     */
    public function dropboxBackup($employee_payroll, $save_xml = true, $save_pdf = true){
        try{
            if(!empty($employee_payroll) && !empty(setting('dropbox_backup')) && !empty(setting('dropbox_access_token'))){
                $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
                $file_xml_pac = $path_xml . $employee_payroll->employeePayrollCfdi->file_xml_pac;
                if (!empty($employee_payroll->employeePayrollCfdi->file_xml_pac) && \Storage::exists($file_xml_pac)) {

                    $folder = !empty(setting('dropbox_folder','')) ? setting('dropbox_folder','') . '/' : setting('dropbox_folder','');
                    $folder .= strtoupper(\Date::parse($employee_payroll->date)->format('F Y'));
                    $folder .= '/' .preg_replace("/[^a-zA-Z0-9\s]/", '', $employee_payroll->company->name);

                    //Guarda XML
                    if($save_xml) {
                        \Storage::disk('dropbox')->putFileAs($folder, new \Illuminate\Http\File(\Storage::path($file_xml_pac)), str_replace('/','',$employee_payroll->name) . '.xml');
                    }

                    //Guarda PDF
                    if($save_pdf) {
                        $pdf = $this->print($employee_payroll, false, true);
                        \Storage::disk('dropbox')->put($folder . '/' . str_replace('/','',$employee_payroll->name) . '.pdf', $pdf);
                    }
                }

            }
        }catch (\Exception $e){
            flash($e->getMessage())->error();
        }
    }


    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function createImportPayrolls()
    {
        $branch_offices = BranchOffice::populateSelect()->pluck('name', 'id');
        $payroll_types = PayrollType::populateSelect()->get()->pluck('name_sat', 'id');
        $employee_categories = EmployeeCategory::populateSelect()->pluck('name', 'id');
        $company = Helper::defaultCompany(); //Empresa
        if(!empty($company->tax_regimen_id2)){
            $tax_regimens = TaxRegimen::populateSelect()->whereIn('id',[$company->tax_regimen_id, $company->tax_regimen_id2])->get()->pluck('name_sat', 'id');
        }else{
            $tax_regimens = collect([]);
        }
        $perceptions = Perception::active()->orderBy('code')->orderBy('name')->get()->pluck('name_sat', 'id');
        $perception_selected = Perception::active()->whereIn('code',['001', '020', '021'])->orderBy('code')->orderBy('name')->get()->pluck('id');

        $deductions = Deduction::active()->orderBy('code')->orderBy('name')->get()->pluck('name_sat', 'id');
        $deduction_selected = Deduction::active()->whereIn('code',['001', '002', '021', '019'])->orderBy('code')->orderBy('name')->get()->pluck('id');

        $other_payment_types = OtherPaymentType::active()->orderBy('code')->orderBy('name')->get()->pluck('name_sat', 'id');
        $other_payment_type_selected = OtherPaymentType::active()->whereIn('code',['002'])->orderBy('code')->orderBy('name')->get()->pluck('id');

        $disability_types = DisabilityType::active()->orderBy('code')->orderBy('name')->get()->pluck('name_sat', 'id');
        $disability_type_selected = DisabilityType::active()->whereIn('code',['00000'])->orderBy('code')->orderBy('name')->get()->pluck('id');

        $import_results [] = (object)[
            'employee_curp' => 'PECJ110313HCMLNS09',
            'employee' => 'Juan Perez Cruz',
            'date_payment' => '30-11-2019',
            'date_start_payment' => '01-11-2019',
            'date_end_payment' => '30-11-2019',
            'payment_days' => '30',
            'others' => '-',
        ];
        $import_results [] = (object)[
            'employee_curp' => 'HEPM110313HCMLNS09',
            'employee' => 'Marcos Hernandez Perez',
            'date_payment' => '30-11-2019',
            'date_start_payment' => '01-11-2019',
            'date_end_payment' => '30-11-2019',
            'payment_days' => '30',
            'others' => '-',
        ];
        $import_results [] = (object)[
            'employee_curp' => 'LUJM110313HCMLNS09',
            'employee' => 'Maria Lugo Juarez',
            'date_payment' => '30-11-2019',
            'date_start_payment' => '01-11-2019',
            'date_end_payment' => '30-11-2019',
            'payment_days' => '30',
            'others' => '- ',
        ];

        return view('base.payrolls.import_payrolls',
            compact('branch_offices', 'payroll_types','import_results','employee_categories','tax_regimens', 'perceptions', 'perception_selected', 'deductions','deduction_selected','other_payment_types','other_payment_type_selected', 'disability_types', 'disability_type_selected'));
    }

    /**
     * Descargar plantilla nomina
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function downloadTemplateImportPayrolls(Request $request)
    {
        //Datos
        $employees = Employee::filter($request->all())->active()->orderBy('sort_order')->orderBy('name')->get();

        //Descargar archivo
        while (ob_get_level()) ob_end_clean();
        ob_start();

        return Excel::download(new PayrollsTemplateImportExport($request, $employees), __('base/payroll.text_template_import') . '-' . config('app.name') . '.xlsx',\Maatwebsite\Excel\Excel::XLSX);

    }

    /**
     * Obtener informacion antes de procesar el archivo
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function beforeImportPayrolls(Request $request)
    {
        //Validaciones
        //Validacion de archivos por extension
        if ($request->hasFile('file_payrolls_import')) {
            $request->merge(['file_payrolls_import_ext' => request()->file('file_payrolls_import')->getClientOriginalExtension()]);
        }
        $validator = \Validator::make($request->all(), [
            'file_payrolls_import' => 'required|max:2048',
            'file_payrolls_import_ext' => 'nullable|in:xls,xlsx'
        ], [
            'file_payrolls_import.*' => __('base/payroll.error_file_payrolls_import'),
            'file_payrolls_import_ext.*' => __('base/payroll.error_file_payrolls_import'),
        ]);
        //Validaciones manuales
        //$validator = \Validator::make([], []);

        //Errores
        if ($validator->fails()) {
            $errors = '';
            foreach ($validator->errors()->all() as $message) {
                $errors = $message;
            }
            return response()->json(['error' => $errors], 422);
        }

        try {
            //Lógica

            //Importar
            $import = new PayrollsTemplateBeforeImportImport;
            Excel::import($import, request()->file('file_payrolls_import'));

            //Mensaje
            return response()->json([
                'success' => 'ok',
                'total_import_payrolls' => sprintf(__('base/payroll.help_import_payrolls_sweet_alert_1'),Helper::numberFormat($import->total_payrolls),Helper::numberFormatMoney($import->subtotal_amount_payrolls),Helper::numberFormatMoney($import->total_amount_discount_payrolls),Helper::numberFormatMoney($import->total_amount_isr_payrolls),Helper::numberFormatMoney($import->total_amount_payrolls)),
            ]);
        }catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 422);
        }
    }


    /**
     * Importar cuotas de nomina desde archivo excel
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     * @throws \Illuminate\Validation\ValidationException
     */
    public function storeImportPayrolls(Request $request)
    {
        //Validacion
        //Validacion de archivos por extension
        if ($request->hasFile('file_payrolls_import')) {
            $request->merge(['file_payrolls_import_ext' => request()->file('file_payrolls_import')->getClientOriginalExtension()]);
        }
        $this->validate($request, [
            'name' => 'required',
            'payroll_type_id' => 'required',
            'branch_office_id' => 'required',
            'file_payrolls_import' => 'required|max:2048',
            'file_payrolls_import_ext' => 'nullable|in:xls,xlsx'
        ], [
            'name.*' => __('base/payroll.error_name'),
            'payroll_type_id.*' => __('base/payroll.error_payroll_type_id'),
            'branch_office_id.*' => __('base/payroll.error_branch_office_id'),
            'file_payrolls_import.*' => __('base/payroll.error_file_payrolls_import'),
            'file_payrolls_import_ext.*' => __('base/payroll.error_file_payrolls_import'),
        ]);

        //filas
        $rows = null;
        $data_payroll = [];
        $data_payrolls = [];

        //Validaciones y arreglo de valores
        try {
            //Lógica

            //Importar
            $import = new PayrollsTemplateImportImport;
            Excel::import($import, request()->file('file_payrolls_import'));

            $rows = $import->rows;

            //Validacion
            foreach ($rows as $key => $row) {
                if (!empty($row['curp'])) {
                    $num_row = $key + 2;

                    if (strpos($row['fecha_de_pago'], "'") || strpos($row['fecha_de_pago'], "-") || strpos($row['fecha_de_pago'], "/")) {
                        $rows[$key]['fecha_de_pago'] = $row['fecha_de_pago'] = str_replace('\'', '', str_replace('/', '-', $row['fecha_de_pago']));
                    } else {
                        $rows[$key]['fecha_de_pago'] = \Date::instance(\PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($row['fecha_de_pago']))->format(setting('date_format', 'd-m-Y'));
                    }
                    if (strpos($row['fecha_inicial_de_pago'], "'") || strpos($row['fecha_inicial_de_pago'], "-") || strpos($row['fecha_inicial_de_pago'], "/")) {
                        $rows[$key]['fecha_inicial_de_pago'] = $row['fecha_inicial_de_pago'] = str_replace('\'', '', str_replace('/', '-', $row['fecha_inicial_de_pago']));
                    } else {
                        $rows[$key]['fecha_inicial_de_pago'] = \Date::instance(\PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($row['fecha_inicial_de_pago']))->format(setting('date_format', 'd-m-Y'));
                    }
                    if (strpos($row['fecha_final_de_pago'], "'") || strpos($row['fecha_final_de_pago'], "-") || strpos($row['fecha_final_de_pago'], "/")) {
                        $rows[$key]['fecha_final_de_pago'] = $row['fecha_final_de_pago'] = str_replace('\'', '', str_replace('/', '-', $row['fecha_final_de_pago']));
                    } else {
                        $rows[$key]['fecha_final_de_pago'] = \Date::instance(\PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($row['fecha_final_de_pago']))->format(setting('date_format', 'd-m-Y'));
                    }

                    \Validator::make($row->toArray(), [
                        'curp' => 'required|size:18',
                        'fecha_de_pago' => 'required|date|date_format:"'.setting('date_format', 'd-m-Y').'"',
                        'fecha_inicial_de_pago' => 'required|date|date_format:"'.setting('date_format', 'd-m-Y').'"',
                        'fecha_final_de_pago' => 'required|date|date_format:"'.setting('date_format', 'd-m-Y').'"|after_or_equal:' . Helper::date(Helper::createDate($row['fecha_inicial_de_pago'])),
                        'dias_pagados' => 'required|numeric|min:0.001',
                    ], [
                        'curp.*' => __('base/employee.error_curp') . sprintf(__('base/payroll.error_row'), $num_row,$row['curp']),
                        'fecha_de_pago.required' => __('base/employee_payroll.error_date_payment') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_de_pago']),
                        'fecha_de_pago.date' => __('base/employee_payroll.error_date_payment_format') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_de_pago']),
                        'fecha_de_pago.date_format' => __('base/employee_payroll.error_date_payment_format') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_de_pago']),
                        'fecha_inicial_de_pago.required' => __('base/employee_payroll.error_date_start_payment') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_inicial_de_pago']),
                        'fecha_inicial_de_pago.date' => __('base/employee_payroll.error_date_start_payment_format') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_inicial_de_pago']),
                        'fecha_inicial_de_pago.date_format' => __('base/employee_payroll.error_date_start_payment_format') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_inicial_de_pago']),
                        'fecha_final_de_pago.required' => __('base/employee_payroll.error_date_end_payment') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_final_de_pago']),
                        'fecha_final_de_pago.date' => __('base/employee_payroll.error_date_end_payment_format') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_final_de_pago']),
                        'fecha_final_de_pago.date_format' => __('base/employee_payroll.error_date_end_payment_format') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_final_de_pago']),
                        'fecha_final_de_pago.after_or_equal' => __('base/employee_payroll.error_date_end_payment_after') . sprintf(__('base/payroll.error_row'), $num_row,$row['fecha_final_de_pago']),
                        'dias_pagados.*' => __('base/payroll.error_payment_days') . sprintf(__('base/payroll.error_row'), $num_row,$row['dias_pagados']),
                    ])->validate();
                }
            }

            //Logica
            $company = Helper::defaultCompany(); //Empresa
            $payment_way =PaymentWay::where('code','=','99')->first();
            $currency =Currency::where('code','=','MXN')->first();
            $payment_method =PaymentMethod::where('code','=','PUE')->first();
            $cfdi_use =CfdiUse::where('code','=','P01')->first();
            if(setting('cfdi_version') == 'cfdi40'){
                $cfdi_use =CfdiUse::where('code','=','CN01')->first();
            }

            //Agrupamos los datos correctamente
            $count=0;
            $amount_discount = 0;
            $amount_untaxed = 0;
            $amount_isr = 0;
            $amount_tax = 0;
            $amount_tax_ret = 0;
            $amount_total = 0;
            $amount_deductions = 0;
            $amount_perceptions = 0;
            $amount_other_payment_types = 0;

            foreach ($rows as $key => $row) {
                if (!empty($row['curp'])) {
                    $num_row = $key + 2;

                    //Empleado
                    $employee = Employee::where('curp', '=', $row['curp'])->where('company_id', '=', Helper::defaultCompany()->id)->first();
                    if (empty($employee)) {
                        throw new \Exception(__('base/payroll.error_employee_id2') . sprintf(__('base/payroll.error_row'), $num_row, $row['empleado']));
                    }
                    if(setting('cfdi_version') == 'cfdi40'){
                        if (empty($employee->tax_regimen_id)) {
                            throw new \Exception(__('base/payroll.error_tax_regimen_employee_id2') . sprintf(__('base/payroll.error_row'), $num_row, $row['empleado']));
                        }
                        if (empty($employee->postcode) && !in_array($employee->taxid, ['XAXX010101000', 'XEXX010101000'])) {
                            throw new \Exception(__('base/payroll.error_postcode_employee') . sprintf(__('base/payroll.error_row'), $num_row, $row['empleado']));
                        }
                    }

                    $keys = array_keys($row->toArray());

                    //Arreglo de percepciones, deducciones y otros pagos e incapacidades
                    $keys_tmp = $keys;
                    $keys_tmp = array_slice($keys_tmp, 6, count($keys_tmp) - 1);

                    for ($i = 0; $i < count($keys_tmp); $i++) {
                        $text = explode('_', $keys_tmp[$i]);
                        $code = $text[0];
                        $type = $text[count($text) - 2];
                        $id = end($text);

                        //Valida que este con la nueva plantilla
                        if(!in_array($type, ['p','d','o']) || !is_numeric($id)){
                            //throw new \Exception(__('base/payroll.error_new_template'));
                        }

                        //Percepciones
                        if($type == 'p'){
                            $amount_untaxed += (double) $row[$keys_tmp[$i]]; //Gravado
                            $amount_perceptions += (double) $row[$keys_tmp[$i]]; //Gravado
                            $i++;
                            $amount_untaxed += (double) $row[$keys_tmp[$i]]; //Exento
                            $amount_perceptions += (double) $row[$keys_tmp[$i]]; //Exento
                            if($code == '019'){
                                $i += 9;
                            }
                        }

                        //Deducciones
                        if($type == 'd'){
                            if (in_array($code, ['002', '101']))
                                $amount_isr += (double)$row[$keys_tmp[$i]];
                            else
                                $amount_discount += (double)$row[$keys_tmp[$i]];

                            $amount_deductions += (double)$row[$keys_tmp[$i]];
                        }

                        //Otros pagos
                        if($type == 'o'){
                            $amount_untaxed += (double)$row[$keys_tmp[$i]];
                            $amount_other_payment_types += (double)$row[$keys_tmp[$i]];
                            if($code == '002'){
                                $i++;
                            }
                        }
                    }


                    if(empty($data_payroll)){
                        $data_payroll = [
                            'created_uid' => \Auth::user()->id,
                            'updated_uid' => \Auth::user()->id,
                            'name' => $request->name,
                            'generated_type' => '',
                            'file' => '',
                            'payroll_type_id' => $request->payroll_type_id,
                            'branch_office_id' => $request->branch_office_id,
                            'date_payment' => Helper::convertDateToSql($row['fecha_de_pago']),
                            'date_start_payment' => Helper::convertDateToSql($row['fecha_inicial_de_pago']),
                            'date_end_payment' => Helper::convertDateToSql($row['fecha_final_de_pago']),
                            'payment_days' => $row['dias_pagados'],
                            'currency_id' => $currency->id,
                            'currency_value' => 1,
                            'amount_discount' => 0,
                            'amount_untaxed' => 0,
                            'amount_isr' => 0,
                            'amount_tax' => 0,
                            'amount_tax_ret' => 0,
                            'amount_total' => 0,
                            'amount_perceptions' => 0,
                            'amount_deductions' => 0,
                            'amount_other_payment_types' => 0,
                            'comment' => '',
                            'company_id' => $company->id,
                            'status' => Payroll::DRAFT,
                            'tax_regimen_id' => $request->tax_regimen_id,
                        ];
                    }
                }
            }

        } catch (\Illuminate\Validation\ValidationException $e ) {
            return back()->withErrors($e->errors());
        } catch (\Exception $e) {
            flash($e->getMessage())->error();
            return back()->withInput();
        }

        //Gran total
        $amount_total = $amount_untaxed-$amount_discount-$amount_isr;
        $data_payroll['amount_untaxed'] = $amount_untaxed;
        $data_payroll['amount_isr'] = $amount_isr;
        $data_payroll['amount_discount'] = $amount_discount;
        $data_payroll['amount_total'] = $amount_total;
        $data_payroll['amount_perceptions'] = $amount_perceptions;
        $data_payroll['amount_deductions'] = $amount_deductions;
        $data_payroll['amount_other_payment_types'] = $amount_other_payment_types;

        //Genera facturas
        $rfc_error = '';
        try {

            if(!empty($data_payroll)) {
                //Registro principal
                $payroll = Payroll::create($data_payroll);

                //
                $tmp_overtimes_type_1 = OvertimesType::where('code', '=', '01')->first();
                $tmp_overtimes_type_2 = OvertimesType::where('code', '=', '02')->first();
                $tmp_overtimes_type_3 = OvertimesType::where('code', '=', '03')->first();

                foreach ($rows as $key => $row) {
                    if (!empty($row['curp'])) {

                        \DB::connection('tenant')->beginTransaction();
                        $rfc_error = ($row['curp'] ?? '') . ' ' . ($row['empleado'] ?? '');

                        //Logica
                        $date = BaseHelper::getDateTimeBranchOffice($request->branch_office_id);
                        $employee = Employee::where('curp', '=', $row['curp'])->where('company_id', '=', Helper::defaultCompany()->id)->first();

                        //Obtiene folio
                        $document_type = Helper::getNextDocumentTypeByCode($this->document_type_code,$company->id, true, $request->branch_office_id);

                        //Guardar
                        //Registro principal
                        $employee_payroll = EmployeePayroll::create([
                            'created_uid' => \Auth::user()->id,
                            'updated_uid' => \Auth::user()->id,
                            'name' => $document_type['name'],
                            'serie' => $document_type['serie'],
                            'folio' => $document_type['folio'],
                            'date' => Helper::dateTimeToSql($date),
                            'date_payment' => Helper::convertDateToSql($row['fecha_de_pago']),
                            'date_start_payment' => Helper::convertDateToSql($row['fecha_inicial_de_pago']),
                            'date_end_payment' => Helper::convertDateToSql($row['fecha_final_de_pago']),
                            'employee_id' => $employee->id,
                            'payroll_type_id' => $request->payroll_type_id,
                            'branch_office_id' => $request->branch_office_id,
                            'payment_way_id' => $payment_way->id,
                            'payment_method_id' => $payment_method->id,
                            'cfdi_use_id' => $cfdi_use->id,
                            'currency_id' => $currency->id,
                            'currency_value' => 1,
                            'base_salary' => $employee->base_salary,
                            'sdi' => $employee->sdi,
                            'amount_discount' => 0,
                            'amount_untaxed' => 0,
                            'amount_isr' => 0,
                            'amount_tax' => 0,
                            'amount_tax_ret' => 0,
                            'amount_total' => 0,
                            'amount_perceptions' => 0,
                            'amount_deductions' => 0,
                            'amount_other_payment_types' => 0,
                            'document_type_id' => $document_type['id'],
                            'cfdi_relation_id' => null,
                            'company_id' => $company->id,
                            'payment_days' => $row['dias_pagados'],
                            'status' => EmployeePayroll::DRAFT,
                            'payroll_id' => $payroll->id,
                            'reference' => !empty($request->reference) ? $request->reference : __('base/employee_payroll.text_reference'),
                            'tax_regimen_id' => $data_payroll['tax_regimen_id'],
                            'tax_regimen_employee_id' => $employee->tax_regimen_id,
                        ]);

                        //Registro de lineas
                        $amount_discount = 0;
                        $amount_untaxed = 0;
                        $amount_isr = 0;
                        $amount_tax = 0;
                        $amount_tax_ret = 0;
                        $amount_total = 0;
                        $amount_deductions = 0;
                        $amount_perceptions = 0;
                        $amount_other_payment_types = 0;

                        $keys = array_keys($row->toArray());

                        //Arreglo de percepciones, deducciones y otros pagos
                        $keys_tmp = $keys;
                        $keys_tmp = array_slice($keys_tmp, 6, count($keys_tmp) - 1);

                        //
                        for ($i = 0; $i < count($keys_tmp); $i++) {
                            $text = explode('_', $keys_tmp[$i]);
                            $code = $text[0];
                            $type = $text[count($text) - 2];
                            $id = end($text);

                            //Valida que este con la nueva plantilla
                            if(!in_array($type, ['p','d','o']) || !is_numeric($id)){
                                //throw new \Exception(__('base/payroll.error_new_template'));
                            }



                            //Percepciones
                            if($type == 'p'){
                                if ($row[$keys_tmp[$i]] != '' || $row[$keys_tmp[$i+1]] != '') {
                                    $amount_untaxed += (double)$row[$keys_tmp[$i]]; //Gravado
                                    $amount_untaxed += (double)$row[$keys_tmp[$i+1]]; //Exento
                                    $amount_perceptions += (double)$row[$keys_tmp[$i]]; //Gravado
                                    $amount_perceptions += (double)$row[$keys_tmp[$i+1]]; //Exento

                                    //
                                    $hr_days_1 = 0;
                                    $hr_overtimes_type_id_1 = $tmp_overtimes_type_1->id ?? null;
                                    $hr_hours_1 = 0;
                                    $hr_amount_paid_1 = 0;
                                    $hr_days_2 = 0;
                                    $hr_overtimes_type_id_2 = $tmp_overtimes_type_2->id ?? null;
                                    $hr_hours_2 = 0;
                                    $hr_amount_paid_2 = 0;
                                    $hr_days_3 = 0;
                                    $hr_overtimes_type_id_3 = $tmp_overtimes_type_3->id ?? null;
                                    $hr_hours_3 = 0;
                                    $hr_amount_paid_3 = 0;

                                    //
                                    $perception = Perception::find($id);

                                    //
                                    if($code == '019'){
                                        $hr_days_1 = (int) $row[$keys_tmp[$i+2]];
                                        $hr_hours_1 = (int) $row[$keys_tmp[$i+3]];
                                        $hr_amount_paid_1 = (double) $row[$keys_tmp[$i+4]];
                                        $hr_days_2 = (int) $row[$keys_tmp[$i+5]];
                                        $hr_hours_2 = (int) $row[$keys_tmp[$i+6]];
                                        $hr_amount_paid_2 = (double) $row[$keys_tmp[$i+7]];
                                        $hr_days_3 = (int) $row[$keys_tmp[$i+8]];
                                        $hr_hours_3 = (int) $row[$keys_tmp[$i+9]];
                                        $hr_amount_paid_3 = (double) $row[$keys_tmp[$i+10]];
                                    }

                                    //Guardar linea
                                    $employee_payroll_line = EmployeePayrollLine::create([
                                        'created_uid' => \Auth::user()->id,
                                        'updated_uid' => \Auth::user()->id,
                                        'employee_payroll_id' => $employee_payroll->id,
                                        'name' => $perception->name,
                                        'code' => !empty($perception->code2) ? $perception->code2 : $perception->code,
                                        'perception_id' => $perception->id,
                                        'amount_taxed' => (double)$row[$keys_tmp[$i]],
                                        'amount_exempt' => (double)$row[$keys_tmp[$i + 1]],
                                        'amount_total' => (double)$row[$keys_tmp[$i]] + (double)$row[$keys_tmp[$i + 1]],
                                        'amount_caused' => 0,
                                        'sort_order' => $key,
                                        'status' => 1,
                                        'hr_days_1' => $hr_days_1,
                                        'hr_overtimes_type_id_1' => $hr_overtimes_type_id_1,
                                        'hr_hours_1' => $hr_hours_1,
                                        'hr_amount_paid_1' => $hr_amount_paid_1,
                                        'hr_days_2' => $hr_days_2,
                                        'hr_overtimes_type_id_2' => $hr_overtimes_type_id_2,
                                        'hr_hours_2' => $hr_hours_2,
                                        'hr_amount_paid_2' => $hr_amount_paid_2,
                                        'hr_days_3' => $hr_days_3,
                                        'hr_overtimes_type_id_3' => $hr_overtimes_type_id_3,
                                        'hr_hours_3' => $hr_hours_3,
                                        'hr_amount_paid_3' => $hr_amount_paid_3
                                    ]);

                                }
                                $i++;
                                if($code == '019'){
                                    $i += 9;
                                }
                            }

                            //Deducciones
                            if($type == 'd'){
                                if ($row[$keys_tmp[$i]] != '') {
                                    $amount_deductions += (double) $row[$keys_tmp[$i]];

                                    //
                                    $deduction = Deduction::find($id);

                                    //Guardar linea
                                    $employee_payroll_line = EmployeePayrollLine::create([
                                        'created_uid' => \Auth::user()->id,
                                        'updated_uid' => \Auth::user()->id,
                                        'employee_payroll_id' => $employee_payroll->id,
                                        'name' => $deduction->name,
                                        'code' => !empty($deduction->code2) ? $deduction->code2 : $deduction->code,
                                        'deduction_id' => $deduction->id,
                                        'amount_taxed' => (double)$row[$keys_tmp[$i]],
                                        'amount_exempt' => 0,
                                        'amount_total' => (double)$row[$keys_tmp[$i]],
                                        'amount_caused' => 0,
                                        'sort_order' => $key,
                                        'status' => 1,
                                    ]);

                                    if (in_array($code, ['002', '101']))
                                        $amount_isr += (double)$row[$keys_tmp[$i]];
                                    else
                                        $amount_discount += (double)$row[$keys_tmp[$i]];
                                }
                            }

                            //Otros pagos
                            if($type == 'o'){
                                if ($row[$keys_tmp[$i]] != '') {
                                    $amount_untaxed += (double)$row[$keys_tmp[$i]];
                                    $amount_other_payment_types += (double)$row[$keys_tmp[$i]];

                                    //
                                    $other_payment_type = OtherPaymentType::find($id);

                                    //
                                    $employee_payroll_line = EmployeePayrollLine::create([
                                        'created_uid' => \Auth::user()->id,
                                        'updated_uid' => \Auth::user()->id,
                                        'employee_payroll_id' => $employee_payroll->id,
                                        'name' => $other_payment_type->name,
                                        'code' => !empty($other_payment_type->code2) ? $other_payment_type->code2 : $other_payment_type->code,
                                        'other_payment_type_id' => $other_payment_type->id,
                                        'amount_taxed' => (double)$row[$keys_tmp[$i]],
                                        'amount_exempt' => 0,
                                        'amount_total' => (double)$row[$keys_tmp[$i]],
                                        'amount_caused' => $code == '002' ?  (double)$row[$keys_tmp[$i + 1]] : 0,
                                        'sort_order' => $key,
                                        'status' => 1,
                                    ]);

                                }
                                if($code == '002'){
                                    $i++;
                                }
                            }

                            //Incapacidades
                            if($type == 'i'){
                                if ($row[$keys_tmp[$i]] != '' || $row[$keys_tmp[$i + 1]] != '') {
                                    //
                                    $disability_type = DisabilityType::find($id);

                                    //Guardar linea
                                    $employee_payroll_line = EmployeePayrollLine::create([
                                        'created_uid' => \Auth::user()->id,
                                        'updated_uid' => \Auth::user()->id,
                                        'employee_payroll_id' => $employee_payroll->id,
                                        'name' => $disability_type->name,
                                        'code' => !empty($disability_type->code2) ? $disability_type->code2 : $disability_type->code,
                                        'disability_type_id' => $disability_type->id,
                                        'disability_days' => (int)$row[$keys_tmp[$i]],
                                        'disability_amount' => (double)$row[$keys_tmp[$i + 1]],
                                        'sort_order' => $key,
                                        'status' => 1,
                                    ]);
                                }
                                $i++;
                            }

                        }


                        //Valida que tenga exista la clase de facturacion
                        $class_cfdi = setting('cfdi_version');
                        if (empty($class_cfdi)) {
                            throw new \Exception(__('general.error_cfdi_version'));
                        }
                        if (!method_exists($this, $class_cfdi)) {
                            throw new \Exception(__('general.error_cfdi_class_exists'));
                        }

                        //Registros de cfdi
                        $employee_payroll_cfdi = EmployeePayrollCfdi::create([
                            'created_uid' => \Auth::user()->id,
                            'updated_uid' => \Auth::user()->id,
                            'employee_payroll_id' => $employee_payroll->id,
                            'name' => $employee_payroll->name,
                            'cfdi_version' => $class_cfdi,
                            'status' => 1,
                        ]);

                        //Gran total
                        $amount_total = $amount_untaxed-$amount_discount-$amount_isr;

                        //Actualiza registro principal con totales
                        $employee_payroll->amount_discount = $amount_discount;
                        $employee_payroll->amount_untaxed = $amount_untaxed;
                        $employee_payroll->amount_isr = $amount_isr;
                        $employee_payroll->amount_tax = $amount_tax;
                        $employee_payroll->amount_tax_ret = $amount_tax_ret;
                        $employee_payroll->amount_total = $amount_total;
                        $employee_payroll->amount_perceptions = $amount_perceptions;
                        $employee_payroll->amount_deductions = $amount_deductions;
                        $employee_payroll->amount_other_payment_types = $amount_other_payment_types;
                        $employee_payroll->update();

                        //Valida Empresa y PAC para timbrado
                        /*PacHelper::validateSatActions($company);*/

                        //Crear XML y timbra
                        $tmp = $this->$class_cfdi($employee_payroll, true);

                        //Guardar registros de CFDI
                        $employee_payroll_cfdi->fill(array_only($tmp, [
                            'pac_id',
                            'cfdi_version',
                            'uuid',
                            'date',
                            'file_xml',
                            'file_xml_pac',
                        ]));
                        $employee_payroll_cfdi->save();

                        //Disminuye folios
                        /*BaseHelper::decrementFolios();*/

                        \DB::connection('tenant')->commit();

                        /*$this->saveCfdiDownloads($employee_payroll);*/

                        //Guardamos facturas creadas
                        $employee_payrolls[] = $employee_payroll;
                    }
                }
            }


            //Mensaje
            flash(__('base/payroll.text_form_success_import'))->success();

        } catch (\Illuminate\Validation\ValidationException $e ) {
            if(!empty($payroll) && $key == 0){
                $payroll->forceDelete();
            }
            \DB::connection('tenant')->rollback();
            if(!empty($rfc_error))
                flash($rfc_error)->error();
            return back()->withErrors($e->errors());
        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            if(!empty($rfc_error))
                flash($rfc_error)->error();
            flash($e->getMessage())->error();
            return back()->withInput();
        }

        //Almacenamiento dropbox
        /*if(!empty($employee_payrolls)){
            foreach($employee_payrolls as $employee_payroll){
                self::dropboxBackup($employee_payroll);
            }
        }*/

        //Redireccion
        return redirect()->route('payrolls.index');
    }

    private function saveCfdiDownloads($employee_payroll, $employee_payroll_cfdi){
        //Directorio de XML
        $path_files = Helper::setDirectory(CfdiDownload::PATH_FILES, $employee_payroll->company->id);
        $tmp_path = str_replace(['files/', '/xml'], '', EmployeePayroll::PATH_XML_FILES);
        if (!\Storage::exists($path_files . '/' . $tmp_path)) {
            \Storage::makeDirectory($path_files . '/' . $tmp_path, 0777, true, true);
        }

        //Genera PDF
        $pdf = $this->print($employee_payroll, false, true);
        $path_xml = Helper::setDirectory(EmployeePayroll::PATH_XML_FILES, $employee_payroll->company_id) . '/';
        $file_xml = $tmp_path . '/' . $employee_payroll_cfdi->uuid . '.xml';
        $file_pdf = str_replace('.xml', '.pdf', $file_xml);
        \Storage::put($path_files . '/' . $file_pdf, $pdf);

        if(!empty($employee_payroll_cfdi->file_xml_pac)){
            \Storage::copy($path_xml . $employee_payroll_cfdi->file_xml_pac, $path_files . '/' . $file_xml);
        }

        //Guarda registro en CFDIS descargados
        $cfdi_download = CfdiDownload::create([
            'created_uid' => \Auth::user()->id,
            'updated_uid' => \Auth::user()->id,
            'type' => 2,
            'uuid' => $employee_payroll_cfdi->uuid,
            'file_xml' => $file_xml,
            'file_pdf' => $file_pdf,
            'file_acuse' => null,
            'status' => 'Vigente',
            'is_cancelable' => 'Cancelable sin aceptación',
            'date_cancel' => null,
            'company_id' => $employee_payroll->company_id,
        ]);
        $data_xml = Helper::parseXmlToArrayCfdi33($path_xml . '/' . $employee_payroll_cfdi->file_xml_pac);
        $data_xml['data']['employee_id'] = $employee_payroll->employee_id;
        $data_xml['data']['caccounting_type_id'] = $employee_payroll->customer->caccounting_type_sale_id ?? null;
        $cfdi_download->fill($data_xml['data']);
        $cfdi_download->date_payment = $employee_payroll->date_payment;
        $cfdi_download->save();
    }
}
