<?php

namespace App\Http\Controllers\Base;

use App\Helpers\Helper;
use App\Models\Base\CfdiDownload;
use App\Models\Base\Pac;
use App\Traits\ConsumesExternalServices;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Sales\CustomerInvoice;
use Illuminate\Support\Facades\Crypt;
use SoapClient;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Process\Process;

class TaxMailboxController extends Controller
{
    use ConsumesExternalServices;

    protected $baseUri;
    protected $token;
    protected $pac;
    protected $company;

    public function __construct()
    {
    }

    /**
     * Resuelve autorizacion
     *
     * @param $queryParams
     * @param $formParams
     * @param $headers
     */
    public function resolveAuthorization(&$queryParams, &$formParams, &$headers)
    {
        if(preg_match('/swsapien/i', $this->pac->code)) {
            $headers['Authorization'] = $this->resolveAccessToken();
        }
    }

    /**
     * Resuelve token
     *
     * @return string
     */
    public function resolveAccessToken()
    {
        if(preg_match('/swsapien/i', $this->pac->code)) {
            return 'Bearer ' . $this->token;
        }
    }

    /**
     * Decodifica la respuesta
     *
     * @param $response
     * @return mixed
     */
    public function decodeResponse($response)
    {
        return json_decode($response);
    }

    /**
     * Show the application dashboard.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $pac = Pac::findOrFail(setting('default_pac_id')); //PAC
        //
        $this->pac = $pac;
        if(preg_match('/swsapien/i', $this->pac->code)) {
            $this->baseUri = $pac->ws_url_cancel;
        }else{
            $this->baseUri = $pac->ws_url_cancel;
        }
        $this->token = $pac->token;

        //
        $this->company = $company = Helper::defaultCompany();

        //Consulta
        $results = [];

        //Logica de datos
        try {
            //Obtiene solicitudes pendientes de cancelacion SW sapien (receptor)
            if(preg_match('/swsapien/i', $this->pac->code)) {
                //Cambia al RFC de pruebas
                if ($this->pac->test) {
                    $this->company->taxid = 'LAN7008173R5';
                }

                //Consume servicio
                $result = $this->makeRequest(
                    'GET',
                    '/pendings/' . $this->company->taxid,
                    [],
                    [
                    ],
                    [],
                    $isJsonRequest = true
                );

                //En caso de error
                if (isset($result->status) && $result->status == 'error') {
                    flash(($result->message ?? '') . ' - ' . ($result->messageDetail ?? ''))->error();
                    return redirect()->route('home');
                }

                //Leer los uuid que trae
                if (!empty($result->data->uuid)) {
                    foreach ($result->data->uuid as $uuid) {
                        //
                        $results [] = $this->createDataCfdi($uuid);
                    }
                }
            }

            //Obtiene solicitudes pendientes de cancelacion Finkok (receptor)
            if(preg_match('/finkok/i', $this->pac->code)) {
                $client = new SoapClient($this->pac->ws_url_cancel);
                $params = [
                    'username' => $this->pac->username,
                    'password' => Crypt::decryptString($this->pac->password),
                    'rtaxpayer_id' => $this->company->taxid,
                ];
                $response = $client->__soapCall('get_pending', ['parameters' => $params]);
                if(isset($response->get_pendingResult->uuids)){
                    foreach ($response->get_pendingResult->uuids as $uuid) {
                        //
                        if(is_array($uuid)){
                            foreach ($uuid as $tmp) {
                                $results [] = $this->createDataCfdi($tmp);
                            }
                        }else{
                            $results [] = $this->createDataCfdi($uuid);
                        }
                    }
                }
                if(isset($response->get_pendingResult->error)){
                    throw new \Exception($response->get_pendingResult->error);
                }
            }

            //Obtiene solicitudes pendientes de cancelacion Timbox (receptor)
            if(preg_match('/timbox/i', $this->pac->code)) {
                $client = new SoapClient($this->pac->ws_url_cancel);
                $params = [
                    'username' => $this->pac->username,
                    'password' => Crypt::decryptString($this->pac->password),
                    'rfc_receptor' => $this->company->taxid,
                    'cert_pem' => \Storage::get($this->company->pathFileCerPem()),
                    'llave_pem' => \Storage::get($this->company->pathFileKeyPem()),
                ];
                $response = $client->__soapCall('consultar_peticiones_pendientes', $params);
                if($response->codestatus == '1100' && isset($response->uuids)){
                    foreach ($response->uuids as $uuid) {
                        //
                        $results [] = $this->createDataCfdi($uuid);
                    }
                }
            }

            //Obtiene solicitudes pendientes de cancelacion Sifei (receptor)
            if(preg_match('/sifei/i', $this->pac->code)) {
                $client = new SoapClient($this->pac->ws_url_cancel);
                $params = [
                    'usuarioSIFEI' => $this->pac->username,
                    'passwordSIFEI' => Crypt::decryptString($this->pac->password),
                    'rfcReceptor' => $this->company->taxid,
                ];
                $response = $client->__soapCall('peticionesPendientes', array('parameters' => $params));
                $xml_response = new \DOMDocument('1.0', 'UTF-8');
                libxml_use_internal_errors(true);
                $xml_response->loadXML($response->return);
                if($xml_response->getElementsByTagName('ObtenerPeticionesPendientesResult')->item(0)->getAttribute('CodEstatus')) {
                    $codigoEstatus = $xml_response->getElementsByTagName('ObtenerPeticionesPendientesResult')->item(0)->getAttribute('CodEstatus');
                    if ($codigoEstatus == '1100') {
                        foreach($xml_response->getElementsByTagName('UUID') as $item){
                            //
                            $results [] = $this->createDataCfdi($item->nodeValue);
                        }
                    }
                }
            }
            //Obtiene solicitudes pendientes de cancelacion timbrador Xpress (receptor)
            if(preg_match('/timbradorXpress/i', $this->pac->code)) {
                //Crear archivos cer.pem para timbrador express sin pubkey
                $path_file_cer = $company->pathFileCer();
                $path_file_cer_pem = $path_file_cer . '.pem';
                $path_file_cer_timexp_pass_pem = $path_file_cer . '.timexp.pem';
                if (\Storage::exists($path_file_cer) && \Storage::exists($path_file_cer_pem) && !\Storage::exists($path_file_cer_timexp_pass_pem)) {
                    $process = new Process('openssl x509 -inform DER -in "' . \Storage::path($path_file_cer) . '" -outform PEM -out "' . \Storage::path($path_file_cer_timexp_pass_pem) . '"');
                    $process->run();
                    if (!$process->isSuccessful()) {
                        throw new ProcessFailedException($process);
                    }
                }

                $client = new SoapClient($this->pac->ws_url);
                $params = [];
                $response = $client->consultarAutorizacionesPendientes(
                    $this->pac->token,
                    \Storage::get($this->company->pathFileKeyPem()),
                    \Storage::get($path_file_cer_timexp_pass_pem)
                );
                if ($response->code != 200) {
                    throw new \Exception($response->message);
                }else{
                    $tmps = explode(',',$response->data);
                    if(!empty($tmps)){
                        foreach($tmps as $uuid){
                            $results [] = $this->createDataCfdi($uuid);
                        }
                    }
                }
            }
        } catch (\SoapFault $e) {
            flash($e->getMessage())->error();
            return redirect()->route('home');
        } catch (\Exception $e) {
            flash($e->getMessage())->error();
            return redirect()->route('home');
        }

        //
        $results = collect($results);

        //Vista
        return view('base.tax_mailboxs.index', compact('results'));
    }

    public function createDataCfdi($uuid){
        $this->company = $company = Helper::defaultCompany();
        //Consulta cfdi por uuid en los cfdis descargados
        $cfdi_download = CfdiDownload::where('uuid','=',$uuid)->where('company_id','=',$this->company->id)->first();
        if(empty($cfdi_download)){
            $cfdi_download = CustomerInvoice::where('company_id','=',$this->company->id)
            ->whereHas('customerInvoiceCfdi', function ($q) use($uuid) {
                $q->where('customer_invoice_cfdis.uuid', '=', $uuid);
            })->first();
        }
        //
        return (object)[
            'date' => $cfdi_download->date ?? '',
            'date_request' => '',
            'cfdi_type' => $cfdi_download->cfdiType->name ?? '',
            'folio' => $cfdi_download->name ?? '',
            'uuid' => $uuid,
            'emisor' => ($cfdi_download->emisor ?? $cfdi_download->customer->name ?? '' ),
            'rfc_emisor' => ($cfdi_download->rfc_emisor ?? $cfdi_download->customer->taxid ?? '' ),
            'amount_total' => $cfdi_download->amount_total ?? 0
        ];
    }

    public function acceptReject(Request $request)
    {
        //
        $pac = Pac::findOrFail(setting('default_pac_id')); //PAC
        //
        $this->pac = $pac;
        if(preg_match('/swsapien/i', $this->pac->code)) {
            $this->baseUri = $pac->ws_url_cancel;
        }else{
            $this->baseUri = $pac->ws_url_cancel;
        }
        $this->token = $pac->token;
        //
        $cfdi_download = $this->createDataCfdi($request->uuid);

        //
        $this->company = $company = Helper::defaultCompany();

        try {
            //Acepta/Rechaza solicitud de cancelacion SW sapien (receptor)
            if(preg_match('/swsapien/i', $this->pac->code)) {
                //Cambia al RFC de pruebas
                if ($this->pac->test) {
                    $this->company->taxid = 'LAN7008173R5';
                }

                //Consume servicio para aceptar cancelacion
                $result = $this->makeRequest(
                    'POST',
                    '/acceptreject/csd',
                    [],
                    [
                        'uuids' => [
                            [
                                'uuid' => $request->uuid,
                                'action' => $request->type == 'A' ? 'Aceptacion' : 'Rechazo'
                            ]
                        ],
                        'rfc' => $this->company->taxid,
                        'password' => Crypt::decryptString($this->company->password_key),
                        'b64Cer' => base64_encode(\Storage::get($this->company->pathFileCer())),
                        'b64Key' => base64_encode(\Storage::get($this->company->pathFileKey())),
                    ],
                    [],
                    $isJsonRequest = true
                );

                //En caso de error
                if (isset($result->status) && $result->status == 'error') {
                    flash(($result->message ?? '') . ' - ' . ($result->messageDetail ?? ''))->error();
                    return redirect()->route('tax-mailbox');
                }
            }

            //Acepta/Rechaza solicitud de cancelacion Finkok (receptor)
            if(preg_match('/finkok/i', $this->pac->code)) {
                //Crear archivos key.finkok.pass.pem con la clave del usuario como lo solicita finkok
                //Convertir en PEM con contraseña
                $path_file_key = $this->company->pathFileKey();
                $path_file_key_pem = $path_file_key . '.pem';
                $path_file_key_finkok_pass_pem = $path_file_key . '.finkok.pass.pem';
                if (\Storage::exists($path_file_key) && \Storage::exists($path_file_key_pem) && !\Storage::exists($path_file_key_finkok_pass_pem)) {
                    $pass = Crypt::decryptString($pac->password);
                    if(PHP_OS_FAMILY !== "Windows"){
                        $pass = str_replace('$','\$', $pass);
                    }
                    $process = new Process('openssl rsa -in "' . \Storage::path($path_file_key_pem) . '" -des3 -out "' . \Storage::path($path_file_key_finkok_pass_pem) . '" -passout pass:"' . $pass.'"');
                    $process->run();
                    if (!$process->isSuccessful()) {
                        throw new ProcessFailedException($process);
                    }
                }
                //
                $uuids = ['uuid' => $request->uuid, 'respuesta' => ($request->type == 'A' ? 'Aceptacion' : 'Rechazo')];
                $uuid_ar = ['UUID_AR' => $uuids];
                $uuids_ar = ['uuids_ar' => $uuid_ar];

                //
                $client = new SoapClient($this->pac->ws_url_cancel);
                $params = [
                    'UUIDS_AR' => $uuids_ar,
                    'username' => $this->pac->username,
                    'password' => Crypt::decryptString($this->pac->password),
                    'rtaxpayer_id' => $this->company->taxid,
                    'cer' => base64_encode(\Storage::get($this->company->pathFileCerPem())),
                    'key' => \Storage::exists($path_file_key_finkok_pass_pem) ? base64_encode(\Storage::get($path_file_key_finkok_pass_pem)) : null,
                ];

                $servicio_out_finkok = false;
                if($servicio_out_finkok){
                    $response = $client->__soapCall('out_accept_reject', ['parameters' => $params]);
                    if($request->type == 'A') {
                        if (isset($response->out_accept_rejectResult->aceptacion->Acepta->status) && $response->out_accept_rejectResult->aceptacion->Acepta->status == '1000') {

                        }else{
                            $error = __('base/tax_mailbox.error_accept_rejected');
                            if(isset($response->out_accept_rejectResult->error))
                                $error = $response->out_accept_rejectResult->error;
                            flash($error)->error();
                            return redirect()->route('tax-mailbox');
                        }
                    }else {
                        if (isset($response->out_accept_rejectResult->rechazo->Rechaza->status) && $response->out_accept_rejectResult->rechazo->Rechaza->status == '1000') {

                        }else{
                            $error = __('base/tax_mailbox.error_accept_rejected');
                            if(isset($response->out_accept_rejectResult->error))
                                $error = $response->out_accept_rejectResult->error;
                            flash($error)->error();
                            return redirect()->route('tax-mailbox');
                        }
                    }
                }else{
                    $response = $client->__soapCall('accept_reject', ['parameters' => $params]);
                    if($request->type == 'A') {
                        if (isset($response->accept_rejectResult->aceptacion->Acepta->status) && $response->accept_rejectResult->aceptacion->Acepta->status == '1000') {

                        }else{
                            $error = __('base/tax_mailbox.error_accept_rejected');
                            if(isset($response->accept_rejectResult->error))
                                $error = $response->accept_rejectResult->error;
                            flash($error)->error();
                            return redirect()->route('tax-mailbox');
                        }
                    }else {
                        if (isset($response->accept_rejectResult->rechazo->Rechaza->status) && $response->accept_rejectResult->rechazo->Rechaza->status == '1000') {

                        }else{
                            $error = __('base/tax_mailbox.error_accept_rejected');
                            if(isset($response->accept_rejectResult->error))
                                $error = $response->accept_rejectResult->error;
                            flash($error)->error();
                            return redirect()->route('tax-mailbox');
                        }
                    }
                }


            }
            //Acepta/Rechaza solicitud de cancelacion Timbox (receptor)
            if(preg_match('/timbox/i', $this->pac->code)) {
                $respuestas = [
                    [
                        'uuid' => $request->uuid,
                        'rfc_emisor' => $cfdi_download->rfc_emisor ?? '',
                        'total' => $cfdi_download->amount_total ?? '',
                        'respuesta' => $request->type
                    ]
                    // ... n folios_respuestas a procesar
                ];

                $client = new SoapClient($this->pac->ws_url_cancel);
                $params = [
                    'username' => $this->pac->username,
                    'password' => Crypt::decryptString($this->pac->password),
                    'rfc_receptor' => $this->company->taxid,
                    'respuestas' => $respuestas,
                    'cert_pem' => \Storage::get($this->company->pathFileCerPem()),
                    'llave_pem' => \Storage::get($this->company->pathFileKeyPem()),
                ];
                $response = $client->__soapCall('procesar_respuesta', $params);
                if(isset($response->folios)){

                }
            }

            //Acepta/Rechaza solicitud de cancelacion sifei (receptor)
            if(preg_match('/sifei/i', $this->pac->code)) {
                $respuestas = [
                    [
                        'uuid' => $request->uuid,
                        'respuesta' => $request->type == 'A' ? 'Aceptacion' : 'Rechazo',
                    ]
                    // ... n folios_respuestas a procesar
                ];

                $client = new SoapClient($this->pac->ws_url_cancel);
                $params = [
                    'usuarioSIFEI' => $this->pac->username,
                    'passwordSIFEI' => Crypt::decryptString($this->pac->password),
                    'rfcReceptor' => $this->company->taxid,
                    'folios' => $respuestas,
                    'pfx' => \Storage::get($this->company->pathFilePfx()),
                    'passwordPfx' => Crypt::decryptString($this->company->password_key),
                ];
                $response = $client->__soapCall('procesarRespuesta', array('parameters' => $params));
                $xml_response = new \DOMDocument('1.0', 'UTF-8');
                libxml_use_internal_errors(true);
                $xml_response->loadXML($response->return);
                if(!empty($xml_response->getElementsByTagName('Folios')->item(0))){

                }
            }
            //Obtiene solicitudes pendientes de cancelacion timbrador Xpress (receptor)
            if(preg_match('/timbradorXpress/i', $this->pac->code)) {
                //Crear archivos cer.pem para timbrador express sin pubkey
                $path_file_cer = $company->pathFileCer();
                $path_file_cer_pem = $path_file_cer . '.pem';
                $path_file_cer_timexp_pass_pem = $path_file_cer . '.timexp.pem';
                if (\Storage::exists($path_file_cer) && \Storage::exists($path_file_cer_pem) && !\Storage::exists($path_file_cer_timexp_pass_pem)) {
                    $process = new Process('openssl x509 -inform DER -in "' . \Storage::path($path_file_cer) . '" -outform PEM -out "' . \Storage::path($path_file_cer_timexp_pass_pem) . '"');
                    $process->run();
                    if (!$process->isSuccessful()) {
                        throw new ProcessFailedException($process);
                    }
                }

                $client = new SoapClient($this->pac->ws_url);
                $params = [];
                $response = $client->autorizarCancelacion(
                    $this->pac->token,
                    \Storage::get($this->company->pathFileKeyPem()),
                    \Storage::get($path_file_cer_timexp_pass_pem),
                    $request->uuid,
                    $request->type == 'A' ? 'Aceptar' : 'Rechazar'
                );
                if ($response->code != 200) {
                    throw new \Exception($response->message);
                }else{

                }
            }

            //Mensaje
            if($request->type == 'A') {
                flash(__('base/tax_mailbox.text_success_accept'))->success();
            }else{
                flash(__('base/tax_mailbox.text_success_rejected'))->success();
            }

            //Redireccion
            return redirect()->route('tax-mailbox');

        } catch (\Exception $e) {
            flash($e->getMessage())->error();
            return redirect()->route('tax-mailbox');
        }
    }
}
