<?php

namespace App\Http\Controllers\Catalogs;

use App\Helpers\Helper;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\Models\Catalogs\AuthorizationAgent;

class AuthorizationAgentController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =AuthorizationAgent::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.authorization_agents.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.authorization_agents.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $request->merge(['code' => $request->name]);
        $this->validate($request, [
            'code' => 'required|unique:tenant.authorization_agents',
        ], [
            'code.required' => __('catalogs/authorization_agent.error_code'),
            'code.unique' => __('catalogs/authorization_agent.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo
        $request->merge(['date_start' => Helper::convertDateToSql($request->date_start)]);
        $request->merge(['date_end' => Helper::convertDateToSql($request->date_end)]);

        //Guardar
        //Registro principal
        $authorization_agent= AuthorizationAgent::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/authorization-agents');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\AuthorizationAgent  $authorization_agent
     * @return \Illuminate\Http\Response
     */
    public function show(AuthorizationAgent $authorization_agent)
    {
        return redirect('/catalogs/authorization-agents');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\AuthorizationAgent  $authorization_agent
     * @return \Illuminate\Http\Response
     */
    public function edit(AuthorizationAgent $authorization_agent)
    {
        return view('catalogs.authorization_agents.edit', compact('authorization_agent'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\AuthorizationAgent  $authorization_agent
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, AuthorizationAgent $authorization_agent)
    {
        //Validacion
        $this->validation($request);
        $request->merge(['code' => $request->name]);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.authorization_agents')->ignore($authorization_agent->id)
            ],
        ], [
            'code.required' => __('catalogs/authorization_agent.error_code'),
            'code.unique' => __('catalogs/authorization_agent.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $request->merge(['date_start' => Helper::convertDateToSql($request->date_start)]);
        $request->merge(['date_end' => Helper::convertDateToSql($request->date_end)]);
        $authorization_agent->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
            'date_start',
            'date_end'
        ]));

        //Guardar
        //Registro principal
        $authorization_agent->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/authorization-agents');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\AuthorizationAgent  $authorization_agent
     * @return \Illuminate\Http\Response
     */
    public function destroy(AuthorizationAgent $authorization_agent)
    {
        //Logica
        $authorization_agent->updated_uid = \Auth::user()->id;
        $authorization_agent->status = 0;
        $authorization_agent->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/authorization-agents');
    }

    public function validation(Request $request)
    {

    }
}
