<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\Bank;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class BankController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = Bank::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.banks.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.banks.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $request->merge(['taxid' => strtoupper(trim($request->taxid))]);
        $this->validation($request);
        $this->validate($request, [
            'code' => 'nullable|unique:tenant.banks|max:3',
        ], [
            'code.max' => __('catalogs/bank.error_code_max'),
            'code.unique' => __('catalogs/bank.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $bank = Bank::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/banks');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\Bank  $bank
     * @return \Illuminate\Http\Response
     */
    public function show(Bank $bank)
    {
        return redirect('/catalogs/banks');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\Bank  $bank
     * @return \Illuminate\Http\Response
     */
    public function edit(Bank $bank)
    {
        return view('catalogs.banks.edit', compact('bank'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\Bank  $bank
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Bank $bank)
    {
        //Validacion
        $request->merge(['taxid' => strtoupper(trim($request->taxid))]);

        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'nullable',
                'max:3',
                Rule::unique('tenant.banks')->ignore($bank->id)
            ],
        ], [
            'code.max' => __('catalogs/bank.error_code_max'),
            'code.unique' => __('catalogs/bank.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $bank->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'taxid',
            'description',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $bank->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/banks');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\Bank  $bank
     * @return \Illuminate\Http\Response
     */
    public function destroy(Bank $bank)
    {
        //Logica
        $bank->updated_uid = \Auth::user()->id;
        $bank->status = 0;
        $bank->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/banks');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $request->merge(['taxid' => strtoupper(trim($request->taxid))]);

        $this->validate($request, [
            'name' => 'required|string',
            'taxid' => ['required', 'regex:/^[A-Z&Ñ]{3,4}[0-9]{2}(0[1-9]|1[012])(0[1-9]|[12][0-9]|3[01])[A-Z0-9]{2}[0-9A]$/i'],
        ], [
            'name.*' => __('catalogs/bank.error_name'),
            'taxid.required' => __('catalogs/bank.error_taxid'),
            'taxid.regex' => __('catalogs/bank.error_taxid_format'),
        ]);
    }
}
