<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\CarType;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class CarTypeController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =CarType::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.car_types.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.car_types.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.car_types',
        ], [
            'code.required' => __('catalogs/car_type.error_code'),
            'code.unique' => __('catalogs/car_type.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $car_type= CarType::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/car-types');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\CarType  $car_type
     * @return \Illuminate\Http\Response
     */
    public function show(CarType $car_type)
    {
        return redirect('/catalogs/car-types');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\CarType  $car_type
     * @return \Illuminate\Http\Response
     */
    public function edit(CarType $car_type)
    {
        return view('catalogs.car_types.edit', compact('car_type'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\CarType  $car_type
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, CarType $car_type)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.car_types')->ignore($car_type->id)
            ],
        ], [
            'code.required' => __('catalogs/car_type.error_code'),
            'code.unique' => __('catalogs/car_type.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $car_type->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $car_type->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/car-types');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\CarType  $car_type
     * @return \Illuminate\Http\Response
     */
    public function destroy(CarType $car_type)
    {
        //Logica
        $car_type->updated_uid = \Auth::user()->id;
        $car_type->status = 0;
        $car_type->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/car-types');
    }

    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/car_type.error_name'),
        ]);
    }
}
