<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\CfdiRelation;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class CfdiRelationController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =CfdiRelation::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.cfdi_relations.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.cfdi_relations.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.cfdi_relations|max:2',
        ], [
            'code.required' => __('catalogs/cfdi_relation.error_code'),
            'code.max' => __('catalogs/cfdi_relation.error_code_max'),
            'code.unique' => __('catalogs/cfdi_relation.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $cfdi_relation= CfdiRelation::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/cfdi-relations');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\CfdiRelation  $cfdi_relation
     * @return \Illuminate\Http\Response
     */
    public function show(CfdiRelation $cfdi_relation)
    {
        return redirect('/catalogs/cfdi-relations');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\CfdiRelation  $cfdi_relation
     * @return \Illuminate\Http\Response
     */
    public function edit(CfdiRelation $cfdi_relation)
    {
        return view('catalogs.cfdi_relations.edit', compact('cfdi_relation'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\CfdiRelation  $cfdi_relation
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, CfdiRelation $cfdi_relation)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                'max:2',
                Rule::unique('tenant.cfdi_relations')->ignore($cfdi_relation->id)
            ],
        ], [
            'code.required' => __('catalogs/cfdi_relation.error_code'),
            'code.max' => __('catalogs/cfdi_relation.error_code_max'),
            'code.unique' => __('catalogs/cfdi_relation.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $cfdi_relation->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $cfdi_relation->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/cfdi-relations');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\CfdiRelation  $cfdi_relation
     * @return \Illuminate\Http\Response
     */
    public function destroy(CfdiRelation $cfdi_relation)
    {
        //Logica
        $cfdi_relation->updated_uid = \Auth::user()->id;
        $cfdi_relation->status = 0;
        $cfdi_relation->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/cfdi-relations');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/cfdi_relation.error_name'),
        ]);
    }
}
