<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\CfdiType;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class CfdiTypeController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =CfdiType::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.cfdi_types.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
        return view('catalogs.cfdi_types.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.cfdi_types|max:2',
        ], [
            'code.required' => __('catalogs/cfdi_type.error_code'),
            'code.max' => __('catalogs/cfdi_type.error_code_max'),
            'code.unique' => __('catalogs/cfdi_type.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $cfdi_type= CfdiType::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/cfdi-types');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\CfdiType  $cfdi_type
     * @return \Illuminate\Http\Response
     */
    public function show(CfdiType $cfdi_type)
    {
        return redirect('/catalogs/cfdi-types');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\CfdiType  $cfdi_type
     * @return \Illuminate\Http\Response
     */
    public function edit(CfdiType $cfdi_type)
    {
        return view('catalogs.cfdi_types.edit', compact('cfdi_type'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\CfdiType  $cfdi_type
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, CfdiType $cfdi_type)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                'max:2',
                Rule::unique('tenant.cfdi_types')->ignore($cfdi_type->id)
            ],
        ], [
            'code.required' => __('catalogs/cfdi_type.error_code'),
            'code.max' => __('catalogs/cfdi_type.error_code_max'),
            'code.unique' => __('catalogs/cfdi_type.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $cfdi_type->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $cfdi_type->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/cfdi-types');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\CfdiType  $cfdi_type
     * @return \Illuminate\Http\Response
     */
    public function destroy(CfdiType $cfdi_type)
    {
        //Logica
        $cfdi_type->updated_uid = \Auth::user()->id;
        $cfdi_type->status = 0;
        $cfdi_type->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/cfdi-types');
    }
    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/cfdi_type.error_name'),
        ]);
    }
}
