<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\CfdiUse;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class CfdiUseController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =CfdiUse::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.cfdi_uses.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.cfdi_uses.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.cfdi_uses|max:3',
        ], [
            'code.required' => __('catalogs/cfdi_use.error_code'),
            'code.max' => __('catalogs/cfdi_use.error_code_max'),
            'code.unique' => __('catalogs/cfdi_use.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $cfdi_use= CfdiUse::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/cfdi-uses');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\CfdiUse  $cfdi_use
     * @return \Illuminate\Http\Response
     */
    public function show(CfdiUse $cfdi_use)
    {
        return redirect('/catalogs/cfdi-uses');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\CfdiUse  $cfdi_use
     * @return \Illuminate\Http\Response
     */
    public function edit(CfdiUse $cfdi_use)
    {
        return view('catalogs.cfdi_uses.edit', compact('cfdi_use'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\CfdiUse  $cfdi_use
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, CfdiUse $cfdi_use)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                'max:3',
                Rule::unique('tenant.cfdi_uses')->ignore($cfdi_use->id)
            ],
        ], [
            'code.required' => __('catalogs/cfdi_use.error_code'),
            'code.max' => __('catalogs/cfdi_use.error_code_max'),
            'code.unique' => __('catalogs/cfdi_use.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $cfdi_use->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $cfdi_use->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/cfdi-uses');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\CfdiUse  $cfdi_use
     * @return \Illuminate\Http\Response
     */
    public function destroy(CfdiUse $cfdi_use)
    {
        //Logica
        $cfdi_use->updated_uid = \Auth::user()->id;
        $cfdi_use->status = 0;
        $cfdi_use->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/cfdi-uses');
    }

    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/cfdi_use.error_name'),
        ]);
    }
}
