<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\City;
use App\Models\Catalogs\Country;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;

class CityController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = City::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.cities.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $countries = Country::populateSelect()->get()->pluck('name_sat', 'id');
        return view('catalogs.cities.create',compact('countries'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $city = City::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/cities');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\City  $city
     * @return \Illuminate\Http\Response
     */
    public function show(City $city)
    {
        return redirect('/catalogs/cities');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\City  $city
     * @return \Illuminate\Http\Response
     */
    public function edit(City $city)
    {
        $countries = Country::populateSelect()->get()->pluck('name_sat', 'id');
        $city->country_id = $city->state->country->id ?? null;
        return view('catalogs.cities.edit', compact('city','countries'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\City  $city
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, City $city)
    {
        //Validacion
        $this->validation($request);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $city->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'state_id',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $city->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/cities');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\City  $city
     * @return \Illuminate\Http\Response
     */
    public function destroy(City $city)
    {
        //Logica
        $city->updated_uid = \Auth::user()->id;
        $city->status = 0;
        $city->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/cities');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
            'country_id' => 'required|integer',
            'state_id' => 'required|integer',
        ], [
            'name.*' => __('catalogs/state.error_name'),
            'country_id.*' => __('catalogs/state.error_country_id'),
            'state_id.*' => __('catalogs/state.error_state_id'),
        ]);
    }
}
