<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\ContainerMaritime;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class ContainerMaritimeController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =ContainerMaritime::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.container_maritimes.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.container_maritimes.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.container_maritimes',
        ], [
            'code.required' => __('catalogs/container_maritime.error_code'),
            'code.unique' => __('catalogs/container_maritime.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $container_maritime= ContainerMaritime::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/container-maritimes');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\ContainerMaritime  $container_maritime
     * @return \Illuminate\Http\Response
     */
    public function show(ContainerMaritime $container_maritime)
    {
        return redirect('/catalogs/container-maritimes');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\ContainerMaritime  $container_maritime
     * @return \Illuminate\Http\Response
     */
    public function edit(ContainerMaritime $container_maritime)
    {
        return view('catalogs.container_maritimes.edit', compact('container_maritime'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\ContainerMaritime  $container_maritime
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, ContainerMaritime $container_maritime)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.container_maritimes')->ignore($container_maritime->id)
            ],
        ], [
            'code.required' => __('catalogs/container_maritime.error_code'),
            'code.unique' => __('catalogs/container_maritime.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $container_maritime->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $container_maritime->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/container-maritimes');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\ContainerMaritime  $container_maritime
     * @return \Illuminate\Http\Response
     */
    public function destroy(ContainerMaritime $container_maritime)
    {
        //Logica
        $container_maritime->updated_uid = \Auth::user()->id;
        $container_maritime->status = 0;
        $container_maritime->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/container-maritimes');
    }

    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/container_maritime.error_name'),
        ]);
    }
}
