<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\ContainerType;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class ContainerTypeController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =ContainerType::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.container_types.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.container_types.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.container_types',
        ], [
            'code.required' => __('catalogs/container_type.error_code'),
            'code.unique' => __('catalogs/container_type.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $container_type= ContainerType::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/container-types');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\ContainerType  $container_type
     * @return \Illuminate\Http\Response
     */
    public function show(ContainerType $container_type)
    {
        return redirect('/catalogs/container-types');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\ContainerType  $container_type
     * @return \Illuminate\Http\Response
     */
    public function edit(ContainerType $container_type)
    {
        return view('catalogs.container_types.edit', compact('container_type'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\ContainerType  $container_type
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, ContainerType $container_type)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.container_types')->ignore($container_type->id)
            ],
        ], [
            'code.required' => __('catalogs/container_type.error_code'),
            'code.unique' => __('catalogs/container_type.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $container_type->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $container_type->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/container-types');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\ContainerType  $container_type
     * @return \Illuminate\Http\Response
     */
    public function destroy(ContainerType $container_type)
    {
        //Logica
        $container_type->updated_uid = \Auth::user()->id;
        $container_type->status = 0;
        $container_type->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/container-types');
    }

    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/container_type.error_name'),
        ]);
    }
}
