<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\ContractType;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class ContractTypeController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = ContractType::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.contract_types.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.contract_types.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.contract_types',
        ], [
            'code.required' => __('catalogs/contract_type.error_code'),
            'code.unique' => __('catalogs/contract_type.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $contract_type = ContractType::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/contract-types');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\ContractType  $contract_type
     * @return \Illuminate\Http\Response
     */
    public function show(ContractType $contract_type)
    {
        return redirect('/catalogs/contract-types');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\ContractType  $contract_type
     * @return \Illuminate\Http\Response
     */
    public function edit(ContractType $contract_type)
    {
        return view('catalogs.contract_types.edit', compact('contract_type'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\ContractType  $contract_type
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, ContractType $contract_type)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.contract_types')->ignore($contract_type->id)
            ],
        ], [
            'code.required' => __('catalogs/contract_type.error_code'),
            'code.unique' => __('catalogs/contract_type.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $contract_type->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $contract_type->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/contract-types');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\ContractType  $contract_type
     * @return \Illuminate\Http\Response
     */
    public function destroy(ContractType $contract_type)
    {
        //Logica
        $contract_type->updated_uid = \Auth::user()->id;
        $contract_type->status = 0;
        $contract_type->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/contract-types');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/contract_type.error_name'),
        ]);
    }
}
