<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\Country;
use App\Models\Catalogs\State;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class CountryController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = Country::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.countries.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.countries.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.countries|max:3',
        ], [
            'code.required' => __('catalogs/country.error_code'),
            'code.max' => __('catalogs/country.error_code_max'),
            'code.unique' => __('catalogs/country.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $country = Country::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/countries');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\Country  $country
     * @return \Illuminate\Http\Response
     */
    public function show(Country $country)
    {
        return redirect('/catalogs/countries');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\Country  $country
     * @return \Illuminate\Http\Response
     */
    public function edit(Country $country)
    {
        return view('catalogs.countries.edit', compact('country'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\Country  $country
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Country $country)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                'max:3',
                Rule::unique('tenant.countries')->ignore($country->id)
            ],
        ], [
            'code.required' => __('catalogs/country.error_code'),
            'code.max' => __('catalogs/country.error_code_max'),
            'code.unique' => __('catalogs/country.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $country->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $country->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/countries');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\Country  $country
     * @return \Illuminate\Http\Response
     */
    public function destroy(Country $country)
    {
        //Logica
        $country->updated_uid = \Auth::user()->id;
        $country->status = 0;
        $country->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/countries');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/country.error_name'),
        ]);
    }

    /**
     * Lista de estados
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function getStates(Request $request)
    {
        // Variables
        $id = $request->id;
        $state_id = $request->state_id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            $states = State::where('country_id', $id)->populateSelect()->pluck('name', 'id');
            $html  = view('layouts.partials.commons.ajax_select')->with('results', $states)->with('selected', $state_id)->render();
            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')],422);
    }
}
