<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\Currency;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class CurrencyController extends Controller
{
    private $list_symbol_position = [];

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->list_symbol_position = ['L' => __('general.text_left'), 'R' => __('general.text_right')];
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = Currency::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.currencies.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $list_symbol_position = $this->list_symbol_position;
        return view('catalogs.currencies.create',compact('list_symbol_position'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.currencies|max:3',
        ], [
            'code.required' => __('catalogs/currency.error_code'),
            'code.max' => __('catalogs/currency.error_code_max'),
            'code.unique' => __('catalogs/currency.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $currency = Currency::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/currencies');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\Currency  $currency
     * @return \Illuminate\Http\Response
     */
    public function show(Currency $currency)
    {
        return redirect('/catalogs/currencies');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\Currency  $currency
     * @return \Illuminate\Http\Response
     */
    public function edit(Currency $currency)
    {
        $list_symbol_position = $this->list_symbol_position;
        return view('catalogs.currencies.edit',compact('currency','list_symbol_position'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\Currency  $currency
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Currency $currency)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                'max:3',
                Rule::unique('tenant.currencies')->ignore($currency->id)
            ],
        ], [
            'code.required' => __('catalogs/currency.error_code'),
            'code.max' => __('catalogs/currency.error_code_max'),
            'code.unique' => __('catalogs/currency.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $currency->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'rate',
            'decimal_place',
            'symbol',
            'symbol_position',
            'decimal_mark',
            'thousands_separator',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $currency->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/currencies');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\Currency  $currency
     * @return \Illuminate\Http\Response
     */
    public function destroy(Currency $currency)
    {
        //Logica
        $currency->updated_uid = \Auth::user()->id;
        $currency->status = 0;
        $currency->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/currencies');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
            'rate' => 'required|numeric|min:0',
            'symbol' => 'required|string|max:3',
            'symbol_position' => 'required|string|max:1',
            'decimal_mark' => 'required|string|max:1',
            'thousands_separator' => 'required|string|max:1',
        ], [
            'name.*' => __('catalogs/currency.error_name'),
            'rate.*' => __('catalogs/currency.error_rate'),
            'symbol.*' => __('catalogs/currency.error_symbol'),
            'symbol_position.*' => __('catalogs/currency.error_symbol_position'),
            'decimal_mark.*' => __('catalogs/currency.error_decimal_mark'),
            'thousands_separator.*' => __('catalogs/currency.error_thousands_separator'),
        ]);
    }

    /**
     * Obtener registro
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCurrency(Request $request)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            $currency = Currency::findOrFail($id);
            $currency->rate = round($currency->rate,4);
            return response()->json($currency,200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

}
