<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\FrequencyPayment;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class FrequencyPaymentController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = FrequencyPayment::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.frequency_payments.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.frequency_payments.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.frequency_payments',
        ], [
            'code.required' => __('catalogs/frequency_payment.error_code'),
            'code.unique' => __('catalogs/frequency_payment.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $frequency_payment = FrequencyPayment::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/frequency-payments');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\FrequencyPayment  $frequency_payment
     * @return \Illuminate\Http\Response
     */
    public function show(FrequencyPayment $frequency_payment)
    {
        return redirect('/catalogs/frequency-payments');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\FrequencyPayment  $frequency_payment
     * @return \Illuminate\Http\Response
     */
    public function edit(FrequencyPayment $frequency_payment)
    {
        return view('catalogs.frequency_payments.edit', compact('frequency_payment'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\FrequencyPayment  $frequency_payment
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, FrequencyPayment $frequency_payment)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.frequency_payments')->ignore($frequency_payment->id)
            ],
        ], [
            'code.required' => __('catalogs/frequency_payment.error_code'),
            'code.unique' => __('catalogs/frequency_payment.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $frequency_payment->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $frequency_payment->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/frequency-payments');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\FrequencyPayment  $frequency_payment
     * @return \Illuminate\Http\Response
     */
    public function destroy(FrequencyPayment $frequency_payment)
    {
        //Logica
        $frequency_payment->updated_uid = \Auth::user()->id;
        $frequency_payment->status = 0;
        $frequency_payment->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/frequency-payments');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/frequency_payment.error_name'),
        ]);
    }
}
