<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\Incoterm;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class IncotermController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =Incoterm::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.incoterms.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.incoterms.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.incoterms',
        ], [
            'code.required' => __('catalogs/incoterm.error_code'),
            'code.unique' => __('catalogs/incoterm.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $incoterm= Incoterm::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/incoterms');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\Incoterm  $incoterm
     * @return \Illuminate\Http\Response
     */
    public function show(Incoterm $incoterm)
    {
        return redirect('/catalogs/incoterms');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\Incoterm  $incoterm
     * @return \Illuminate\Http\Response
     */
    public function edit(Incoterm $incoterm)
    {
        return view('catalogs.incoterms.edit', compact('incoterm'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\Incoterm  $incoterm
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Incoterm $incoterm)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.incoterms')->ignore($incoterm->id)
            ],
        ], [
            'code.required' => __('catalogs/incoterm.error_code'),
            'code.unique' => __('catalogs/incoterm.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $incoterm->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $incoterm->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/incoterms');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\Incoterm  $incoterm
     * @return \Illuminate\Http\Response
     */
    public function destroy(Incoterm $incoterm)
    {
        //Logica
        $incoterm->updated_uid = \Auth::user()->id;
        $incoterm->status = 0;
        $incoterm->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/incoterms');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/incoterm.error_name'),
        ]);
    }
}
