<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\JobRiskClasse;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class JobRiskClasseController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = JobRiskClasse::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.job_risk_classes.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.job_risk_classes.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.job_risk_classes',
        ], [
            'code.required' => __('catalogs/job_risk_classe.error_code'),
            'code.unique' => __('catalogs/job_risk_classe.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $job_risk_classe = JobRiskClasse::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/job-risk-classes');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\JobRiskClasse  $job_risk_classe
     * @return \Illuminate\Http\Response
     */
    public function show(JobRiskClasse $job_risk_classe)
    {
        return redirect('/catalogs/job-risk-classes');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\JobRiskClasse  $job_risk_classe
     * @return \Illuminate\Http\Response
     */
    public function edit(JobRiskClasse $job_risk_classe)
    {
        return view('catalogs.job_risk_classes.edit', compact('job_risk_classe'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\JobRiskClasse  $job_risk_classe
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, JobRiskClasse $job_risk_classe)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.job_risk_classes')->ignore($job_risk_classe->id)
            ],
        ], [
            'code.required' => __('catalogs/job_risk_classe.error_code'),
            'code.unique' => __('catalogs/job_risk_classe.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $job_risk_classe->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $job_risk_classe->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/job-risk-classes');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\JobRiskClasse  $job_risk_classe
     * @return \Illuminate\Http\Response
     */
    public function destroy(JobRiskClasse $job_risk_classe)
    {
        //Logica
        $job_risk_classe->updated_uid = \Auth::user()->id;
        $job_risk_classe->status = 0;
        $job_risk_classe->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/job-risk-classes');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/job_risk_classe.error_name'),
        ]);
    }
}
