<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\MeasurePackaging;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class MeasurePackagingController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =MeasurePackaging::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.measure_packagings.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.measure_packagings.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.measure_packagings',
        ], [
            'code.required' => __('catalogs/measure_packaging.error_code'),
            'code.unique' => __('catalogs/measure_packaging.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $measure_packaging= MeasurePackaging::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/measure-packagings');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\MeasurePackaging  $measure_packaging
     * @return \Illuminate\Http\Response
     */
    public function show(MeasurePackaging $measure_packaging)
    {
        return redirect('/catalogs/measure-packagings');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\MeasurePackaging  $measure_packaging
     * @return \Illuminate\Http\Response
     */
    public function edit(MeasurePackaging $measure_packaging)
    {
        return view('catalogs.measure_packagings.edit', compact('measure_packaging'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\MeasurePackaging  $measure_packaging
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, MeasurePackaging $measure_packaging)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.measure_packagings')->ignore($measure_packaging->id)
            ],
        ], [
            'code.required' => __('catalogs/measure_packaging.error_code'),
            'code.unique' => __('catalogs/measure_packaging.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $measure_packaging->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $measure_packaging->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/measure-packagings');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\MeasurePackaging  $measure_packaging
     * @return \Illuminate\Http\Response
     */
    public function destroy(MeasurePackaging $measure_packaging)
    {
        //Logica
        $measure_packaging->updated_uid = \Auth::user()->id;
        $measure_packaging->status = 0;
        $measure_packaging->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/measure-packagings');
    }

    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/measure_packaging.error_name'),
        ]);
    }

    /**
     * Autoacompletar select2 de clientes
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function autocomplete(Request $request)
    {
        //Variables
        $term = $request->term;

        //Logica
        if ($request->ajax() && !empty($term)) {
            $tmp = MeasurePackaging::filter(['filter_search_select2' => $term])->active()->sortable('name')->limit(16)->get();
            $results = [];
            if ($tmp->isNotEmpty()) {
                foreach ($tmp as $result) {
                    $results[] = [
                        'id' => $result->id,
                        'text' => $result->text_select2,
                        'description' => $result->description_select2
                    ];
                }
            }
            return response()->json($results, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }
}
