<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\OvertimesType;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class OvertimesTypeController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = OvertimesType::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.overtimes_types.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.overtimes_types.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.overtimes_types',
        ], [
            'code.required' => __('catalogs/overtimes_type.error_code'),
            'code.unique' => __('catalogs/overtimes_type.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $overtimes_type = OvertimesType::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/overtimes-types');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\OvertimesType  $overtimes_type
     * @return \Illuminate\Http\Response
     */
    public function show(OvertimesType $overtimes_type)
    {
        return redirect('/catalogs/overtimes-types');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\OvertimesType  $overtimes_type
     * @return \Illuminate\Http\Response
     */
    public function edit(OvertimesType $overtimes_type)
    {
        return view('catalogs.overtimes_types.edit', compact('overtimes_type'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\OvertimesType  $overtimes_type
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, OvertimesType $overtimes_type)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.overtimes_types')->ignore($overtimes_type->id)
            ],
        ], [
            'code.required' => __('catalogs/overtimes_type.error_code'),
            'code.unique' => __('catalogs/overtimes_type.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $overtimes_type->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $overtimes_type->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/overtimes-types');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\OvertimesType  $overtimes_type
     * @return \Illuminate\Http\Response
     */
    public function destroy(OvertimesType $overtimes_type)
    {
        //Logica
        $overtimes_type->updated_uid = \Auth::user()->id;
        $overtimes_type->status = 0;
        $overtimes_type->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/overtimes-types');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/overtimes_type.error_name'),
        ]);
    }
}
