<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\PaymentTerm;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class PaymentTermController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = PaymentTerm::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.payment_terms.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.payment_terms.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $payment_term = PaymentTerm::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/payment-terms');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\PaymentTerm  $payment_term
     * @return \Illuminate\Http\Response
     */
    public function show(PaymentTerm $payment_term)
    {
        return redirect('/catalogs/payment-terms');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\PaymentTerm  $payment_term
     * @return \Illuminate\Http\Response
     */
    public function edit(PaymentTerm $payment_term)
    {
        return view('catalogs.payment_terms.edit', compact('payment_term'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\PaymentTerm  $payment_term
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, PaymentTerm $payment_term)
    {
        //Validacion
        $this->validation($request);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['days' => (int)$request->days]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $payment_term->fill($request->only([
            'updated_uid',
            'name',
            'days',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $payment_term->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/payment-terms');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\PaymentTerm  $payment_term
     * @return \Illuminate\Http\Response
     */
    public function destroy(PaymentTerm $payment_term)
    {
        //Logica
        $payment_term->updated_uid = \Auth::user()->id;
        $payment_term->status = 0;
        $payment_term->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/payment-terms');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
            'days' => 'required|integer',
        ], [
            'name.*' => __('catalogs/payment_term.error_name'),
            'days.*' => __('catalogs/payment_term.error_days'),
        ]);
    }
}
