<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\PaymentWay;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class PaymentWayController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =PaymentWay::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.payment_ways.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.payment_ways.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.payment_ways|max:2',
        ], [
            'code.required' => __('catalogs/payment_way.error_code'),
            'code.max' => __('catalogs/payment_way.error_code_max'),
            'code.unique' => __('catalogs/payment_way.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $payment_way= PaymentWay::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/payment-ways');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\PaymentWay  $payment_way
     * @return \Illuminate\Http\Response
     */
    public function show(PaymentWay $payment_way)
    {
        return redirect('/catalogs/payment-ways');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\PaymentWay  $payment_way
     * @return \Illuminate\Http\Response
     */
    public function edit(PaymentWay $payment_way)
    {
        return view('catalogs.payment_ways.edit', compact('payment_way'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\PaymentWay  $payment_way
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, PaymentWay $payment_way)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                'max:2',
                Rule::unique('tenant.payment_ways')->ignore($payment_way->id)
            ],
        ], [
            'code.required' => __('catalogs/payment_way.error_code'),
            'code.max' => __('catalogs/payment_way.error_code_max'),
            'code.unique' => __('catalogs/payment_way.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $payment_way->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'patron_cuenta_ordenante',
            'patron_cuenta_beneficiaria',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $payment_way->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/payment-ways');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\PaymentWay  $payment_way
     * @return \Illuminate\Http\Response
     */
    public function destroy(PaymentWay $payment_way)
    {
        //Logica
        $payment_way->updated_uid = \Auth::user()->id;
        $payment_way->status = 0;
        $payment_way->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/payment-ways');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/payment_way.error_name'),
        ]);
    }
}
