<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\PayrollType;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class PayrollTypeController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = PayrollType::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.payroll_types.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.payroll_types.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.payroll_types',
        ], [
            'code.required' => __('catalogs/payroll_type.error_code'),
            'code.unique' => __('catalogs/payroll_type.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $payroll_type = PayrollType::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/payroll-types');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\PayrollType  $payroll_type
     * @return \Illuminate\Http\Response
     */
    public function show(PayrollType $payroll_type)
    {
        return redirect('/catalogs/payroll-types');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\PayrollType  $payroll_type
     * @return \Illuminate\Http\Response
     */
    public function edit(PayrollType $payroll_type)
    {
        return view('catalogs.payroll_types.edit', compact('payroll_type'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\PayrollType  $payroll_type
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, PayrollType $payroll_type)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.payroll_types')->ignore($payroll_type->id)
            ],
        ], [
            'code.required' => __('catalogs/payroll_type.error_code'),
            'code.unique' => __('catalogs/payroll_type.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $payroll_type->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $payroll_type->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/payroll-types');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\PayrollType  $payroll_type
     * @return \Illuminate\Http\Response
     */
    public function destroy(PayrollType $payroll_type)
    {
        //Logica
        $payroll_type->updated_uid = \Auth::user()->id;
        $payroll_type->status = 0;
        $payroll_type->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/payroll-types');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/payroll_type.error_name'),
        ]);
    }
}
