<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\Pediment;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class PedimentController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =Pediment::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.pediments.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.pediments.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.pediments',
        ], [
            'code.required' => __('catalogs/pediment.error_code'),
            'code.unique' => __('catalogs/pediment.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $pediment= Pediment::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/pediments');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\Pediment  $pediment
     * @return \Illuminate\Http\Response
     */
    public function show(Pediment $pediment)
    {
        return redirect('/catalogs/pediments');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\Pediment  $pediment
     * @return \Illuminate\Http\Response
     */
    public function edit(Pediment $pediment)
    {
        return view('catalogs.pediments.edit', compact('pediment'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\Pediment  $pediment
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Pediment $pediment)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.pediments')->ignore($pediment->id)
            ],
        ], [
            'code.required' => __('catalogs/pediment.error_code'),
            'code.unique' => __('catalogs/pediment.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $pediment->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $pediment->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/pediments');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\Pediment  $pediment
     * @return \Illuminate\Http\Response
     */
    public function destroy(Pediment $pediment)
    {
        //Logica
        $pediment->updated_uid = \Auth::user()->id;
        $pediment->status = 0;
        $pediment->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/pediments');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/pediment.error_name'),
        ]);
    }
}
