<?php

namespace App\Http\Controllers\Catalogs;

use App\Exports\ProductsExport;
use App\Helpers\Helper;
use App\Models\Catalogs\Product;
use App\Models\Catalogs\SatProduct;
use App\Models\Catalogs\Tax;
use App\Models\Catalogs\UnitMeasure;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\ProductTemplateImportExport;
use App\Imports\ProductTemplateBeforeImportImport;
use App\Imports\ProductTemplateImportImport;

class ProductController extends Controller
{
    private $import_results = [];

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {

        $this->import_results [] = (object)[
            'code' => 'P0001',
            'name' => 'Caja de archivo muerto',
            'description' => 'Caja de archivo muerto',
            'model' => '',
            'manufacturer' => '',
            'unit_measure' => 'H87-Pieza',
            'sat_product' => '01010101',
            'price' => '525.00',
            'taxes' => 'IVA 16%',
        ];
        $this->import_results [] = (object)[
            'code' => 'P0002',
            'name' => 'Teclado',
            'description' => 'Teclado USB, alambrico, estandar, cable de 1,5m',
            'model' => 'K270',
            'manufacturer' => 'Logitech',
            'unit_measure' => 'H87-Pieza',
            'sat_product' => '43202205',
            'price' => '203.65',
            'taxes' => 'IVA 16%',
        ];
        $this->import_results [] = (object)[
            'code' => '',
            'name' => 'Servicio de administración',
            'description' => 'Servicio de administración',
            'model' => '',
            'manufacturer' => '',
            'unit_measure' => 'E48-Unidad de servicio ',
            'sat_product' => '84111500',
            'price' => '0',
            'taxes' => 'IVA 16%',
        ];
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);
        if (!isset($request->filter_status) && $request->filter_status != 'all') {
            $request->request->add([
                'filter_status' => '1'
            ]);
        }

        //Consulta
        $results = Product::filter($request->all());
        if(isset($request->filter_status) && $request->filter_status != 'all'){
            $results = $results->where('products.status','=',(int)$request->filter_status);
        }
        $results = $results->with('unitMeasure')
            ->with('satProduct')
            ->sortable('name')
            ->paginate($limit);

        //Vista
        return view('catalogs.products.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $taxes = Tax::populateSelect()->pluck('name', 'id');
        return view('catalogs.products.create', compact('taxes'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);

        \DB::connection('tenant')->beginTransaction();
        try {
            //Logica
            $company = Helper::firstCompany(); //Empresa
            if(!empty(setting('products_per_taxid'))){
                $company = Helper::defaultCompany(); //Empresa
            }
            $request->merge(['created_uid' => \Auth::user()->id]);
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['price_decimal_place' => (double)$request->price_decimal_place]);
            $request->merge(['price' => (double)$request->price]);
            $request->merge(['quota_ieps' => (double)$request->quota_ieps]);
            $request->merge(['status' => 1]); //Por default activo
            $request->merge(['company_id' => $company->id]);
            $request->merge(['tax_object' => empty($request->taxes) ? '01' : '02']);
            //Si suben una imagen
            if ($request->hasFile('file_image')) {
                $image = Helper::uploadFileImage('file_image', Product::PATH_IMAGES,$company->id);
                $request->merge(['image' => $image]);
            }

            //Guardar
            //Registro principal
            $product = Product::create($request->input());

            //Impuestos
            if (!empty($request->taxes)) {
                $product->taxes()->sync($request->taxes);
            } else {
                $product->taxes()->sync([]);
            }

            \DB::connection('tenant')->commit();

            //Mensaje
            flash(__('general.text_form_success_add'))->success();

            //Redireccion
            return redirect('/catalogs/products');
        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            flash($e->getMessage())->error();
            return back()->withInput();
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\Product $product
     * @return \Illuminate\Http\Response
     */
    public function show(Product $product)
    {
        return redirect('/catalogs/products');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\Product $product
     * @return \Illuminate\Http\Response
     */
    public function edit(Product $product)
    {
        $taxes = Tax::populateSelect()->pluck('name', 'id');
        return view('catalogs.products.edit', compact('product', 'taxes'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  \App\Models\Catalogs\Product $product
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Product $product)
    {
        //Validacion
        $this->validation($request);

        \DB::connection('tenant')->beginTransaction();
        try {
            //Logica
            $request->merge(['updated_uid' => \Auth::user()->id]);
            $request->merge(['price_decimal_place' => (double)$request->price_decimal_place]);
            $request->merge(['price' => (double)$request->price]);
            $request->merge(['quota_ieps' => (double)$request->quota_ieps]);
            $request->merge(['status' => !empty($request->status) ? 1 : 0]);
            $request->merge(['tax_object' => empty($request->taxes) ? '01' : '02']);
            $product->fill($request->only([
                'updated_uid',
                'name',
                'image',
                'code',
                'description',
                'model',
                'manufacturer',
                'price',
                'unit_measure_id',
                'sat_product_id',
                'comment',
                'sort_order',
                'status',
                'tariff_fraction_id',
                'uom_custom_id',
                'price_decimal_place',
                'quota_ieps',
                'tax_object'
            ]));

            //Si suben una imagen
            if ($request->hasFile('file_image')) {
                //Si ya tenia un archivo lo eliminamos
                if (!empty($product->image)) {
                    \Storage::delete(Helper::setDirectory(Product::PATH_IMAGES,$product->company_id) . '/' . $product->image);
                }
                $image = Helper::uploadFileImage('file_image', Product::PATH_IMAGES,$product->company_id);
                $product->image = $image;
            } else {
                $product->image = $request->image; //si no tiene archivo sobreescribimos elque tenia
            }

            //Guardar
            //Registro principal
            $product->save();

            //Impuestos
            if (!empty($request->taxes)) {
                $product->taxes()->sync($request->taxes);
            } else {
                $product->taxes()->sync([]);
            }

            \DB::connection('tenant')->commit();

            //Mensaje
            flash(__('general.text_form_success_edit'))->success();

            //Redireccion
            return redirect('/catalogs/products');
        } catch (\Exception $e) {
            \DB::connection('tenant')->rollback();
            flash($e->getMessage())->error();
            return back()->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\Product $product
     * @return \Illuminate\Http\Response
     */
    public function destroy(Product $product)
    {
        //Logica
        $product->updated_uid = \Auth::user()->id;
        $product->status = 0;
        $product->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/products');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'file_image' => 'nullable|mimes:jpeg,jpg,png|max:2048',
            'name' => 'required|string',
            'description' => 'required',
            'unit_measure_id' => 'required|integer',
            'sat_product_id' => 'required|integer',
        ], [
            'file_image.*' => __('catalogs/product.error_image'),
            'name.*' => __('catalogs/product.error_name'),
            'description.*' => __('catalogs/product.error_description'),
            'unit_measure_id.*' => __('catalogs/product.error_unit_measure_id'),
            'sat_product_id.*' => __('catalogs/product.error_sat_product_id'),
        ]);
    }

    /**
     * Obtener registro
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getProduct(Request $request)
    {
        //Variables
        $id = $request->id;

        //Logica
        if ($request->ajax() && !empty($id)) {
            $product = Product::findOrFail($id);
            $product->price = Helper::numberFormat($product->price, !empty($product->price_decimal_place) ? $product->price_decimal_place : \App\Helpers\Helper::companyProductPriceDecimalPlace(),false);
            $product->taxes_1 = $product->taxes()->pluck('id', 'id');
            $product->unit_measure_description_select2 = $product->unitMeasure->description_select2;
            $product->sat_product_description_select2 = $product->satProduct->description_select2;
            unset($product->unitMeasure);
            unset($product->satProduct);
            $product->tariff_fraction_description_select2 = $product->tariffFraction->description_select2 ?? '';
            $product->uom_custom_description_select2 = $product->uomCustom->description_select2 ?? '';
            unset($product->tariffFraction);
            unset($product->uomCustom);
            return response()->json($product, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Autoacompletar select2 de productos
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function autocomplete(Request $request)
    {
        //Variables
        $term = $request->term;

        //Logica
        if ($request->ajax() && !empty($term)) {
            $tmp = Product::filter(['filter_search_select2' => $term])->active()->sortable('name')->limit(16)->get();
            $results = [];
            if ($tmp->isNotEmpty()) {
                foreach ($tmp as $result) {
                    $results[] = [
                        'id' => $result->id,
                        'text' => $result->text_select2,
                        'description' => $result->description_select2
                    ];
                }
            }
            return response()->json($results, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para buscar products
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function modalSearch(Request $request)
    {
        //Variables
        $active_btn_select = (!empty($request->active_btn_select) ? $request->active_btn_select : '');
        $row_origin = (isset($request->row_origin) ? $request->row_origin : '');
        //En caso de estar vacio le asignamos un datos para mostrar los clientes con "a"
        $request->merge(['filter_search' => !empty($request->filter_search) ? $request->filter_search : '' ]);

        //Logica
        if ($request->ajax()) {
            $results = Product::filter($request->all())->active()->sortable('name')->limit(10)->get();

            //Busca cliente
            $html_products = view('layouts.partials.products.search',
                compact('results','active_btn_select','row_origin'))->render();

            //modal de buscar
            $html = view('layouts.partials.products.modal_search',
                compact('html_products','active_btn_select','row_origin'))->render();


            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Modal para buscar productos
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     * @throws \Throwable
     */
    public function search(Request $request)
    {
        //Variables
        $active_btn_select = (!empty($request->active_btn_select) ? $request->active_btn_select : '');
        $row_origin = (isset($request->row_origin) ? $request->row_origin : '');
        //En caso de estar vacio le asignamos un datos para mostrar los clientes con "a"
        $request->merge(['filter_search' => !empty($request->filter_search) ? $request->filter_search : '' ]);

        //Logica
        if ($request->ajax()) {

            $results = Product::filter($request->all())->active()->sortable('name')->limit(10)->get();

            //Busca cliente
            $html = view('layouts.partials.products.search',
                compact('results','active_btn_select','row_origin'))->render();

            return response()->json(['html' => $html]);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }

    /**
     * Exportar datos a excel
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function exportToExcel(Request $request)
    {
        while (ob_get_level()) ob_end_clean();
        ob_start();

        return Excel::download(new ProductsExport($request),
            __('catalogs/product.document_title') . '-' . config('app.name') . '.xlsx');
    }


    /**
     * Importar viviendas desde archivo
     *
     * @return \Illuminate\Http\Response
     */
    public function createImport()
    {
        //Datos
        $taxes = Tax::populateSelect()->pluck('name', 'id');
        $import_results = $this->import_results;

        //Vista
        return view('catalogs.products.import',compact('taxes','import_results'));
    }

    /**
     * Descargar plantilla
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function downloadTemplateImport(Request $request)
    {
        //Datos
        $import_results = $this->import_results;

        //Descargar archivo
        while (ob_get_level()) ob_end_clean();
        ob_start();

        return Excel::download(new ProductTemplateImportExport($request, $import_results),
            __('catalogs/product.text_template_import') . '-' . config('app.name') . '.xlsx',\Maatwebsite\Excel\Excel::XLSX);
    }

    /**
     * Obtener informacion antes de procesar el archivo
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function beforeImport(Request $request)
    {
        //Validaciones
        //Validacion de archivos por extension
        if ($request->hasFile('file_products_import')) {
            $request->merge(['file_products_import_ext' => request()->file('file_products_import')->getClientOriginalExtension()]);
        }
        $validator = \Validator::make($request->all(), [
            'file_products_import' => 'required|max:2048',
            'file_products_import_ext' => 'nullable|in:xls,xlsx'
        ], [
            'file_products_import.*' => __('catalogs/product.error_file_products_import'),
            'file_products_import_ext.*' => __('catalogs/product.error_file_products_import'),
        ]);
        //Validaciones manuales
        //$validator = \Validator::make([], []);

        //Errores
        if ($validator->fails()) {
            $errors = '';
            foreach ($validator->errors()->all() as $message) {
                $errors = $message;
            }
            return response()->json(['error' => $errors], 422);
        }

        try {
            //Lógica

            //Importar
            $import = new ProductTemplateBeforeImportImport;
            Excel::import($import, request()->file('file_products_import'));

            //Mensaje
            return response()->json([
                'success' => 'ok',
                'total_import_products' => sprintf(__('catalogs/product.help_import_sweet_alert_1'),Helper::numberFormat($import->total_products)),
            ]);
        }catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 422);
        }
    }

    /**
     * Importar viviendas desde archivo
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     * @throws \Illuminate\Validation\ValidationException
     */
    public function storeImport(Request $request)
    {
        //Validacion
        //Validacion de archivos por extension
        if ($request->hasFile('file_products_import')) {
            $request->merge(['file_products_import_ext' => request()->file('file_products_import')->getClientOriginalExtension()]);
        }
        $this->validate($request, [
            'file_products_import' => 'required||max:2048',
            'file_products_import_ext' => 'nullable|in:xls,xlsx'
        ], [
            'file_products_import.*' => __('catalogs/product.error_file_products_import'),
            'file_products_import_ext.*' => __('catalogs/product.error_file_products_import'),
        ]);

        try {
            //Lógica

            //Importar
            Excel::import(new ProductTemplateImportImport(), request()->file('file_products_import'));

            //Mensaje
            flash(__('general.text_form_success_import'))->success();

            //Redireccion
            return redirect()->route('products.index');

        } catch (\Illuminate\Validation\ValidationException $e ) {
            return back()->withErrors($e->errors());
        } catch (\Exception $e) {
            flash($e->getMessage())->error();
            return back()->withInput();
        }
    }
}
