<?php

namespace App\Http\Controllers\Catalogs;

use App\Helpers\Helper;
use App\Models\Catalogs\Project;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class ProjectController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =Project::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.projects.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
        return view('catalogs.projects.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);

        //Logica
        $company = Helper::firstCompany(); //Empresa
        if(!empty(setting('projects_per_taxid'))){
            $company = Helper::defaultCompany(); //Empresa
        }
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo
        $request->merge(['company_id' => $company->id]);

        //Guardar
        //Registro principal
        $project= Project::create($request->input());

        //Si suben una imagen
        if ($request->hasFile('file_logo_docs')) {
            $logo_docs = Helper::uploadFileImage('file_logo_docs', Project::PATH_LOGO_DOCS,$company->id);
            $project->logo_docs = $logo_docs;
            $project->save();
        }

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/projects');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\Project  $project
     * @return \Illuminate\Http\Response
     */
    public function show(Project $project)
    {
        return redirect('/catalogs/projects');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\Project  $project
     * @return \Illuminate\Http\Response
     */
    public function edit(Project $project)
    {
        return view('catalogs.projects.edit', compact('project'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\Project  $project
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Project $project)
    {
        //Validacion
        $this->validation($request);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $project->fill($request->only([
            'updated_uid',
            'name',
            'description',
            'logo_docs',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $project->save();

        //Si suben una imagen
        if ($request->hasFile('file_logo_docs')) {
            //Si ya tenia un archivo lo eliminamos
            if (!empty($project->logo_docs)) {
                \Storage::delete(Helper::setDirectory(Project::PATH_LOGO_DOCS,$project->company_id) . '/' . $project->logo_docs);
            }
            $logo_docs = Helper::uploadFileImage('file_logo_docs', Project::PATH_LOGO_DOCS,$project->company_id);
            $project->logo_docs = $logo_docs;
            $project->save();
        }

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/projects');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\Project  $project
     * @return \Illuminate\Http\Response
     */
    public function destroy(Project $project)
    {
        //Logica
        $project->updated_uid = \Auth::user()->id;
        $project->status = 0;
        $project->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/projects');
    }
    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
            'file_logo_docs' => 'nullable|mimes:jpeg,jpg,png|max:2048',
        ], [
            'name.*' => __('catalogs/project.error_name'),
            'file_logo_docs.*' => __('catalogs/project.error_logo_docs'),
        ]);
    }
}
