<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\RecruitmentRegime;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class RecruitmentRegimeController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = RecruitmentRegime::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.recruitment_regimes.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.recruitment_regimes.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.recruitment_regimes',
        ], [
            'code.required' => __('catalogs/recruitment_regime.error_code'),
            'code.unique' => __('catalogs/recruitment_regime.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $recruitment_regime = RecruitmentRegime::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/recruitment-regimes');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\RecruitmentRegime  $recruitment_regime
     * @return \Illuminate\Http\Response
     */
    public function show(RecruitmentRegime $recruitment_regime)
    {
        return redirect('/catalogs/recruitment-regimes');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\RecruitmentRegime  $recruitment_regime
     * @return \Illuminate\Http\Response
     */
    public function edit(RecruitmentRegime $recruitment_regime)
    {
        return view('catalogs.recruitment_regimes.edit', compact('recruitment_regime'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\RecruitmentRegime  $recruitment_regime
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, RecruitmentRegime $recruitment_regime)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.recruitment_regimes')->ignore($recruitment_regime->id)
            ],
        ], [
            'code.required' => __('catalogs/recruitment_regime.error_code'),
            'code.unique' => __('catalogs/recruitment_regime.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $recruitment_regime->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $recruitment_regime->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/recruitment-regimes');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\RecruitmentRegime  $recruitment_regime
     * @return \Illuminate\Http\Response
     */
    public function destroy(RecruitmentRegime $recruitment_regime)
    {
        //Logica
        $recruitment_regime->updated_uid = \Auth::user()->id;
        $recruitment_regime->status = 0;
        $recruitment_regime->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/recruitment-regimes');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/recruitment_regime.error_name'),
        ]);
    }
}
