<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\RightsWay;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class RightsWayController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =RightsWay::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.rights_ways.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.rights_ways.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.rights_ways',
        ], [
            'code.required' => __('catalogs/rights_way.error_code'),
            'code.unique' => __('catalogs/rights_way.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $rights_way= RightsWay::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/rights-ways');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\RightsWay  $rights_way
     * @return \Illuminate\Http\Response
     */
    public function show(RightsWay $rights_way)
    {
        return redirect('/catalogs/rights-ways');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\RightsWay  $rights_way
     * @return \Illuminate\Http\Response
     */
    public function edit(RightsWay $rights_way)
    {
        return view('catalogs.rights_ways.edit', compact('rights_way'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\RightsWay  $rights_way
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, RightsWay $rights_way)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.rights_ways')->ignore($rights_way->id)
            ],
        ], [
            'code.required' => __('catalogs/rights_way.error_code'),
            'code.unique' => __('catalogs/rights_way.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $rights_way->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
            'from',
            'to',
            'grant_receive',
            'concessionaire',
        ]));

        //Guardar
        //Registro principal
        $rights_way->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/rights-ways');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\RightsWay  $rights_way
     * @return \Illuminate\Http\Response
     */
    public function destroy(RightsWay $rights_way)
    {
        //Logica
        $rights_way->updated_uid = \Auth::user()->id;
        $rights_way->status = 0;
        $rights_way->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/rights-ways');
    }

    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/rights_way.error_name'),
        ]);
    }
}
