<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\SatProductStcc;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class SatProductStccController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =SatProductStcc::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.sat_product_stccs.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.sat_product_stccs.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.sat_product_stccs',
        ], [
            'code.required' => __('catalogs/sat_product_stcc.error_code'),
            'code.unique' => __('catalogs/sat_product_stcc.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $sat_product_stcc= SatProductStcc::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/sat-product-stccs');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\SatProductStcc  $sat_product_stcc
     * @return \Illuminate\Http\Response
     */
    public function show(SatProductStcc $sat_product_stcc)
    {
        return redirect('/catalogs/sat-product-stccs');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\SatProductStcc  $sat_product_stcc
     * @return \Illuminate\Http\Response
     */
    public function edit(SatProductStcc $sat_product_stcc)
    {
        return view('catalogs.sat_product_stccs.edit', compact('sat_product_stcc'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\SatProductStcc  $sat_product_stcc
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, SatProductStcc $sat_product_stcc)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.sat_product_stccs')->ignore($sat_product_stcc->id)
            ],
        ], [
            'code.required' => __('catalogs/sat_product_stcc.error_code'),
            'code.unique' => __('catalogs/sat_product_stcc.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $sat_product_stcc->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $sat_product_stcc->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/sat-product-stccs');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\SatProductStcc  $sat_product_stcc
     * @return \Illuminate\Http\Response
     */
    public function destroy(SatProductStcc $sat_product_stcc)
    {
        //Logica
        $sat_product_stcc->updated_uid = \Auth::user()->id;
        $sat_product_stcc->status = 0;
        $sat_product_stcc->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/sat-product-stccs');
    }

    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/sat_product_stcc.error_name'),
        ]);
    }

    /**
     * Autoacompletar select2 de clientes
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function autocomplete(Request $request)
    {
        //Variables
        $term = $request->term;

        //Logica
        if ($request->ajax() && !empty($term)) {
            $tmp = SatProductStcc::filter(['filter_search_select2' => $term])->active()->sortable('name')->limit(16)->get();
            $results = [];
            if ($tmp->isNotEmpty()) {
                foreach ($tmp as $result) {
                    $results[] = [
                        'id' => $result->id,
                        'text' => $result->text_select2,
                        'description' => $result->description_select2
                    ];
                }
            }
            return response()->json($results, 200);
        }

        return response()->json(['error' => __('general.error_general')], 422);
    }
}
