<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\ServiceType;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class ServiceTypeController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =ServiceType::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.service_types.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.service_types.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.service_types',
        ], [
            'code.required' => __('catalogs/service_type.error_code'),
            'code.unique' => __('catalogs/service_type.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $service_type= ServiceType::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/service-types');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\ServiceType  $service_type
     * @return \Illuminate\Http\Response
     */
    public function show(ServiceType $service_type)
    {
        return redirect('/catalogs/service-types');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\ServiceType  $service_type
     * @return \Illuminate\Http\Response
     */
    public function edit(ServiceType $service_type)
    {
        return view('catalogs.service_types.edit', compact('service_type'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\ServiceType  $service_type
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, ServiceType $service_type)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.service_types')->ignore($service_type->id)
            ],
        ], [
            'code.required' => __('catalogs/service_type.error_code'),
            'code.unique' => __('catalogs/service_type.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $service_type->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $service_type->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/service-types');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\ServiceType  $service_type
     * @return \Illuminate\Http\Response
     */
    public function destroy(ServiceType $service_type)
    {
        //Logica
        $service_type->updated_uid = \Auth::user()->id;
        $service_type->status = 0;
        $service_type->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/service-types');
    }

    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/service_type.error_name'),
        ]);
    }
}
