<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\SettingTransport;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class SettingTransportController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results =SettingTransport::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.setting_transports.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.setting_transports.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.setting_transports',
        ], [
            'code.required' => __('catalogs/setting_transport.error_code'),
            'code.unique' => __('catalogs/setting_transport.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $setting_transport= SettingTransport::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/setting-transports');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\SettingTransport  $setting_transport
     * @return \Illuminate\Http\Response
     */
    public function show(SettingTransport $setting_transport)
    {
        return redirect('/catalogs/setting-transports');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\SettingTransport  $setting_transport
     * @return \Illuminate\Http\Response
     */
    public function edit(SettingTransport $setting_transport)
    {
        return view('catalogs.setting_transports.edit', compact('setting_transport'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\SettingTransport  $setting_transport
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, SettingTransport $setting_transport)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.setting_transports')->ignore($setting_transport->id)
            ],
        ], [
            'code.required' => __('catalogs/setting_transport.error_code'),
            'code.unique' => __('catalogs/setting_transport.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $setting_transport->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
            'axles',
            'tires'
        ]));

        //Guardar
        //Registro principal
        $setting_transport->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/setting-transports');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\SettingTransport  $setting_transport
     * @return \Illuminate\Http\Response
     */
    public function destroy(SettingTransport $setting_transport)
    {
        //Logica
        $setting_transport->updated_uid = \Auth::user()->id;
        $setting_transport->status = 0;
        $setting_transport->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/setting-transports');
    }

    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/setting_transport.error_name'),
        ]);
    }
}
