<?php

namespace App\Http\Controllers\Catalogs;

use App\Models\Catalogs\SourceResource;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\Rule;

class SourceResourceController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //Variables
        $limit = ($request->has('limit') ? $request->get('limit') : 100);

        //Consulta
        $results = SourceResource::filter($request->all())->sortable('general')->paginate($limit);

        //Vista
        return view('catalogs.source_resources.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('catalogs.source_resources.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => 'required|unique:tenant.source_resources',
        ], [
            'code.required' => __('catalogs/source_resource.error_code'),
            'code.unique' => __('catalogs/source_resource.error_code_unique'),
        ]);

        //Logica
        $request->merge(['created_uid' => \Auth::user()->id]);
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => 1]); //Por default activo

        //Guardar
        //Registro principal
        $source_resource = SourceResource::create($request->input());

        //Mensaje
        flash(__('general.text_form_success_add'))->success();

        //Redireccion
        return redirect('/catalogs/source-resources');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Catalogs\SourceResource  $source_resource
     * @return \Illuminate\Http\Response
     */
    public function show(SourceResource $source_resource)
    {
        return redirect('/catalogs/source-resources');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Catalogs\SourceResource  $source_resource
     * @return \Illuminate\Http\Response
     */
    public function edit(SourceResource $source_resource)
    {
        return view('catalogs.source_resources.edit', compact('source_resource'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Catalogs\SourceResource  $source_resource
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, SourceResource $source_resource)
    {
        //Validacion
        $this->validation($request);
        $this->validate($request, [
            'code' => [
                'required',
                Rule::unique('tenant.source_resources')->ignore($source_resource->id)
            ],
        ], [
            'code.required' => __('catalogs/source_resource.error_code'),
            'code.unique' => __('catalogs/source_resource.error_code_unique'),
        ]);

        //Logica
        $request->merge(['updated_uid' => \Auth::user()->id]);
        $request->merge(['sort_order' => (int)$request->sort_order]);
        $request->merge(['status' => !empty($request->status) ? 1 : 0]);
        $source_resource->fill($request->only([
            'updated_uid',
            'name',
            'code',
            'sort_order',
            'status',
        ]));

        //Guardar
        //Registro principal
        $source_resource->save();

        //Mensaje
        flash(__('general.text_form_success_edit'))->success();

        //Redireccion
        return redirect('/catalogs/source-resources');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Catalogs\SourceResource  $source_resource
     * @return \Illuminate\Http\Response
     */
    public function destroy(SourceResource $source_resource)
    {
        //Logica
        $source_resource->updated_uid = \Auth::user()->id;
        $source_resource->status = 0;
        $source_resource->save();

        //Mensaje
        flash(__('general.text_form_success_delete'))->success();

        //Redireccion
        return redirect('/catalogs/source-resources');
    }

    /**
     * Validacion de formulario
     *
     * @param Request $request
     */
    public function validation(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|string',
        ], [
            'name.*' => __('catalogs/source_resource.error_name'),
        ]);
    }
}
